"""
#--------------------------------------------------------------------------#
# Copyright (c) 2025, Ciena Corporation                                    #
# All rights reserved.                                                     #
#                                                                          #
#     _______ _____ __    __ ___                                           #
#    / _ __(_) ___//  |  / // _ |                                          #
#   / /   / / /__ / /|| / // / ||                                          #
#  / /___/ / /__ / / ||/ // /__||                                          #
# /_____/_/_____/_/  |__//_/   ||                                          #
#                                                                          #
# Distributed as Ciena-Customer confidential.                              #
#                                                                          #
#--------------------------------------------------------------------------#
"""

import jsondiff
import re

from typing import Any, List, Optional

from django.contrib.auth.decorators import user_passes_test
from django.core.exceptions import PermissionDenied
from rest_framework.response import Response
from rest_framework.status import is_success, HTTP_204_NO_CONTENT, HTTP_400_BAD_REQUEST

from database_manager import database_manager


def get_nested_value(_dict, path: List[Any], default: Optional[Any] = None) -> Any:
    """ Read the value of an attribute in a nested dictionary """
    val = _dict
    for elem in path:
        try:
            val = val[elem]
        except (KeyError, IndexError):
            val = default
            break

    return val


def load_mongo_query_parameter(parameter) -> Optional[dict]:
    """ Create a json formatted query object from a URL query parameter

        NOTE: Does not allow for '=' or ',' to be used in field names or values.
        These characters are currently used as delimiters.

        :raises: ValueError if query or projection is improperly formatted
    """
    document = {}
    if parameter is not None:
        for item in str(parameter).strip().split(","):
            item = item.strip().split("=")
            if len(item) >= 2:
                key = item[0]
                raw_value = "=".join(item[1:])

                # Cast the value to its appropriate type for schema validation
                if raw_value.isdigit():
                    value = int(raw_value)
                elif raw_value.lower() == "true" or raw_value.lower() == "false":
                    value = raw_value.lower() == "true"
                else:
                    value = raw_value
                document[key] = value
            else:
                raise ValueError

    if len(document.keys()) == 0:
        document = None

    return document


class PonManagerApiResponse(Response):
    """ Extends the rest_framework.Response class to create Tibit formatted API responses """

    data_change = ""

    def __init__(self, status, status_text=None, details=None, data=None, new_data=None, old_data=None, data_change=None, *args, **kwargs):
        """ Creates a new custom response object to be returned to the user after middleware

        :param status: The HTTP response status code
        :param status_text: Custom status text string for the final response content (not used in GETs)
        :param details: Details text or dictionary when a warning or error occurs
        :param data: The response data to be passed to the client
        :param new_data: The new data from after the requested change was made (only for PUT/POST/PATCH)
        :param old_data: The old data from before the requested change was made (only for PUT/POST/PATCH)
        """
        super().__init__(status=status, data=data, *args, **kwargs)
        if not status_text:
            if is_success(status):
                status_text = "success"
            else:
                status_text = "fail"

        if status != 204:
            response_data = {"status": status_text}
            if details:
                if type(details) is str:
                    details = {"message": details}
                response_data['details'] = details
            if data is not None:
                response_data['data'] = data
            self.data = response_data

        self.status_code = status

        # Determine if a data change has occurred in the case of a PUT/POST/PATCH request and set the log message contents
        if data_change is None:
            if new_data is not None:
                if old_data is None:
                    old_data = new_data

                if self.status_code == 201:
                    difference = new_data
                    data_label = "created"
                else:
                    if old_data == {}:
                        difference = jsondiff.diff(old_data, new_data, syntax='explicit')
                    else:
                        difference = jsondiff.diff(old_data, new_data, syntax='symmetric')
                    if difference == {}:
                        difference = None
                    data_label = "updated"

                self.data_change = "{}: {}".format(data_label, difference)
        else:
            self.data_change = data_change


def validate_query_params(collection, schema=None):
    """ Decorator to parse and validate GET 'query' and 'projection' query parameters """

    def wrapper(func):
        def wrapped_f(self, request, *args, **kwargs):
            # Pass query dictionaries through JSON validation and return error if improper key value pairs
            try:
                query = load_mongo_query_parameter(request.GET.get('query'))
            except ValueError:
                return PonManagerApiResponse(status=HTTP_400_BAD_REQUEST, details="Query parameter must be a comma separated list of '<key>=<value>' strings")
            if query:
                valid, details = database_manager.validate_query(collection=collection, query_params=query, schema=schema)
                if not valid:
                    return PonManagerApiResponse(status=HTTP_400_BAD_REQUEST, details=details)

            # Pass projection dictionaries through JSON validation and return error if improper key value pairs
            try:
                projection = load_mongo_query_parameter(request.GET.get('projection'))
            except ValueError:
                return PonManagerApiResponse(status=HTTP_400_BAD_REQUEST, details="Projection parameter must be a comma separated list of '<key>=<value>' strings where value is '0' or '1'")
            if projection:
                valid, details = database_manager.validate_projection(collection=collection, query_params=projection, schema=schema)
                if not valid:
                    return PonManagerApiResponse(status=HTTP_400_BAD_REQUEST, details=details)

            # Extract sort parameter and validate
            sort = []
            try:
                sort_dict = load_mongo_query_parameter(request.GET.get('sort'))
                if sort_dict:
                    # Construct sort parameter into mongo format
                    for key, value in sort_dict.items():
                        value = int(value)
                        sort_dict[key] = value
                        sort.append((key, value))
                    # Validate sort parameter
                    valid, details = database_manager.validate_sort(collection=collection, query_params=sort_dict, schema=schema)
                    if not valid:
                        return PonManagerApiResponse(status=HTTP_400_BAD_REQUEST, details=details)
            except ValueError:
                return PonManagerApiResponse(status=HTTP_400_BAD_REQUEST, details="Sort parameter must be a comma separated list of '<key>=<value>' strings where value is '-1' or '1'")

            # Extract limit parameter and validate
            try:
                limit = request.GET.get('limit')
                # Ignore if limit parameter is not given
                if limit:
                    limit = int(limit)
                    if limit < 0:
                        raise ValueError
                else:
                    # Default limit of 1000 documents
                    limit = 1000
            except ValueError:
                return PonManagerApiResponse(status=HTTP_400_BAD_REQUEST, details="Limit parameter must be an integer greater than or equal to 0")

            # Extract skip parameter and validate
            try:
                skip = request.GET.get('skip')
                # Ignore if skip parameter is not given
                if skip:
                    skip = int(skip)
                    if skip < 0:
                        raise ValueError
                else:
                    skip = 0
            except ValueError:
                return PonManagerApiResponse(status=HTTP_400_BAD_REQUEST,
                                             details="Skip parameter must be an integer greater than or equal to 0")

            # Extract next parameter and validate it is a valid _id for the collection
            next = request.GET.get('next')
            if next:
                valid, details = database_manager.validate_query(collection=collection, query_params={"_id": next}, schema=schema)
                if not valid:
                    return PonManagerApiResponse(status=HTTP_400_BAD_REQUEST, details=details)

            # Extract distinct parameter and validate
            distinct = request.GET.get('distinct')
            if distinct:
                valid, details = database_manager.validate_path(collection, distinct)
                if not valid:
                    return PonManagerApiResponse(status=HTTP_400_BAD_REQUEST, details=details)

            return func(self, request, query, projection, sort, limit, skip, next, distinct, *args, **kwargs)

        return wrapped_f

    return wrapper


def validate_data(collection, resource_id_param, validate_required=True, strict_validation=False, schema=None):
    """ Decorator to parse and validate the request body's 'data' field from POST/PUT/PATCH """

    def wrapper(func):
        def wrapped_f(self, request, *args, **kwargs):
            # Payload has 'data' dictionary wrapper
            data = get_nested_value(request.data, ["data"])
            # Payload does not have 'data' dictionary wrapper
            if data is None:
                data = request.data


            if resource_id_param:
                data['_id'] = kwargs[resource_id_param]
            valid, details = database_manager.validate(
                collection=collection,
                document=data,
                validate_required=validate_required,
                strict_validation=strict_validation,
                schema=schema)
            if valid:
                response = func(self, request, data, *args, **kwargs)
                # Update the response if there was a JSON Schema validation warning
                if details and response.status_code != HTTP_204_NO_CONTENT:
                    response.data["status"] = "validation warning"
                    response.data["details"] = details
            else:
                response = PonManagerApiResponse(status=HTTP_400_BAD_REQUEST, details=details)
            return response

        return wrapped_f

    return wrapper


def permission_required_any_of(perms, raise_exception=False):
    """
    Mimics Django's django.contrib.auth.decorators.permission_required decorator
    to allow for an OR condition. The user will 'pass the test' if they have at
    least one of the permissions given in the perms list.
    """

    def check_perms(user):
        if isinstance(perms, str):
            permissions = (perms,)
        else:
            permissions = perms

        user_email = user.email
        user_permissions = database_manager.get_user_permissions(user_email)

        # First check if the user has any of the permissions (even anon users)
        for perm in permissions:
            if perm in user_permissions:
                return True

        # In case the 403 handler should be called raise the exception
        if raise_exception:
            raise PermissionDenied

        # As the last resort, show the login form
        return False

    return user_passes_test(check_perms, login_url=None)


def permission_required(permission, raise_exception=False):
    """
    Mimics Django's django.contrib.auth.decorators.permission_required decorator
    to allow for an OR condition. The user will 'pass the test' if they have at
    least one of the permissions given in the perms list.
    """

    def check_perm(user):
        if hasattr(user, 'email'):
            user_email = user.email
            permissions = database_manager.get_user_permissions(user_email)
            if permission in permissions:
                return True
            # In case the 403 handler should be called raise the exception
            if raise_exception:
                raise PermissionDenied
        else:
            if raise_exception:
                raise PermissionDenied
        # As the last resort, show the login form
        return False

    return user_passes_test(check_perm, login_url=None)


def validate_mac(mac):
    return re.match("^([0-9a-f]{2}(:[0-9a-f]{2}){5})$", mac)

def validate_onu_ssn(ssn):
    return re.match("^([A-Z]{4}[0-9a-f]{8})$", ssn)

# Validate PON Controller, OLT, and Switch Device IDs, which have one of
# the following formats: MAC Address or 'Default'
def validate_device_id(id):
    return validate_mac(id) or id == 'Default'

# Validate Device IDs, which have one of the following formats: GPON
# Serial Number, EPON MAC Address, or 'Default'
def validate_onu_id(id):
    return validate_onu_ssn(id) or validate_mac(id) or id == 'Default'

