"""
#--------------------------------------------------------------------------#
# Copyright (c) 2025, Ciena Corporation                                    #
# All rights reserved.                                                     #
#                                                                          #
#     _______ _____ __    __ ___                                           #
#    / _ __(_) ___//  |  / // _ |                                          #
#   / /   / / /__ / /|| / // / ||                                          #
#  / /___/ / /__ / / ||/ // /__||                                          #
# /_____/_/_____/_/  |__//_/   ||                                          #
#                                                                          #
# Distributed as Ciena-Customer confidential.                              #
#                                                                          #
#--------------------------------------------------------------------------#
"""

import pymongo.errors

from django.contrib.auth.mixins import LoginRequiredMixin
from django.utils.decorators import method_decorator
from rest_framework.exceptions import APIException
from rest_framework.fields import JSONField
from rest_framework.generics import GenericAPIView
from drf_spectacular.utils import extend_schema, OpenApiParameter, OpenApiResponse
from drf_spectacular.types import OpenApiTypes
from rest_framework import status

from database_manager import database_manager
from utils.schema_helpers import MongoProjectionParameter, MongoQueryParameter, OpenApiParameter, ResponseExample
from utils.tools import PonManagerApiResponse, get_nested_value, validate_data, validate_query_params, \
    permission_required_any_of, validate_onu_id, permission_required
from utils.serializers import schema, get_schema



# ==================================================
# ============== One ONU State View ================
# ==================================================
class OneState(LoginRequiredMixin, GenericAPIView):
    raise_exception = True
    queryset = ''

    swaggerSchema = get_schema('ONU-STATE')

    @extend_schema(
        operation_id="get_one_onu_state",
        responses={
            200: OpenApiResponse(response=schema(swaggerSchema),
                                 description='OK'),
        },
        tags=['onu state'],
        summary="Get the state for the specified ONU",
        description=" "
    )
    @method_decorator(permission_required('can_read_network_onus', raise_exception=True))
    def get(self, request, onu_id, version):
        """Get the state for the specified ONU"""
        # Validate ONU ID
        if not validate_onu_id(onu_id):
            response = PonManagerApiResponse(status=status.HTTP_400_BAD_REQUEST, details="Invalid ONU ID format")
        else:
            res_data = database_manager.find_one(database_id=request.session.get('database'), collection="ONU-STATE",
                                                 query={"_id": onu_id})
            if res_data:
                response = PonManagerApiResponse(status=status.HTTP_200_OK, data=res_data)
            else:
                response = PonManagerApiResponse(status=status.HTTP_404_NOT_FOUND, details={
                    "message": "ONU ID " + str(onu_id) + " has no state document"})

        return response

    @extend_schema(
        operation_id="delete_one_onu_state",
        responses=None,
        tags=['onu state'],
        summary="Delete the state of the specified ONU",
        description=" "
    )
    @method_decorator(permission_required('can_delete_network_onus', raise_exception=True))
    def delete(self, request, onu_id, version):
        """Delete the state of the specified ONU"""
        database_manager.delete_one(database_id=request.session.get('database'), collection="ONU-STATE",
                                    query={"_id": onu_id})

        return PonManagerApiResponse(status=status.HTTP_204_NO_CONTENT)


# ==================================================
# =============== ONU States View ==================
# ==================================================
class States(LoginRequiredMixin, GenericAPIView):
    raise_exception = True
    queryset = ''

    swaggerSchema = get_schema('ONU-STATE')

    @extend_schema(
        operation_id="get_onu_states",
        parameters=[MongoQueryParameter(), MongoProjectionParameter()],
        responses={
            200: OpenApiResponse(response=schema(swaggerSchema),
                                 description='OK'),
        },
        tags=['onu state'],
        summary="Get the states for all ONUs",
        description=" "
    )
    @method_decorator(permission_required('can_read_network_onus', raise_exception=True))
    @validate_query_params(collection="ONU-STATE")
    def get(self, request, query, projection, sort, limit, skip, next, distinct, version):
        """Get the states for all ONUs"""
        if distinct:
            res_data = database_manager.distinct(database_id=request.session.get('database'), collection="ONU-STATE",
                                                 query=query, distinct=distinct)
        else:
            res_data = database_manager.find(database_id=request.session.get('database'), collection="ONU-STATE",
                                             query=query, projection=projection, sort=sort, limit=limit, skip=skip,
                                             next=next)
        return PonManagerApiResponse(status=status.HTTP_200_OK, data=res_data)


# ==================================================
# ========== One ONU Configuration View ============
# ==================================================
class OneConfiguration(LoginRequiredMixin, GenericAPIView):
    raise_exception = True
    queryset = ''

    swaggerSchema = get_schema('ONU-CFG')

    @extend_schema(
        operation_id="get_one_onu_config",
        responses={
            200: OpenApiResponse(response=schema(swaggerSchema),
                                 description='OK'),
        },
        tags=['onu config'],
        summary="Get the config for the specified ONU",
        description=" "
    )
    @method_decorator(permission_required('can_read_network_onus', raise_exception=True))
    def get(self, request, onu_id, version):
        """Get the config for the specified ONU"""
        # Validate ONU ID
        if not validate_onu_id(onu_id):
            response = PonManagerApiResponse(status=status.HTTP_400_BAD_REQUEST,
                                             details={"message": "Invalid ONU ID format"})
        else:
            res_data = database_manager.find_one(database_id=request.session.get('database'), collection="ONU-CFG",
                                                 query={"_id": onu_id})
            if res_data:
                response = PonManagerApiResponse(status=status.HTTP_200_OK, data=res_data)
            else:
                response = PonManagerApiResponse(status=status.HTTP_404_NOT_FOUND, details={
                    "message": "ONU ID " + str(onu_id) + " has no configuration document"})

        return response

    swaggerSchema = get_schema('ONU-CFG')

    @extend_schema(
        operation_id="put_one_onu_config",
        request={
            "application/json": schema(swaggerSchema),
        },
        responses={
            201: OpenApiResponse(response=schema(swaggerSchema),
                                 description='Created'),
        },
        tags=['onu config'],
        summary="Update the config for the specified ONU",
        description=" "
    )
    @method_decorator(
        permission_required_any_of(['can_update_network_onus', 'can_create_network_onus'], raise_exception=True))
    @validate_data(collection="ONU-CFG", resource_id_param="onu_id")
    def put(self, request, data, onu_id, version):
        """Update the config for the specified ONU"""
        old_document = database_manager.find_one_and_replace(database_id=request.session.get('database'),
                                                             collection="ONU-CFG", query={"_id": onu_id},
                                                             new_document=data)
        if old_document is None:
            status_code = status.HTTP_201_CREATED
        else:
            status_code = status.HTTP_200_OK

        return PonManagerApiResponse(status=status_code, new_data=data, old_data=old_document)

    @extend_schema(
        operation_id="delete_one_onu_config",
        responses=None,
        tags=['onu config'],
        summary="Delete the config of the specified ONU",
        description=" "
    )
    @method_decorator(permission_required('can_delete_network_onus', raise_exception=True))
    def delete(self, request, onu_id, version):
        """Delete the config of the specified ONU"""
        database_manager.delete_one(database_id=request.session.get('database'), collection="ONU-CFG",
                                    query={"_id": onu_id})

        return PonManagerApiResponse(status=status.HTTP_204_NO_CONTENT)


# ==================================================
# =========== ONU Configurations View ==============
# ==================================================
class Configurations(LoginRequiredMixin, GenericAPIView):
    raise_exception = True
    queryset = ''

    swaggerSchema = get_schema('ONU-CFG')

    @extend_schema(
        operation_id="get_onu_configs",
        responses={
            200: OpenApiResponse(response=schema(swaggerSchema),
                                 description='OK'),
        },
        tags=['onu config'],
        summary="Get the configs for all ONUs",
        description=" "
    )
    @method_decorator(permission_required('can_read_network_onus', raise_exception=True))
    @validate_query_params(collection="ONU-CFG")
    def get(self, request, query, projection, sort, limit, skip, next, distinct, version):
        """Get the configs for all ONUs"""
        if distinct:
            res_data = database_manager.distinct(database_id=request.session.get('database'), collection="ONU-CFG",
                                                 query=query, distinct=distinct)
        else:
            res_data = database_manager.find(database_id=request.session.get('database'), collection="ONU-CFG",
                                             query=query, projection=projection, sort=sort, limit=limit, skip=skip,
                                             next=next)

        return PonManagerApiResponse(status=status.HTTP_200_OK, data=res_data)

    swaggerSchema = get_schema('ONU-CFG')

    @extend_schema(
        operation_id="post_onu_config",
        request={
            "application/json": schema(swaggerSchema),
        },
        responses={
            201: OpenApiResponse(response=schema(swaggerSchema),
                                 description='Created'),
        },
        tags=['onu config'],
        summary="Create the provided ONU config",
        description=" "
    )
    @method_decorator(permission_required('can_create_network_onus', raise_exception=True))
    @validate_data(collection="ONU-CFG", resource_id_param=None)
    def post(self, request, data, version):
        """Create the provided ONU config"""
        try:
            database_manager.insert_one(database_id=request.session.get('database'), collection="ONU-CFG",
                                        document=data)
            response = PonManagerApiResponse(status=status.HTTP_201_CREATED, new_data=data, old_data=None)
        except pymongo.errors.DuplicateKeyError:
            onu_id = get_nested_value(data, ["_id"], None)
            response = PonManagerApiResponse(status=status.HTTP_409_CONFLICT,
                                             details={"message": f"ONU configuration with id {onu_id} already exists"})

        return response


# ==================================================
# ======= One ONU Alarm Configuration View =========
# ==================================================
class OneAlarmConfiguration(LoginRequiredMixin, GenericAPIView):
    raise_exception = True
    queryset = ''

    swaggerSchema = get_schema('ONU-ALARM-CFG')

    @extend_schema(
        operation_id="get_one_onu_alarm_config",
        responses={
            200: OpenApiResponse(response=schema(swaggerSchema),
                                 description='OK'),
        },
        tags=['onu alarm config'],
        summary="Get the specified ONU Alarm Config",
        description=" "
    )
    @method_decorator(permission_required('can_read_global_config_alarms', raise_exception=True))
    def get(self, request, cfg_id, version):
        """Get the specified ONU Alarm Config"""
        res_data = database_manager.find_one(database_id=request.session.get('database'), collection="ONU-ALARM-CFG",
                                             query={"_id": cfg_id})
        if res_data:
            response = PonManagerApiResponse(status=status.HTTP_200_OK, data=res_data)
        else:
            response = PonManagerApiResponse(status=status.HTTP_404_NOT_FOUND, details={
                "message": "CFG ID " + str(cfg_id) + " has no alarm configuration document"})

        return response

    swaggerSchema = get_schema('ONU-ALARM-CFG')

    @extend_schema(
        operation_id="put_one_onu_alarm_config",
        request={
            "application/json": schema(swaggerSchema),
        },
        responses={
            201: OpenApiResponse(response=schema(swaggerSchema),
                                 description='Created'),
        },
        tags=['onu alarm config'],
        summary="Update the config for the specified ONU Alarm Config",
        description=" "
    )
    @method_decorator(permission_required_any_of(['can_update_global_config_alarms', 'can_create_global_config_alarms'],
                                                 raise_exception=True))
    @validate_data(collection="ONU-ALARM-CFG", resource_id_param="cfg_id")
    def put(self, request, data, cfg_id, version):
        """Update the config for the specified ONU Alarm Config"""
        old_document = database_manager.find_one_and_replace(database_id=request.session.get('database'),
                                                             collection="ONU-ALARM-CFG", query={"_id": cfg_id},
                                                             new_document=data)
        if old_document is None:
            status_code = status.HTTP_201_CREATED
        else:
            status_code = status.HTTP_200_OK

        return PonManagerApiResponse(status=status_code, new_data=data, old_data=old_document)

    @extend_schema(
        operation_id="delete_one_onu_alarm_config",
        responses=None,
        tags=['onu alarm config'],
        summary="Delete the specified ONU Alarm Config",
        description=" "
    )
    @method_decorator(permission_required('can_delete_global_config_alarms', raise_exception=True))
    def delete(self, request, cfg_id, version):
        """Delete the specified ONU Alarm Config"""
        database_manager.delete_one(database_id=request.session.get('database'), collection="ONU-ALARM-CFG",
                                    query={"_id": cfg_id})

        return PonManagerApiResponse(status=status.HTTP_204_NO_CONTENT)


# ==================================================
# ======== ONU Alarm Configurations View ===========
# ==================================================
class AlarmConfigurations(LoginRequiredMixin, GenericAPIView):
    raise_exception = True
    queryset = ''

    swaggerSchema = get_schema('ONU-ALARM-CFG')

    @extend_schema(
        operation_id="get_onu_alarm_configs",
        responses={
            200: OpenApiResponse(response=schema(swaggerSchema),
                                 description='OK'),
        },
        tags=['onu alarm config'],
        summary="Get all ONU Alarm Configs",
        description=" "
    )
    @method_decorator(permission_required('can_read_global_config_alarms', raise_exception=True))
    @validate_query_params(collection="ONU-ALARM-CFG")
    def get(self, request, query, projection, sort, limit, skip, next, distinct, version):
        """Get all ONU Alarm Configs"""
        if distinct:
            res_data = database_manager.distinct(database_id=request.session.get('database'),
                                                 collection="ONU-ALARM-CFG",
                                                 query=query, distinct=distinct)
        else:
            res_data = database_manager.find(database_id=request.session.get('database'), collection="ONU-ALARM-CFG",
                                             query=query, projection=projection, sort=sort, limit=limit, skip=skip,
                                             next=next)

        return PonManagerApiResponse(status=status.HTTP_200_OK, data=res_data)

    swaggerSchema = get_schema('ONU-ALARM-CFG')

    @extend_schema(
        operation_id="post_onu_alarm_config",
        request={
            "application/json": schema(swaggerSchema),
        },
        responses={
            201: OpenApiResponse(response=schema(swaggerSchema),
                                 description='Created'),
        },
        tags=['onu alarm config'],
        summary="Create the provided ONU Alarm Config",
        description=" "
    )
    @method_decorator(permission_required('can_create_global_config_alarms', raise_exception=True))
    @validate_data(collection="ONU-ALARM-CFG", resource_id_param=None)
    def post(self, request, data, version):
        """Create the provided ONU Alarm Config"""
        try:
            database_manager.insert_one(database_id=request.session.get('database'), collection="ONU-ALARM-CFG",
                                        document=data)
            response = PonManagerApiResponse(status=status.HTTP_201_CREATED, new_data=data, old_data=None)
        except pymongo.errors.DuplicateKeyError:
            doc_id = get_nested_value(data, ["_id"], None)
            response = PonManagerApiResponse(status=status.HTTP_409_CONFLICT, details={
                "message": f"ONU alarm configuration with id {doc_id} already exists"})

        return response


# ==================================================
# ======= One ONU Alarm History State View =========
# ==================================================
class OneAlarmHistoryState(LoginRequiredMixin, GenericAPIView):
    raise_exception = True
    queryset = ''

    swaggerSchema = get_schema('ONU-ALARM-HIST-STATE')

    @extend_schema(
        operation_id="get_one_onu_alarm_history",
        responses={
            200: OpenApiResponse(response=schema(swaggerSchema),
                                 description='OK'),
        },
        tags=['onu alarm history'],
        summary="Get the specified ONU Alarm History State",
        description=" "
    )
    @method_decorator(permission_required('can_read_network_onus', raise_exception=True))
    def get(self, request, onu_id, version):
        """Get the specified ONU Alarm History State"""
        # Validate ONU ID
        if not validate_onu_id(onu_id):
            response = PonManagerApiResponse(status=status.HTTP_400_BAD_REQUEST, details="Invalid ONU ID format")
        else:
            res_data = database_manager.find_one(database_id=request.session.get('database'),
                                                 collection="ONU-ALARM-HIST-STATE", query={"_id": onu_id})
            if res_data:
                response = PonManagerApiResponse(status=status.HTTP_200_OK, data=res_data)
            else:
                response = PonManagerApiResponse(status=status.HTTP_404_NOT_FOUND, details={
                    "message": "ONU ID " + str(onu_id) + " has no alarm history document"})

        return response

    @extend_schema(
        operation_id="delete_one_onu_alarm_history",
        responses=None,
        tags=['onu alarm history'],
        summary="Delete the specified ONU Alarm Config",
        description=" "
    )
    @method_decorator(permission_required('can_delete_network_onus', raise_exception=True))
    def delete(self, request, onu_id, version):
        """Delete the specified ONU Alarm Config"""
        database_manager.delete_one(database_id=request.session.get('database'), collection="ONU-ALARM-HIST-STATE",
                                    query={"_id": onu_id})

        return PonManagerApiResponse(status=status.HTTP_204_NO_CONTENT)


# ==================================================
# ======== ONU Alarm History States View ===========
# ==================================================
class AlarmHistoryStates(LoginRequiredMixin, GenericAPIView):
    raise_exception = True
    queryset = ''

    swaggerSchema = get_schema('ONU-ALARM-HIST-STATE')

    @extend_schema(
        operation_id="get_onu_alarm_history",
        responses={
            200: OpenApiResponse(response=schema(swaggerSchema),
                                 description='OK'),
        },
        tags=['onu alarm history'],
        summary="Get all ONU Alarm History States",
        description=" "
    )
    @method_decorator(permission_required('can_read_network_onus', raise_exception=True))
    @validate_query_params(collection="ONU-ALARM-HIST-STATE")
    def get(self, request, query, projection, sort, limit, skip, next, distinct, version):
        """Get all ONU Alarm History States"""
        if distinct:
            res_data = database_manager.distinct(database_id=request.session.get('database'),
                                                 collection="ONU-ALARM-HIST-STATE",
                                                 query=query, distinct=distinct)
        else:
            res_data = database_manager.find(database_id=request.session.get('database'),
                                             collection="ONU-ALARM-HIST-STATE",
                                             query=query, projection=projection, sort=sort, limit=limit, skip=skip,
                                             next=next)

        return PonManagerApiResponse(status=status.HTTP_200_OK, data=res_data)


# ==================================================
# ============ One ONU Statistics View =============
# ==================================================
class Statistics(LoginRequiredMixin, GenericAPIView):
    raise_exception = True
    queryset = ''

    swaggerSchema = get_schema('ONU-STATE')

    @extend_schema(
        operation_id="get_one_onu_stats",
        parameters=[
            OpenApiParameter(name="time-start", description="UTC timestamp to begin getting stats at",
                             type=OpenApiTypes.DATETIME, required=True),
            OpenApiParameter(name="time-end", description="UTC timestamp to stop getting stats at",
                             type=OpenApiTypes.DATETIME)
        ],
        responses={
            200: OpenApiResponse(response=schema(swaggerSchema),
                                 description='OK'),
        },
        tags=['onu stats'],
        summary="Get the statistics of the specified ONU between the start and end times",
        description=" "
    )
    @method_decorator(permission_required('can_read_network_onus', raise_exception=True))
    def get(self, request, onu_id, version):
        """Get the statistics of the specified ONU between the start and end times"""
        start_time = request.GET.get('start-time', None)
        end_time = request.GET.get('end-time', None)

        # Return missing parameter response if start time is undefined
        if start_time is None:
            response = PonManagerApiResponse(status=status.HTTP_400_BAD_REQUEST,
                                             details={"message": "Parameter 'start-time' is required"})
        # Validate ONU ID
        elif not validate_onu_id(onu_id):
            response = PonManagerApiResponse(status=status.HTTP_400_BAD_REQUEST,
                                             details={"message": "Invalid ONU ID format"})
        else:
            database = database_manager.get_database(request.session.get('database'))
            try:
                state_data = database_manager.find_one(database_id=request.session.get('database'),
                                                       collection="ONU-STATE",
                                                       query={"_id": onu_id}, projection={"_id": 0, "CNTL.Version": 1})
                # Pre-provisioned devices will return state_data = None ; if statement avoids 500 error
                if state_data:
                    cntl_version = get_nested_value(state_data, ["CNTL", "Version"], "")
                    sub_three_one_version = False

                    if float(cntl_version[1:4]) < 3.1:
                        sub_three_one_version = True

                    if sub_three_one_version:
                        collection = database.get_collection("STATS-ONU-{}".format(onu_id.replace(":", "")))
                        if end_time is None:
                            res_data = list(collection.find({"_id": {"$gte": start_time}}).limit(10000))
                        else:
                            res_data = list(
                                collection.find({"_id": {"$gte": start_time, "$lte": end_time}}).limit(10000))
                    else:
                        # For new versions of the DB
                        collection = database.get_collection("STATS-ONU")
                        if end_time is None:
                            res_data = list(collection.find({
                                "$and": [
                                    {"device ID": onu_id},
                                    {"valid": True},
                                    {"_id": {"$gte": start_time}},
                                ]
                            }).limit(10000))
                        else:
                            res_data = list(collection.find({
                                "$and": [
                                    {"device ID": onu_id},
                                    {"valid": True},
                                    {"_id": {"$gte": start_time, "$lte": end_time}}
                                ]
                            }).limit(10000))
                else:
                    res_data = None

            except (ConnectionRefusedError, pymongo.errors.PyMongoError) as e:
                raise APIException(detail=f"MongoDB error: {str(e)}")

            if res_data:
                # Add ONU ID to response format for easier handling in UI
                for block in res_data:
                    block['mac_address'] = onu_id
                response = PonManagerApiResponse(status=status.HTTP_200_OK, data=res_data)
            else:
                response = PonManagerApiResponse(status=status.HTTP_404_NOT_FOUND, details={
                    "message": "ONU ID " + str(onu_id) + " has no statistics documents"})

        return response

    @extend_schema(
        operation_id="delete_one_onu_stats",
        responses=None,
        tags=['onu stats'],
        summary="Delete the Statistics of the specified ONU",
        description=" "
    )
    @method_decorator(permission_required('can_delete_network_onus', raise_exception=True))
    def delete(self, request, onu_id, version):
        """Delete the Statistics of the specified ONU"""
        database = database_manager.get_database(request.session.get('database'))

        try:
            state_data = database_manager.find_one(database_id=request.session.get('database'), collection="ONU-STATE",
                                                   query={"_id": onu_id}, projection={"_id": 0, "CNTL.Version": 1})
            cntl_version = get_nested_value(state_data, ["CNTL", "Version"], "")
            sub_three_one_version = False

            if float(cntl_version[1:4]) < 3.1:
                sub_three_one_version = True

            if sub_three_one_version:
                collection = database.get_collection("STATS-ONU-{}".format(onu_id.replace(":", "")))
                collection.drop()
            else:
                # For new versions of the DB
                collection = database.get_collection("STATS-ONU")
                collection.update_many({"device ID": onu_id}, {"$set": {"valid": False}})

        except (ConnectionRefusedError, pymongo.errors.PyMongoError) as e:
            raise APIException(detail=f"MongoDB error: {str(e)}")

        return PonManagerApiResponse(status=status.HTTP_204_NO_CONTENT)


# ==================================================
# ============ One ONU Logs View =============
# ==================================================
class Logs(LoginRequiredMixin, GenericAPIView):
    raise_exception = True
    queryset = ''

    swaggerSchema = get_schema('ONU-STATE')

    @extend_schema(
        operation_id="get_one_onu_logs",
        parameters=[
            OpenApiParameter(name="time-start", description="UTC timestamp to begin getting stats at",
                             type=OpenApiTypes.DATETIME, required=True),
            OpenApiParameter(name="time-end", description="UTC timestamp to stop getting stats at",
                             type=OpenApiTypes.DATETIME)
        ],
        responses={
            200: OpenApiResponse(response=schema(swaggerSchema),
                                 description='OK'),
        },
        tags=['onu logs'],
        summary="Get the logs of the specified ONU between the start and end times",
        description=" "
    )
    @method_decorator(permission_required('can_read_network_onus', raise_exception=True))
    def get(self, request, onu_id, version):
        """Get the logs of the specified ONU between the start and end times"""
        start_time = request.GET.get('start-time', None)
        end_time = request.GET.get('end-time', None)

        # Return missing parameter response if start time is undefined
        if start_time is None:
            response = PonManagerApiResponse(status=status.HTTP_400_BAD_REQUEST,
                                             details={"message": "Parameter 'start-time' is required"})
        # Validate ONU ID
        elif not validate_onu_id(onu_id):
            response = PonManagerApiResponse(status=status.HTTP_400_BAD_REQUEST,
                                             details={"message": "Invalid ONU ID format"})
        else:
            database = database_manager.get_database(request.session.get('database'))

            try:
                state_data = database_manager.find_one(database_id=request.session.get('database'),
                                                       collection="ONU-STATE",
                                                       query={"_id": onu_id}, projection={"_id": 0, "CNTL.Version": 1})

                # If no state is found ONU is likely pre-provisioned or detached, so there is no state document.
                if not state_data:
                    return PonManagerApiResponse(status=status.HTTP_404_NOT_FOUND,
                                                 details={"message": "ONU ID " + str(onu_id) + " has no log document"})
                cntl_version = get_nested_value(state_data, ["CNTL", "Version"], "")
                sub_three_one_version = False

                if float(cntl_version[1:4]) < 3.1:
                    sub_three_one_version = True

                if sub_three_one_version:
                    collection = database.get_collection("SYSLOG-ONU-{}".format(onu_id.replace(":", "")))
                    if end_time is None:
                        res_data = list(collection.find({"time": {"$gte": start_time}},
                                                        {"_id": 0, "device ID": 0}))
                    else:
                        res_data = list(collection.find({"time": {"$gte": start_time, "$lte": end_time}},
                                                        {"_id": 0, "device ID": 0}))
                else:
                    collection = database.get_collection("SYSLOG-ONU")
                    if end_time is None:
                        res_data = list(collection.find({
                            "$and": [
                                {"device ID": onu_id},
                                {"valid": True},
                                {"time": {"$gte": start_time}},
                            ]
                        }, {"_id": 0, "device ID": 0}).limit(10000))
                    else:
                        res_data = list(collection.find({
                            "$and": [
                                {"device ID": onu_id},
                                {"valid": True},
                                {"time": {"$gte": start_time, "$lte": end_time}}
                            ]
                        }, {"_id": 0, "device ID": 0}).limit(10000))

            except (ConnectionRefusedError, pymongo.errors.PyMongoError) as e:
                raise APIException(detail=f"MongoDB error: {str(e)}")
            if res_data:
                response = PonManagerApiResponse(status=status.HTTP_200_OK, data=res_data)
            else:
                response = PonManagerApiResponse(status=status.HTTP_404_NOT_FOUND,
                                                 details={"message": "ONU ID " + str(onu_id) + " has no logs document"})

        return response

    @extend_schema(
        operation_id="delete_one_onu_logs",
        responses=None,
        tags=['onu logs'],
        summary="Delete the Logs of the specified ONU",
        description=" "
    )
    @method_decorator(permission_required('can_delete_network_onus', raise_exception=True))
    def delete(self, request, onu_id, version):
        """Delete the Logs of the specified ONU"""
        database = database_manager.get_database(request.session.get('database'))

        try:
            state_data = database_manager.find_one(database_id=request.session.get('database'), collection="ONU-STATE",
                                                   query={"_id": onu_id}, projection={"_id": 0, "CNTL.Version": 1})
            cntl_version = get_nested_value(state_data, ["CNTL", "Version"], "")
            sub_three_one_version = False

            if float(cntl_version[1:4]) < 3.1:
                sub_three_one_version = True

            if sub_three_one_version:
                collection = database.get_collection("SYSLOG-ONU-{}".format(onu_id.replace(":", "")))
                collection.drop()
            else:
                # For new versions of the DB
                collection = database.get_collection("SYSLOG-ONU")
                collection.update_many({"device ID": onu_id}, {"$set": {"valid": False}})

        except (ConnectionRefusedError, pymongo.errors.PyMongoError) as e:
            raise APIException(detail=f"MongoDB error: {str(e)}")

        return PonManagerApiResponse(status=status.HTTP_204_NO_CONTENT)


# ==================================================
# ========== One ONU MIB Reset State View ==========
# ==================================================
class OneMibResetState(LoginRequiredMixin, GenericAPIView):
    raise_exception = True
    queryset = ''

    swaggerSchema = get_schema('ONU-MIB-RST-STATE')

    @extend_schema(
        operation_id="get_one_onu_mib_rst_state",
        responses={
            200: OpenApiResponse(response=schema(swaggerSchema),
                                 description='OK'),
        },
        tags=['onu mib reset state'],
        summary="Get the MIB Reset State of the specified ONU",
        description=" "
    )
    @method_decorator(permission_required('can_read_network_onus', raise_exception=True))
    def get(self, request, onu_id, version):
        """Get the MIB Reset State of the specified ONU"""
        # Validate ONU ID
        if not validate_onu_id(onu_id):
            response = PonManagerApiResponse(status=status.HTTP_400_BAD_REQUEST,
                                             details={"message": "Invalid ONU ID format"})
        else:
            res_data = database_manager.find_one(database_id=request.session.get('database'),
                                                 collection="ONU-MIB-RST-STATE", query={"_id": onu_id})
            if res_data:
                response = PonManagerApiResponse(status=status.HTTP_200_OK, data=res_data)
            else:
                response = PonManagerApiResponse(status=status.HTTP_404_NOT_FOUND, details={
                    "message": "ONU ID " + str(onu_id) + " has no MIB reset state document"})

        return response

    @extend_schema(
        operation_id="delete_one_onu_mib_rst_state",
        responses=None,
        tags=['onu mib reset state'],
        summary="Delete the MIB Reset State of the specified ONU",
        description=" "
    )
    @method_decorator(permission_required('can_delete_network_onus', raise_exception=True))
    def delete(self, request, onu_id, version):
        """Delete the MIB Reset State of the specified ONU"""
        database_manager.delete_one(database_id=request.session.get('database'), collection="ONU-MIB-RST-STATE",
                                    query={"_id": onu_id})

        return PonManagerApiResponse(status=status.HTTP_204_NO_CONTENT)


# ==================================================
# ============ ONU MIB Reset States View ===========
# ==================================================
class MibResetStates(LoginRequiredMixin, GenericAPIView):
    raise_exception = True
    queryset = ''

    swaggerSchema = get_schema('ONU-MIB-RST-STATE')

    @extend_schema(
        operation_id="get_onu_mib_rst_states",
        responses={
            200: OpenApiResponse(response=schema(swaggerSchema),
                                 description='OK'),
        },
        tags=['onu mib reset state'],
        summary="Get the MIB Reset States of all ONUs",
        description=" "
    )
    @method_decorator(permission_required('can_read_network_onus', raise_exception=True))
    @validate_query_params(collection="ONU-MIB-RST-STATE")
    def get(self, request, query, projection, sort, limit, skip, next, distinct, version):
        """Get the MIB Reset States of all ONUs"""
        if distinct:
            res_data = database_manager.distinct(database_id=request.session.get('database'),
                                                 collection="ONU-MIB-RST-STATE",
                                                 query=query, distinct=distinct)
        else:
            res_data = database_manager.find(database_id=request.session.get('database'),
                                             collection="ONU-MIB-RST-STATE",
                                             query=query, projection=projection, sort=sort, limit=limit, skip=skip,
                                             next=next)

        return PonManagerApiResponse(status=status.HTTP_200_OK, data=res_data)


# ==================================================
# ========= One ONU MIB Current State View =========
# ==================================================
class OneMibCurrentState(LoginRequiredMixin, GenericAPIView):
    raise_exception = True
    queryset = ''

    swaggerSchema = get_schema('ONU-MIB-CUR-STATE')

    @extend_schema(
        operation_id="get_one_onu_mib_cur_state",
        responses={
            200: OpenApiResponse(response=schema(swaggerSchema),
                                 description='OK'),
        },
        tags=['onu mib current state'],
        summary="Get the MIB Current State of the specified ONU",
        description=" "
    )
    @method_decorator(permission_required('can_read_network_onus', raise_exception=True))
    def get(self, request, onu_id, version):
        """Get the MIB Current State of the specified ONU"""
        # Validate ONU ID
        if not validate_onu_id(onu_id):
            response = PonManagerApiResponse(status=status.HTTP_400_BAD_REQUEST,
                                             details={"message": "Invalid ONU ID format"})
        else:
            res_data = database_manager.find_one(database_id=request.session.get('database'),
                                                 collection="ONU-MIB-CUR-STATE", query={"_id": onu_id})
            if res_data:
                response = PonManagerApiResponse(status=status.HTTP_200_OK, data=res_data)
            else:
                response = PonManagerApiResponse(status=status.HTTP_404_NOT_FOUND, details={
                    "message": "ONU ID " + str(onu_id) + " has no MIB current state document"})

        return response

    @extend_schema(
        operation_id="delete_one_onu_mib_cur_state",
        responses=None,
        tags=['onu mib current state'],
        summary="Delete the MIB Current State of the specified ONU",
        description=" "
    )
    @method_decorator(permission_required('can_delete_network_onus', raise_exception=True))
    def delete(self, request, onu_id, version):
        """Delete the MIB Current State of the specified ONU"""
        database_manager.delete_one(database_id=request.session.get('database'), collection="ONU-MIB-CUR-STATE",
                                    query={"_id": onu_id})

        return PonManagerApiResponse(status=status.HTTP_204_NO_CONTENT)


# ==================================================
# =========== ONU MIB Current States View ==========
# ==================================================
class MibCurrentStates(LoginRequiredMixin, GenericAPIView):
    raise_exception = True
    queryset = ''

    swaggerSchema = get_schema('ONU-MIB-CUR-STATE')

    @extend_schema(
        operation_id="get_onu_mib_cur_states",
        responses={
            200: OpenApiResponse(response=schema(swaggerSchema),
                                 description='OK'),
        },
        tags=['onu mib current state'],
        summary="Get the MIB Current States of all ONUs",
        description=" "
    )
    @method_decorator(permission_required('can_read_network_onus', raise_exception=True))
    @validate_query_params(collection="ONU-MIB-CUR-STATE")
    def get(self, request, query, projection, sort, limit, skip, next, distinct, version):
        """Get the MIB Current States of all ONUs"""
        if distinct:
            res_data = database_manager.distinct(database_id=request.session.get('database'),
                                                 collection="ONU-MIB-CUR-STATE",
                                                 query=query, distinct=distinct)
        else:
            res_data = database_manager.find(database_id=request.session.get('database'),
                                             collection="ONU-MIB-CUR-STATE",
                                             query=query, projection=projection, sort=sort, limit=limit, skip=skip,
                                             next=next)

        return PonManagerApiResponse(status=status.HTTP_200_OK, data=res_data)


# ==================================================
# ============= One ONU CPE State View =============
# ==================================================
class OneCpeState(LoginRequiredMixin, GenericAPIView):
    raise_exception = True
    queryset = ''

    swaggerSchema = get_schema('ONU-CPE-STATE')

    @extend_schema(
        operation_id="get_one_onu_cpe_state",
        responses={
            200: OpenApiResponse(response=schema(swaggerSchema),
                                 description='OK'),
        },
        tags=['onu cpe state'],
        summary="Get the CPE State of the specified ONU",
        description=" "
    )
    @method_decorator(permission_required('can_read_network_onus', raise_exception=True))
    def get(self, request, onu_id, version):
        """Get the CPE State of the specified ONU"""
        # Validate ONU ID
        if not validate_onu_id(onu_id):
            response = PonManagerApiResponse(status=status.HTTP_400_BAD_REQUEST,
                                             details={"message": "Invalid ONU ID format"})
        else:
            res_data = database_manager.find_one(database_id=request.session.get('database'),
                                                 collection="ONU-CPE-STATE", query={"_id": onu_id})
            if res_data:
                response = PonManagerApiResponse(status=status.HTTP_200_OK, data=res_data)
            else:
                response = PonManagerApiResponse(status=status.HTTP_404_NOT_FOUND, details={
                    "message": "ONU ID " + str(onu_id) + " has no cpe state document"})

        return response

    @extend_schema(
        operation_id="delete_one_onu_cpe_state",
        responses=None,
        tags=['onu cpe state'],
        summary="Delete the specified ONU CPE State",
        description=" "
    )
    @method_decorator(permission_required('can_delete_network_onus', raise_exception=True))
    def delete(self, request, onu_id, version):
        """Delete the specified ONU CPE State"""
        database_manager.delete_one(database_id=request.session.get('database'), collection="ONU-CPE-STATE",
                                    query={"_id": onu_id})

        return PonManagerApiResponse(status=status.HTTP_204_NO_CONTENT)


# ==================================================
# =============== ONU CPE States View ==============
# ==================================================
class CpeStates(LoginRequiredMixin, GenericAPIView):
    raise_exception = True
    queryset = ''

    swaggerSchema = get_schema('ONU-CPE-STATE')

    @extend_schema(
        operation_id="get_onu_cpe_states",
        responses={
            200: OpenApiResponse(response=schema(swaggerSchema),
                                 description='OK'),
        },
        tags=['onu cpe state'],
        summary="Get the CPE States of all ONUs",
        description=" "
    )
    @method_decorator(permission_required('can_read_network_onus', raise_exception=True))
    @validate_query_params(collection="ONU-CPE-STATE")
    def get(self, request, query, projection, sort, limit, skip, next, distinct, version):
        """Get the CPE States of all ONUs"""
        if distinct:
            res_data = database_manager.distinct(database_id=request.session.get('database'),
                                                 collection="ONU-CPE-STATE",
                                                 query=query, distinct=distinct)
        else:
            res_data = database_manager.find(database_id=request.session.get('database'), collection="ONU-CPE-STATE",
                                             query=query, projection=projection, sort=sort, limit=limit, skip=skip,
                                             next=next)

        return PonManagerApiResponse(status=status.HTTP_200_OK, data=res_data)


# ==================================================
# ========= One ONU Automation State View ==========
# ==================================================
class OneAutomationState(LoginRequiredMixin, GenericAPIView):
    raise_exception = True
    queryset = ''

    swaggerSchema = get_schema('ONU-AUTO-STATE')

    @extend_schema(
        operation_id="get_one_onu_automation_state",
        responses={
            200: OpenApiResponse(response=schema(swaggerSchema),
                                 description='OK'),
        },
        tags=['onu automation state'],
        summary="Get the Automation State of the specified ONU",
        description=" "
    )
    @method_decorator(permission_required('can_read_network_onus', raise_exception=True))
    def get(self, request, onu_id, version):
        """Get the Automation State of the specified ONU"""
        # Validate ONU ID
        if not validate_onu_id(onu_id):
            response = PonManagerApiResponse(status=status.HTTP_400_BAD_REQUEST,
                                             details={"message": "Invalid ONU ID format"})
        else:
            res_data = database_manager.find_one(database_id=request.session.get('database'),
                                                 collection="ONU-AUTO-STATE", query={"_id": onu_id})
            if res_data:
                response = PonManagerApiResponse(status=status.HTTP_200_OK, data=res_data)
            else:
                response = PonManagerApiResponse(status=status.HTTP_404_NOT_FOUND, details={
                    "message": "ONU ID " + str(onu_id) + " has no automation state document"})

        return response

    @extend_schema(
        operation_id="delete_one_onu_automation_state",
        responses=None,
        tags=['onu automation state'],
        summary="Delete the specified ONU Automation State",
        description=" "
    )
    @method_decorator(permission_required('can_delete_network_onus', raise_exception=True))
    def delete(self, request, onu_id, version):
        """Delete the specified ONU Automation State"""
        database_manager.delete_one(database_id=request.session.get('database'), collection="ONU-AUTO-STATE",
                                    query={"_id": onu_id})

        return PonManagerApiResponse(status=status.HTTP_204_NO_CONTENT)


# ==================================================
# =========== ONU Automation States View ===========
# ==================================================
class AutomationStates(LoginRequiredMixin, GenericAPIView):
    raise_exception = True
    queryset = ''

    swaggerSchema = get_schema('ONU-AUTO-STATE')

    @extend_schema(
        operation_id="get_onu_automation_states",
        responses={
            200: OpenApiResponse(response=schema(swaggerSchema),
                                 description='OK'),
        },
        tags=['onu automation state'],
        summary="Get the Automation States of all ONUs",
        description=" "
    )
    @method_decorator(permission_required('can_read_network_onus', raise_exception=True))
    # TODO @validate_query_params(collection="ONU-AUTO-STATE")
    def get(self, request, version):
        """Get the Automation States of all ONUs"""
        res_data = database_manager.find(database_id=request.session.get('database'), collection="ONU-AUTO-STATE")

        return PonManagerApiResponse(status=status.HTTP_200_OK, data=res_data)


# ==================================================
# ========= One ONU Automation Config View ==========
# ==================================================
class OneAutomationConfig(LoginRequiredMixin, GenericAPIView):
    raise_exception = True
    queryset = ''

    swaggerSchema = get_schema('ONU-AUTO-CFG')

    @extend_schema(
        operation_id="get_one_onu_automation_config",
        responses={
            200: OpenApiResponse(response=schema(swaggerSchema),
                                 description='OK'),
        },
        tags=['onu automation config'],
        summary="Get the specified ONU Automation Config",
        description=" "
    )
    @method_decorator(permission_required('can_read_network_onus', raise_exception=True))
    def get(self, request, cfg_id, version):
        """Get the specified ONU Automation Config"""
        res_data = database_manager.find_one(database_id=request.session.get('database'), collection="ONU-AUTO-CFG",
                                             query={"_id": cfg_id})
        if res_data:
            response = PonManagerApiResponse(status=status.HTTP_200_OK, data=res_data)
        else:
            response = PonManagerApiResponse(status=status.HTTP_404_NOT_FOUND, details={
                "message": "CFG ID " + str(cfg_id) + " has no automation state document"})

        return response

    swaggerSchema = get_schema('ONU-AUTO-CFG')

    @extend_schema(
        operation_id="put_one_onu_automation_config",
        request={
            "application/json": schema(swaggerSchema),
        },
        responses={
            201: OpenApiResponse(response=schema(swaggerSchema),
                                 description='Created'),
        },
        tags=['onu automation config'],
        summary="Update the config for the specified ONU Automation Config",
        description=" "
    )
    @method_decorator(
        permission_required_any_of(['can_update_network_onus', 'can_create_network_onus'], raise_exception=True))
    @validate_data(collection="ONU-AUTO-CFG", resource_id_param="cfg_id")
    def put(self, request, data, cfg_id, version):
        """Update the config for the specified ONU Automation Config"""
        if data is None:
            response = PonManagerApiResponse(status=status.HTTP_400_BAD_REQUEST, details={
                "message": "Request body must be of format '{ data: <ONU-AUTO-CFG> }'"})
        else:
            data['_id'] = cfg_id
            old_document = database_manager.find_one_and_replace(database_id=request.session.get('database'),
                                                                 collection="ONU-AUTO-CFG", query={"_id": cfg_id},
                                                                 new_document=data)
            if old_document is None:
                response = PonManagerApiResponse(status=status.HTTP_201_CREATED, new_data=data)
            else:
                response = PonManagerApiResponse(status=status.HTTP_200_OK, new_data=data, old_data=old_document)

        return response

    @extend_schema(
        operation_id="delete_one_onu_automation_config",
        responses=None,
        tags=['onu automation config'],
        summary="Delete the specified ONU Automation Config",
        description=" "
    )
    @method_decorator(permission_required('can_delete_network_onus', raise_exception=True))
    def delete(self, request, cfg_id, version):
        """Delete the specified ONU Automation Config"""
        database_manager.delete_one(database_id=request.session.get('database'), collection="ONU-AUTO-CFG",
                                    query={"_id": cfg_id})

        return PonManagerApiResponse(status=status.HTTP_204_NO_CONTENT)


# ==================================================
# ========== ONU Automation Configs View ===========
# ==================================================
class AutomationConfigs(LoginRequiredMixin, GenericAPIView):
    raise_exception = True
    queryset = ''

    swaggerSchema = get_schema('ONU-AUTO-CFG')

    @extend_schema(
        operation_id="get_onu_automation_configs",
        responses={
            200: OpenApiResponse(response=schema(swaggerSchema),
                                 description='OK'),
        },
        tags=['onu automation config'],
        summary="Get the Automation Configs of all ONUs",
        description=" "
    )
    @method_decorator(permission_required('can_read_network_onus', raise_exception=True))
    # TODO @validate_query_params(collection="ONU-AUTO-CFG")
    def get(self, request, version):
        """Get the Automation Configs of all ONUs"""
        res_data = database_manager.find(database_id=request.session.get('database'), collection="ONU-AUTO-CFG")

        return PonManagerApiResponse(status=status.HTTP_200_OK, data=res_data)

    swaggerSchema = get_schema('ONU-AUTO-CFG')

    @extend_schema(
        operation_id="post_onu_automation_config",
        request={
            "application/json": schema(swaggerSchema),
        },
        responses={
            201: OpenApiResponse(response=schema(swaggerSchema),
                                 description='Created'),
        },
        tags=['onu automation config'],
        summary="Create the provided ONU Automation Config",
        description=" "
    )
    @method_decorator(permission_required('can_create_network_onus', raise_exception=True))
    # TODO @validate_data(collection="ONU-AUTO-CFG", resource_id_param=None)
    def post(self, request, version):
        """Create the provided ONU Automation Config"""
        try:
            data = get_nested_value(request.data, ["data"])
            database_manager.insert_one(database_id=request.session.get('database'), collection="ONU-AUTO-CFG",
                                        document=data)
            response = PonManagerApiResponse(status=status.HTTP_201_CREATED, new_data=data, old_data=None)
        except pymongo.errors.DuplicateKeyError:
            doc_id = get_nested_value(data, ["_id"], None)
            response = PonManagerApiResponse(status=status.HTTP_409_CONFLICT, details={
                "message": f"ONU Automation configuration with id {doc_id} already exists"})

        return response


# ==================================================
# ======= Global ONU Automation Config View ========
# ==================================================
class GlobalAutomationConfig(LoginRequiredMixin, GenericAPIView):
    raise_exception = True
    queryset = ''

    swaggerSchema = get_schema('ONU-AUTO-CFG')

    @extend_schema(
        operation_id="get_global_onu_automation_config",
        responses={
            200: OpenApiResponse(response=schema(swaggerSchema),
                                 description='OK'),
        },
        tags=['onu automation config'],
        summary="Get the Global ONU Automation Config",
        description=" "
    )
    @method_decorator(permission_required('can_read_automation', raise_exception=True))
    def get(self, request, version):
        """Get the Global ONU Automation Config"""
        res_data = database_manager.find_one(database_id=request.session.get('database'), collection="ONU-AUTO-CFG",
                                             query={"_id": "Global"})

        return PonManagerApiResponse(status=status.HTTP_200_OK, data=res_data)

    swaggerSchema = get_schema('ONU-AUTO-CFG')

    @extend_schema(
        operation_id="put_global_onu_automation_config",
        request={
            "application/json": schema(swaggerSchema),
        },
        responses={
            201: OpenApiResponse(response=schema(swaggerSchema),
                                 description='Created'),
        },
        tags=['onu automation config'],
        summary="Update the Global ONU Automation Config",
        description=" "
    )
    @method_decorator(
        permission_required_any_of(['can_update_automation', 'can_create_automation'], raise_exception=True))
    @validate_data(collection="ONU-AUTO-CFG", resource_id_param=None)
    def put(self, request, data, version):
        """Update the Global ONU Automation Config"""
        if data is None:
            response = PonManagerApiResponse(status=status.HTTP_400_BAD_REQUEST, details={
                "message": "Request body must be of format '{ data: <ONU-AUTO-CFG> }'"})
        else:
            old_document = database_manager.find_one_and_replace(database_id=request.session.get('database'),
                                                                 collection="ONU-AUTO-CFG", query={"_id": "Global"},
                                                                 new_document=data)
            if old_document is None:
                response = PonManagerApiResponse(status=status.HTTP_201_CREATED, new_data=data)
            else:
                response = PonManagerApiResponse(status=status.HTTP_200_OK, new_data=data, old_data=old_document)

        return response

    swaggerSchema = get_schema('ONU-AUTO-CFG')

    @extend_schema(
        operation_id="post_global_onu_automation_config",
        request={
            "application/json": schema(swaggerSchema),
        },
        responses={
            201: OpenApiResponse(response=schema(swaggerSchema),
                                 description='Created'),
        },
        tags=['onu automation config'],
        summary="Create the provided ONU Automation Global Config",
        description=" "
    )
    @method_decorator(permission_required('can_create_automation', raise_exception=True))
    # TODO @validate_data(collection="ONU-AUTO-CFG", resource_id_param=None)
    def post(self, request, version):
        """Create the provided ONU Automation Global Config"""
        try:
            data = get_nested_value(request.data, ["data"])
            database_manager.insert_one(database_id=request.session.get('database'), collection="ONU-AUTO-CFG",
                                        document=data)
            response = PonManagerApiResponse(status=status.HTTP_201_CREATED, new_data=data, old_data=None)
        except pymongo.errors.DuplicateKeyError:
            response = PonManagerApiResponse(status=status.HTTP_409_CONFLICT, details={
                "message": f"ONU Automation configuration with id Global already exists"})

        return response

    @extend_schema(
        operation_id="delete_global_onu_automation_config",
        parameters=[
            OpenApiParameter(name="step", description="Automation step to delete template from",
                             type=OpenApiTypes.STR, required=True),
            OpenApiParameter(name="name", description="Automation template to delete from the specified step",
                             type=OpenApiTypes.STR, required=True)
        ],
        responses=None,
        tags=['onu automation config'],
        summary="Delete the Global ONU Automation Config",
        description=" "
    )
    @method_decorator(permission_required('can_delete_automation', raise_exception=True))
    def delete(self, request, version):
        """Delete the Global ONU Automation Config"""
        step = request.GET.get('step', None)
        name = request.GET.get('name', None)

        # Return missing parameter response if step or name are undefined
        if step is None:
            response = PonManagerApiResponse(status=status.HTTP_400_BAD_REQUEST,
                                             details={"message": "Parameter 'step' is required"})
        elif name is None:
            response = PonManagerApiResponse(status=status.HTTP_400_BAD_REQUEST,
                                             details={"message": "Parameter 'name' is required"})
        else:
            database = database_manager.get_database(request.session.get('database'))
            collection = database.get_collection("ONU-AUTO-CFG")
            try:
                if step.upper() == "IDENTIFY":
                    collection.update_one({'_id': "Global"}, {"$pull": {"IDENTIFY.Mapping": {"Description": name}}})
                elif step.upper() == "SERVICE_PKG":
                    collection.update_one({'_id': "Global"}, {"$unset": {
                        f"SERVICE.PACKAGES.{name}": "",
                        f"AUTH.{name}": "",
                        f"VLAN.{name}": "",
                        f"SLA.{name}": "",
                        f"DHCP.{name}": ""
                    }, })
                else:
                    collection.update_one({'_id': "Global"}, {"$unset": {f"{step.upper()}.{name}": ""}})
            except (ConnectionRefusedError, pymongo.errors.PyMongoError) as e:
                raise APIException(detail=f"MongoDB error: {str(e)}")

            response = PonManagerApiResponse(status=status.HTTP_204_NO_CONTENT)

        return response


# ==================================================
# ================ ONU Reset View ==================
# ==================================================
class Reset(LoginRequiredMixin, GenericAPIView):
    raise_exception = True
    queryset = ''

    swaggerSchema = get_schema('ONU-STATE')

    @extend_schema(
        operation_id="get_onu_reset_status",
        responses={
            200: OpenApiResponse(response=schema(swaggerSchema),
                                 description='OK'),
        },
        tags=['onu reset'],
        summary="Check the reset status of the specified ONU",
        description=" "
    )
    @method_decorator(permission_required('can_read_network_onus', raise_exception=True))
    def get(self, request, onu_id, version):
        """ Check the reset status of the specified ONU """
        database = database_manager.get_database(request.session.get("database"))
        collection = database.get_collection("ONU-STATE")
        try:
            result = list(collection.aggregate([
                {"$match": {"_id": onu_id}},
                {"$lookup": {"from": "ONU-CFG", "localField": "_id", "foreignField": "_id", "as": "CFG"}},
                {"$addFields": {"CFG": {"$arrayElemAt": ["$CFG", 0]}}},
                {"$project": {"_id": 0, "ONU.Reset Count": 1, "CFG.ONU.Reset Count": 1}}
            ]))
        except (ConnectionRefusedError, pymongo.errors.PyMongoError) as e:
            raise APIException(detail=f"MongoDB error: {str(e)}")

        res_data = {
            "Pending": False
        }
        if len(result) > 0 and result[0]["ONU"] and result[0]["CFG"]:
            # Return reset pending status as false if State or Config are not found
            state_count = get_nested_value(result, path=[0, "ONU", "Reset Count"], default=0)
            cfg_count = get_nested_value(result, path=[0, "CFG", "ONU", "Reset Count"], default=0)
            res_data["Pending"] = state_count != cfg_count

        return PonManagerApiResponse(status=status.HTTP_200_OK, data=res_data)

    swaggerSchema = get_schema('ONU-CFG')

    @extend_schema(
        operation_id="reset_onu",
        request={
            "application/json": schema(swaggerSchema),
        },
        responses={
            201: OpenApiResponse(response=schema(swaggerSchema),
                                 description='Created'),
        },
        tags=['onu reset'],
        summary="Trigger a Reset on the specified ONU",
        description=" "
    )
    @method_decorator(
        permission_required_any_of(['can_update_network_onus', 'can_create_network_onus'], raise_exception=True))
    def put(self, request, onu_id, version):
        """ Trigger a Reset on the specified ONU """
        update_document = {"$inc": {"ONU.Reset Count": 1, "ONU.CFG Change Count": 1}}
        update_result = database_manager.update_one(database_id=request.session.get("database"), collection="ONU-CFG",
                                                    query={"_id": onu_id}, update_document=update_document)
        if update_result.matched_count == 0:
            response = PonManagerApiResponse(status=status.HTTP_404_NOT_FOUND,
                                             details={"message": f"ONU {onu_id}'s configuration was not found"})
        else:
            response = PonManagerApiResponse(status=status.HTTP_200_OK, new_data=update_document, old_data={})

        return response
