"""
#--------------------------------------------------------------------------#
# Copyright (c) 2025, Ciena Corporation                                    #
# All rights reserved.                                                     #
#                                                                          #
#     _______ _____ __    __ ___                                           #
#    / _ __(_) ___//  |  / // _ |                                          #
#   / /   / / /__ / /|| / // / ||                                          #
#  / /___/ / /__ / / ||/ // /__||                                          #
# /_____/_/_____/_/  |__//_/   ||                                          #
#                                                                          #
# Distributed as Ciena-Customer confidential.                              #
#                                                                          #
#--------------------------------------------------------------------------#
"""

import copy
import traceback
import pymongo.errors

from django.contrib.auth.mixins import LoginRequiredMixin
from django.utils.decorators import method_decorator
from django.contrib import messages
from rest_framework.exceptions import APIException
from rest_framework.fields import JSONField, ChoiceField, BooleanField
from drf_spectacular.utils import extend_schema, OpenApiParameter, OpenApiResponse
from drf_spectacular.types import OpenApiTypes
from rest_framework import status
from rest_framework.generics import GenericAPIView

from database_manager import database_manager
from utils.schema_helpers import ResponseExample
from utils.tools import get_nested_value, PonManagerApiResponse, validate_query_params, validate_data, \
    permission_required_any_of, load_mongo_query_parameter, validate_device_id, permission_required
from utils.serializers import schema, get_schema, get_unversioned_schema



# ==================================================
# ============== One OLT State View ================
# ==================================================
class OneState(LoginRequiredMixin, GenericAPIView):
    raise_exception = True
    queryset = ''

    swaggerSchema = get_schema('OLT-STATE')

    @extend_schema(
        operation_id="get_one_olt_state",
        responses={
            200: OpenApiResponse(response=schema(swaggerSchema),
                                 description='OK'),
        },
        tags=['olt state'],
        summary="Get the state for the specified OLT",
        description=" "
    )
    @method_decorator(permission_required('can_read_network_olts', raise_exception=True))
    def get(self, request, olt_id, version):
        """Get the state for the specified OLT"""
        # Verify validity of olt_id
        if not validate_device_id(olt_id):
            response = PonManagerApiResponse(status=status.HTTP_400_BAD_REQUEST, details="Invalid OLT ID format")
        else:
            res_data = database_manager.find_one(database_id=request.session.get('database'), collection="OLT-STATE",
                                                 query={"_id": olt_id})
            if res_data:
                # Cannot add to projection here (after R2.3.0) because there cannot be a mix of 0's and 1's in the mongo projection except for "_id"
                if "_internal" in res_data:
                    del res_data["_internal"]
                response = PonManagerApiResponse(status=status.HTTP_200_OK, data=res_data)
            else:
                response = PonManagerApiResponse(status=status.HTTP_404_NOT_FOUND, details={"message":"OLT ID "+str(olt_id)+" has no state document"})

        return response

    @extend_schema(
        operation_id="delete_one_olt_state",
        responses=None,
        tags=['olt state'],
        summary="Delete the state of the specified OLT",
        description=" "
    )
    @method_decorator(permission_required('can_delete_network_olts', raise_exception=True))
    def delete(self, request, olt_id, version):
        """Delete the state of the specified OLT"""
        database_manager.delete_one(database_id=request.session.get('database'), collection="OLT-STATE",
                                    query={"_id": olt_id})

        return PonManagerApiResponse(status=status.HTTP_204_NO_CONTENT)


# ==================================================
# =============== OLT States View ==================
# ==================================================
class States(LoginRequiredMixin, GenericAPIView):
    raise_exception = True
    queryset = ''

    swaggerSchema = get_schema('OLT-STATE')

    @extend_schema(
        operation_id="get_olt_states",
        responses={
            200: OpenApiResponse(response=schema(swaggerSchema),
                                 description='OK'),
        },
        tags=['olt state'],
        summary="Get the states for all OLTs",
        description=" "
    )
    @method_decorator(permission_required('can_read_network_olts', raise_exception=True))
    @validate_query_params(collection="OLT-STATE")
    def get(self, request, query, projection, sort, limit, skip, next, distinct, version):
        """Get the states for all OLTs"""
        if distinct:
            res_data = database_manager.distinct(database_id=request.session.get('database'), collection="OLT-STATE",
                                                 query=query, distinct=distinct)
        # Filter out '_internal' field
        else:
            if projection:
                keys = list(projection.keys())
                if '_id' not in keys:
                    if any(projection[key] == 0 for key in keys):
                        projection['_internal'] = 0
                else:
                    keys.remove('_id')
                    if len(keys) == 0 and projection['_id'] == 0:
                        projection['_internal'] = 0
                    elif any(projection[key] == 0 for key in keys):
                        projection['_internal'] = 0
            else:
                projection = {'_internal': 0}

            res_data = database_manager.find(database_id=request.session.get('database'), collection="OLT-STATE",
                                             query=query, projection=projection, sort=sort, limit=limit, skip=skip,
                                             next=next)

        return PonManagerApiResponse(status=status.HTTP_200_OK, data=res_data)


# ==================================================
# ========== One OLT Configuration View ============
# ==================================================
class OneConfiguration(LoginRequiredMixin, GenericAPIView):
    raise_exception = True
    queryset = ''

    swaggerSchema = get_schema('OLT-CFG')

    @extend_schema(
        operation_id="get_one_olt_config",
        responses={
            200: OpenApiResponse(response=schema(swaggerSchema),
                                 description='OK'),
        },
        tags=['olt config'],
        summary="Get the config for the specified OLT",
        description=" "
    )
    @method_decorator(permission_required('can_read_network_olts', raise_exception=True))
    def get(self, request, olt_id, version):
        """Get the config for the specified OLT"""
        # Verify validity of olt_id
        if not validate_device_id(olt_id):
            response = PonManagerApiResponse(status=status.HTTP_400_BAD_REQUEST, details={"message":"Invalid OLT ID format"})
        else:
            res_data = database_manager.find_one(database_id=request.session.get('database'), collection="OLT-CFG", query={"_id": olt_id})
            if res_data:
                response = PonManagerApiResponse(status=status.HTTP_200_OK, data=res_data)
            else:
                response = PonManagerApiResponse(status=status.HTTP_404_NOT_FOUND, details={"message":"OLT ID "+str(olt_id)+" has no configuration document"})


        return response

    @extend_schema(
        operation_id="put_one_olt_config",
        request={
            "application/json": schema(swaggerSchema),
        },
        responses={
            201: OpenApiResponse(response=schema(swaggerSchema),
                                 description='Created'),
        },
        tags=['olt config'],
        summary="Update the config for the specified OLT",
        description=" "
    )
    @method_decorator(permission_required_any_of(['can_update_network_olts', 'can_create_network_olts'],
                                                 raise_exception=True))
    @validate_data(collection="OLT-CFG", resource_id_param="olt_id")
    def put(self, request, data, olt_id, version):
        """Update the config for the specified OLT"""
        data['OLT']['CFG Change Count'] += 1
        old_document = database_manager.find_one_and_replace(database_id=request.session.get('database'),
                                                             collection="OLT-CFG", query={"_id": olt_id},
                                                             new_document=data)
        if old_document is None:
            status_code = status.HTTP_201_CREATED
        else:
            status_code = status.HTTP_200_OK
        return PonManagerApiResponse(status=status_code, new_data=data, old_data=old_document)

    @extend_schema(
        operation_id="delete_one_olt_config",
        responses=None,
        tags=['olt config'],
        summary="Delete the config of the specified OLT",
        description=" "
    )
    @method_decorator(permission_required('can_delete_network_olts', raise_exception=True))
    def delete(self, request, olt_id, version):
        """Delete the config of the specified OLT"""
        database_manager.delete_one(database_id=request.session.get('database'), collection="OLT-CFG",
                                    query={"_id": olt_id})

        return PonManagerApiResponse(status=status.HTTP_204_NO_CONTENT)


# ==================================================
# =========== OLT Configurations View ==============
# ==================================================
class Configurations(LoginRequiredMixin, GenericAPIView):
    raise_exception = True
    queryset = ''

    swaggerSchema = get_schema('OLT-CFG')

    @extend_schema(
        operation_id="get_olt_configs",
        responses={
            200: OpenApiResponse(response=schema(swaggerSchema),
                                 description='OK'),
        },
        tags=['olt config'],
        summary="Get the configs for all OLTs",
        description=" "
    )
    @method_decorator(permission_required('can_read_network_olts', raise_exception=True))
    @validate_query_params(collection="OLT-CFG")
    def get(self, request, query, projection, sort, limit, skip, next, distinct, version):
        """Get the configs for all OLTs"""
        if distinct:
            res_data = database_manager.distinct(database_id=request.session.get('database'), collection="OLT-CFG",
                                                 query=query, distinct=distinct)
        else:
            res_data = database_manager.find(database_id=request.session.get('database'), collection="OLT-CFG",
                                             query=query, projection=projection, sort=sort, limit=limit, skip=skip,
                                             next=next)
        return PonManagerApiResponse(status=status.HTTP_200_OK, data=res_data)

    @extend_schema(
        operation_id="post_olt_config",
        request={
            "application/json": schema(swaggerSchema),
        },
        responses={
            201: OpenApiResponse(response=schema(swaggerSchema),
                                 description='Created'),
        },
        tags=['olt config'],
        summary="Create the provided OLT config",
        description=" "
    )
    @method_decorator(permission_required('can_create_network_olts', raise_exception=True))
    @validate_data(collection="OLT-CFG", resource_id_param=None)
    def post(self, request, data, version):
        """Create the provided OLT config"""
        try:
            database_manager.insert_one(database_id=request.session.get('database'), collection="OLT-CFG",
                                        document=data)
            response = PonManagerApiResponse(status=status.HTTP_201_CREATED, new_data=data, old_data=None)
        except pymongo.errors.DuplicateKeyError:
            olt_id = get_nested_value(data, ["_id"], None)
            response = PonManagerApiResponse(status=status.HTTP_409_CONFLICT,
                                             details={"message":f"OLT configuration with id {olt_id} already exists"})

        return response


# ==================================================
# ================ OLT Debug View ==================
# ==================================================
class Debug(LoginRequiredMixin, GenericAPIView):
    raise_exception = True
    queryset = ''

    # TODO figure out what collection this is
    swaggerSchema = get_schema('OLT-STATE')

    @extend_schema(
        operation_id="get_olt_debug",
        parameters=[
            OpenApiParameter(name="limit", description="Maximum Number of Debug Dumps to Return",
                             type=OpenApiTypes.NUMBER),
            OpenApiParameter(name="projection", description="Fields to Return.", type=OpenApiTypes.STR),
            OpenApiParameter(name="sort", description="Fields to sort in descending or ascending.",
                             type=OpenApiTypes.STR),
            OpenApiParameter(name="skip", description="Number of items to be skipped at beginning of response",
                             type=OpenApiTypes.NUMBER),
            OpenApiParameter(name="time-start", description="UTC timestamp to begin getting stats at",
                             type=OpenApiTypes.DATETIME, required=True),
            OpenApiParameter(name="time-end", description="UTC timestamp to stop getting stats at",
                             type=OpenApiTypes.DATETIME)
        ],
        responses={
            200: OpenApiResponse(response=schema(swaggerSchema),
                                 description='OK'),
        },
        tags=['olt debug'],
        summary="Get the debug data for an OLT",
        description=" "
    )
    @method_decorator(permission_required('can_read_network_olts', raise_exception=True))
    def get(self, request, olt_id, version):
        """Get the debug data for an OLT"""
        # Can't use validate_query_params and other decorators because using a query within the path
        # Must manually put query params into dictionaries

        start_time = request.GET.get('time-start', None)
        end_time = request.GET.get('time-end', None)
        limit = int(request.GET.get('limit', 0))
        skip = int(request.GET.get('skip', 0))
        projection = load_mongo_query_parameter(request.GET.get('projection'))
        sort_dict = load_mongo_query_parameter(request.GET.get('sort'))

        query = {"device ID": olt_id, "valid": True}
        if start_time is not None and end_time is None:
            query["Time"] = {"$gte": start_time}
        elif end_time is not None and start_time is None:
            query["Time"] = {"$lte": end_time}
        elif start_time is not None and end_time is not None:
            temp_dict = {"$lte": end_time, "$gte": start_time}
            query["Time"] = temp_dict

        try:
            sort = []
            if sort_dict:
                for key, value in sort_dict.items():
                    value = int(value)
                    sort_dict[key] = value
                    sort.append((key, value))

            res_data = database_manager.find(database_id=request.session.get('database'), collection="DEBUG-OLT",
                                             query=query, sort=sort, limit=limit,
                                             skip=skip, projection=projection)
            return PonManagerApiResponse(status=status.HTTP_200_OK, data=res_data)
        except Exception as err:
            messages.error(self.request, f"{type(err)}")
            messages.error(self.request, traceback.format_exc(), extra_tags='stacktrace')
            return PonManagerApiResponse(status=status.HTTP_500_INTERNAL_SERVER_ERROR)

    @extend_schema(
        operation_id="delete_olt_debug",
        responses=None,
        tags=['olt debug'],
        summary="Delete the debug data for an OLT",
        description=" "
    )
    @method_decorator(permission_required('can_delete_network_olts', raise_exception=True))
    def delete(self, request, olt_id, version):
        """Delete the debug data for an OLT"""
        try:
            database = database_manager.get_database(request.session.get('database'))
            collection = database.get_collection("DEBUG-OLT")
            collection.update_many({"device ID": olt_id}, {"$set": {"valid": False}})

        except (ConnectionRefusedError, pymongo.errors.PyMongoError) as e:
            raise APIException(detail={"message":f"MongoDB error: {str(e)}"})

        return PonManagerApiResponse(status=status.HTTP_204_NO_CONTENT)

# ==================================================
# ======= One OLT Alarm Configuration View =========
# ==================================================
class OneAlarmConfiguration(LoginRequiredMixin, GenericAPIView):
    raise_exception = True
    queryset = ''

    swaggerSchema = get_schema('OLT-ALARM-CFG')

    @extend_schema(
        operation_id="get_one_olt_alarm_config",
        responses={
            200: OpenApiResponse(response=schema(swaggerSchema),
                                 description='OK'),
        },
        tags=['olt alarm config'],
        summary="Get the specified OLT Alarm Config",
        description=" "
    )
    @method_decorator(permission_required('can_read_global_config_alarms', raise_exception=True))
    def get(self, request, cfg_id, version):
        """Get the specified OLT Alarm Config"""
        res_data = database_manager.find_one(database_id=request.session.get('database'), collection="OLT-ALARM-CFG",
                                             query={"_id": cfg_id})
        if res_data:
            response = PonManagerApiResponse(status=status.HTTP_200_OK, data=res_data)
        else:
            response = PonManagerApiResponse(status=status.HTTP_404_NOT_FOUND, details={"message":"CFG ID "+str(cfg_id)+" has no alarm configuration document"})

        return response

    @extend_schema(
        operation_id="put_one_olt_alarm_config",
        request={
            "application/json": schema(swaggerSchema),
        },
        responses={
            201: OpenApiResponse(response=schema(swaggerSchema),
                                 description='Created'),
        },
        tags=['olt alarm config'],
        summary="Update the config for the specified OLT Alarm Config",
        description=" "
    )
    @method_decorator(permission_required_any_of(
        ['can_update_global_config_alarms', 'can_create_global_config_alarms'],
        raise_exception=True))
    @validate_data(collection="OLT-ALARM-CFG", resource_id_param="cfg_id")
    def put(self, request, data, cfg_id, version):
        """Update the config for the specified OLT Alarm Config"""
        old_document = database_manager.find_one_and_replace(database_id=request.session.get('database'),
                                                             collection="OLT-ALARM-CFG", query={"_id": cfg_id},
                                                             new_document=data)
        if old_document is None:
            status_code = status.HTTP_201_CREATED
        else:
            status_code = status.HTTP_200_OK

        return PonManagerApiResponse(status=status_code, new_data=data, old_data=old_document)

    @extend_schema(
        operation_id="delete_one_olt_alarm_config",
        responses=None,
        tags=['olt alarm config'],
        summary="Delete the specified OLT Alarm Config",
        description=" "
    )
    @method_decorator(permission_required('can_delete_global_config_alarms', raise_exception=True))
    def delete(self, request, cfg_id, version):
        """Delete the specified OLT Alarm Config"""
        database_manager.delete_one(database_id=request.session.get('database'), collection="OLT-ALARM-CFG",
                                    query={"_id": cfg_id})

        return PonManagerApiResponse(status=status.HTTP_204_NO_CONTENT)


# ==================================================
# ======== OLT Alarm Configurations View ===========
# ==================================================
class AlarmConfigurations(LoginRequiredMixin, GenericAPIView):
    raise_exception = True
    queryset = ''

    swaggerSchema = get_schema('OLT-ALARM-CFG')

    @extend_schema(
        operation_id="get_olt_alarm_configs",
        responses={
            200: OpenApiResponse(response=schema(swaggerSchema),
                                 description='OK'),
        },
        tags=['olt alarm config'],
        summary="Get all OLT Alarm Configs",
        description=" "
    )
    @method_decorator(permission_required('can_read_global_config_alarms', raise_exception=True))
    @validate_query_params(collection="OLT-ALARM-CFG")
    def get(self, request, query, projection, sort, limit, skip, next, distinct, version):
        """Get all OLT Alarm Configs"""
        if distinct:
            res_data = database_manager.distinct(database_id=request.session.get('database'),
                                                 collection="OLT-ALARM-CFG",
                                                 query=query, distinct=distinct)
        else:
            res_data = database_manager.find(database_id=request.session.get('database'), collection="OLT-ALARM-CFG",
                                             query=query, projection=projection, sort=sort, limit=limit, skip=skip,
                                             next=next)

        return PonManagerApiResponse(status=status.HTTP_200_OK, data=res_data)

    @extend_schema(
        operation_id="post_olt_alarm_config",
        request={
            "application/json": schema(swaggerSchema),
        },
        responses={
            201: OpenApiResponse(response=schema(swaggerSchema),
                                 description='Created'),
        },
        tags=['olt alarm config'],
        summary="Create the provided OLT Alarm Config",
        description=" "
    )
    @method_decorator(permission_required('can_create_global_config_alarms', raise_exception=True))
    @validate_data(collection="OLT-ALARM-CFG", resource_id_param=None)
    def post(self, request, data, version):
        """Create the provided OLT Alarm Config"""
        try:
            database_manager.insert_one(database_id=request.session.get('database'), collection="OLT-ALARM-CFG",
                                        document=data)
            response = PonManagerApiResponse(status=status.HTTP_201_CREATED, new_data=data, old_data=None)
        except pymongo.errors.DuplicateKeyError:
            doc_id = get_nested_value(data, ["_id"], None)
            response = PonManagerApiResponse(status=status.HTTP_409_CONFLICT,
                                             details={"message":f"OLT alarm configuration with id {doc_id} already exists"})

        return response


# ==================================================
# ======= One OLT Alarm History State View =========
# ==================================================
class OneAlarmHistoryState(LoginRequiredMixin, GenericAPIView):
    raise_exception = True
    queryset = ''

    swaggerSchema = get_schema('OLT-ALARM-HIST-STATE')

    @extend_schema(
        operation_id="get_one_olt_alarm_history",
        responses={
            200: OpenApiResponse(response=schema(swaggerSchema),
                                 description='OK'),
        },
        tags=['olt alarm history'],
        summary="Get the specified OLT Alarm History State",
        description=" "
    )
    @method_decorator(permission_required('can_read_network_olts', raise_exception=True))
    def get(self, request, olt_id, version):
        """Get the specified OLT Alarm History State"""
        # Verify validity of olt_id
        if not validate_device_id(olt_id):
            response = PonManagerApiResponse(status=status.HTTP_400_BAD_REQUEST, details={"message":"Invalid OLT ID format"})
        else:
            res_data = database_manager.find_one(database_id=request.session.get('database'), collection="OLT-ALARM-HIST-STATE", query={"_id": olt_id})
            if res_data:
                response = PonManagerApiResponse(status=status.HTTP_200_OK, data=res_data)
            else:
                response = PonManagerApiResponse(status=status.HTTP_404_NOT_FOUND, details={"message":"OLT ID "+str(olt_id)+" has no alarm history document"})

        return response

    @extend_schema(
        operation_id="delete_one_olt_alarm_history",
        responses=None,
        tags=['olt alarm history'],
        summary="Delete the specified OLT Alarm Config",
        description=" "
    )
    @method_decorator(permission_required('can_delete_network_olts', raise_exception=True))
    def delete(self, request, olt_id, version):
        """Delete the specified OLT Alarm Config"""
        database_manager.delete_one(database_id=request.session.get('database'), collection="OLT-ALARM-HIST-STATE",
                                    query={"_id": olt_id})

        return PonManagerApiResponse(status=status.HTTP_204_NO_CONTENT)


# ==================================================
# ======== OLT Alarm History States View ===========
# ==================================================
class AlarmHistoryStates(LoginRequiredMixin, GenericAPIView):
    raise_exception = True
    queryset = ''

    swaggerSchema = get_schema('OLT-ALARM-HIST-STATE')

    @extend_schema(
        operation_id="get_olt_alarm_history",
        responses={
            200: OpenApiResponse(response=schema(swaggerSchema),
                                 description='OK'),
        },
        tags=['olt alarm history'],
        summary="Get all OLT Alarm History States",
        description=" "
    )
    @method_decorator(permission_required('can_read_network_olts', raise_exception=True))
    @validate_query_params(collection="OLT-ALARM-HIST-STATE")
    def get(self, request, query, projection, sort, limit, skip, next, distinct, version):
        """Get all OLT Alarm History States"""
        if distinct:
            res_data = database_manager.distinct(database_id=request.session.get('database'),
                                                 collection="OLT-ALARM-HIST-STATE",
                                                 query=query, distinct=distinct)
        else:
            res_data = database_manager.find(database_id=request.session.get('database'),
                                             collection="OLT-ALARM-HIST-STATE",
                                             query=query, projection=projection, sort=sort, limit=limit, skip=skip,
                                             next=next)

        return PonManagerApiResponse(status=status.HTTP_200_OK, data=res_data)


# ==================================================
# ============ One OLT Statistics View =============
# ==================================================
class Statistics(LoginRequiredMixin, GenericAPIView):
    raise_exception = True
    queryset = ''

    swaggerSchema = get_schema('OLT-STATE')

    @extend_schema(
        operation_id="get_one_olt_stats",
        parameters=[
            OpenApiParameter(name="time-start", description="UTC timestamp to begin getting stats at",
                             type=OpenApiTypes.DATETIME, required=True),
            OpenApiParameter(name="time-end", description="UTC timestamp to stop getting stats at",
                             type=OpenApiTypes.DATETIME)
        ],
        responses={
            200: OpenApiResponse(response=schema(swaggerSchema),
                                 description='OK'),
        },
        tags=['olt stats'],
        summary="Get the statistics of the specified OLT between the start and end times",
        description=" "
    )
    @method_decorator(permission_required('can_read_network_olts', raise_exception=True))
    def get(self, request, olt_id, version):
        """Get the statistics of the specified OLT between the start and end times"""
        start_time = request.GET.get('start-time', None)
        end_time = request.GET.get('end-time', None)

        # Return missing parameter response if start time is undefined
        if start_time is None:
            response = PonManagerApiResponse(status=status.HTTP_400_BAD_REQUEST,
                                             details={"message": "Parameter 'start-time' is required"})
        # Verify validity of olt_id
        elif not validate_device_id(olt_id):
            response = PonManagerApiResponse(status=status.HTTP_400_BAD_REQUEST, details={"message":"Invalid OLT ID format"})
        else:
            database = database_manager.get_database(request.session.get('database'))
            try:
                state_data = database_manager.find_one(database_id=request.session.get('database'),
                                                       collection="OLT-STATE",
                                                       query={"_id": olt_id}, projection={"_id": 0, "CNTL.Version": 1})

                # If no state is found OLT is likely pre-provisioned, won't have stats
                if state_data is None:
                    return PonManagerApiResponse(status=status.HTTP_404_NOT_FOUND,
                                                 details={"message": "OLT ID " + str(
                                                     olt_id) + " has no statistics document"})
                cntl_version = get_nested_value(state_data, ["CNTL", "Version"], "")
                sub_three_one_version = False

                if float(cntl_version[1:4]) < 3.1:
                    sub_three_one_version = True

                if sub_three_one_version:
                    collection = database.get_collection("STATS-OLT-{}".format(olt_id.replace(":", "")))
                    if end_time is None:
                        res_data = list(collection.find({"_id": {"$gte": start_time}}).limit(10000))
                    else:
                        res_data = list(collection.find({"_id": {"$gte": start_time, "$lte": end_time}}).limit(10000))
                else:
                    # For new versions of the DB
                    collection = database.get_collection("STATS-OLT")
                    if end_time is None:
                        res_data = list(collection.find({
                            "$and": [
                                {"device ID": olt_id},
                                {"valid": True},
                                {"_id": {"$gte": start_time}},
                            ]
                        }).limit(10000))
                    else:
                        res_data = list(collection.find({
                            "$and": [
                                {"device ID": olt_id},
                                {"valid": True},
                                {"_id": {"$gte": start_time, "$lte": end_time}}
                            ]
                        }).limit(10000))

            except (ConnectionRefusedError, pymongo.errors.PyMongoError) as e:
                raise APIException(detail=f"MongoDB error: {str(e)}")
            if res_data:
                # Add OLT ID to response format for easier handling in UI
                for block in res_data:
                    block['mac_address'] = olt_id
                response = PonManagerApiResponse(status=status.HTTP_200_OK, data=res_data)
            else:
                response = PonManagerApiResponse(status=status.HTTP_404_NOT_FOUND, details={"message":"OLT ID "+str(olt_id)+" has no statistics document"})

        return response

    @extend_schema(
        operation_id="delete_one_olt_stats",
        responses=None,
        tags=['olt stats'],
        summary="Delete the Statistics of the specified OLT",
        description=" "
    )
    @method_decorator(permission_required('can_delete_network_olts', raise_exception=True))
    def delete(self, request, olt_id, version):
        """Delete the Statistics of the specified OLT"""
        database = database_manager.get_database(request.session.get('database'))

        try:
            state_data = database_manager.find_one(database_id=request.session.get('database'), collection="OLT-STATE",
                                                   query={"_id": olt_id}, projection={"_id": 0, "CNTL.Version": 1})

            # If no state is found OLT is likely pre-provisioned, won't have stats
            if state_data is None:
                return PonManagerApiResponse(status=status.HTTP_404_NOT_FOUND,
                                             details={
                                                 "message": "OLT ID " + str(olt_id) + " has no statistics document"})
            cntl_version = get_nested_value(state_data, ["CNTL", "Version"], "")
            sub_three_one_version = False

            if float(cntl_version[1:4]) < 3.1:
                sub_three_one_version = True

            if sub_three_one_version:
                collection = database.get_collection("STATS-OLT-{}".format(olt_id.replace(":", "")))
                collection.drop()
            else:
                # For new versions of the DB
                collection = database.get_collection("STATS-OLT")
                collection.update_many({"device ID": olt_id}, {"$set": {"valid": False}})

        except (ConnectionRefusedError, pymongo.errors.PyMongoError) as e:
            raise APIException(detail=f"MongoDB error: {str(e)}")

        return PonManagerApiResponse(status=status.HTTP_204_NO_CONTENT)


# ==================================================
# ============ One OLT Logs View =============
# ==================================================
class Logs(LoginRequiredMixin, GenericAPIView):
    raise_exception = True
    queryset = ''

    swaggerSchema = get_schema('OLT-STATE')

    @extend_schema(
        operation_id="get_one_olt_logs",
        parameters=[
            OpenApiParameter(name="time-start", description="UTC timestamp to begin getting stats at",
                             type=OpenApiTypes.DATETIME, required=True),
            OpenApiParameter(name="time-end", description="UTC timestamp to stop getting stats at",
                             type=OpenApiTypes.DATETIME)
        ],
        responses={
            200: OpenApiResponse(response=schema(swaggerSchema),
                                 description='OK'),
        },
        tags=['olt stats'],
        summary="Get the logs of the specified OLT between the start and end times",
        description=" "
    )
    @method_decorator(permission_required('can_read_network_olts', raise_exception=True))
    def get(self, request, olt_id, version):
        """Get the logs of the specified OLT between the start and end times"""
        start_time = request.GET.get('start-time', None)
        end_time = request.GET.get('end-time', None)

        # Return missing parameter response if start time is undefined
        if start_time is None:
            response = PonManagerApiResponse(status=status.HTTP_400_BAD_REQUEST,
                                             details={"message": "Parameter 'start-time' is required"})
        # Verify validity of olt_id
        elif not validate_device_id(olt_id):
            response = PonManagerApiResponse(status=status.HTTP_400_BAD_REQUEST, details={"message":"Invalid OLT ID format"})
        else:
            database = database_manager.get_database(request.session.get('database'))

            try:
                state_data = database_manager.find_one(database_id=request.session.get('database'),
                                                       collection="OLT-STATE",
                                                       query={"_id": olt_id}, projection={"_id": 0, "CNTL.Version": 1})

                # If no state is found OLT is likely pre-provisioned, won't have stats
                if state_data is None:
                    return PonManagerApiResponse(status=status.HTTP_404_NOT_FOUND,
                                                 details={"message": "OLT ID " + str(
                                                     olt_id) + " has no logs document"})
                cntl_version = get_nested_value(state_data, ["CNTL", "Version"], "")
                sub_three_one_version = False

                if float(cntl_version[1:4]) < 3.1:
                    sub_three_one_version = True

                if sub_three_one_version:
                    collection = database.get_collection("SYSLOG-OLT-{}".format(olt_id.replace(":", "")))
                    if end_time is None:
                        res_data = list(collection.find({"time": {"$gte": start_time}},
                                                        {"_id": 0, "device ID": 0}))
                    else:
                        res_data = list(collection.find({"time": {"$gte": start_time, "$lte": end_time}},
                                                        {"_id": 0, "device ID": 0}))
                else:
                    collection = database.get_collection("SYSLOG-OLT")
                    if end_time is None:
                        res_data = list(collection.find({
                            "$and": [
                                {"device ID": olt_id},
                                {"valid": True},
                                {"time": {"$gte": start_time}},
                            ]
                        }, {"_id": 0, "device ID": 0}).limit(10000))
                    else:
                        res_data = list(collection.find({
                            "$and": [
                                {"device ID": olt_id},
                                {"valid": True},
                                {"time": {"$gte": start_time, "$lte": end_time}}
                            ]
                        }, {"_id": 0, "device ID": 0}).limit(10000))

            except (ConnectionRefusedError, pymongo.errors.PyMongoError) as e:
                raise APIException(detail=f"MongoDB error: {str(e)}")
            if res_data:
                response = PonManagerApiResponse(status=status.HTTP_200_OK, data=res_data)
            else:
                response = PonManagerApiResponse(status=status.HTTP_404_NOT_FOUND, details={"message":"OLT ID "+str(olt_id)+" has no logs document"})

        return response

    @extend_schema(
        operation_id="delete_one_olt_logs",
        responses=None,
        tags=['olt logs'],
        summary="Delete the Logs of the specified OLT",
        description=" "
    )
    @method_decorator(permission_required('can_delete_network_olts', raise_exception=True))
    def delete(self, request, olt_id, version):
        """Delete the Logs of the specified OLT"""
        database = database_manager.get_database(request.session.get('database'))

        try:
            state_data = database_manager.find_one(database_id=request.session.get('database'), collection="OLT-STATE",
                                                   query={"_id": olt_id}, projection={"_id": 0, "CNTL.Version": 1})

            # If no state is found OLT is likely pre-provisioned, won't have stats
            if state_data is None:
                return PonManagerApiResponse(status=status.HTTP_404_NOT_FOUND,
                                             details={
                                                 "message": "OLT ID " + str(olt_id) + " has no logs document"})
            cntl_version = get_nested_value(state_data, ["CNTL", "Version"], "")
            sub_three_one_version = False

            if float(cntl_version[1:4]) < 3.1:
                sub_three_one_version = True

            if sub_three_one_version:
                collection = database.get_collection("SYSLOG-OLT-{}".format(olt_id.replace(":", "")))
                collection.drop()
            else:
                # For new versions of the DB
                collection = database.get_collection("SYSLOG-OLT")
                collection.update_many({"device ID": olt_id}, {"$set": {"valid": False}})

        except (ConnectionRefusedError, pymongo.errors.PyMongoError) as e:
            raise APIException(detail=f"MongoDB error: {str(e)}")

        return PonManagerApiResponse(status=status.HTTP_204_NO_CONTENT)


# ==================================================
# ========= One OLT Automation State View ==========
# ==================================================
class OneAutomationState(LoginRequiredMixin, GenericAPIView):
    raise_exception = True
    queryset = ''

    swaggerSchema = get_schema('OLT-AUTO-STATE')

    @extend_schema(
        operation_id="get_one_olt_automation_state",
        responses={
            200: OpenApiResponse(response=schema(swaggerSchema),
                                 description='OK'),
        },
        tags=['olt automation state'],
        summary="Get the Automation State of the specified OLT",
        description=" "
    )
    @method_decorator(permission_required('can_read_network_olts', raise_exception=True))
    def get(self, request, olt_id, version):
        """Get the Automation State of the specified OLT"""
        # Verify validity of olt_id
        if not validate_device_id(olt_id):
            response = PonManagerApiResponse(status=status.HTTP_400_BAD_REQUEST, details="Invalid OLT ID format")
        else:
            res_data = database_manager.find_one(database_id=request.session.get('database'), collection="OLT-AUTO-STATE", query={"_id": olt_id})
            if res_data:
                response = PonManagerApiResponse(status=status.HTTP_200_OK, data=res_data)
            else:
                response = PonManagerApiResponse(status=status.HTTP_404_NOT_FOUND, details={"message":"OLT ID "+str(olt_id)+" has no automation state document"})

        return response

    @extend_schema(
        operation_id="delete_one_olt_automation_state",
        responses=None,
        tags=['olt automation state'],
        summary="Delete the specified OLT Automation State",
        description=" "
    )
    @method_decorator(permission_required('can_delete_network_olts', raise_exception=True))
    def delete(self, request, olt_id, version):
        """Delete the specified OLT Automation State"""
        database_manager.delete_one(database_id=request.session.get('database'), collection="OLT-AUTO-STATE",
                                    query={"_id": olt_id})

        return PonManagerApiResponse(status=status.HTTP_204_NO_CONTENT)


# ==================================================
# =========== OLT Automation States View ===========
# ==================================================
class AutomationStates(LoginRequiredMixin, GenericAPIView):
    raise_exception = True
    queryset = ''

    swaggerSchema = get_schema('OLT-AUTO-STATE')

    @extend_schema(
        operation_id="get_olt_automation_states",
        responses={
            200: OpenApiResponse(response=schema(swaggerSchema),
                                 description='OK'),
        },
        tags=['olt automation state'],
        summary="Get the Automation States of all OLTs",
        description=" "
    )
    @method_decorator(permission_required('can_read_network_olts', raise_exception=True))
    # TODO @validate_query_params(collection="OLT-AUTO-STATE")
    def get(self, request, version):
        """Get the Automation States of all OLTs"""
        res_data = database_manager.find(database_id=request.session.get('database'), collection="OLT-AUTO-STATE")

        return PonManagerApiResponse(status=status.HTTP_200_OK, data=res_data)


# ==================================================
# ========= One OLT Automation Config View ==========
# ==================================================
class OneAutomationConfig(LoginRequiredMixin, GenericAPIView):
    raise_exception = True
    queryset = ''

    swaggerSchema = get_schema('OLT-AUTO-CFG')

    @extend_schema(
        operation_id="get_one_olt_automation_config",
        responses={
            200: OpenApiResponse(response=schema(swaggerSchema),
                                 description='OK'),
        },
        tags=['olt automation config'],
        summary="Get the specified OLT Automation Config",
        description=" "
    )
    @method_decorator(permission_required('can_read_network_olts', raise_exception=True))
    def get(self, request, cfg_id, version):
        """Get the specified OLT Automation Config"""
        res_data = database_manager.find_one(database_id=request.session.get('database'), collection="OLT-AUTO-CFG", query={"_id": cfg_id})
        if res_data:
            response = PonManagerApiResponse(status=status.HTTP_200_OK, data=res_data)
        else:
            response = PonManagerApiResponse(status=status.HTTP_404_NOT_FOUND, details={"message":"CFG ID "+str(cfg_id)+" has no automation configuration document"})

        return response

    @extend_schema(
        operation_id="put_one_olt_automation_config",
        request={
            "application/json": schema(swaggerSchema),
        },
        responses={
            201: OpenApiResponse(response=schema(swaggerSchema),
                                 description='Created'),
        },
        tags=['olt automation config'],
        summary="Update the config for the specified OLT Automation Config",
        description=" "
    )
    @method_decorator(permission_required_any_of(['can_update_network_olts', 'can_create_network_olts'],
                                                 raise_exception=True))
    @validate_data(collection="OLT-AUTO-CFG", resource_id_param="cfg_id")
    def put(self, request, data, cfg_id, version):
        """Update the config for the specified OLT Automation Config"""
        data['_id'] = cfg_id
        old_document = database_manager.find_one_and_replace(database_id=request.session.get('database'),
                                                             collection="OLT-AUTO-CFG", query={"_id": cfg_id},
                                                             new_document=data)

        if old_document is None:
            response = PonManagerApiResponse(status=status.HTTP_201_CREATED, new_data=data)
        else:
            response = PonManagerApiResponse(status=status.HTTP_200_OK, new_data=data, old_data=old_document)

        return response

    @extend_schema(
        operation_id="delete_one_olt_automation_config",
        responses=None,
        tags=['olt automation config'],
        summary="Delete the specified OLT Automation Config",
        description=" "
    )
    @method_decorator(permission_required('can_delete_network_olts', raise_exception=True))
    def delete(self, request, cfg_id, version):
        """Delete the specified OLT Automation Config"""
        database_manager.delete_one(database_id=request.session.get('database'), collection="OLT-AUTO-CFG",
                                    query={"_id": cfg_id})

        return PonManagerApiResponse(status=status.HTTP_204_NO_CONTENT)


# ==================================================
# ========== OLT Automation Configs View ===========
# ==================================================
class AutomationConfigs(LoginRequiredMixin, GenericAPIView):
    raise_exception = True
    queryset = ''

    swaggerSchema = get_schema('OLT-AUTO-CFG')

    @extend_schema(
        # parameters=
        operation_id="get_olt_automation_configs",
        responses={
            200: OpenApiResponse(response=schema(swaggerSchema),
                                 description='OK'),
        },
        tags=['olt automation config'],
        summary="Get the Automation Configs of all OLTs",
        description=" "
    )
    @method_decorator(permission_required('can_read_network_olts', raise_exception=True))
    # TODO @validate_query_params(collection="OLT-AUTO-CFG")
    def get(self, request, version):
        """Get the Automation Configs of all OLTs"""
        res_data = database_manager.find(database_id=request.session.get('database'), collection="OLT-AUTO-CFG")

        return PonManagerApiResponse(status=status.HTTP_200_OK, data=res_data)

    @extend_schema(
        operation_id="post_olt_automation_config",
        request={
            "application/json": schema(swaggerSchema),
        },
        responses={
            201: OpenApiResponse(response=schema(swaggerSchema),
                                 description='Created'),
        },
        tags=['olt automation config'],
        summary="Create the provided OLT Automation Config",
        description=" "
    )
    @method_decorator(permission_required('can_create_network_olts', raise_exception=True))
    # TODO @validate_data(collection="OLT-AUTO-CFG", resource_id_param=None)
    def post(self, request, version):
        """Create the provided OLT Automation Config"""
        try:
            data = get_nested_value(request.data, ["data"])
            database_manager.insert_one(database_id=request.session.get('database'), collection="OLT-AUTO-CFG",
                                        document=data)
            response = PonManagerApiResponse(status=status.HTTP_201_CREATED, new_data=data, old_data=None)
        except pymongo.errors.DuplicateKeyError:
            doc_id = get_nested_value(data, ["_id"], None)
            response = PonManagerApiResponse(status=status.HTTP_409_CONFLICT,
                                             details={"message":f"OLT Automation configuration with id {doc_id} already exists"})

        return response


# ==================================================
# ======= Global OLT Automation Config View ========
# ==================================================
class GlobalAutomationConfig(LoginRequiredMixin, GenericAPIView):
    raise_exception = True
    queryset = ''

    swaggerSchema = get_schema('OLT-AUTO-CFG')

    @extend_schema(
        operation_id="get_global_olt_automation_config",
        responses={
            200: OpenApiResponse(response=schema(swaggerSchema),
                                 description='OK'),
        },
        tags=['olt automation config'],
        summary="Get the Global OLT Automation Config",
        description=" "
    )
    @method_decorator(permission_required('can_read_automation', raise_exception=True))
    def get(self, request, version):
        """Get the Global OLT Automation Config"""
        res_data = database_manager.find_one(database_id=request.session.get('database'), collection="OLT-AUTO-CFG",
                                             query={"_id": "Global"})

        return PonManagerApiResponse(status=status.HTTP_200_OK, data=res_data)

    @extend_schema(
        operation_id="put_global_olt_automation_config",
        request={
            "application/json": schema(swaggerSchema),
        },
        responses={
            201: OpenApiResponse(response=schema(swaggerSchema),
                                 description='Created'),
        },
        tags=['olt automation config'],
        summary="Update the Global OLT Automation Config",
        description=" "
    )
    @method_decorator(
        permission_required_any_of(['can_update_automation', 'can_create_automation'],
                                   raise_exception=True))
    @validate_data(collection="OLT-AUTO-CFG", resource_id_param=None)
    def put(self, request, data, version):
        """Update the Global OLT Automation Config"""
        if data is None:
            response = PonManagerApiResponse(status=status.HTTP_400_BAD_REQUEST,
                                             details={"message":"Request body must be of format '{ data: <OLT-AUTO-CFG> }'"})
        else:
            old_document = database_manager.find_one_and_replace(database_id=request.session.get('database'),
                                                                 collection="OLT-AUTO-CFG", query={"_id": "Global"},
                                                                 new_document=data)

        if old_document is None:
            response = PonManagerApiResponse(status=status.HTTP_201_CREATED, new_data=data)
        else:
            response = PonManagerApiResponse(status=status.HTTP_200_OK, new_data=data, old_data=old_document)

        return response

    @extend_schema(
        operation_id="post_global_olt_automation_config",
        request={
            "application/json": schema(swaggerSchema),
        },
        responses={
            201: OpenApiResponse(response=schema(swaggerSchema),
                                 description='Created'),
        },
        tags=['olt automation config'],
        summary="Create the provided OLT Automation Global Config",
        description=" "
    )
    @method_decorator(permission_required('can_create_automation', raise_exception=True))
    # TODO @validate_data(collection="OLT-AUTO-CFG", resource_id_param=None)
    def post(self, request, version):
        """Create the provided OLT Automation Global Config"""
        try:
            data = get_nested_value(request.data, ["data"])
            database_manager.insert_one(database_id=request.session.get('database'), collection="OLT-AUTO-CFG",
                                        document=data)
            response = PonManagerApiResponse(status=status.HTTP_201_CREATED, new_data=data, old_data=None)
        except pymongo.errors.DuplicateKeyError:
            response = PonManagerApiResponse(status=status.HTTP_409_CONFLICT,
                                             details={"message":f"OLT Automation configuration with id Global already exists"})

        return response

    @extend_schema(
        operation_id="delete_global_olt_automation_config",
        parameters=[
            OpenApiParameter(name="step", description="Automation step to delete template from",
                             type=OpenApiTypes.STR, required=True),
            OpenApiParameter(name="name", description="Automation template to delete from the specified step",
                             type=OpenApiTypes.STR, required=True)
        ],
        responses=None,
        tags=['olt automation config'],
        summary="Delete the Global OLT Automation Config",
        description=" "
    )
    @method_decorator(permission_required('can_delete_automation', raise_exception=True))
    def delete(self, request, version):
        """Delete the Global OLT Automation Config"""
        step = request.GET.get('step', None)
        name = request.GET.get('name', None)

        # Return missing parameter response if step or name are undefined
        if step is None:
            response = PonManagerApiResponse(status=status.HTTP_400_BAD_REQUEST,
                                             details={"message": "Parameter 'step' is required"})
        elif name is None:
            response = PonManagerApiResponse(status=status.HTTP_400_BAD_REQUEST,
                                             details={"message": "Parameter 'name' is required"})
        else:
            database = database_manager.get_database(request.session.get('database'))
            collection = database.get_collection("OLT-AUTO-CFG")
            try:
                if step.upper() == "IDENTIFY":
                    collection.update_one({'_id': "Global"}, {"$pull": {"IDENTIFY.Mapping": {"Description": name}}})
                else:
                    collection.update_one({'_id': "Global"}, {"$unset": {f"{step.upper()}.{name}": ""}})
            except (ConnectionRefusedError, pymongo.errors.PyMongoError) as e:
                raise APIException(detail=f"MongoDB error: {str(e)}")

            response = PonManagerApiResponse(status=status.HTTP_204_NO_CONTENT)

        return response


# ==================================================
# ================ OLT Reset View ==================
# ==================================================
class Reset(LoginRequiredMixin, GenericAPIView):
    raise_exception = True
    queryset = ''

    swaggerSchema = get_schema('OLT-CFG')

    @extend_schema(
        operation_id="get_olt_reset_status",
        request=None,
        responses={
            200: OpenApiResponse(response=schema(swaggerSchema),
                                 description='OK'),
        },
        tags=['olt reset'],
        summary="Check the reset status of the specified OLT",
        description=" "
    )
    @method_decorator(permission_required('can_read_network_olts', raise_exception=True))
    def get(self, request, olt_id, version):
        """ Check the reset status of the specified OLT """
        # Verify validity of olt_id
        if not validate_device_id(olt_id):
            response = PonManagerApiResponse(status=status.HTTP_400_BAD_REQUEST, details={"message":"Invalid OLT ID format"})
        else:
            database = database_manager.get_database(request.session.get("database"))
            collection = database.get_collection("OLT-STATE")
            try:
                result = list(collection.aggregate([
                    {"$match": {"_id": olt_id}},
                    {"$lookup": {"from": "OLT-CFG", "localField": "_id", "foreignField": "_id", "as": "CFG"}},
                    {"$addFields": {"CFG": {"$arrayElemAt": ["$CFG", 0]}}},
                    {"$project": {"_id": 0, "OLT.Reset Count": 1, "CFG.OLT.Reset Count": 1}}
                ]))
            except (ConnectionRefusedError, pymongo.errors.PyMongoError) as e:
                raise APIException(detail=f"MongoDB error: {str(e)}")

            res_data = {
                "Pending": False
            }
            if len(result) > 0 and result[0]["OLT"] and result[0]["CFG"]:
                # Return pending status as false if State or Config are not found
                state_count = get_nested_value(result, path=[0, "OLT", "Reset Count"], default=0)
                cfg_count = get_nested_value(result, path=[0, "CFG", "OLT", "Reset Count"], default=0)
                res_data["Pending"] = state_count != cfg_count
            response = PonManagerApiResponse(status=status.HTTP_200_OK, data=res_data)

        return response

    @extend_schema(
        operation_id="reset_olt",
        request=None,
        responses={
            200: OpenApiResponse(response=schema(swaggerSchema),
                                 description='OK'),
        },
        tags=['olt reset'],
        summary="Trigger a Reset on the specified OLT",
        description=" "
    )
    @method_decorator(permission_required_any_of(['can_update_network_olts', 'can_create_network_olts'],
                                                 raise_exception=True))
    def put(self, request, olt_id, version):
        """ Trigger a Reset on the specified OLT """
        update_document = {"$inc": {"OLT.Reset Count": 1, "OLT.CFG Change Count": 1}}
        update_result = database_manager.update_one(database_id=request.session.get("database"), collection="OLT-CFG",
                                                    query={"_id": olt_id}, update_document=update_document)
        if update_result.matched_count == 0:
            response = PonManagerApiResponse(status=status.HTTP_404_NOT_FOUND,
                                             details={"message": f"OLT {olt_id}'s configuration was not found"})
        else:
            response = PonManagerApiResponse(status=status.HTTP_200_OK, new_data=update_document, old_data={})

        return response


# ==================================================
# == OLT Allow ONU Registration Schema Definition ==
# ==================================================
ALLOW_ONU_REG_SCHEMA = {
    "properties": {
        "onu-id": {
            "anyOf": [
                {"$ref": "COMMON-TYPES.json#/definitions/MAC Address"},
                {"$ref": "COMMON-TYPES.json#/definitions/ONU Serial Number"},
                {"const": "ALL"}
            ]
        }
    }
}


# ==================================================
# ======== OLT Allow ONU Registration View =========
# ==================================================
class AllowRegistration(LoginRequiredMixin, GenericAPIView):
    raise_exception = True
    queryset = ''

    swaggerSchema = get_schema('OLT-STATE')

    @extend_schema(
        operation_id="get_olt_allow_onu_registration",
        responses={
            200: OpenApiResponse(response=schema(swaggerSchema),
                                 description='OK'),
        },
        tags=['olt registration'],
        summary="Check if the specified ONU can be allowed to register",
        description=" "
    )
    @method_decorator(permission_required('can_read_network_olts', raise_exception=True))
    def get(self, request, olt_id, version):
        """ Check if the specified ONU can be allowed to register """
        # Verify validity of olt_id
        if not validate_device_id(olt_id):
            response = PonManagerApiResponse(status=status.HTTP_400_BAD_REQUEST, details={"message":"Invalid OLT ID format"})
        else:
            database = database_manager.get_database(request.session.get('database'))
            olt_state_collection = database.get_collection("OLT-STATE")
            try:
                cursor = list(olt_state_collection.aggregate([
                    {"$match": {"_id": olt_id}},
                    {"$lookup": {"from": "OLT-CFG", "localField": "_id", "foreignField": "_id", "as": "CFG"}},
                    {"$addFields": {"CFG": {"$arrayElemAt": ["$CFG", 0]}}},
                    {"$project": {"_id": 0, "OLT.Reg Allow ONU": 1, "OLT.Reg Allow Count": 1, "CFG.OLT.Reg Allow ONU": 1,
                                  "CFG.OLT.Reg Allow Count": 1}}
                ]))
            except (ConnectionRefusedError, pymongo.errors.PyMongoError) as e:
                raise APIException(detail=f"MongoDB error: {str(e)}")

            res_data = {
                "Pending": False
            }
            if len(cursor) > 0:
                for olt in cursor:
                    state_allow_onu = olt['OLT']['Reg Allow ONU']
                    config_allow_onu = olt['CFG']['OLT']['Reg Allow ONU']
                    state_allow_count = olt['OLT']['Reg Allow Count']
                    config_allow_count = olt['CFG']['OLT']['Reg Allow Count']

                    if config_allow_onu != state_allow_onu or config_allow_count != state_allow_count:
                        res_data["Pending"] = True
                        break
            response = PonManagerApiResponse(status=status.HTTP_200_OK, data=res_data)

        return response

    swaggerSchema = get_unversioned_schema('ALLOW-ONU-REG')

    @extend_schema(
        operation_id="olt_allow_onu_registration",
        request={
            "application/json": schema(swaggerSchema),
        },
        responses={
            201: OpenApiResponse(response=schema(swaggerSchema),
                                 description='Created'),
        },
        tags=['olt registration'],
        summary="Allow registration for the specified ONU",
        description=" "
    )
    @method_decorator(permission_required_any_of(['can_update_network_olts', 'can_create_network_olts'],
                                                 raise_exception=True))
    @validate_data(collection="OLT-CFG", resource_id_param=None, schema=ALLOW_ONU_REG_SCHEMA)
    def put(self, request, data, olt_id, version):
        """ Allow registration for the specified ONU """
        allow_onu = get_nested_value(data, ["onu-id"])
        if allow_onu is None:
            response = PonManagerApiResponse(status=status.HTTP_400_BAD_REQUEST, details={
                "message": "Request body must be of format '{ data: { onu-id: <ONU MAC Address/SSN> | 'ALL' } }'"})
        else:
            update_document = {"$set": {"OLT.Reg Allow ONU": allow_onu},
                               "$inc": {"OLT.Reg Allow Count": 1, "OLT.CFG Change Count": 1}}
            update_result = database_manager.update_one(database_id=request.session.get("database"),
                                                        collection="OLT-CFG",
                                                        query={"_id": olt_id}, update_document=update_document)
            if update_result.matched_count == 0:
                response = PonManagerApiResponse(status=status.HTTP_404_NOT_FOUND,
                                                 details={"message": f"OLT {olt_id}'s configuration was not found"})
            else:
                response = PonManagerApiResponse(status=status.HTTP_200_OK, new_data=update_document, old_data={})

        return response


# ==================================================
# ==== OLT Disable/Enable ONU Schema Definition ====
# ==================================================
DISABLE_ONU_SCHEMA = {
    "properties": {
        "disable": {
            "type": "boolean"
        }
    }
}


# ==================================================
# ========== OLT Disable/Enable ONU View ===========
# ==================================================
class DisableOnu(LoginRequiredMixin, GenericAPIView):
    raise_exception = True
    queryset = ''

    swaggerSchema = get_schema('OLT-STATE')

    @extend_schema(
        operation_id="get_olt_onu_disabled_status",
        responses={
            200: OpenApiResponse(response=schema(swaggerSchema),
                                 description='OK'),
        },
        tags=['olt disable onu'],
        summary="Get the disabled status for the ONU in the OLT's inventory",
        description=" "
    )
    def get(self, request, olt_id, onu_id, version):
        """ Get the disabled status for the ONU in the OLT's inventory """
        # Verify validity of olt_id
        if not validate_device_id(olt_id):
            response = PonManagerApiResponse(status=status.HTTP_400_BAD_REQUEST, details={"message":"Invalid OLT ID format"})
        else:
            database = database_manager.get_database(request.session.get('database'))
            olt_state_collection = database.get_collection("OLT-STATE")
            try:
                result = list(olt_state_collection.aggregate([
                    {"$match": {"_id": olt_id}},
                    {"$lookup": {"from": "OLT-CFG", "localField": "_id", "foreignField": "_id", "as": "CFG"}},
                    {"$addFields": {"CFG": {"$arrayElemAt": ["$CFG", 0]}}},
                    {"$project": {"_id": 0}}
                ]))
            except (ConnectionRefusedError, pymongo.errors.PyMongoError) as e:
                raise APIException(detail=f"MongoDB error: {str(e)}")

            res_data = {
                "Pending": False
            }
            # Return pending status as false if State or Config are not found
            if len(result) > 0 and result[0]["ONUs"] and result[0]["CFG"]:
                state_disable_count = get_nested_value(result, path=[0, "ONUs", onu_id, "Disable Count"], default=0)
                cfg_disable_count = get_nested_value(result, path=[0, "CFG", "ONUs", onu_id, "Disable Count"], default=0)
                state_enable_count = get_nested_value(result, path=[0, "ONUs", onu_id, "Enable Count"], default=0)
                cfg_enable_count = get_nested_value(result, path=[0, "CFG", "ONUs", onu_id, "Enable Count"], default=0)
                res_data["Pending"] = state_disable_count != cfg_disable_count or state_enable_count != cfg_enable_count
            response = PonManagerApiResponse(status=status.HTTP_200_OK, data=res_data)

        return response

    swaggerSchema = get_unversioned_schema('DISABLE-ONU')

    @extend_schema(
        operation_id="olt_disable_onu",
        request={
            "application/json": schema(swaggerSchema),
        },
        responses={
            201: OpenApiResponse(response=schema(swaggerSchema),
                                 description='Created'),
        },
        tags=['olt disable onu'],
        summary="Update the disable/enable signal for the ONU in the OLT's inventory",
        description=" "
    )
    @method_decorator(permission_required_any_of(['can_update_network_olts', 'can_create_network_olts'],
                                                 raise_exception=True))
    @validate_data(collection="OLT-CFG", resource_id_param=None, schema=DISABLE_ONU_SCHEMA)
    def put(self, request, data, olt_id, onu_id, version):
        """ Update the disable/enable signal for the ONU in the OLT's inventory """
        disable = get_nested_value(data, ["disable"])
        if disable is None:
            response = PonManagerApiResponse(status=status.HTTP_400_BAD_REQUEST, details={
                "message": "Request body must be of format '{ data: { disable: <bool> } }'"})
        else:
            if disable:
                update_document = {"$set": {f"ONUs.{onu_id}.Disable": True},
                                   "$inc": {f"ONUs.{onu_id}.Disable Count": 1, "OLT.CFG Change Count": 1}}
            else:
                update_document = {"$set": {f"ONUs.{onu_id}.Disable": False},
                                   "$inc": {f"ONUs.{onu_id}.Enable Count": 1, "OLT.CFG Change Count": 1}}

            update_result = database_manager.update_one(database_id=request.session.get("database"),
                                                        collection="OLT-CFG",
                                                        query={"_id": olt_id}, update_document=update_document)
            if update_result.matched_count == 0:
                response = PonManagerApiResponse(status=status.HTTP_404_NOT_FOUND,
                                                 details={"message": f"OLT {olt_id}'s configuration was not found"})
            else:
                response = PonManagerApiResponse(status=status.HTTP_200_OK, new_data=update_document, old_data={})

        return response


# ==================================================
# ========= OLT Broadcast Enable ONUs View =========
# ==================================================
class BroadcastEnableOnus(LoginRequiredMixin, GenericAPIView):
    raise_exception = True
    queryset = ''

    swaggerSchema = get_schema('OLT-STATE')

    @extend_schema(
        operation_id="get_olt_broadcast_enable_status",
        responses={
            200: OpenApiResponse(response=schema(swaggerSchema),
                                 description='OK'),
        },
        tags=['olt broadcast enable onu'],
        summary="Get the status of the OLTs broadcast enable signal",
        description=" "
    )
    def get(self, request, olt_id, version):
        """ Get the status of the OLTs broadcast enable signal """
        # Verify validity of olt_id
        if not validate_device_id(olt_id):
            response = PonManagerApiResponse(status=status.HTTP_400_BAD_REQUEST, details={"message":"Invalid OLT ID format"})
        else:
            database = database_manager.get_database(request.session.get('database'))
            olt_state_collection = database.get_collection("OLT-STATE")
            try:
                result = list(olt_state_collection.aggregate([
                    {"$match": {"_id": olt_id}},
                    {"$lookup": {"from": "OLT-CFG", "localField": "_id", "foreignField": "_id", "as": "CFG"}},
                    {"$addFields": {"CFG": {"$arrayElemAt": ["$CFG", 0]}}},
                    {"$project": {"_id": 0}}
                ]))
            except (ConnectionRefusedError, pymongo.errors.PyMongoError) as e:
                raise APIException(detail=f"MongoDB error: {str(e)}")

            res_data = {
                "Pending": False
            }
            if len(result) > 0 and result[0]["ONUs"] and result[0]["CFG"]:
                # Return pending status as false if State or Config are not found
                state_count = get_nested_value(result, path=[0, "ONU", "Enable All Serial Numbers Count"], default=0)
                cfg_count = get_nested_value(result, path=[0, "CFG", "ONU", "Enable All Serial Numbers Count"], default=0)
                res_data["Pending"] = state_count != cfg_count
            response = PonManagerApiResponse(status=status.HTTP_200_OK, data=res_data)

        return response

    @extend_schema(
        operation_id="olt_broadcast_enable_onus",
        request={
            "application/json": schema(swaggerSchema),
        },
        responses={
            201: OpenApiResponse(response=schema(swaggerSchema),
                                 description='Created'),
        },
        tags=['olt broadcast enable onu'],
        summary="Update the OLT's enable all ONUs count",
        description=" "
    )
    @method_decorator(permission_required_any_of(['can_update_network_olts', 'can_create_network_olts'],
                                                 raise_exception=True))
    def put(self, request, olt_id, version):
        """ - Update the OLT's enable all ONUs count
            - Update each ONU's Disable field to false
        """
        set_query = {}
        inventoried_onus = database_manager.find_one(database_id=request.session.get('database'), collection="OLT-CFG",
                                               query={"_id": olt_id}, projection={"_id": 0, "ONUs": 1})

        if inventoried_onus is not None:
            for sn, data in inventoried_onus.items():
                for nested_sn, nested_data in data.items():
                    set_query[f"ONUs.{nested_sn}.Disable"] = False

        update_document = {"$inc": {"ONU.Enable All Serial Numbers Count": 1, "OLT.CFG Change Count": 1},
                           "$set": set_query}
        update_result = database_manager.update_one(database_id=request.session.get("database"), collection="OLT-CFG",
                                                    query={"_id": olt_id}, update_document=update_document)
        if update_result.matched_count == 0:
            response = PonManagerApiResponse(status=status.HTTP_404_NOT_FOUND,
                                             details={"message": f"OLT {olt_id}'s configuration was not found"})
        else:
            response = PonManagerApiResponse(status=status.HTTP_200_OK, new_data=update_document, old_data={})

        return response


# ==================================================
# ======== OLT Protection Sync OLTs View =========
# ==================================================
class ProtectionSyncOlts(LoginRequiredMixin, GenericAPIView):
    raise_exception = True
    queryset = ''

    swaggerSchema = get_schema('OLT-CFG')

    @extend_schema(
        operation_id="get_olt_protection_sync_olts",
        exclude=True,
        responses={
            200: OpenApiResponse(response=schema(swaggerSchema),
                                 description='OK'),
        },
        tags=['olt'],
        summary="Get the Protection Peer for the specified OLT",
        description=" "
    )
    @method_decorator(permission_required('can_read_network_olts', raise_exception=True))
    def get(self, request, olt_id, version):
        """ Get the Protection Peer for the specified OLT """
        # Verify validity of olt_id
        if not validate_device_id(olt_id):
            response = PonManagerApiResponse(status=status.HTTP_400_BAD_REQUEST, details="Invalid OLT ID format")
        else:
            res_data = database_manager.find_one(database_id=request.session.get('database'), collection="OLT-CFG", query={"_id": olt_id}, projection={"_id": 0, "Protection.Peer": 1})
            if res_data:
                response = PonManagerApiResponse(status=status.HTTP_200_OK, data=res_data)
            else:
                response = PonManagerApiResponse(status=status.HTTP_404_NOT_FOUND, details={"message":"OLT ID "+str(olt_id)+" has no protection peer document"})

        return response

    @extend_schema(
        operation_id="olt_protection_sync_olts",
        request=None,
        exclude=True,
        responses={
            201: OpenApiResponse(response=schema(swaggerSchema),
                                 description='Created'),
        },
        tags=['olt'],
        summary="Apply specified OLTs config to the configured Protection Peer OLT",
        description=" "
    )
    @method_decorator(permission_required_any_of(['can_update_network_olts', 'can_create_network_olts'],
                                                 raise_exception=True))
    def put(self, request, olt_id, version):
        """ Apply specified OLTs config to the configured Protection Peer OLT """

        # Find the specified OLTs protection peers OLT-CFG document, and add it to results
        database = database_manager.get_database(request.session.get("database"))
        collection = database.get_collection("OLT-CFG")
        try:
            olt_cfg_peer_aggregate = list(collection.aggregate([
                {"$match": {"_id": olt_id}},
                {"$lookup": {"from": "OLT-CFG", "localField": "Protection.Peer", "foreignField": "_id",
                             "as": "PEER_CFG"}},
                {"$lookup": {"from": "OLT-STATE", "localField": "Protection.Peer", "foreignField": "_id",
                             "as": "PEER_STATE"}},
                {"$addFields": {"PEER_CFG": {"$arrayElemAt": ["$PEER_CFG", 0]}}},
                {"$addFields": {"PEER_STATE": {"$arrayElemAt": ["$PEER_STATE", 0]}}}
            ]))[0]
        except (ConnectionRefusedError, pymongo.errors.PyMongoError) as e:
            raise APIException(detail=f"MongoDB error: {str(e)}")

        if 'PEER_CFG' not in olt_cfg_peer_aggregate:
            # The 'PEER_CFG' property will be missing in a few scenarios:
            # If the configuration file for the specified OLTs protection peer was not found. Expected behavior should result in error.
            # If the OLT does not have a protection peer configured. Expected behavior should result in success.
            # If the OLT CFG is an old document not containing the Protection.Peer field. Expected behavior should result in success.
            response = PonManagerApiResponse(status=status.HTTP_200_OK)
        else:
            try:
                ### Syncing ONUs object ###
                # Loop through ONUs that should be removed from peer
                onus_inventoried_on_source_olt = copy.deepcopy(olt_cfg_peer_aggregate['ONUs']).keys()
                onus_inventoried_on_peer_olt = copy.deepcopy(olt_cfg_peer_aggregate['PEER_CFG']['ONUs']).keys()
                for onu in onus_inventoried_on_peer_olt:
                    if onu not in onus_inventoried_on_source_olt:
                        del olt_cfg_peer_aggregate['PEER_CFG']['ONUs'][onu]

                # Loop through ONUs that should exist in peer
                for onu in olt_cfg_peer_aggregate['ONUs']:
                    # If the ONU doesn't exist in the peer, we must add it
                    if onu not in olt_cfg_peer_aggregate['PEER_CFG']['ONUs']:
                        # Copy ONU inventory record from source OLT to peer OLT
                        olt_cfg_peer_aggregate['PEER_CFG']['ONUs'][onu] = olt_cfg_peer_aggregate['ONUs'][onu]

                        # If the Peer OLT has enable/disable counts in state, use those
                        if 'PEER_STATE' in olt_cfg_peer_aggregate and onu in olt_cfg_peer_aggregate['PEER_STATE']['ONUs']:
                            if 'Enable Count' in olt_cfg_peer_aggregate['PEER_STATE']['ONUs'][onu]:
                                olt_cfg_peer_aggregate['PEER_CFG']['ONUs'][onu]['Enable Count'] = \
                                olt_cfg_peer_aggregate['PEER_STATE']['ONUs'][onu]['Enable Count']
                            else:
                                olt_cfg_peer_aggregate['PEER_CFG']['ONUs'][onu]['Enable Count'] = 0
                            if 'Disable Count' in olt_cfg_peer_aggregate['PEER_STATE']['ONUs'][onu]:
                                olt_cfg_peer_aggregate['PEER_CFG']['ONUs'][onu]['Disable Count'] = \
                                olt_cfg_peer_aggregate['PEER_STATE']['ONUs'][onu]['Disable Count']
                            else:
                                olt_cfg_peer_aggregate['PEER_CFG']['ONUs'][onu]['Disable Count'] = 0
                        # If not, if not, write zeros for counts
                        else:
                            olt_cfg_peer_aggregate['PEER_CFG']['ONUs'][onu]['Enable Count'] = 0
                            olt_cfg_peer_aggregate['PEER_CFG']['ONUs'][onu]['Disable Count'] = 0

                    # Peer OLT has ONU in inventory. We must still update everything except counts
                    else:
                        # Setting Enable Count
                        if 'Enable Count' in olt_cfg_peer_aggregate['PEER_CFG']['ONUs'][onu]:
                            current_enable_count = olt_cfg_peer_aggregate['PEER_CFG']['ONUs'][onu]['Enable Count']
                        elif 'PEER_STATE' in olt_cfg_peer_aggregate and onu in olt_cfg_peer_aggregate['PEER_STATE'][
                            'ONUs'] and 'Enable Count' in olt_cfg_peer_aggregate['PEER_STATE']['ONUs'][onu]:
                            current_enable_count = olt_cfg_peer_aggregate['PEER_STATE']['ONUs'][onu]['Enable Count']
                        else:
                            current_enable_count = 0

                        if 'Disable Count' in olt_cfg_peer_aggregate['PEER_CFG']['ONUs'][onu]:
                            current_disable_count = olt_cfg_peer_aggregate['PEER_CFG']['ONUs'][onu]['Disable Count']
                        elif 'PEER_STATE' in olt_cfg_peer_aggregate and onu in olt_cfg_peer_aggregate['PEER_STATE'][
                            'ONUs'] and 'Disable Count' in olt_cfg_peer_aggregate['PEER_STATE']['ONUs'][onu]:
                            current_disable_count = olt_cfg_peer_aggregate['PEER_STATE']['ONUs'][onu]['Disable Count']
                        else:
                            current_disable_count = 0

                        olt_cfg_peer_aggregate['PEER_CFG']['ONUs'][onu] = olt_cfg_peer_aggregate['ONUs'][onu]
                        olt_cfg_peer_aggregate['PEER_CFG']['ONUs'][onu]['Enable Count'] = current_enable_count
                        olt_cfg_peer_aggregate['PEER_CFG']['ONUs'][onu]['Disable Count'] = current_disable_count

                # Fields to match the Protection Partner
                protection_fields = {
                    "EPON.Discovery Period": olt_cfg_peer_aggregate['EPON']['Discovery Period'],
                    "EPON.Encryption": olt_cfg_peer_aggregate['EPON']['Encryption'],
                    "EPON.Encryption Key Time": olt_cfg_peer_aggregate['EPON']['Encryption Key Time'],
                    "EPON.FEC": olt_cfg_peer_aggregate['EPON']['FEC'],
                    "EPON.Grant Spacing": olt_cfg_peer_aggregate['EPON']['Grant Spacing'],
                    "EPON.Laser OFF": olt_cfg_peer_aggregate['EPON']['Laser OFF'],
                    "EPON.Laser ON": olt_cfg_peer_aggregate['EPON']['Laser ON'],
                    "EPON.Max Frame Size": olt_cfg_peer_aggregate['EPON']['Max Frame Size'],
                    "EPON.Sync Time": olt_cfg_peer_aggregate['EPON']['Sync Time'],
                    "GPON.Discovery Period": olt_cfg_peer_aggregate['GPON']['Discovery Period'],
                    "GPON.Downstream Fec": olt_cfg_peer_aggregate['GPON']['Downstream Fec'],
                    "GPON.Encryption": olt_cfg_peer_aggregate['GPON']['Encryption'],
                    "GPON.Error Det Min Sample": olt_cfg_peer_aggregate['GPON']['Error Det Min Sample'],
                    "GPON.Error Det Max Ratio": olt_cfg_peer_aggregate['GPON']['Error Det Max Ratio'],
                    "GPON.Guard Time": olt_cfg_peer_aggregate['GPON']['Guard Time'],
                    "GPON.Max Frame Size": olt_cfg_peer_aggregate['GPON']['Max Frame Size'],
                    "GPON.Upstream FEC 0": olt_cfg_peer_aggregate['GPON']['Upstream FEC 0'],
                    "GPON.Upstream FEC 1": olt_cfg_peer_aggregate['GPON']['Upstream FEC 1'],
                    "GPON.Upstream FEC 2": olt_cfg_peer_aggregate['GPON']['Upstream FEC 2'],
                    "GPON.Upstream FEC 3": olt_cfg_peer_aggregate['GPON']['Upstream FEC 3'],
                    "GPON.Upstream Preamble 0": olt_cfg_peer_aggregate['GPON']['Upstream Preamble 0'],
                    "GPON.Upstream Preamble 1": olt_cfg_peer_aggregate['GPON']['Upstream Preamble 1'],
                    "GPON.Upstream Preamble 2": olt_cfg_peer_aggregate['GPON']['Upstream Preamble 2'],
                    "GPON.Upstream Preamble 3": olt_cfg_peer_aggregate['GPON']['Upstream Preamble 3'],
                    "MAC Learning": olt_cfg_peer_aggregate['MAC Learning'],
                    "NNI Networks": olt_cfg_peer_aggregate['NNI Networks'],
                    "OLT.OLT-ALARM-CFG": olt_cfg_peer_aggregate['OLT']['OLT-ALARM-CFG'],
                    "OLT.PON Enable": olt_cfg_peer_aggregate['OLT']['PON Enable'],
                    "OLT.PON Mode": olt_cfg_peer_aggregate['OLT']['PON Mode'],
                    "OLT.Max Round Trip Time": olt_cfg_peer_aggregate['OLT']['Max Round Trip Time'],
                    "ONU.Max FW Upgrades": olt_cfg_peer_aggregate['ONU']['Max FW Upgrades'],
                    "ONUs": olt_cfg_peer_aggregate['PEER_CFG']['ONUs'],
                }

                # Get CFG version
                cntl_version = olt_cfg_peer_aggregate['CNTL']['CFG Version']
                # Add correct fields for backwards compatability. NOTE: 'Inactive Period' fields have been removed permanently but were in 4.1 source code
                if float(cntl_version[1:4]) > 2.3:
                    protection_fields["GPON.Encryption Key Time"] = olt_cfg_peer_aggregate['GPON']['Encryption Key Time']
                if float(cntl_version[1:4]) > 3.0:
                    protection_fields["OLT.Shared Downstream Policer"] = olt_cfg_peer_aggregate['OLT']['Shared Downstream Policer']
                if float(cntl_version[1:4]) > 3.1:
                    protection_fields["GPON.Error Det Min HEC Sample"] = olt_cfg_peer_aggregate['GPON']['Error Det Min HEC Sample']
                    protection_fields["GPON.Error Det Max HEC Ratio"] = olt_cfg_peer_aggregate['GPON']['Error Det Max HEC Ratio']
                if float(cntl_version[1:4]) > 3.2:
                    protection_fields["OLT.PON Speeds"] = olt_cfg_peer_aggregate['OLT']['PON Speeds']
                if float(cntl_version[1:4]) > 4.0:
                    protection_fields["GPON.PON-TAG"] = olt_cfg_peer_aggregate['GPON']['PON-TAG']
                    protection_fields["OLT.Campus Mode"] = olt_cfg_peer_aggregate['OLT']['Campus Mode']
                    protection_fields["GPON.Upstream Delimiter 0"] = olt_cfg_peer_aggregate['GPON']['Upstream Delimiter 0']
                    protection_fields["GPON.Upstream Preamble Pattern 0"] = olt_cfg_peer_aggregate['GPON']['Upstream Preamble Pattern 0']
                if float(cntl_version[1:4]) >= 5.0:
                    protection_fields["Protection.NNI RX LOS Pulse"] = olt_cfg_peer_aggregate['Protection']['NNI RX LOS Pulse']
                    protection_fields["Protection.ONU Fail Limit"] = olt_cfg_peer_aggregate['Protection']['ONU Fail Limit']

                update_document = {
                    "$set": protection_fields,
                    "$inc": {
                        "OLT.CFG Change Count": 1
                    }
                }
            except (ConnectionRefusedError, pymongo.errors.PyMongoError) as e:
                raise APIException(detail=f"MongoDB error: {str(e)}")
            except Exception as e:
                raise APIException(detail=f"MongoDB error: {str(e)}")

            update_result = database_manager.update_one(database_id=request.session.get("database"),
                                                        collection="OLT-CFG",
                                                        query={"_id": olt_cfg_peer_aggregate['Protection']['Peer']},
                                                        update_document=update_document)
            if update_result.matched_count == 0:
                response = PonManagerApiResponse(status=status.HTTP_404_NOT_FOUND, details={
                    "message": f"Peer OLT {olt_id}'s configuration was not found"})
            else:
                response = PonManagerApiResponse(status=status.HTTP_200_OK, new_data=update_document, old_data={})

        return response


# ==================================================
# ======== OLT Protection Remove OLT View =========
# ==================================================
class RemoveFromProtectionPair(LoginRequiredMixin, GenericAPIView):
    raise_exception = True
    queryset = ''

    @extend_schema(
        operation_id="olt_protection_sync_olts",
        request=None,
        exclude=True,
        responses={
            200: ResponseExample(200),
            403: ResponseExample(403),
            404: ResponseExample(404),
            500: ResponseExample(500),
        },
        tags=['olt', 'protection', 'remove', 'put']
    )
    @method_decorator(permission_required_any_of(['can_update_network_olts', 'can_create_network_olts'], raise_exception=True))
    def put(self, request, olt_id, version):
        """ Remove OLTs from the protection pair and set the standby's PON Enable to False """
        database = database_manager.get_database(request.session.get("database"))
        collection = database.get_collection("OLT-CFG")
        olt_cfg1 = collection.find_one({'_id': olt_id})

        if olt_cfg1:
            peer_id = olt_cfg1['Protection']['Peer']
            database_manager.update_many(database_id=request.session.get("database"),
                                         collection="OLT-CFG",
                                         query={'_id': {'$in': [olt_id, peer_id]}},
                                         update_document={'$set': {'Protection.Peer': ''}})

        database_manager.update_one(database_id=request.session.get("database"),
                                    collection="OLT-CFG",
                                    query={'_id': olt_id},
                                    update_document={'$set': {'OLT.PON Enable': False}})

        response = PonManagerApiResponse(status=status.HTTP_200_OK)

        return response
