"""
#--------------------------------------------------------------------------#
# Copyright (c) 2025, Ciena Corporation                                    #
# All rights reserved.                                                     #
#                                                                          #
#     _______ _____ __    __ ___                                           #
#    / _ __(_) ___//  |  / // _ |                                          #
#   / /   / / /__ / /|| / // / ||                                          #
#  / /___/ / /__ / / ||/ // /__||                                          #
# /_____/_/_____/_/  |__//_/   ||                                          #
#                                                                          #
# Distributed as Ciena-Customer confidential.                              #
#                                                                          #
#--------------------------------------------------------------------------#
"""
import base64
import datetime
import hashlib
import json
import sys
import pymongo
import pymongo.errors
import fcntl
import DBdjango
from threading import Lock
from typing import List, Tuple
from urllib.parse import quote_plus
from statistics import mean

from django.contrib.auth.models import User, Group
from pymongo import MongoClient, uri_parser, ReturnDocument
from pymongo.database import Database
from pymongo.results import UpdateResult
from rest_framework.exceptions import APIException

from api.settings import IN_PRODUCTION, PONMGR_OPT_PATH, PONMGR_ETC_PATH
from utils.database_init import DatabaseInit
from json_validator import JsonSchemaValidator
from log.DatabaseLogHandler import DatabaseLogHandler
from mongo_heartbeat import MongoServerHeartbeatHandler
from connection_pool_statistics import CustomConnectionPoolLogger
from command_statistics import CustomCommandLogger
from log.PonManagerLogger import pon_manager_logger
from database_stats_writer import database_stats_writer
from manage import BUILDING_DOCUMENTATION
from utils.interval_timer import database_stats_timer
from file_watcher import databases_file_watcher
from ipaddress import ip_address, IPv6Address
from utils.keyring_helper import get_mongo_password
from django.core.signing import BadSignature
from django.core import signing
from django.utils.encoding import force_str
from backends.mongo_sessions import BSONSerializer

def _format_ipv6_address(host):
    """ Tests if address is of IPv6 type and if so adds [] around the host name.
        If the host name already has [] around it then it will return False when checking if v6 and skip formatting. xs
     """
    try:
        isIpv6 = isinstance(ip_address(host), IPv6Address)
    except ValueError:
        isIpv6 = False

    if isIpv6:
        host = "[{}]".format(host)

    return host


class DatabaseManager:
    """ Creates and handles database connections for PON Manager REST API """

    def __init__(self):
        """
        Initializes the Database Manager

        Reads the databases, user_database, and user_migration JSON files to load database connections
        Creates a single MongoClient to the user database, and one MongoClient PER entry in the databases file
        """
        self._databases = {}
        self._databases_lock = Lock()
        self._file_lock = Lock()
        self._json_schema_validator = None
        self._session_remote_addr = {}
        self._databases_json_store = {}
        # Sets default SYSLOG-ACTIONS collection size
        self._COLLECTION_CAP_BYTES_DEFAULT = 50000000
        pon_manager_logger.info("PON Manager REST API started")

        self.command_logger = CustomCommandLogger()
        self.connection_pool_logger = CustomConnectionPoolLogger()
        self.heartbeat_handler = MongoServerHeartbeatHandler()

        self._WRITE_INTERVAL_TIMER_SEC = 30

        if IN_PRODUCTION:
            self._DATABASES_FILE = "/var/www/html/api/databases.json"
            self._USER_DATABASE_FILE = "/var/www/html/api/user_database.json"
            self._USER_MIGRATION_FILE = "/var/www/html/api/user_migration.json"
            self._SCHEMA_FILES = "/var/www/html/api/schema_files"
            self.production_branding_controls = "/var/www/html/ponmgr/assets/branding/custom-controls.json"
        else:
            self._DATABASES_FILE = "databases.json"
            self._USER_DATABASE_FILE = "user_database.json"
            self._USER_MIGRATION_FILE = "user_migration.json"
            self._SCHEMA_FILES = "schema_files"
            self.development_branding_controls = "../src/assets/branding/custom-controls.json"

        self._db_init = DatabaseInit(self._DATABASES_FILE, self._USER_DATABASE_FILE)
        user_database_json = self._db_init.user_database_fields

        self.logo_setter_unlocked = self.get_branding_control_settings()

        self._user_migration_options = self.read_json_file(self._USER_MIGRATION_FILE)

        # Connect to user database
        try:
            temp_user_db, heart_beat, command_stats, connection_pool_stats = self._create_connection(user_database_json)
            if temp_user_db is not None:
                self._USER_DATABASE = temp_user_db, heart_beat
                pon_manager_logger.info("User database is active")
            else:
                pon_manager_logger.critical("User database is NOT active. Exiting...")
                sys.exit(1)
        except (pymongo.errors.ConfigurationError, pymongo.errors.ServerSelectionTimeoutError) as e:
            pon_manager_logger.critical(f"User database connection error: {e}\nExiting...")
            sys.exit(1)

        # Do migrations for user database
        try:
            self._perform_migrations()
        except Exception as e:
            pon_manager_logger.error(f"Error migrating user options to user database: {e}")

        # Add the database log handler to the logger
        self.database_logger = DatabaseLogHandler(self.user_database)
        pon_manager_logger.add_handler(self.database_logger)

        # Add the user database to the database stats writer
        database_stats_writer.set_user_database(self.user_database)

        # Connect to each PON Controller database
        self.create_database_connections()

        # Instantiate JSON validators for each collection
        self._json_schema_validator = JsonSchemaValidator(self._SCHEMA_FILES)

        # Write database statistics to user database every 30 seconds
        database_stats_timer.set(self._WRITE_INTERVAL_TIMER_SEC, self.write_database_stats)
        # File observer for databases.json
        databases_file_watcher.start(self.sync_databases)

    @property
    def user_database(self):
        return self._USER_DATABASE[0]

    def get_ponmgr_cfg(self, query=None, projection=None):
        """ Retrieves the chosen tibit settings document """
        if query is None:
            query = {"_id": "Default"}
        collection = self._USER_DATABASE[0].get_collection("PONMGR-CFG")
        return collection.find_one(query, projection)

    def set_ponmgr_cfg(self, document):
        """ Sets the pon manager configuration """
        collection = self._USER_DATABASE[0].get_collection("PONMGR-CFG")
        query = {"_id": "Default"}
        result = collection.update_one(filter=query, update={"$set": document}, upsert=False)
        return result

    def read_json_file(self, file_path: str) -> dict:
        """ Reads the json contents of the file specified in the given path

        :param file_path: Path to the json file to read
        :return json file contents as dict
        :raises FileNotFoundError, JSONDecodeError
        """
        with self._file_lock:
            with open(file_path, 'r') as file:
                file_contents = json.load(file)

        return file_contents

    def update_json_file_key(self, file_path: str, key: str, new_dict):
        """ Updates the document of the given key in the JSON file at file_path

        :param file_path: Path to the json file to read
        :param key: The JSON key to modify or create
        :param new_dict: The JSON body. If None, deletes the key from the dictionary
        :raises FileNotFoundError, JSONDecodeError
        """
        with self._file_lock:
            with open(file_path, 'r+') as file:
                try:
                    # Lock the file
                    fcntl.lockf(file, fcntl.LOCK_EX, fcntl.LOCK_NB)
                    file_contents = json.load(file)
                    if new_dict is None:
                        del file_contents[key]
                    else:
                        file_contents[key] = new_dict
                    file.truncate(0)    # Clear the file
                    file.seek(0)    # Point to the start of the file
                    json.dump(obj=file_contents, fp=file, indent=4)
                    # update database init object after modifying databases document
                    if "databases.json" in file_path:
                        self._db_init.reload(databases_json_content=file_contents)
                    # Unlock the file
                    fcntl.lockf(file, fcntl.LOCK_UN)
                except:
                    fcntl.lockf(file, fcntl.LOCK_UN)

    def get_user_database_compression_value(self):
        """ Gets the value of 'compression' from the user_database.json file """
        compression = False

        try:
            with open(f"{PONMGR_OPT_PATH}/api/user_database.json" if IN_PRODUCTION else "user_database.json") as json_file:
                user_db_data = json.load(json_file)
                if user_db_data and "compression" in user_db_data:
                    compression = user_db_data["compression"]
        except FileNotFoundError:
            compression = False

        return compression

    def get_user_database_password_opts(self):
        """ Gets the value of 'compression' from the user_database.json file """
        password_opts = {}

        try:
            with open(f"{PONMGR_OPT_PATH}/api/user_database.json" if IN_PRODUCTION else "user_database.json") as json_file:
                user_db_data = json.load(json_file)
                if user_db_data and "password_opts" in user_db_data:
                    password_opts = user_db_data["password_opts"]

        except FileNotFoundError:
            password_opts = {}

        return password_opts

    def set_session_remote_addr(self, session_key, remote_addr):
        """ Sets the selected database for the given session """
        self._session_remote_addr[session_key] = remote_addr

    def get_session_remote_addr(self, session_key):
        """ Gets the selected database for the given session """
        return self._session_remote_addr[session_key]

    def remove_session_remote_addr(self, session_key):
        """ Deletes the selected database for the given session """
        del self._session_remote_addr[session_key]

    def purge_radius_users(self):
        collection = self.user_database.get_collection("django_session")
        try:
            sessions_encoded = list(collection.find({}))
            collection = self.user_database.get_collection("auth_user")
            radius_users = list(collection.find({'database_type':'radius'}, {"id": 1}))
            active_user_ids = []

            for session in sessions_encoded:
                try:
                    session_data = signing.loads(force_str(session), salt='django.contrib.sessions.SessionStore', serializer=BSONSerializer)
                except BadSignature:
                    session_data = {}
                if '_auth_user_id' in session_data:
                    active_user_ids.append(int(session_data['_auth_user_id']))

            users_to_purge = []
            for user in radius_users:
                if user['id'] not in active_user_ids:
                    users_to_purge.append(user['id'])
            collection.delete_many({'id': {'$in': users_to_purge}})
            collection = self.user_database.get_collection("auth_user_groups")
            collection.delete_many({'user_id': {'$in': users_to_purge}})
        except Exception as e:
            raise APIException(detail=f"EXCEPTION: {str(e)}")

    def set_users_database_type(self, user_email, type):
        """ Sets the user type to radius """
        collection = self.user_database.get_collection("auth_user")
        try:
            result = collection.update_one({"email": user_email}, {"$set": {"database_type": type}})
        except (ConnectionRefusedError, pymongo.errors.PyMongoError) as e:
            raise APIException(detail=f"MongoDB error: {str(e)}")
        return result

    def get_users_database_type(self, user_email):
        """ Gets the user type to radius """
        collection = self.user_database.get_collection("auth_user")
        try:
            result = collection.find_one({"email": user_email}, {"database_type":1, "_id":0, "email":1})
        except (ConnectionRefusedError, pymongo.errors.PyMongoError) as e:
            raise APIException(detail=f"MongoDB error: {str(e)}")
        return result

    def set_users_selected_database(self, user_email, database_id) -> UpdateResult:
        """ Sets the users active database in the user database """
        collection = self.user_database.get_collection("auth_user")
        try:
            result = collection.update_one({"email": user_email}, {"$set": {"active_database": database_id}})
        except (ConnectionRefusedError, pymongo.errors.PyMongoError) as e:
            raise APIException(detail=f"MongoDB error: {str(e)}")
        return result

    def get_users_selected_database(self, user_email):
        """ Get the users active PON Controller database selection from the user database.
            Raises KeyError if the active_database fetched is not found or is not an active database connection
        """
        collection = self.user_database.get_collection("auth_user")

        try:
            result = collection.find_one({"email": user_email}, {"active_database": 1, "_id": 0})
        except (ConnectionRefusedError, pymongo.errors.PyMongoError) as e:
            raise APIException(detail=f"MongoDB error: {str(e)}")

        if result is None or "active_database" not in result or result['active_database'] not in self._databases.keys():
            raise KeyError

        return result['active_database']

    def get_user_session_expiry(self, email) -> int:
        """ Get user session expiry age from user database """

        try:
            expiration_amounts = list(self.user_database["auth_user"].aggregate([
                {
                    '$match': {
                        'email': email
                    }
                }, {
                    '$lookup': {
                        'from': 'auth_user_groups',
                        'localField': 'id',
                        'foreignField': 'user_id',
                        'as': 'roles'
                    }
                }, {
                    '$unwind': {
                        'path': '$roles',
                        'includeArrayIndex': 'string',
                        'preserveNullAndEmptyArrays': False
                    }
                }, {
                    '$lookup': {
                        'from': 'auth_group',
                        'localField': 'roles.group_id',
                        'foreignField': 'id',
                        'as': 'group'
                    }
                }, {
                    '$project': {
                        'Timeout': {
                            '$arrayElemAt': [
                                '$group.User Session Expiry Age Timeout', 0
                            ]
                        },
                        'Override': {
                            '$arrayElemAt': [
                                '$group.User Session Expiry Age Timeout Override', 0
                            ]
                        }
                    }
                }, {
                    '$sort': {
                        'Timeout': -1
                    }
                }
            ]))
            expiration = None

            if len(expiration_amounts) > 0:
                for exp_dict in expiration_amounts:
                    if "Timeout" in exp_dict:
                        expiration = exp_dict["Timeout"]
                        break
            if expiration is None:
                collection = self.user_database["PONMGR-CFG"]
                expiration = collection.find_one({"_id": "Default"})["User Session Expiry Age Timeout"]

        except (ConnectionRefusedError, pymongo.errors.PyMongoError) as e:
            raise APIException(detail=f"MongoDB error: {str(e)}")

        return int(expiration)

    def get_user_groups(self, user_email):
        """
        Get all the active users and the groups they belong to
        """
        response = []
        try:
            groups_data = list(self.user_database["auth_user"].aggregate([
                {
                    '$match': {
                        'email': user_email
                    }
                }, {
                    '$lookup': {
                        'from': 'auth_user_groups',
                        'localField': 'id',
                        'foreignField': 'user_id',
                        'as': 'user-role relationships'
                    }
                }, {
                    '$unwind': {
                        'path': '$user-role relationships',
                        'preserveNullAndEmptyArrays': False
                    }
                }, {
                    '$lookup': {
                        'from': 'auth_group',
                        'localField': 'user-role relationships.group_id',
                        'foreignField': 'id',
                        'as': 'roles'
                    }
                }, {
                    '$unwind': {
                        'path': '$roles',
                        'preserveNullAndEmptyArrays': False
                    }
                }, {
                    '$group': {
                        '_id': '',
                        'groups': {
                            '$addToSet': '$roles.name'
                        }
                    }
                }
            ]))
            response = groups_data[0]['groups']
        except Exception as err:
            raise APIException(detail=f"EXCEPTION (get_user_groups): {str(err)}")

        return response

    def get_user_permissions(self, user_email):
        """
        Returns permissions a user has. (Copy of get_user_permissions from db connector)
        """
        response = []
        try:
            permission_data = list(self.user_database["auth_user"].aggregate([
                {
                    '$match': {
                        'email': user_email
                    }
                }, {
                    '$lookup': {
                        'from': 'auth_user_groups',
                        'localField': 'id',
                        'foreignField': 'user_id',
                        'as': 'user-role relationships'
                    }
                }, {
                    '$unwind': {
                        'path': '$user-role relationships',
                        'preserveNullAndEmptyArrays': False
                    }
                }, {
                    '$lookup': {
                        'from': 'auth_group_permissions',
                        'localField': 'user-role relationships.group_id',
                        'foreignField': 'group_id',
                        'as': 'permissions'
                    }
                }, {
                    '$unwind': {
                        'path': '$permissions',
                        'preserveNullAndEmptyArrays': False
                    }
                }, {
                    '$lookup': {
                        'from': 'auth_permission',
                        'localField': 'permissions.permission_id',
                        'foreignField': 'id',
                        'as': 'permissions'
                    }
                }, {
                    '$unwind': {
                        'path': '$permissions',
                        'preserveNullAndEmptyArrays': False
                    }
                }, {
                    '$group': {
                        '_id': '',
                        'permissions': {
                            '$addToSet': '$permissions.codename'
                        }
                    }
                }
            ]))
            # Checks if global branding permissions are locked.
            if self.logo_setter_unlocked is False:
                permission_data[0]['permissions'] = list(filter(lambda x: "can_update_other_branding" not in x, permission_data[0]['permissions']))

            response = permission_data[0]['permissions']
        except Exception as err:
            raise APIException(detail=f"EXCEPTION (get_user_permissions): {str(e)}")

        return response

    def get_session_key_expiry_date(self, session_key):
        # response = [status.HTTP_200_OK, ""]

        try:
            collection = self.user_database["django_session"]
            expiry_date = list(collection.find({'session_key': session_key}))
            if len(expiry_date) > 0:
                return expiry_date[0]['expire_date']
            else: #Collection empty
                return None
        except Exception as err: #Database is offline
            return 0

    def get_branding_control_settings(self):
        """
        Checks the custom-controls.json file to see if users of the PDM can change the logo.

            :return true or false
        """
        try:
            if IN_PRODUCTION:
                with open(self.production_branding_controls, 'r', encoding='utf-8') as control_file:
                    control_data = json.load(control_file)
                    can_brand = control_data["enable-app-logos"] == True
            else:
                with open(self.development_branding_controls, 'r', encoding='utf-8') as control_file:
                    control_data = json.load(control_file)
                    can_brand = control_data["enable-app-logos"] == True
        except Exception as e:
            raise APIException(detail=f"EXCEPTION (get_branding_control_settings): {str(e)}")

        return can_brand

    def list_databases(self):
        """ Gives a list of databases currently active

        :return List of database IDs available
        """
        return self._databases.keys()

    def get_all_databases(self):
        """ Gives a list of databases currently active

        :return Dictionary of database connections
        """
        return self._databases

    def get_databases_json(self):
        """ Gives a dictionary of all active databases parameters

        :return Dictionary of databases information
        """
        self._db_init.reload()
        return self._db_init.databases

    def get_database(self, database_id: str) -> Database:
        """ Get the connection to the database specified by the database_id

        :param database_id: ID of the database reference to get
        :return Database object of the database connection
        :raises KeyError, APIException
        """
        if not self._databases[database_id][1].is_alive:
            raise APIException(detail=self._databases[database_id][1].details)
        return self._databases[database_id][0]

    def mongo_server_is_active(self, database_id: str) -> bool:
        """ Check the given database connection is alive

        :param database_id: ID of the database reference to check
        :return bool describing if the mongo server connection is ok
        :raises KeyError if the database_id is not found in the active list
        """
        return self._databases[database_id][1].is_alive

    def mongo_server_get_status(self, database_id: str) -> str:
        """ Check the status of the given mongo server

        :param database_id: ID of the database reference to check
        :return str status of mongo server connection
        :raises KeyError if the database_id is not found in the active list
        """
        status = self._databases[database_id][1].details
        heartbeat = self._databases[database_id][1]
        alive = False
        if heartbeat.is_replica_set:
            # Check if the replica set has a primary member online
            if self._databases[database_id][0].client.primary is not None:
                alive = True
        else:   # Other connection types
            alive = heartbeat.is_alive

        if alive:
            try:
                with pymongo.timeout(5):
                    collections = self._databases[database_id][0].list_collection_names()
                    if 'CNTL-CFG' not in collections:
                        status = f"Database '{heartbeat.name}' is Not Populated"
                    else:
                        status = 'Online'
            except (pymongo.errors.OperationFailure, pymongo.errors.ServerSelectionTimeoutError):
                status = f"Authentication failed for Database '{heartbeat.name}'"
            except pymongo.errors.PyMongoError as e:
                status = f"Error: {e}"
        return status

    def mongo_server_get_version(self, database_id):
        if self._databases[database_id][1].is_alive:
            try:
                with pymongo.timeout(1):
                    db_version = self._databases[database_id][0].command({'buildInfo': 1})['version']
            except pymongo.errors.PyMongoError:
                db_version = '-'
        else:
            db_version = '-'

        return db_version


    def add_database(self, database_id: str, database_json: dict):
        """ Create a new database connection and add the information to the databases file

        :param database_id: ID of the new database
        :param database_json: Dictionary of the new database connection parameters
        :raises DuplicateKeyError if the database ID is already in use
        """
        with self._databases_lock:
            # Test if IPv6 and if so add []
            database_json['host'] = _format_ipv6_address(database_json['host'])

            if database_id in self._databases.keys():
                raise pymongo.errors.DuplicateKeyError(error="Database ID already in use")
            else:
                new_database, heart_beat, command_stats, connection_pool_stats = self._create_connection(database_json, pc_database_id=database_id)
                if new_database is not None:
                    self.update_json_file_key(self._DATABASES_FILE, database_id, database_json)
                    self._databases[database_id] = new_database, heart_beat, command_stats, connection_pool_stats
                    # Remove old database state if existing, get new stats
                    self.clear_database_stats(database_id)
                else:
                    raise APIException("Failed to create database connection. See logs for details.")

    def edit_database(self, database_id: str, database_json: dict) -> str:
        """ Update an existing database connection and update the information in the databases file

        :param database_id: ID of the new database
        :param database_json: Dictionary of the new database connection parameters
        :return str Describing the status of the edit or error encountered
        """
        with self._databases_lock:
            if database_id in self._databases.keys():
                result = "Updated"
            else:
                result = "Created"

            new_database, heart_beat, command_stats, connection_pool_stats = self._create_connection(database_json, pc_database_id=database_id)
            if new_database is not None:
                # Test if IPv6 and if so add []
                database_json['host'] = _format_ipv6_address(database_json['host'])

                self.update_json_file_key(self._DATABASES_FILE, database_id, database_json)
                self._databases[database_id] = new_database, heart_beat, command_stats, connection_pool_stats
                # Clear old database state document
                self.clear_database_stats(database_id)
            else:
                raise APIException("Failed to edit database connection. See logs for details.")

        return result

    def remove_database(self, database_id: str):
        """ Deletes the given database reference from the list of connections

        :param database_id: ID of the database reference to delete
        """
        with self._databases_lock:
            self.update_json_file_key(file_path=self._DATABASES_FILE, key=database_id, new_dict=None)
            if database_id in self._databases.keys():
                # Make sure to stop the heartbeat and close the MongoClient
                # The explicit stop is only to prevent a failed heartbeat message after the connection is closed
                self._databases[database_id][1].stop()
                self._databases[database_id][0].client.close()
                del self._databases[database_id]
                # Clear old database state document from DATABASE-STATE
                self.clear_database_stats(database_id)

    def get_database_stats(self, database_id):
        try:
            collection = self.user_database.get_collection("DATABASE-STATE")
            # Get the DATABASE-STATE document for the given database ID
            state_doc = collection.find_one(filter={'_id': database_id})
            if state_doc:
                # Calculate Connections.Count Average
                if 'Connections' in state_doc:
                    if 'Count Average Samples' in state_doc['Connections']:
                        state_doc['Connections']['Count Average'] = round(
                            mean(state_doc['Connections']['Count Average Samples']))
                        del state_doc['Connections']['Count Average Samples']

                # Calculate Connections.Count Current
                if 'Connections' in state_doc:
                    if 'Count Current Samples' in state_doc['Connections']:
                        state_doc['Connections']['Count Current'] = round(
                            mean(state_doc['Connections']['Count Current Samples']))
                        del state_doc['Connections']['Count Current Samples']

                # Calculate Commands.Average Latency
                if 'Commands' in state_doc and isinstance(state_doc['Commands'], dict):
                    for name, cmd in state_doc['Commands'].items():
                        if 'Average Latency Samples' in cmd:
                            cmd['Average Latency'] = round(mean(cmd['Average Latency Samples']))
                            del cmd['Average Latency Samples']

                # Calculate Servers.Latency
                if 'Servers' in state_doc and isinstance(state_doc['Servers'], list):
                    for server in state_doc['Servers']:
                        if 'Latency Samples' in server:
                            server['Latency'] = round(mean(server['Latency Samples']))
                            del server['Latency Samples']

        except (ConnectionRefusedError, pymongo.errors.PyMongoError) as e:
            raise APIException(detail=f"MongoDB error: {str(e)}")
        return state_doc

    def write_database_stats(self):
        try:
            with self._databases_lock:
                for database_id in self._databases:
                    server_statistics = self._databases[database_id][1].get_statistics()
                    command_statistics = self._databases[database_id][2].get_statistics()
                    connection_statistics = self._databases[database_id][3].get_statistics()

                    #  Get type and latency for each server
                    servers = self._databases[database_id][0].client.topology_description.server_descriptions()
                    all_servers = {}
                    for host, server in servers.items():
                        host_name = str(host[0]) + ":" + str(host[1])
                        latency = 0
                        if server.round_trip_time is not None and server.round_trip_time != 0:
                            latency = round(server.round_trip_time * 1000, 0)  # convert from seconds to milliseconds
                        all_servers[host_name] = {
                            'latency': latency,
                            'type': server.server_type_name
                        }

                    # Set Type and Latency for Server with matching host name
                    for server in server_statistics:
                        server_id = server['Host']
                        if server_id in all_servers:
                            server['Type'] = all_servers[server_id]['type']
                            server['Latency'] = all_servers[server_id]['latency']

                    # Write database stats to DATABASE-STATE
                    database_stats_writer.write_statistics(database_id, connection_statistics, server_statistics, command_statistics)
        except (ConnectionRefusedError, pymongo.errors.PyMongoError) as e:
            raise APIException(detail=f"MongoDB error: {str(e)}")

    def clear_database_stats(self, database_id):
        try:
            database_collection = self.user_database['DATABASE-STATE']
            database_collection.delete_one({'_id': database_id})
            # Refresh stats
            database_stats_timer.do_function()
        except (ConnectionRefusedError, pymongo.errors.PyMongoError) as e:
            raise APIException(detail=f"MongoDB error: {str(e)}")

    def sync_databases(self):
        with self._databases_lock:
            try:
                # Remove connections not found in databases.json
                databases_json = self.read_json_file(self._DATABASES_FILE)
                connections_to_remove = {x: self._databases.keys() for x in set(self._databases.keys())
                                         - set(databases_json.keys())}
                for database_id in connections_to_remove:
                    self._databases[database_id][1].stop()
                    self._databases[database_id][0].client.close()
                    del self._databases[database_id]
            except FileNotFoundError:
                pon_manager_logger.error("Databases json file not found")

            # Add any new or modified connections
            self.create_database_connections()

            # Sync update object in views
            DBdjango.views.start_up()

    def create_database_connections(self):
        """ Creates PON Controller database connections for self._databases """
        try:
            databases_json = self._db_init.databases
            for database_id in databases_json.keys():
                create_connection = False
                if database_id in self._databases_json_store:
                    old_database_data = self._databases_json_store[database_id]
                    for attribute in databases_json[database_id]:
                        # Create a new connection if any attributes have been modified
                        if databases_json[database_id][attribute] != old_database_data[attribute]:
                            create_connection = True
                            break
                else:  # New database
                    create_connection = True

                if create_connection:
                    self.command_logger = CustomCommandLogger()
                    self.connection_pool_logger = CustomConnectionPoolLogger()
                    self.heartbeat_handler = MongoServerHeartbeatHandler()
                    try:
                        client, heart_beat, command_stats, connection_pool_stats = self._create_connection(databases_json[database_id],
                                                                     pc_database_id=database_id)
                        if client is not None:
                            self._databases[database_id] = client, heart_beat, command_stats, connection_pool_stats
                            pon_manager_logger.info(f"Database {database_id} active")
                    except pymongo.errors.ConfigurationError as e:
                        pon_manager_logger.error(f"Failed to connect to database {database_id}. Database URI error: {e}")
            # store databases.json content for comparison
            self._databases_json_store = databases_json
        except FileNotFoundError:
            pon_manager_logger.error("Databases json file not found")

    def find(self, database_id: str, collection: str, query=None, projection=None, sort=None, limit=0, skip=0,
             next=None):
        """ Preforms the collection.find operation

        :param database_id: The string ID of the database
        :param collection: The string name of the collection to access
        :param query: Custom query filter document
        :param projection: Query projection document
        :param sort: List of Tuples with the key and 1 or -1 in order of the key to sort on first
        :param limit: The total number of documents to return
        :param skip: The number of documents to omit from the beginning of the results
        :param next: The _id to being the query at
        :return List of retrieved documents
        """
        return self._find(database_id=database_id, collection=collection, query=query, projection=projection, sort=sort,
                          limit=limit, skip=skip, next=next, one=False)

    def find_one(self, database_id: str, collection: str, query: dict, projection=None):
        """ Preforms the collection.find_one operation

        :param database_id: The string ID of the database
        :param collection: The string name of the collection to access
        :param query: Custom query filter document
        :param projection: Query projection document
        :return Dictionary of retrieved document
        """
        return self._find(database_id=database_id, collection=collection, query=query, projection=projection, one=True)

    def distinct(self, database_id: str, collection: str, query=None, distinct=None):
        """ Preforms the collection.distinct operation

        :param database_id: The string ID of the database
        :param collection: The string name of the collection to access
        :param query: Custom query filter document
        :param distinct: The Field to get unique values of
        :return Dictionary with array of unique values
        """
        return self._distinct(database_id=database_id, collection=collection, query=query, field=distinct)

    def update_many(self, database_id: str, collection: str, query: dict, update_document: dict, upsert=False):
        """ Preforms the collection.update_many operation

        :param database_id: The string ID of the database
        :param collection: The string name of the collection to access
        :param query: Custom query filter document
        :param update_document: The update options dictionary
        :param upsert: Insert the document if it does not exist if set to true
        :return UpdateResult object
        """
        return self._update(database_id=database_id, collection=collection, query=query,
                            update_document=update_document, upsert=upsert, many=True)

    def update_one(self, database_id: str, collection: str, query: dict, update_document: dict, upsert=False):
        """ Preforms the collection.update_one operation

        :param database_id: The string ID of the database
        :param collection: The string name of the collection to access
        :param query: Custom query filter document
        :param update_document: The update options dictionary
        :param upsert: Insert the document if it does not exist if set to true
        :return UpdateResult object
        """
        return self._update(database_id=database_id, collection=collection, query=query,
                            update_document=update_document, upsert=upsert, many=False)

    def insert_many(self, database_id: str, collection: str, documents: List[dict]):
        """ Preforms the collection.insert_many operation

        :param database_id: The string ID of the database
        :param collection: The string name of the collection to access
        :param documents: List of documents to insert into the collection
        :return InsertManyResult object
        """
        return self._insert(database_id=database_id, collection=collection, documents=documents, many=True)

    def insert_one(self, database_id: str, collection: str, document: dict):
        """ Preforms the collection.insert_one operation

        :param database_id: The string ID of the database
        :param collection: The string name of the collection to access
        :param document: Document to insert into the collection
        :return InsertOneResult object
        """
        return self._insert(database_id=database_id, collection=collection, documents=document, many=False)

    def find_one_and_update(self, database_id: str, collection: str, query: dict, update_document: dict,
                            projection=None, return_doc=ReturnDocument.BEFORE, upsert=False):
        """ Preforms the collection.find_one_and_update operation

        :param database_id: The string ID of the database
        :param collection: The string name of the collection to access
        :param query: Custom query filter document
        :param update_document: The update options dictionary
        :param projection: Projection document for the find operation
        :param return_doc: ReturnDocument.BEFORE or ReturnDocument.AFTER
        :param upsert: Insert the document if it does not exist if set to true
        :return UpdateResult object
        """
        database = self.get_database(database_id)
        collection = database.get_collection(collection)

        try:
            return_doc = collection.find_one_and_update(filter=query, update=update_document, projection=projection,
                                                        return_document=return_doc, upsert=upsert)
        except (ConnectionRefusedError, pymongo.errors.PyMongoError) as e:
            raise APIException(detail=f"MongoDB error: {str(e)}")

        return return_doc

    def update_checksum(self, database, collection_name, file_id, filename, file_data):
        """ Adds to the FILE-STATE collection

        :param database: The database object in use
        :param collection_name: collection name that was uploaded to
        :param file_id: _id of the file uploaded to mongodb
        :param filename: filename of the file uploaded
        :param file_data: data of the filed uploaded
        """
        try:
            collection = database.get_collection('FILE-STATE')

            if True:
                raw_hash = hashlib.sha256()
                raw_hash.update(file_data)
                checksum = raw_hash.hexdigest()

                data = {
                    'File ID': file_id,
                    'File Type': collection_name,
                    'Upload Status': {
                        'Details': '',
                        "Progress": "100%",
                        "Status": "Uploaded",
                        "Time": datetime.datetime.now().strftime('%Y-%m-%d %H:%M:%S.%f')
                    },
                    'filename': filename,
                    'Checksum': checksum
                }

                update = {'$set': data}

                collection.update_one(filter={'_id': f'{collection_name}-{file_id}'}, update=update, upsert=True)
        except Exception as e:
            raise APIException(detail=f"MongoDB error: {str(e)}")

    def find_one_and_replace(self, database_id: str, collection: str, query: dict, new_document: dict, projection=None,
                             return_doc=ReturnDocument.BEFORE, upsert=True):
        """ Preforms the collection.find_one_and_replace operation

        :param database_id: The string ID of the database
        :param collection: The string name of the collection to access
        :param query: Custom query filter document
        :param new_document: The new document
        :param projection: Projection document for the find operation
        :param return_doc: ReturnDocument.BEFORE or ReturnDocument.AFTER
        :param upsert: Insert the document if it does not exist if set to true
        :return UpdateResult object
        """
        database = self.get_database(database_id)
        collection = database.get_collection(collection)

        try:
            return_doc = collection.find_one_and_replace(filter=query, replacement=new_document, projection=projection,
                                                         return_document=return_doc, upsert=upsert)
        except (ConnectionRefusedError, pymongo.errors.PyMongoError) as e:
            raise APIException(detail=f"MongoDB error: {str(e)}")

        return return_doc

    def replace_one(self, database_id: str, collection: str, query: dict, new_document: dict, upsert=False):
        """ Preforms the collection.replace_one operation

        :param database_id: The string ID of the database
        :param collection: The string name of the collection to access
        :param query: Custom query filter document
        :param new_document: The document to insert
        :param upsert: Insert the document if it does not exist if set to true
        :return UpdateResult object
        """
        database = self.get_database(database_id)
        collection = database.get_collection(collection)

        try:
            replace_result = collection.replace_one(filter=query, replacement=new_document, upsert=upsert)
        except (ConnectionRefusedError, pymongo.errors.PyMongoError) as e:
            raise APIException(detail=f"MongoDB error: {str(e)}")

        return replace_result

    def delete_many(self, database_id: str, collection: str, query: dict):
        """ Preforms the collection.delete_many operation

        :param database_id: The string ID of the database
        :param collection: The string name of the collection to access
        :param query: Custom query filter document
        """
        self._delete(database_id=database_id, collection=collection, query=query, many=True)

    def delete_one(self, database_id: str, collection: str, query: dict):
        """ Preforms the collection.delete_one operation

        :param database_id: The string ID of the database
        :param collection: The string name of the collection to access
        :param query: Custom query filter document
        """
        self._delete(database_id=database_id, collection=collection, query=query, many=False)

    def validate(self, collection: str, document: dict, validate_required: bool = True, strict_validation: bool = False, schema: dict = None) -> Tuple[
        bool, dict]:
        """ Validates a document against the JSON schema defined for a collection

        :param collection: The string name of the collection to validate against
        :param document: The document to validate
        :param validate_required: Check for missing required fields in the document (set to False for PATCH)
        :param strict_validation: Return 400 Error when a required field is missing
        :param schema: Schema to validate against (overrides collection schema)
        :return
            A tuple (bool, dict), where the boolean returns 'True' if the document
            is valid and 'False' otherwise. The dict returns HTTP response data, if
            the document fails JSON validation and '{}' otherwise.
        """
        schema_version = None
        if "CNTL" in document and "CFG Version" in document["CNTL"]:
            schema_version = self._json_schema_validator.get_schema_version_for_doc_version(
                document["CNTL"]["CFG Version"])
        else:
            # If the configuration version isn't present in the document, validate the document against
            # the 'current' (or latest) schema.
            schema_version = 'current'

        if schema_version:
            is_valid, details = self._json_schema_validator.validate(collection, document, validate_required,
                                                                     strict_validation, schema_version, schema)
            if not is_valid and details:
                print(f"ERROR: JSON validation failed for {details['collection']}, doc_id {details['id']}.")
                print(f"ERROR: {details['message']}")
                err_msg = self._json_schema_validator.json_dumps(details, sort_keys=False)
                err_msg = err_msg.replace("\n", "\nERROR: ")
                print(f"ERROR: details = {err_msg}")
            elif details and details['level'] == 'warning':
                print(
                    f"WARNING: JSON validation warning for {details['collection']}, doc_id {details['id']}, {details['message']}.")
                # Warnings are considered success
        else:
            # Skip validation if no validator exists for the specified document version
            is_valid = True
            details = {}

        return is_valid, details

    def validate_path(self, collection: str, path: str, schema: dict = None) -> Tuple[bool, dict]:
        """ Validates a MongoDB path against the JSON schema defined for a collection

        :param collection: The string name of the collection to validate against
        :param path: A path to validate
        :param schema: Schema to validate against (overrides collection schema)
        :return
            A tuple (bool, dict), where the boolean returns 'True' if the path
            is valid and 'False' otherwise. The dict returns HTTP response data, if
            the path fails JSON validation and '{}' otherwise.
        """
        valid = True
        details = {}
        valid, details = self._json_schema_validator.validate_path(collection, path, schema=schema)
        if not valid and 'bad value' in details:
            print(
                f"ERROR: Invalid query value for {details['collection']}, path {details['path']}, value {details['bad value']}.")
            print(f"ERROR: {details['message']}")
            err_msg = self._json_schema_validator.json_dumps(details, sort_keys=False)
            err_msg = err_msg.replace("\n", "\nERROR: ")
            print(f"ERROR: details = {err_msg}")
        elif not valid:
            print(f"ERROR: Invalid query path for {details['collection']}, path {details['path']}.")
            err_msg = self._json_schema_validator.json_dumps(details, sort_keys=False)
            err_msg = err_msg.replace("\n", "\nERROR: ")
            print(f"ERROR: details = {err_msg}")

        return valid, details

    def validate_query(self, collection: str, query_params: dict, schema: dict = None) -> Tuple[bool, dict]:
        """ Validates a MongoDB query path and value against the JSON schema defined for a collection

        :param collection: The string name of the collection to validate against
        :param query_params: A dictionary of query paths and values to validate
        :param schema: Schema to validate against (overrides collection schema)
        :return
            A tuple (bool, dict), where the boolean returns 'True' if the path or value
            is valid and 'False' otherwise. The dict returns HTTP response data, if
            the path or value fails JSON validation and '{}' otherwise.
        """
        valid = True
        details = {}
        for key, value in query_params.items():
            valid, details = self._json_schema_validator.validate_path(collection, key, value, schema=schema)
            if not valid and 'bad value' in details:
                print(
                    f"ERROR: Invalid query value for {details['collection']}, path {details['path']}, value {details['bad value']}.")
                print(f"ERROR: {details['message']}")
                err_msg = self._json_schema_validator.json_dumps(details, sort_keys=False)
                err_msg = err_msg.replace("\n", "\nERROR: ")
                print(f"ERROR: details = {err_msg}")
                break
            elif not valid:
                print(f"ERROR: Invalid query path for {details['collection']}, path {details['path']}.")
                err_msg = self._json_schema_validator.json_dumps(details, sort_keys=False)
                err_msg = err_msg.replace("\n", "\nERROR: ")
                print(f"ERROR: details = {err_msg}")
                break

        return valid, details

    def validate_projection(self, collection: str, query_params: dict, schema: dict = None) -> Tuple[bool, dict]:
        """ Validates a MongoDB projection path and value against the JSON schema defined for a collection

        :param collection: The string name of the collection to validate against
        :param query_params: A dictionary of query paths and values to validate
        :param schema: Schema to validate against (overrides collection schema)
        :return
            A tuple (bool, dict), where the boolean returns 'True' if the path or value
            is valid and 'False' otherwise. The dict returns HTTP response data, if
            the path or value fails JSON validation and '{}' otherwise.
        """
        valid = True
        details = {}
        for key, value in query_params.items():
            valid, details = self._json_schema_validator.validate_path(collection, key, None, schema=schema)
            if not valid:
                # ERROR - Invalid projection path
                details['message'] = f"Invalid projection path for {details['collection']}, path {details['path']}."
                print(f"ERROR: {details['message']}")
                err_msg = self._json_schema_validator.json_dumps(details, sort_keys=False)
                err_msg = err_msg.replace("\n", "\nERROR: ")
                print(f"ERROR: details = {err_msg}")
                break
            elif not isinstance(value, int) or value < 0 or value > 1:
                # ERROR - Invalid projection value
                valid = False
                details = {
                    "level": "error",
                    "message": f"Invalid projection value for {collection}, path {key}, value {value}.",
                    "collection": collection,
                    "path": key,
                    "bad value": value
                }
                print(f"ERROR: {details['message']}")
                err_msg = self._json_schema_validator.json_dumps(details, sort_keys=False)
                err_msg = err_msg.replace("\n", "\nERROR: ")
                print(f"ERROR: details = {err_msg}")
                break

        return valid, details

    def validate_sort(self, collection: str, query_params: dict, schema: dict = None) -> Tuple[bool, dict]:
        """ Validates a MongoDB sort path and value against the JSON schema defined for a collection

        :param collection: The string name of the collection to validate against
        :param query_params: A dictionary of query paths and values to validate
        :param schema: Schema to validate against (overrides collection schema)
        :return
            A tuple (bool, dict), where the boolean returns 'True' if the path or value
            is valid and 'False' otherwise. The dict returns HTTP response data, if
            the path or value fails JSON validation and '{}' otherwise.
        """
        valid = True
        details = {}
        for key, value in query_params.items():
            valid, details = self._json_schema_validator.validate_path(collection, key, None, schema=schema)
            if not valid:
                # ERROR - Invalid sort path
                details['message'] = f"Invalid sort path for {details['collection']}, path {details['path']}."
                print(f"ERROR: {details['message']}")
                err_msg = self._json_schema_validator.json_dumps(details, sort_keys=False)
                err_msg = err_msg.replace("\n", "\nERROR: ")
                print(f"ERROR: details = {err_msg}")
                break
            elif not isinstance(value, int) or (value != -1 and value != 1):
                # ERROR - Invalid sort value
                valid = False
                details = {
                    "level": "error",
                    "message": f"Invalid sort value for {collection}, path {key}, value {value}.",
                    "collection": collection,
                    "path": key,
                    "bad value": value
                }
                print(f"ERROR: {details['message']}")
                err_msg = self._json_schema_validator.json_dumps(details, sort_keys=False)
                err_msg = err_msg.replace("\n", "\nERROR: ")
                print(f"ERROR: details = {err_msg}")
                break

        return valid, details

    def _find(self, database_id: str, collection: str, query: dict, projection=None, sort=None, limit=0, skip=0,
              next=None, one=True):
        database = self.get_database(database_id)
        collection = database.get_collection(collection)

        try:
            if one:
                result = collection.find_one(filter=query, projection=projection)
            else:
                if sort is None or len(sort) == 0:
                    sort = [("_id", 1)]
                if next:
                    if query is None:
                        query = {}
                    if any(map(lambda tup: tup[0] == '_id' and tup[1] == -1, sort)):
                        query["_id"] = {"$lt": next}
                    else:
                        query["_id"] = {"$gt": next}
                result = list(collection.find(filter=query, projection=projection, limit=limit, skip=skip, sort=sort))
        except (ConnectionRefusedError, pymongo.errors.PyMongoError) as e:
            raise APIException(detail=f"MongoDB error: {str(e)}")

        return result

    def _distinct(self, database_id=None, collection=None, query=None, field=None):
        database = self.get_database(database_id)
        collection = database.get_collection(collection)

        try:
            result = collection.distinct(field, query)
        except (ConnectionRefusedError, pymongo.errors.PyMongoError) as e:
            raise APIException(detail=f"MongoDB error: {str(e)}")

        return result

    def _update(self, database_id: str, collection: str, query: dict, update_document: dict, upsert: bool, many: bool):
        database = self.get_database(database_id)
        collection = database.get_collection(collection)

        try:
            if many:
                update_result = collection.update_many(filter=query, update=update_document, upsert=upsert)
            else:
                update_result = collection.update_one(filter=query, update=update_document, upsert=upsert)
        except (ConnectionRefusedError, pymongo.errors.PyMongoError) as e:
            raise APIException(detail=f"MongoDB error: {str(e)}")

        return update_result

    def _insert(self, database_id: str, collection: str, documents, many: bool):
        database = self.get_database(database_id)
        collection = database.get_collection(collection)

        try:
            if many:
                insert_result = collection.insert_many(documents=documents)
            else:
                insert_result = collection.insert_one(document=documents)
        except pymongo.errors.DuplicateKeyError as e:
            # Allow DuplicateKeyError to pass through for handling in the view
            raise e
        except (ConnectionRefusedError, pymongo.errors.PyMongoError) as e:
            raise APIException(detail=f"MongoDB error: {str(e)}")

        return insert_result

    def _delete(self, database_id: str, collection: str, query: dict, many=False):
        database = self.get_database(database_id)
        collection = database.get_collection(collection)

        try:
            if many:
                collection.delete_many(filter=query)
            else:
                collection.delete_one(filter=query)
        except (ConnectionRefusedError, pymongo.errors.PyMongoError) as e:
            raise APIException(detail=f"MongoDB error: {str(e)}")

    def _create_connection(self, parameters_json: dict, pc_database_id=None) -> Tuple[
        Database, MongoServerHeartbeatHandler, CustomCommandLogger, CustomConnectionPoolLogger]:
        """ Connect to the given Mongo server and return a database reference object
            NOTE: Returned database connection should be verified to not be None in case of configuration error

        :param parameters_json: JSON dict of the MongoDB connection information
        :return Database object of the specified Mongo database
        :raises ConfigurationError if final URI is invalid
        """

        # pon controller databases need to retrieve keyring paths from user_databases.json
        if parameters_json['auth_enable'] == True and 'password_opts' in parameters_json \
                and 'type' in parameters_json['password_opts'] and parameters_json['password_opts']['type'] == 'keyring':
            if 'keyring_path' not in parameters_json['password_opts'] or 'keyring_key_path' not in parameters_json['password_opts']:
                opts = self.get_user_database_password_opts()
                parameters_json['password_opts']['keyring_path'] = opts['keyring_path']
                parameters_json['password_opts']['keyring_key_path'] = opts['keyring_key_path']

        mongodb_uri = parameters_json['db_uri']
        auth_source = parameters_json['auth_db']
        auth_enable = parameters_json['auth_enable']
        username = parameters_json['username']
        if not auth_enable:
            password = parameters_json['password']
        elif pc_database_id:
            parameters_json['password_opts'] = self.get_user_database_password_opts()
            # specify the username if calling for a pc database
            password = get_mongo_password(parameters_json, username=f'databases.{pc_database_id}.password')
        else:
            # default user_database behavior if auth enabled
            password = get_mongo_password(parameters_json)
        dns_srv = parameters_json['dns_srv']
        host = parameters_json['host']
        port = parameters_json['port']
        name = parameters_json['name']
        replica_set_enable = parameters_json['replica_set_enable']
        replica_set_hosts = parameters_json['replica_set_hosts']
        replica_set_name = parameters_json['replica_set_name']
        tls_enable = parameters_json['tls_enable']
        ca_cert_path = parameters_json['ca_cert_path']

        # Use URI if defined
        if mongodb_uri == '':
            if dns_srv:
                mongodb_uri = "mongodb+srv://"
            else:
                mongodb_uri = "mongodb://"

            options_tokens = []

            # Auth options
            if auth_enable:
                mongodb_uri += "{}:{}@".format(
                    quote_plus(username),
                    quote_plus(password)
                )
                options_tokens.append("authSource={}".format(auth_source))

            # Test if address is of IPv6 type
            try:
                isIpv6 = isinstance(ip_address(host), IPv6Address)
            except ValueError:
                isIpv6 = False

            # There are three types of connections
            #   Direct Connection   - connect directly to MongoDB server.
            #   Replica Set         - connect to a cluster of MongoDB servers providing redundancy.
            #   DNS Seed List (SRV) - connect using connection information from a DNS SRV record.
            if dns_srv:
                # DNS Seed List (SRV) - use the 'host' field only
                if isIpv6:
                    mongodb_uri += "[{}]".format(host)
                else:
                    mongodb_uri += "{}".format(host)
            elif replica_set_enable:
                # Replica Set Connections - use the 'replica_set_hosts' field
                # The 'replica_set_hosts' field is a list of hosts that are part of the Replica Set.
                for index, host in enumerate(replica_set_hosts):
                    try:
                        isIpv6 = isinstance(ip_address(host), IPv6Address)
                    except ValueError:
                        isIpv6 = False
                    if isIpv6:
                        replica_set_hosts[index] = "[{}]".format(host)

                mongodb_uri += ",".join(replica_set_hosts)
            else:
                # Direct Connection - use the 'host' and 'port' fields
                if isIpv6:
                    mongodb_uri += "[{}]:{}".format(host, port)
                else:
                    mongodb_uri += "{}:{}".format(host, port)

            # Append Replica Set information
            if replica_set_enable:
                options_tokens.append("replicaSet={}".format(replica_set_name))

            if tls_enable:
                options_tokens.append("tls=true")
                options_tokens.append("tlsCAFile={}".format(ca_cert_path))
            elif dns_srv:
                # MongoDB automatically enables encryption (ssl=true) with SRV. If encryption is
                # disabled, explicitly disable encryption via the URI.
                options_tokens.append("tls=false")

            # Always add the directConnection flag, which is set to 'true' when replicaSets are disabled for the MongodB connection
            options_tokens.append("directConnection={}".format("false" if replica_set_enable else "true"))

            if len(options_tokens) > 0:
                mongodb_uri += "/?"
                mongodb_uri += "&".join(options_tokens)

        mongo_client = None
        heart_beat = None
        command_stats = None
        connection_pool_stats = None

        # Validate the URI
        try:
            uri_parser.parse_uri(mongodb_uri)

            is_replica_set = replica_set_enable or "replicaSet=" in mongodb_uri
            mongo_server_heartbeat = MongoServerHeartbeatHandler(is_replica_set=is_replica_set, name=name)
            custom_command = CustomCommandLogger()
            custom_connection_pool = CustomConnectionPoolLogger()
            if self.get_user_database_compression_value():
                mongo_client = MongoClient(mongodb_uri, event_listeners=[mongo_server_heartbeat, custom_command, custom_connection_pool], compressors='snappy', serverSelectionTimeoutMS=10000).get_database(name=name)
            else:
                mongo_client = MongoClient(mongodb_uri, event_listeners=[mongo_server_heartbeat, custom_command, custom_connection_pool], serverSelectionTimeoutMS=10000).get_database(name=name)
            heart_beat = mongo_server_heartbeat
            command_stats = custom_command
            connection_pool_stats = custom_connection_pool
        except pymongo.errors.ConfigurationError as e:
            print(f"Failed to connect to MongoDB server at {mongodb_uri}. Database URI error: {e}")
        except FileNotFoundError:
            print(f"Failed to connect to MongoDB server at {mongodb_uri}. CA Certificate file was not found.")

        return mongo_client, heart_beat, command_stats, connection_pool_stats

    def _perform_migrations(self):
        """ Performing Django User Migration manually (instead of; python manage.py makemigrations(migrate) command(s)) if not already done """
        collections = self.user_database.list_collection_names()

        # Rebranding collections from tibit_* to generic names
        if "PONMGR-CFG" not in collections and "tibit_settings" in collections:
            self.user_database['tibit_settings'].aggregate([{"$match": {}}, {"$out": "PONMGR-CFG"}])
            collections = self.user_database.list_collection_names()
            if 'PONMGR-CFG' not in collections:
                pon_manager_logger.error('Error cloning "tibit_settings" collection')
        if "ponmgr_migrations" in collections and "tibit_migrations" in collections:
            self.user_database['tibit_migrations'].aggregate([{"$match": {}}, {"$out": "ponmgr_migrations"}])
            collections = self.user_database.list_collection_names()
            if 'ponmgr_migrations' not in collections:
                pon_manager_logger.error('Error renaming "tibit_migrations" collection to "ponmgr_migrations"')



        # SCHEMA
        if not "__schema__" in collections:  # If the collection does not exist, create it
            self.user_database.create_collection('__schema__')
        schema_collection = self.user_database['__schema__']
        for migration_document in self._user_migration_options[
            '__schema__']:  # Looping through all schema documents that need to exist, as defined in 'user_migration.json'
            schema_document = schema_collection.find_one(
                {'name': migration_document['name']})  # Find schema document in MongoDB by name
            if schema_document is None:  # If document was not found, insert it
                result = schema_collection.insert_one(migration_document)
                if not result.acknowledged:
                    pon_manager_logger.error("Error adding schema document to database")
            if migration_document[
                'name'] != 'django_session':  # Here, begin validating sequence number is correct. django_session schema document does not have a sequence.
                sequenced_collection = self.user_database[migration_document['name']]
                highest_sequence_document = sequenced_collection.find_one({"$query": {}, "$orderby": {"id": -1}}, {
                    "id": 1})  # Based on the name field of the schema document, search the associated collection for the highest 'id' value. Only return the id field as that is all I need. This is where the sequence should start
                if highest_sequence_document is not None:  # Continue if we found a document
                    if type(highest_sequence_document['id']) is int:  # Continue if 'id' is an integer
                        if int(highest_sequence_document["id"]) != schema_document['auto'][
                            'seq']:  # Only update schema document if the sequence value does not match what it should be, the highest value
                            result = schema_collection.update_one({'name': migration_document['name']}, {"$set": {
                                'auto': {'field_names': ['id'], 'seq': int(highest_sequence_document[
                                                                               "id"])}}})  # Update schema document with highest sequence value
                            if not result.acknowledged:
                                pon_manager_logger.error("Error updating schema document in database")

        # SYSLOG-ACTIONS
        if not "SYSLOG-ACTIONS" in collections:
            self.user_database.create_collection('SYSLOG-ACTIONS')
        # This caps the SYSLOG-ACTIONS collection if it is not capped to 50MB
        collection_stats = self.user_database.command("collstats", "SYSLOG-ACTIONS")
        if "capped" not in collection_stats.keys() or not collection_stats["capped"]:
            collection_cap_size = self._COLLECTION_CAP_BYTES_DEFAULT
            settings_doc = self.user_database.get_collection("PONMGR-CFG").find_one({"_id": "Default"},
                                                                                        {"SYSLOG-ACTIONS Max Bytes": 1})
            if settings_doc is not None and settings_doc != {}:
                if settings_doc["SYSLOG-ACTIONS Max Bytes"] is not None:
                    try:
                        collection_cap_size = int(settings_doc["SYSLOG-ACTIONS Max Bytes"])
                    except ValueError:
                        pon_manager_logger.warning(
                            f"Could not parse max collection size for \"SYSLOG-ACTIONS\" collection. Using default of {self._COLLECTION_CAP_BYTES_DEFAULT} bytes.")

            self.user_database.command({"convertToCapped": "SYSLOG-ACTIONS", "size": collection_cap_size})
            pon_manager_logger.info(
                f"Converted collection \"SYSLOG-ACTIONS\" to capped collection with max size of {collection_cap_size} bytes")

        # AUTH_GROUP
        if not "auth_group" in collections:
            self.user_database.create_collection('auth_group')
        collection = self.user_database['auth_group']
        for document in self._user_migration_options['auth_group']:
            exists = collection.find_one(document)
            if exists is None:
                result = collection.replace_one({'id': document['id']}, document, upsert=True)
                if not result.acknowledged:
                    pon_manager_logger.error("Error replacing auth group document in database")

        # AUTH_GROUP_PERMISSIONS
        if not "auth_group_permissions" in collections:
            self.user_database.create_collection('auth_group_permissions')
        collection = self.user_database['auth_group_permissions']
        try:
            max_id = collection.find_one(sort=[("id", pymongo.DESCENDING)])["id"] + 1
        except:
            max_id = 1
        for document in self._user_migration_options['auth_group_permissions']:
            exists = collection.find_one({"group_id": document["group_id"], "permission_id": document["permission_id"]})
            if exists is None:
                document["id"] = max_id
                max_id += 1
                result = collection.insert_one(document)
                if not result.acknowledged:
                    pon_manager_logger.error("Error adding permissions document to database")

        # AUTH_PERMISSION
        if not "auth_permission" in collections:
            self.user_database.create_collection('auth_permission')
        collection = self.user_database['auth_permission']
        for document in self._user_migration_options['auth_permission']:
            exists = collection.find_one(document)
            if exists is None:
                result = collection.replace_one({'id': document['id']}, document, upsert=True)
                if not result.acknowledged:
                    pon_manager_logger.error("Error replacing auth permission document in database")

        # AUTH_USER
        if not "auth_user" in collections:
            self.user_database.create_collection('auth_user')

        # AUTH_USER_GROUPS
        if not "auth_user_groups" in collections:
            self.user_database.create_collection('auth_user_groups')

        # AUTH_USER_USER_PERMISSIONS
        if not "auth_user_user_permissions" in collections:
            self.user_database.create_collection('auth_user_user_permissions')

        # DJANGO_ADMIN_LOG
        if not "django_admin_log" in collections:
            self.user_database.create_collection('django_admin_log')

        # DJANGO_CONTENT_TYPE
        if not "django_content_type" in collections:
            self.user_database.create_collection('django_content_type')
        collection = self.user_database['django_content_type']
        for document in self._user_migration_options['django_content_type']:
            exists = collection.find_one(document)
            if exists is None:
                result = collection.replace_one({'id': document['id']}, document, upsert=True)
                if not result.acknowledged:
                    pon_manager_logger.error("Error replacing django content type document in database")

        # DJANGO_MIGRATIONS
        if not "django_migrations" in collections:
            self.user_database.create_collection('django_migrations')
        collection = self.user_database['django_migrations']
        for document in self._user_migration_options['django_migrations']:
            exists = collection.find_one(document)
            if exists is None:
                result = collection.replace_one({'id': document['id']}, document, upsert=True)
                if not result.acknowledged:
                    pon_manager_logger.error("Error replacing django migrations document in database")

        # DJANGO_SESSION
        if not "django_session" in collections:
            self.user_database.create_collection('django_session')

        # PONMGR-CFG
        if not "PONMGR-CFG" in collections:
            self.user_database.create_collection('PONMGR-CFG')
        collection = self.user_database['PONMGR-CFG']
        for migration_document in self._user_migration_options['PONMGR-CFG']:
            found_document_in_db = collection.find_one({'_id': 'Default'})
            if found_document_in_db is None:
                result = collection.replace_one({'_id': migration_document['_id']}, migration_document, upsert=True)
                if not result.acknowledged:
                    pon_manager_logger.error("Error replacing migration document in database")
                # Migrate old session expiry age if the deprecated record exists
                found_document_in_db = collection.find_one({'_id': 'USER_SESSION_EXPIRY_AGE'})
                if found_document_in_db and found_document_in_db['timeout']:
                    result = collection.update_one({'_id': migration_document['_id']}, {
                        "$set": {"User Session Expiry Age Timeout": found_document_in_db['timeout']}})
                    if not result.acknowledged:
                        pon_manager_logger.error("Error updating migration document in database")
            else:  # Default document exists, need to verify all properties are present
                for key in migration_document:  # Looping through all values that should exist in the document from migration.json
                    if key not in found_document_in_db:  # Check if the document in DB doesn't have key:value pair that it should
                        result = collection.update_one({'_id': migration_document['_id']},
                                              {"$set": {key: migration_document[key]}})
                        if not result.acknowledged:
                            pon_manager_logger.error("Error updating migration document in database")
                    else:  # The property in mongoDB does exist. We need to verify that the property has all the correct properties
                        if type(found_document_in_db[
                                    key]) is dict:  # We only need to verify contents of a property if it's an object/dict
                            for found_document_in_db_property_key in migration_document[
                                key]:  # Looping through all the properties of the nested object/dict
                                if found_document_in_db_property_key not in found_document_in_db[
                                    key]:  # If the nested object/dict is missing a property, insert it
                                    property_name = key + '.' + found_document_in_db_property_key  # Forming the property name to be used in Mongo query
                                    result = collection.update_one({'_id': migration_document['_id']}, {"$set": {
                                        property_name: migration_document[key][found_document_in_db_property_key]}})
                                    if not result.acknowledged:
                                        pon_manager_logger.error("Error updating migration document in database")

        # TIBIT_MIGRATIONS
        if not "ponmgr_migrations" in collections:
            self.user_database.create_collection('ponmgr_migrations')

        # Migrating R1.3.1 and below users to R2.0.0 users
        collection = self.user_database['ponmgr_migrations']
        user_migration_0001 = collection.find_one({"_id": 'auth_user_1'})
        if user_migration_0001 is None:
            if len(User.objects.filter(groups__name='Administrators')) == 0:
                for user in User.objects.all():
                    user.groups.add(Group.objects.get(name="Administrators"))
                    user.is_staff = False
                    user.save()

            # Creating migration record
            collection = self.user_database['ponmgr_migrations']
            migration_record = self._user_migration_options['ponmgr_migrations']['auth_user_1']
            migration_record.update({"applied": datetime.datetime.now()})
            result = collection.insert_one(migration_record)
            if not result.acknowledged:
                pon_manager_logger.error("Error adding migration record to database")

        # Migrating R2.2.0 and below users to have lowercase email and username
        user_migration_0002 = collection.find_one({"_id": 'auth_user_2'})
        if user_migration_0002 is None:
            for user in User.objects.all():
                user.email = user.email.lower()
                user.username = user.username.lower()
                user.save()

            # Creating migration record
            collection = self.user_database['ponmgr_migrations']
            migration_record = self._user_migration_options['ponmgr_migrations']['auth_user_2']
            migration_record.update({"applied": datetime.datetime.now()})
            result = collection.insert_one(migration_record)
            if not result.acknowledged:
                pon_manager_logger.error("Error adding migration record to database")

        # Migrating #R3.1.0 and below groups to have view permissions
        auth_group_permission_migration_0001 = collection.find_one({"_id": 'auth_group_permissions_1'})
        if auth_group_permission_migration_0001 is None:
            group_ids = list(self.user_database['auth_group'].find({}, {"id": 1, "_id": 0}))
            permission_ids = [57, 58, 59, 60, 61, 62, 63, 64, 65, 66, 67, 68, 69,
                              70]  # permission_ids for newly added view permissions
            all_group_permissions = self.user_database['auth_group_permissions'].find()
            has_new_permissions = False
            for permission in all_group_permissions:
                if permission["group_id"] != 1 and permission["group_id"] != 2:  # not admin or read only
                    if permission["permission_id"] in permission_ids:
                        has_new_permissions = True
                        break

            if not has_new_permissions:  # Need to add view permissions
                # Grab highest 'id' as to not overwrite any ids
                permission_counter = \
                self.user_database['auth_group_permissions'].find_one(sort=[("id", pymongo.DESCENDING)])["id"] + 1

                for group in group_ids:  # Loop through each existing group/role
                    group_id = group["id"]  # id number of group/role

                    if group_id != 1 and group_id != 2:  # not admin or read only
                        has_admin_read = self.user_database['auth_group_permissions'].find_one(
                            {"group_id": group_id, "permission_id": 41})

                        for permission in permission_ids:  # Loop through new permissions to add
                            if (
                                    permission == 57 and has_admin_read) or permission != 57:  # only add 'admin view' if group has 'read admin'
                                new_permission = {
                                    "id": permission_counter,
                                    "group_id": group_id,
                                    "permission_id": permission
                                }
                                result = self.user_database['auth_group_permissions'].insert_one(new_permission)
                                if not result.acknowledged:
                                    pon_manager_logger.error("Error adding permissions document to database")
                                permission_counter += 1

            # Creating migration record
            collection = self.user_database['ponmgr_migrations']
            migration_record = self._user_migration_options['ponmgr_migrations']['auth_group_permissions_1']
            migration_record.update({"applied": datetime.datetime.now()})
            result = collection.insert_one(migration_record)
            if not result.acknowledged:
                pon_manager_logger.error("Error adding migration record to database")

        # R3.2.0 Adding Session Expiry Timeout and Override fields to Auth_Group
        collection = self.user_database['auth_group']
        missing_role_timeout_fields = list(collection.find({"User Session Expiry Age Timeout": {'$exists': False}}))
        if len(missing_role_timeout_fields) > 0:
            global_timeout = self.user_database['PONMGR-CFG'].find_one({'_id': 'Default'})['User Session Expiry Age Timeout']
            update = {"$set": {"User Session Expiry Age Timeout": global_timeout, "User Session Expiry Age Timeout Override": False}}
            result = collection.update_many({"User Session Expiry Age Timeout": {'$exists': False}}, update, upsert=True)
            if not result.acknowledged:
                pon_manager_logger.error("Error updating timeout settings in database")

        # Migrating R4.0.0 and below groups to have other.search permissions
        collection = self.user_database['ponmgr_migrations']
        auth_group_permission_migration_0002 = collection.find_one({"_id": 'auth_group_permissions_2'})
        if auth_group_permission_migration_0002 is None:
            group_ids = list(self.user_database['auth_group'].find({}, {"id": 1, "_id": 0}))
            write_permission_ids = [71, 73, 74]  # ids for other.search write permissions
            read_permission_ids = [72, 75]  # ids for other.search read only permissions
            all_group_permissions = self.user_database['auth_group_permissions'].find()
            has_new_permissions = False

            for permission in all_group_permissions:
                if permission["group_id"] != 1 and permission["group_id"] != 2:  # not admin or read only
                    if permission["permission_id"] in read_permission_ids or permission["permission_id"] in write_permission_ids:
                        has_new_permissions = True
                        break

            if not has_new_permissions:  # Need to add new permissions
                # Grab highest 'id' as to not overwrite any ids
                permission_counter = \
                    self.user_database['auth_group_permissions'].find_one(sort=[("id", pymongo.DESCENDING)])[
                        "id"] + 1

                for group in group_ids:  # Loop through each existing group/role
                    group_id = group["id"]  # id number of group/role

                    if group_id != 1 and group_id != 2: # all other roles
                        for permission in read_permission_ids:  # Loop through new permissions to add
                            new_permission = {
                                "id": permission_counter,
                                "group_id": group_id,
                                "permission_id": permission
                            }
                            result = self.user_database['auth_group_permissions'].insert_one(new_permission)
                            if not result.acknowledged:
                                pon_manager_logger.error("Error adding permissions document to database")
                            permission_counter += 1

            # Creating migration record
            collection = self.user_database['ponmgr_migrations']
            migration_record = self._user_migration_options['ponmgr_migrations']['auth_group_permissions_2']
            migration_record.update({"applied": datetime.datetime.now()})
            result = collection.insert_one(migration_record)
            if not result.acknowledged:
                pon_manager_logger.error("Error adding migration record to database")

        # Cascading Feature Migration
        auth_group_permission_migration_0003 = collection.find_one({"_id": 'auth_group_permissions_3'})
        if auth_group_permission_migration_0003 is None:
            group_ids = list(self.user_database['auth_group'].find({}, {"id": 1, "_id": 0}))
            new_permission_ids = [76, 80]
            all_group_permissions = self.user_database['auth_group_permissions'].find()
            has_new_permissions = False
            for permission in all_group_permissions:
                if permission["group_id"] != 1 and permission["group_id"] != 2:  # not admin or read only
                    if permission["permission_id"] in new_permission_ids:
                        has_new_permissions = True
                        break

            if not has_new_permissions:  # Need to add view permissions
                # Grab highest 'id' as to not overwrite any ids
                permission_counter = \
                    self.user_database['auth_group_permissions'].find_one(sort=[("id", pymongo.DESCENDING)])[
                        "id"] + 1

                for group in group_ids:  # Loop through each existing group/role
                    group_id = group["id"]  # id number of group/role

                    if group_id != 1 and group_id != 2:  # not admin or read only
                        for permission in new_permission_ids:
                            new_permission = {
                                "id": permission_counter,
                                "group_id": group_id,
                                "permission_id": permission
                            }
                            result = self.user_database['auth_group_permissions'].insert_one(new_permission)
                            if not result.acknowledged:
                                pon_manager_logger.error("Error adding permissions document to database")
                            permission_counter += 1

                # Creating migration record
                collection = self.user_database['ponmgr_migrations']
                migration_record = self._user_migration_options['ponmgr_migrations']['auth_group_permissions_3']
                migration_record.update({"applied": datetime.datetime.now()})
                result = collection.insert_one(migration_record)
                if not result.acknowledged:
                    pon_manager_logger.error("Error adding migration record to database")

        # Migrating R5.2.0 and below groups to have global_config.tasks permissions
        collection = self.user_database['ponmgr_migrations']
        auth_group_permission_migration_0004 = collection.find_one({"_id": 'auth_group_permissions_4'})
        if auth_group_permission_migration_0004 is None:
            group_ids = list(self.user_database['auth_group'].find({}, {"id": 1, "_id": 0}))
            write_permission_ids = [81, 83, 84]  # ids for other.global_config.tasks permissions
            read_permission_ids = [82, 85]  # ids for global_config.tasks read only permissions
            all_group_permissions = self.user_database['auth_group_permissions'].find()
            has_new_permissions = False

            for permission in all_group_permissions:
                if permission["group_id"] != 1 and permission["group_id"] != 2:  # not admin or read only
                    if permission["permission_id"] in read_permission_ids or permission[
                        "permission_id"] in write_permission_ids:
                        has_new_permissions = True
                        break

            if not has_new_permissions:  # Need to add new permissions
                # Grab highest 'id' as to not overwrite any ids
                permission_counter = \
                    self.user_database['auth_group_permissions'].find_one(sort=[("id", pymongo.DESCENDING)])[
                        "id"] + 1

                for group in group_ids:  # Loop through each existing group/role
                    group_id = group["id"]  # id number of group/role

                    if group_id != 1 and group_id != 2:  # all other roles
                        for permission in read_permission_ids:  # Loop through new permissions to add
                            new_permission = {
                                "id": permission_counter,
                                "group_id": group_id,
                                "permission_id": permission
                            }
                            result = self.user_database['auth_group_permissions'].insert_one(new_permission)
                            if not result.acknowledged:
                                pon_manager_logger.error("Error adding permissions document to database")
                            permission_counter += 1

            # Creating migration record
            collection = self.user_database['ponmgr_migrations']
            migration_record = self._user_migration_options['ponmgr_migrations']['auth_group_permissions_4']
            migration_record.update({"applied": datetime.datetime.now()})
            result = collection.insert_one(migration_record)
            if not result.acknowledged:
                pon_manager_logger.error("Error adding migration record to database")


        pon_manager_logger.info("User database migrations complete")


    def get_in_production(self):
        return IN_PRODUCTION

if BUILDING_DOCUMENTATION:
	print("Skipping Database Manager for Documentation")
	database_manager = None
else:
	database_manager = DatabaseManager()

