"""
#--------------------------------------------------------------------------#
# Copyright (c) 2025, Ciena Corporation                                    #
# All rights reserved.                                                     #
#                                                                          #
#     _______ _____ __    __ ___                                           #
#    / _ __(_) ___//  |  / // _ |                                          #
#   / /   / / /__ / /|| / // / ||                                          #
#  / /___/ / /__ / / ||/ // /__||                                          #
# /_____/_/_____/_/  |__//_/   ||                                          #
#                                                                          #
# Distributed as Ciena-Customer confidential.                              #
#                                                                          #
#--------------------------------------------------------------------------#

Connector for using MongoDB calls in Django API
"""
import copy
import traceback
import pymongo
import json
import gridfs
import base64
import datetime
import inspect
import os
import io
import httpagentparser
import sys
import re

from bson import ObjectId
from bson.binary import Binary
from PIL import Image
from django.core.signing import BadSignature
from pymongo import ReturnDocument, UpdateOne
from resizeimage import resizeimage
from rest_framework import status
from pymongo.errors import *
from django.core import signing
from django.contrib import messages
from django.utils.encoding import force_str
from backends.mongo_sessions import BSONSerializer
from math import ceil

from api.settings import IN_PRODUCTION

from database_manager import database_manager
from utils.tools import get_nested_value


class MongoDBConnector:

    def __init__(self, database_id: str):
        """Initialize the MongoDBConnector with database host info"""
        self.database_id = database_id
        self.database = database_manager.get_all_databases()[database_id][0]
        self.user_database = database_manager.user_database

        # For development
        self.development_databases_files = "databases.json"
        self.development_logo_directory = "../src/assets/branding/"
        self.development_footer_text = "../src/assets/branding/"
        # For production
        self.production_databases_files = "/var/www/html/api/databases.json"
        self.production_logo_directory = "/var/www/html/ponmgr/assets/branding/"
        self.production_footer_text = "/var/www/html/ponmgr/assets/branding/"
        self.database_seed_base_dir = "/var/www/html/api/databaseSeedFiles/"  # instead of having two distinct production and development variables, like above, the value of this variable is modified to be production or development

        if IN_PRODUCTION:
            with open("/var/www/html/api/user_database.json"):
                pass
            with open("/var/www/html/api/user_migration.json"):
                pass
            self.database_seed_base_dir = "/var/www/html/api/databaseSeedFiles/"
            self.production_branding_controls = "/var/www/html/ponmgr/assets/branding/custom-controls.json"
        else:
            with open("user_database.json"):
                pass
            with open("user_migration.json"):
                pass
            self.database_seed_base_dir = "databaseSeedFiles/"
            self.development_branding_controls = "../src/assets/branding/custom-controls.json"

        self.logo_setter_unlocked = self.get_branding_control_settings()

    def seed_database(self, request, to_include):
        """Insert default documents and files into database collections"""
        response = [status.HTTP_200_OK, "Database was successfully initialized.", None, {}]
        user_email = request.user.email

        try:
            database_manager.get_database(self.database_id)
            # Insert default alarm configuration documents
            self.initialize_alarm_configs_collection(to_include, "CNTL-ALARM-CFG")
            self.initialize_alarm_configs_collection(to_include, "OLT-ALARM-CFG")
            self.initialize_alarm_configs_collection(to_include, "ONU-ALARM-CFG")

            # Insert default device configuration documents
            self.initialize_device_configs_collection(to_include, "CNTL-CFG")
            self.initialize_device_configs_collection(to_include, "OLT-CFG")
            self.initialize_device_configs_collection(to_include, "ONU-CFG")
            self.initialize_device_configs_collection(to_include, "SWI-CFG")

            # Insert default/global PON Automation configuration documents
            self.initialize_pon_auto_configs_collection(to_include, "CNTL-AUTO-CFG")
            self.initialize_pon_auto_configs_collection(to_include, "OLT-AUTO-CFG")
            self.initialize_pon_auto_configs_collection(to_include, "ONU-AUTO-CFG")
            self.initialize_pon_auto_configs_collection(to_include, "SWI-AUTO-CFG")
            self.initialize_pon_auto_configs_collection(to_include, "AUTO-CFG")
            self.initialize_pon_auto_configs_collection(to_include, "AUTO-TASK-CFG")

            # Insert default SLA, Downstream mapping, and Service configuration documents
            self.initialize_sla_configs_collection(to_include)
            self.initialize_ds_map_configs_collection(to_include)
            self.initialize_service_configs_collection(to_include)

            # Upload default firmware and picture files
            self.initialize_olt_firmware_bucket(to_include)
            self.initialize_onu_firmware_bucket(to_include)
            self.initialize_picture_bucket(to_include)

            response[2] = to_include.keys()
        except Exception as err:
            messages.error(request, f"{type(err)}")
            messages.error(request, traceback.format_exc(), extra_tags='stacktrace')
            self.status_log("EXCEPTION (seed_database)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not seed database due to {type(err).__name__}::{sys.exc_info()[1]}", None, None]

        action_data = {"old": "none", "new": "seed files", "collection": "databaseSeedFiles/", "user": user_email}
        self.sys_log(action_data, "write", response[0] == status.HTTP_200_OK)

        return response

    def set_seed_pon_mode(self, request, json_seed):
        directory = self.database_seed_base_dir + "documents/deviceConfigs/olts/"
        response = [status.HTTP_200_OK, "PON Mode Seed was successfully set.", None, {}]
        try:
            if directory != "":
                for filename in os.listdir(directory):
                    if filename.lower().endswith(".json"):
                        with open(directory + filename, 'r') as json_file:
                            new_doc = json.load(json_file)
                            old_doc = copy.deepcopy(new_doc)
                        if "OLT" in new_doc:
                            new_doc['OLT']['PON Mode'] = json_seed
                        with open(directory + filename, 'w') as json_file:
                            json_file.write(json.dumps(new_doc, indent=4))
                        return [status.HTTP_200_OK, "PON Mode Seed was successfully set.", new_doc, old_doc]
        except Exception as err:
            messages.error(request, f"{type(err)}")
            messages.error(request, traceback.format_exc(), extra_tags='stacktrace')
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not change seed pon mode due to {type(err).__name__}::{sys.exc_info()[1]}", None, None]
        return response

    def initialize_alarm_configs_collection(self, collections_to_include, collection):
        """
        Insert all default alarm configuration documents into the specified database collection.
        :param collections_to_include: list of collections and documents to be initialized
        :param collection: the alarm configuration collection to initialize. (CNTL-ALARM-CFG, OLT-ALARM-CFG, or ONU-ALARM-CFG)
        :return: None
        """
        directory = ""

        # Set directory
        if collection == "CNTL-ALARM-CFG":
            directory = self.database_seed_base_dir + "documents/alarmConfigs/controllers/"
        if collection == "OLT-ALARM-CFG":
            directory = self.database_seed_base_dir + "documents/alarmConfigs/olts/"
        if collection == "ONU-ALARM-CFG":
            directory = self.database_seed_base_dir + "documents/alarmConfigs/onus/"

        if os.path.isdir(directory):
            file_list = os.listdir(directory)
            if file_list:
                for filename in os.listdir(directory):
                    # Strip filename
                    doc_id = filename[14:-5]
                    if collection == "CNTL-ALARM-CFG":
                        doc_id = doc_id[1:]
                    # Insert document if to be included
                    if collection in collections_to_include and "-ALARM-CFG" in filename and filename.lower().endswith(
                            ".json"):
                        with open(directory + filename) as json_file:
                            self.database[collection].replace_one({"_id": doc_id}, json.load(json_file), upsert=True)

    def initialize_device_configs_collection(self, collections_to_include, collection):
        """
        Insert all default device configuration documents into the specified database collection.
        :param collections_to_include: list of collections and documents to be initialized
        :param collection: the device configuration collection to initialize. (CNTL-CFG, OLT-CFG, ONU-CFG, or SWI-CFG)
        :return: None
        """
        directory = ""

        # Set directory
        if collection == "CNTL-CFG":
            directory = self.database_seed_base_dir + "documents/deviceConfigs/controllers/"
        if collection == "OLT-CFG":
            directory = self.database_seed_base_dir + "documents/deviceConfigs/olts/"
        if collection == "ONU-CFG":
            directory = self.database_seed_base_dir + "documents/deviceConfigs/onus/"
        if collection == "SWI-CFG":
            directory = self.database_seed_base_dir + "documents/deviceConfigs/switches/"

        if os.path.isdir(directory):
            file_list = os.listdir(directory)
            if file_list:
                if directory != "":
                    for filename in os.listdir(directory):
                        # Strip filename
                        doc_id = filename[8:-5]
                        if collection == "CNTL-CFG":
                            doc_id = doc_id[1:]
                        # Insert document if to be included
                        if collection in collections_to_include and "-CFG" in filename and filename.lower().endswith(".json"):
                            with open(directory + filename) as json_file:
                                # Add index for NETCONF.Name if not already in collection
                                if "NETCONF.Name_1" not in self.database[collection].index_information():
                                    self.database[collection].create_index("NETCONF.Name", unique=True)
                                self.database[collection].replace_one({"_id": doc_id}, json.load(json_file), upsert=True)

    def initialize_pon_auto_configs_collection(self, collections_to_include, collection):
        """
        Insert all default/global pon automation configuration documents into the specified database collection.
        :param collections_to_include: list of collections and documents to be initialized
        :param collection: the device configuration collection to initialize. (CNTL-AUTO-CFG, OLT-AUTO-CFG, ONU-AUTOCFG, SWI-AUTO-CFG, or AUTO-CFG)
        :return: None
        """
        directory = ""

        # Set directory
        if collection == "CNTL-AUTO-CFG":
            directory = self.database_seed_base_dir + "documents/ponAutoConfigs/controllers/"
        if collection == "OLT-AUTO-CFG":
            directory = self.database_seed_base_dir + "documents/ponAutoConfigs/olts/"
        if collection == "ONU-AUTO-CFG":
            directory = self.database_seed_base_dir + "documents/ponAutoConfigs/onus/"
        if collection == "SWI-AUTO-CFG":
            directory = self.database_seed_base_dir + "documents/ponAutoConfigs/switches/"
        if collection == "AUTO-CFG":
            directory = self.database_seed_base_dir + "documents/ponAutoConfigs/automation/"
        if collection == "AUTO-TASK-CFG":
            directory = self.database_seed_base_dir + "documents/ponAutoConfigs/tasks/"

        if os.path.isdir(directory):
            file_list = os.listdir(directory)
            if file_list:
                if directory != "":
                    for filename in os.listdir(directory):
                        # Strip filename
                        if collection == "AUTO-CFG":
                            doc_id = filename[9:-5]
                        else:
                            doc_id = filename[13:-5]
                            if collection == "CNTL-AUTO-CFG" or collection == "AUTO-TASK-CFG":
                                doc_id = doc_id[1:]
                        # Insert document if to be included
                        if collection in collections_to_include and "-CFG" in filename and filename.lower().endswith(".json"):
                            with open(directory + filename) as json_file:
                                self.database[collection].replace_one({"_id": doc_id}, json.load(json_file), upsert=True)

    def initialize_sla_configs_collection(self, documents_to_include):
        """
        Insert all default SLA configuration documents into the SLA-CFG collection.
        :param documents_to_include: list of which SLA documents to initialize
        :return: None
        """
        directory = self.database_seed_base_dir + "documents/slaConfigs/"
        if os.path.isdir(directory):
            file_list = os.listdir(directory)
            if file_list:
                for filename in os.listdir(directory):
                    # Strip filename
                    doc_id = filename[8:-5]
                    # Insert document if to be included
                    if doc_id in documents_to_include and "SLA-CFG" in filename and filename.lower().endswith(".json"):
                        with open(directory + filename) as json_file:
                            self.database["SLA-CFG"].replace_one({"_id": doc_id}, json.load(json_file), upsert=True)

    def initialize_ds_map_configs_collection(self, documents_to_include):
        """
        Insert all default DS-MAP-CFG configuration documents into the DS-MAP-CFG collection.
        :param documents_to_include: list of which DS-MAP-CFG documents to initialize
        :return: None
        """
        directory = self.database_seed_base_dir + "documents/dsMapConfigs/"
        if os.path.isdir(directory):
            file_list = os.listdir(directory)
            if file_list:
                for filename in os.listdir(directory):
                    # Strip filename
                    doc_id = filename[11:-5]
                    # Insert document if to be included
                    if doc_id in documents_to_include and "DS-MAP-CFG" in filename and filename.lower().endswith(".json"):
                        with open(directory + filename) as json_file:
                            self.database["DS-MAP-CFG"].replace_one({"_id": doc_id}, json.load(json_file), upsert=True)

    def initialize_service_configs_collection(self, documents_to_include):
        """
        Insert all default Service configuration documents into the SRV-CFG collection.
        :param documents_to_include: list of which Service configuration documents to initialize
        :return: None
        """
        directory = self.database_seed_base_dir + "documents/srvConfigs/"
        if os.path.isdir(directory):
            file_list = os.listdir(directory)
            if file_list:
                for filename in os.listdir(directory):
                    # Strip document
                    doc_id = filename[4:-9]
                    # Insert document if to be included
                    if doc_id in documents_to_include and "SRV-" in filename and filename.lower().endswith(".json"):
                        with open(directory + filename) as json_file:
                            self.database["SRV-CFG"].replace_one({"_id": doc_id}, json.load(json_file), upsert=True)

    def initialize_olt_firmware_bucket(self, files_to_include):
        """
        Upload all default OLT firmware files into the OLT-FIRMWARE bucket.
        :param files_to_include: list of which files to initialize
        :return: None
        """
        fs = gridfs.GridFSBucket(db=self.database, bucket_name="OLT-FIRMWARE")
        directory = self.database_seed_base_dir + "firmwares/olts/"
        if os.path.isdir(directory):
            file_list = os.listdir(directory)
            if file_list:
                for filename in os.listdir(directory):
                    # Strip filename
                    file_id = os.path.splitext(filename)[0]
                    # Exclude unexpected files
                    if filename in files_to_include.keys():
                        # Delete old file if it exists
                        if fs.find().collection.count_documents({"_id": file_id}) > 0:
                            fs.delete(file_id=file_id)
                        if self.database['OLT-FIRMWARE.chunks'].count_documents({'files_id': filename}) > 0:
                            self.database['OLT-FIRMWARE.chunks'].delete_many({'files_id': filename})

                        file_data = Binary(open(directory + filename, mode='rb').read())

                        # Upload new file
                        up = fs.open_upload_stream_with_id(file_id=file_id, filename=filename,
                                                           metadata=files_to_include[filename])
                        up.write(file_data)
                        up.close()
                        database_manager.update_checksum(self.database,'OLT-FIRMWARE', file_id, filename, file_data)


    def initialize_onu_firmware_bucket(self, files_to_include):
        """
        Upload all default ONU firmware files into the ONU-FIRMWARE bucket.
        :param files_to_include: list of which files to initialize
        :return: None
        """
        fs = gridfs.GridFSBucket(db=self.database, bucket_name="ONU-FIRMWARE")
        directory = self.database_seed_base_dir + "firmwares/onus/"
        if os.path.isdir(directory):
            file_list = os.listdir(directory)
            if file_list:
                for filename in os.listdir(directory):
                    # Exclude unexpected files
                    if filename in files_to_include.keys():
                        # Delete old file if it exists
                        if fs.find().collection.count_documents({"_id": filename}) > 0:
                            fs.delete(file_id=filename)
                        if self.database['ONU-FIRMWARE.chunks'].count_documents({'files_id': filename}) > 0:
                            self.database['ONU-FIRMWARE.chunks'].delete_many({'files_id': filename})

                        file_data = Binary(open(directory + filename, mode='rb').read())

                        # Upload new file
                        up = fs.open_upload_stream_with_id(file_id=filename, filename=filename,
                                                           metadata=files_to_include[filename])
                        up.write(file_data)
                        up.close()
                        database_manager.update_checksum(self.database,'ONU-FIRMWARE', filename, filename, file_data)

    def initialize_picture_bucket(self, files_to_include):
        """
        Upload all default device pictures into the PICTURES bucket.
        Scales all pictures to a maximum height of 200 pixels.
        :param files_to_include: list of which files to initialize
        :return: None
        """
        fs = gridfs.GridFSBucket(db=self.database, bucket_name="PICTURES")
        directory = self.database_seed_base_dir + "pictures/"
        if os.path.isdir(directory):
            file_list = os.listdir(directory)
            if file_list:
                for device_directory in os.listdir(directory):
                    if device_directory in ["controllers", "olts", "onus", "switches"]:
                        for filename in os.listdir(directory + device_directory):
                            # Strip filename for file_id
                            file_id = os.path.splitext(filename)[0]
                            if filename in files_to_include.keys():
                                # Delete old picture if it exists
                                if fs.find().collection.count_documents({"_id": file_id}) > 0:
                                    fs.delete(file_id=file_id)
                                if self.database['PICTURES.chunks'].count_documents({'files_id': filename}) > 0:
                                    self.database['PICTURES.chunks'].delete_many({'files_id': filename})

                                # Read file
                                image_bytes = io.BytesIO()
                                binary = Binary(open(directory + device_directory + "/" + filename, mode='rb').read())

                                with Image.open(io.BytesIO(binary)) as image:
                                    # Shrink to height of 200 if greater than 200 pixels tall
                                    if image.size[1] > 400:
                                        file_data = resizeimage.resize_height(image, 400)
                                        file_data.save(image_bytes, format='PNG')
                                        image_bytes = image_bytes.getvalue()
                                    else:
                                        image_bytes = binary

                                # Upload new picture
                                up = fs.open_upload_stream_with_id(file_id=file_id, filename=filename,
                                                                   metadata=files_to_include[filename])
                                up.write(image_bytes)
                                up.close()
                                database_manager.update_checksum(self.database, 'PICTURES', file_id, filename, image_bytes)

    def get_seed_documents_info(self):
        """Returns the filenames and collections for all files in the documents directory"""
        data = []
        for collection_type in os.listdir(self.database_seed_base_dir + "documents/"):
            if collection_type in ["deviceConfigs", "alarmConfigs", "srvConfigs", "slaConfigs", 'dsMapConfigs', 'ponAutoConfigs']:
                # Determine collection to insert to and filename
                for device_type in os.listdir(self.database_seed_base_dir + "documents/" + collection_type):
                    collection = ""
                    if collection_type == "slaConfigs":
                        if device_type.lower().endswith('.json'):
                            filename = self.database_seed_base_dir + "documents/" + collection_type + "/" + device_type
                            if "SLA-CFG" in filename:
                                data.append((filename, "SLA-CFG"))
                    elif collection_type == "dsMapConfigs":
                        if device_type.lower().endswith('.json'):
                            filename = self.database_seed_base_dir + "documents/" + collection_type + "/" + device_type
                            if "DS-MAP-CFG" in filename:
                                data.append((filename, "DS-MAP-CFG"))
                    elif collection_type == "srvConfigs":
                        if device_type.lower().endswith('.json'):
                            filename = self.database_seed_base_dir + "documents/" + collection_type + "/" + device_type
                            if "SRV-" in filename:
                                data.append((filename, "SRV-CFG"))
                    elif collection_type == "alarmConfigs":
                        if device_type in ["olts", "onus", "controllers"]:
                            for file in os.listdir(
                                    self.database_seed_base_dir + "documents/" + collection_type + "/" + device_type + "/"):
                                filename = self.database_seed_base_dir + "documents/" + collection_type + "/" + device_type + "/" + file
                                if "-CFG" in filename:
                                    if device_type == "controllers":
                                        collection = "CNTL-ALARM-CFG"
                                    elif device_type == "olts":
                                        collection = "OLT-ALARM-CFG"
                                    elif device_type == "onus":
                                        collection = "ONU-ALARM-CFG"
                                    data.append((filename, collection))
                    elif collection_type == "deviceConfigs":
                        if device_type in ["olts", "onus", "controllers", "switches"]:
                            for file in os.listdir(
                                    self.database_seed_base_dir + "documents/" + collection_type + "/" + device_type + "/"):
                                filename = self.database_seed_base_dir + "documents/" + collection_type + "/" + device_type + "/" + file
                                if "-CFG" in filename:
                                    if device_type == "controllers":
                                        collection = "CNTL-CFG"
                                    elif device_type == "olts":
                                        collection = "OLT-CFG"
                                    elif device_type == "onus":
                                        collection = "ONU-CFG"
                                    elif device_type == "switches":
                                        collection = "SWI-CFG"
                                    data.append((filename, collection))
                    elif collection_type == "ponAutoConfigs":
                        if device_type in ["olts", "onus", "controllers", "switches", "automation", "tasks"]:
                            for file in os.listdir(
                                    self.database_seed_base_dir + "documents/" + collection_type + "/" + device_type + "/"):
                                filename = self.database_seed_base_dir + "documents/" + collection_type + "/" + device_type + "/" + file
                                if "-CFG" in filename:
                                    if device_type == "controllers":
                                        collection = "CNTL-AUTO-CFG"
                                    elif device_type == "olts":
                                        collection = "OLT-AUTO-CFG"
                                    elif device_type == "onus":
                                        collection = "ONU-AUTO-CFG"
                                    elif device_type == "switches":
                                        collection = "SWI-AUTO-CFG"
                                    elif device_type == "automation":
                                        collection = "AUTO-CFG"
                                    elif device_type == "tasks":
                                        collection = "AUTO-TASK-CFG"
                                    data.append((filename, collection))
        return data

    """
    Database Seeding
    """

    def get_seed_files(self, request, user_email: str):
        """Gives the names and metadata of all files to seed a database with

        Args:
            user_email: The email of the user requesting the information

        Returns:
            Response object containing the filenames and metadata for all files to seed a database with
        """

        response = [status.HTTP_200_OK,
                    {
                        "pictures": [], "oltFirmwares": [], "onuFirmwares": [],
                        "CFG": [], "ALARM-CFG": [], "SLA-CFG": [], "DS-MAP-CFG": [],
                        "SRV-CFG": [], "AUTO-CFG": []
                    }]

        try:
            with open(self.database_seed_base_dir + "metadata.json", "r") as json_file:
                metadata_file = json.load(json_file)
            for directory in os.listdir(self.database_seed_base_dir):
                if directory == "firmwares" or directory == "pictures":
                    current_dir = self.database_seed_base_dir + directory
                    for device_directory in os.listdir(current_dir):
                        current_dir = self.database_seed_base_dir + directory + "/" + device_directory
                        if os.path.isdir(current_dir):
                            for filename in os.listdir(current_dir):
                                file_to_be = directory + "/" + device_directory + "/" + filename
                                if file_to_be in metadata_file:
                                    value = {
                                        "filename": filename,
                                        "metadata": metadata_file[file_to_be]
                                    }
                                    if directory == "firmwares":
                                        if device_directory == "olts":
                                            response[1]["oltFirmwares"].append(value)
                                        else:
                                            response[1]["onuFirmwares"].append(value)
                                    else:
                                        response[1]["pictures"].append(value)
            for (filename, collection) in self.get_seed_documents_info():
                doc_id = str(filename)[str(filename).rfind("/") + 1:]

                if collection == "SLA-CFG":
                    response[1][collection].append({"collection": collection, "docId": doc_id[8:-5]})
                if collection == "DS-MAP-CFG":
                    response[1][collection].append({"collection": collection, "docId": doc_id[11:-5]})
                elif collection == "SRV-CFG":
                    response[1][collection].append({"collection": collection, "docId": doc_id[4:-9]})
                elif "ALARM" in collection:
                    response[1]["ALARM-CFG"].append(
                        {"collection": collection, "docId": doc_id[doc_id.rfind("-") + 1:-5]})
                elif collection == "SWI-CFG" or collection == "CNTL-CFG" or collection == "OLT-CFG" or collection == "ONU-CFG":
                    response[1]["CFG"].append({"collection": collection, "docId": doc_id[doc_id.rfind("-") + 1:-5]})
                elif collection == "SWI-AUTO-CFG" or collection == "CNTL-AUTO-CFG" \
                        or collection == "OLT-AUTO-CFG" or collection == "ONU-AUTO-CFG" or collection == "AUTO-CFG" \
                        or collection == "AUTO-TASK-CFG":
                    response[1]["AUTO-CFG"].append({"collection": collection, "docId": doc_id[doc_id.rfind("-") + 1:-5]})
        except Exception as err:
            self.status_log("EXCEPTION (get_seed_files)", sys.exc_info()[1])
            messages.error(request, f"{type(err)}")
            messages.error(request, traceback.format_exc(), extra_tags='stacktrace')
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get seed files due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "databaseSeedFiles/", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response


    """
    Database Collections
    """

    def check_collection_existence(self, request, user_email: str, collection_names):
        response = [status.HTTP_200_OK, []]

        collection_names = collection_names.split(",")

        try:
            database_manager.get_database(self.database_id)
            db_collections = self.database.list_collection_names()

            for collection in collection_names:
                collection_document = {'name': collection, 'exists': False}
                if collection in db_collections:
                    collection_document['exists'] = True

                response[1].append(collection_document)

        except Exception as err:
            self.status_log("EXCEPTION (check_collection_existence)", sys.exc_info()[1])
            messages.error(request, f"{type(err)}")
            messages.error(request, traceback.format_exc(), extra_tags='stacktrace')
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not check collection existence due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "COLLECTIONS", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response


    """
    Delete devices
    """

    # Remove all data for the specified ONU from database
    def delete_onu_all(self, request, onu_id, delete_from_olt_inv):
        response = [status.HTTP_200_OK, f"All ONU {onu_id} data was deleted."]
        user_email = request.user.email

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["ONU-STATE"]
            collection.delete_one({"_id": onu_id})
            collection = self.database["ONU-CFG"]
            collection.delete_one({"_id": onu_id})
            collection = self.database["STATS-ONU-{}".format(onu_id.replace(":", ""))]
            collection.drop()
            collection = self.database["SYSLOG-ONU-{}".format(onu_id.replace(":", ""))]
            collection.drop()
            collection = self.database["ONU-MIB-CUR-STATE"]
            collection.delete_one({"_id": onu_id})
            collection = self.database["ONU-MIB-RST-STATE"]
            collection.delete_one({"_id": onu_id})
            collection = self.database["ONU-CPE-STATE"]
            collection.delete_one({"_id": onu_id})
            collection = self.database["ONU-AUTO-CFG"]
            collection.delete_one({"_id": onu_id})
            collection = self.database["ONU-AUTO-STATE"]
            collection.delete_one({"_id": onu_id})
            collection = self.database["VCM-STATE"]
            collection.delete_one({"_id": onu_id})

            # Removes from parent OLT-CFG
            if (delete_from_olt_inv == "true"):
                collection = self.database["OLT-CFG"]
                querystring = "ONUs." + onu_id
                collection.update_many({querystring: {'$exists': 1}},
                                       {"$unset": {querystring: ""}, "$inc": {"OLT.CFG Change Count": 1}})

        except Exception as err:
            self.status_log("EXCEPTION (delete_onu_all)", sys.exc_info()[1])
            messages.error(request, f"{type(err)}")
            messages.error(request, traceback.format_exc(), extra_tags='stacktrace')
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not delete all ONU {onu_id} data due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {
            "collection": f"ONU-STATE, ONU-CPE-STATE, ONU-CFG, STATS-ONU-{onu_id.replace(':', '')}, SYSLOG-ONU-{onu_id.replace(':', '')}, ONU-MIB-CUR-STATE, ONU-MIB-RST-STATE",
            "deleted": response[0] == status.HTTP_200_OK, "user": user_email}
        self.sys_log(action_data, "delete", response[0] == status.HTTP_200_OK)

        return response

    def delete_many_onus(self, request, ids, delete_from_olt_inv):
        response = [status.HTTP_200_OK, f"All ONUs data was deleted."]
        user_email = request.user.email

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["ONU-STATE"]
            collection.delete_many({"_id": {'$in': ids}})
            collection = self.database["ONU-CFG"]
            collection.delete_many({"_id": {'$in': ids}})
            for onu_id in ids:
                collection = self.database["STATS-ONU-{}".format(onu_id.replace(":", ""))]
                collection.drop()
                collection = self.database["SYSLOG-ONU-{}".format(onu_id.replace(":", ""))]
                collection.drop()
            collection = self.database["ONU-MIB-CUR-STATE"]
            collection.delete_many({"_id": {'$in': ids}})
            collection = self.database["ONU-MIB-RST-STATE"]
            collection.delete_many({"_id": {'$in': ids}})
            collection = self.database["ONU-CPE-STATE"]
            collection.delete_many({"_id": {'$in': ids}})
            collection = self.database["ONU-AUTO-CFG"]
            collection.delete_many({"_id": {'$in': ids}})
            collection = self.database["ONU-AUTO-STATE"]
            collection.delete_many({"_id": {'$in': ids}})
            collection = self.database["VCM-STATE"]
            collection.delete_many({"_id": {'$in': ids}})

            # Removes from parent OLT-CFG
            if delete_from_olt_inv == "true":
                for onu_id in ids:
                    collection = self.database["OLT-CFG"]
                    querystring = "ONUs." + onu_id
                    collection.update_many({querystring: {'$exists': 1}},
                                           {"$unset": {querystring: ""}, "$inc": {"OLT.CFG Change Count": 1}})

        except Exception as err:
            self.status_log("EXCEPTION (delete_onu_all)", sys.exc_info()[1])
            messages.error(request, f"{type(err)}")
            messages.error(request, traceback.format_exc(), extra_tags='stacktrace')
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not delete all ONUs bulk data due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {
            "collection": f"ONU-STATE, ONU-CPE-STATE, ONU-CFG, STATS-ONU-bulk, SYSLOG-ONU-bulk, ONU-MIB-CUR-STATE, ONU-MIB-RST-STATE",
            "deleted": response[0] == status.HTTP_200_OK, "user": user_email}
        self.sys_log(action_data, "delete", response[0] == status.HTTP_200_OK)

        return response

    # Remove all data for the specified ONU from database
    def delete_onu_logs(self, request, onu_id):
        response = [status.HTTP_200_OK, "All ONU {} logs were deleted.".format(onu_id)]
        user_email = request.user.email

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["SYSLOG-ONU-{}".format(onu_id.replace(":", ""))]
            collection.drop()
        except Exception as err:
            self.status_log("EXCEPTION (delete_onu_logs)", sys.exc_info()[1])
            messages.error(request, f"{type(err)}")
            messages.error(request, traceback.format_exc(), extra_tags='stacktrace')
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not delete ONU {onu_id} logs due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "SYSLOG-ONU-{}".format(onu_id.replace(":", "")),
                       "deleted": response[0] == status.HTTP_200_OK, "user": user_email}
        self.sys_log(action_data, "delete", response[0] == status.HTTP_200_OK)

        return response

    # Remove all data for the specified ONU from database
    def delete_onu_stats(self, request, onu_id):
        response = [status.HTTP_200_OK, "All ONU {} statistics were deleted.".format(onu_id)]
        user_email = request.user.email

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["STATS-ONU-{}".format(onu_id.replace(":", ""))]
            collection.drop()
        except Exception as err:
            self.status_log("EXCEPTION (delete_onu_stats)", sys.exc_info()[1])
            messages.error(request, f"{type(err)}")
            messages.error(request, traceback.format_exc(), extra_tags='stacktrace')
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not delete ONU {onu_id} statistics due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "STATS-ONU-{}".format(onu_id.replace(":", "")),
                       "deleted": response[0] == status.HTTP_200_OK, "user": user_email}
        self.sys_log(action_data, "delete", response[0] == status.HTTP_200_OK)

        return response

    # Remove all data for the specified OLT from database
    def delete_olt_all(self, request, olt_id, delete_from_all_devices):
        response = [status.HTTP_200_OK, f"All OLT {olt_id} data was deleted."]
        user_email = request.user.email

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["OLT-STATE"]
            collection.delete_one({"_id": olt_id})
            collection = self.database["OLT-CFG"]
            collection.delete_one({"_id": olt_id})
            collection = self.database["STATS-OLT-{}".format(olt_id.replace(":", ""))]
            collection.drop()
            collection = self.database["SYSLOG-OLT-{}".format(olt_id.replace(":", ""))]
            collection.drop()
            collection = self.database["OLT-AUTO-CFG"]
            collection.delete_one({"_id": olt_id})
            collection = self.database["OLT-AUTO-STATE"]
            collection.delete_one({"_id": olt_id})

            if delete_from_all_devices == 'true':
                # Remove from switch inventories
                swi_collection = self.database["SWI-CFG"]
                filter_document = {f'OLTs.{olt_id}': {'$exists': True}}
                update_document = {'$unset': {f'OLTs.{olt_id}': ''}}
                swi_collection.update_many(filter=filter_document, update=update_document, upsert=False)

                # Remove from controller inventories
                cntl_collection = self.database["CNTL-CFG"]
                filter_document = {'$or': [
                    {'OLTs.Primary': olt_id},
                    {'OLTs.Secondary': olt_id},
                    {'OLTs.Excluded': olt_id},
                ]}
                update_document = {'$pull': {
                    'OLTs.Primary': olt_id,
                    'OLTs.Secondary': olt_id,
                    'OLTs.Excluded': olt_id
                }}
                cntl_collection.update_many(filter=filter_document, update=update_document, upsert=False)

                # Remove from ONUs
                onu_collection = self.database["ONU-CFG"]
                filter_document = {'OLT.MAC Address': olt_id}
                update_document = {'$pull': {'OLT.MAC Address': olt_id}}
                onu_collection.update_many(filter=filter_document, update=update_document, upsert=False)

        except Exception as err:
            self.status_log("EXCEPTION (delete_olt_all)", sys.exc_info()[1])
            messages.error(request, f"{type(err)}")
            messages.error(request, traceback.format_exc(), extra_tags='stacktrace')
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not delete all OLT {olt_id} data due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {
            "collection": f"OLT-STATE, OLT-CFG, STATS-OLT-{olt_id.replace(':', '')}, SYSLOG-OLT-{olt_id.replace(':', '')}",
            "deleted": response[0] == status.HTTP_200_OK, "user": user_email}
        self.sys_log(action_data, "delete", response[0] == status.HTTP_200_OK)

        return response

    def delete_many_olts(self, request, ids, delete_from_all_devices):
        response = [status.HTTP_200_OK, f"All OLTs data were deleted."]
        user_email = request.user.email

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["OLT-STATE"]
            collection.delete_many({"_id": {'$in': ids}})
            collection = self.database["OLT-CFG"]
            collection.delete_many({"_id": {'$in': ids}})
            for olt_id in ids:
                collection = self.database["STATS-OLT-{}".format(olt_id.replace(":", ""))]
                collection.drop()
                collection = self.database["SYSLOG-OLT-{}".format(olt_id.replace(":", ""))]
                collection.drop()
            collection = self.database["OLT-AUTO-CFG"]
            collection.delete_many({"_id": {'$in': ids}})
            collection = self.database["OLT-AUTO-STATE"]
            collection.delete_many({"_id": {'$in': ids}})

            if delete_from_all_devices == 'true':
                for olt_id in ids:
                    # Remove from switch inventories
                    swi_collection = self.database["SWI-CFG"]
                    filter_document = {f'OLTs.{olt_id}': {'$exists': True}}
                    update_document = {'$unset': {f'OLTs.{olt_id}': ''}}
                    swi_collection.update_many(filter=filter_document, update=update_document, upsert=False)

                    # Remove from controller inventories
                    cntl_collection = self.database["CNTL-CFG"]
                    filter_document = {'$or': [
                        {'OLTs.Primary': olt_id},
                        {'OLTs.Secondary': olt_id},
                        {'OLTs.Excluded': olt_id},
                    ]}
                    update_document = {'$pull': {
                        'OLTs.Primary': olt_id,
                        'OLTs.Secondary': olt_id,
                        'OLTs.Excluded': olt_id
                    }}
                    cntl_collection.update_many(filter=filter_document, update=update_document, upsert=False)

                    # Remove from ONUs
                    onu_collection = self.database["ONU-CFG"]
                    filter_document = {'OLT.MAC Address': olt_id}
                    update_document = {'$pull': {'OLT.MAC Address': olt_id}}
                    onu_collection.update_many(filter=filter_document, update=update_document, upsert=False)

        except Exception as err:
            self.status_log("EXCEPTION (delete_olt_all)", sys.exc_info()[1])
            messages.error(request, f"{type(err)}")
            messages.error(request, traceback.format_exc(), extra_tags='stacktrace')
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not delete all bulk OLT data due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {
            "collection": f"OLT-STATE, OLT-CFG, STATS-OLT-bulk, SYSLOG-OLT-bulk",
            "deleted": response[0] == status.HTTP_200_OK, "user": user_email}
        self.sys_log(action_data, "delete", response[0] == status.HTTP_200_OK)

        return response

    # Remove all logs for the specified OLT from database
    def delete_olt_logs(self, request, olt_id, user_email):
        response = [status.HTTP_200_OK, f"All OLT {olt_id} logs were deleted."]
        user_email = request.user.email

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["SYSLOG-OLT-{}".format(olt_id.replace(":", "").replace(":", ""))]
            collection.drop()
        except Exception as err:
            self.status_log("EXCEPTION (delete_olt_logs)", sys.exc_info()[1])
            messages.error(request, f"{type(err)}")
            messages.error(request, traceback.format_exc(), extra_tags='stacktrace')
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not delete OLT {olt_id} logs due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": f"SYSLOG-OLT-{olt_id.replace(':', '')}",
                       "deleted": response[0] == status.HTTP_200_OK, "user": user_email}
        self.sys_log(action_data, "delete", response[0] == status.HTTP_200_OK)

        return response

    # Remove all stats for the specified OLT from database
    def delete_olt_stats(self, request, olt_id):
        response = [status.HTTP_200_OK, f"All OLT {olt_id} statistics were deleted."]
        user_email = request.user.email

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["STATS-OLT-{}".format(olt_id.replace(":", ""))]
            collection.drop()
        except Exception as err:
            self.status_log("EXCEPTION (delete_olt_stats)", sys.exc_info()[1])
            messages.error(request, f"{type(err)}")
            messages.error(request, traceback.format_exc(), extra_tags='stacktrace')
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not delete OLT {olt_id} statistics due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": f"STATS-OLT-{olt_id.replace(':', '')}",
                       "deleted": response[0] == status.HTTP_200_OK, "user": user_email}
        self.sys_log(action_data, "delete", response[0] == status.HTTP_200_OK)

        return response

    # Remove all data for the specified CNTL from database
    def delete_cntl(self, request, cntl_id):
        response = [status.HTTP_200_OK, f"All PON Controller {cntl_id} data was deleted."]
        user_email = request.user.email

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["CNTL-STATE"]
            collection.delete_one({"_id": cntl_id})
            collection = self.database["CNTL-CFG"]
            collection.delete_one({"_id": cntl_id})
            collection = self.database["STATS-CNTL-{}".format(cntl_id.replace(":", ""))]
            collection.drop()
            collection = self.database["SYSLOG-CNTL-{}".format(cntl_id.replace(":", ""))]
            collection.drop()
            collection = self.database["CNTL-AUTO-CFG"]
            collection.delete_one({"_id": cntl_id})
            collection = self.database["CNTL-AUTO-STATE"]
            collection.delete_one({"_id": cntl_id})
        except Exception as err:
            self.status_log("EXCEPTION (delete_cntl_all)", sys.exc_info()[1])
            messages.error(request, f"{type(err)}")
            messages.error(request, traceback.format_exc(), extra_tags='stacktrace')
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not delete all PON Controller {cntl_id} data due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": f"CNTL-STATE, CNTL-CFG, SYSLOG-CNTL-{cntl_id.replace(':', '')}",
                       "deleted": response[0] == status.HTTP_200_OK, "user": user_email}
        self.sys_log(action_data, "delete", response[0] == status.HTTP_200_OK)

        return response

    # Remove all data for the specified CNTL from database
    def delete_cntl_logs(self, request, cntl_id):
        response = [status.HTTP_200_OK, f"All PON Controller {cntl_id} logs were deleted."]
        user_email = request.user.email

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["SYSLOG-CNTL-{}".format(cntl_id.replace(":", ""))]
            collection.drop()
        except Exception as err:
            self.status_log("EXCEPTION (delete_cntl_logs)", sys.exc_info()[1])
            messages.error(request, f"{type(err)}")
            messages.error(request, traceback.format_exc(), extra_tags='stacktrace')
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not delete PON Controller {cntl_id} logs due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": f"SYSLOG-CNTL-{cntl_id.replace(':', '')}",
                       "deleted": response[0] == status.HTTP_200_OK, "user": user_email}
        self.sys_log(action_data, "delete", response[0] == status.HTTP_200_OK)

        return response

    # Remove all stats for the specified PON Controller from database
    def delete_cntl_stats(self, request, cntl_id):
        response = [status.HTTP_200_OK, f"All PON Controller {cntl_id} statistics were deleted."]
        user_email = request.user.email

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["STATS-CNTL-{}".format(cntl_id.replace(":", ""))]
            collection.drop()
        except Exception as err:
            self.status_log("EXCEPTION (delete_cntl_stats)", sys.exc_info()[1])
            messages.error(request, f"{type(err)}")
            messages.error(request, traceback.format_exc(), extra_tags='stacktrace')
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not delete PON Controller {cntl_id} statistics due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": f"STATS-CNTL-{cntl_id.replace(':', '')}",
                       "deleted": response[0] == status.HTTP_200_OK, "user": user_email}
        self.sys_log(action_data, "delete", response[0] == status.HTTP_200_OK)

        return response

    # Remove all data for the specified Switch from database
    def delete_switch_all(self, request, switch_id):
        response = [status.HTTP_200_OK, f"All Switch {switch_id} data was deleted."]
        user_email = request.user.email

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["SWI-CFG"]
            collection.delete_one({"_id": switch_id})
            collection = self.database["STATS-SWI-{}".format(switch_id.replace(":", ""))]
            collection.drop()
            collection = self.database["SYSLOG-SWI-{}".format(switch_id.replace(":", ""))]
            collection.drop()
            collection = self.database["SWI-AUTO-CFG"]
            collection.delete_one({"_id": switch_id})
            collection = self.database["SWI-AUTO-STATE"]
            collection.delete_one({"_id": switch_id})
        except Exception as err:
            self.status_log("EXCEPTION (delete_switch_all)", sys.exc_info()[1])
            messages.error(request, f"{type(err)}")
            messages.error(request, traceback.format_exc(), extra_tags='stacktrace')
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not delete all Switch {switch_id} data due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": f"SWI-CFG, SYSLOG-SWI-{switch_id.replace(':', '')}",
                       "deleted": response[0] == status.HTTP_200_OK, "user": user_email}
        self.sys_log(action_data, "delete", response[0] == status.HTTP_200_OK)

        return response

    """
    Get all image names
    """

    def get_image_names(self, request):
        response = [status.HTTP_200_OK, []]
        user_email = request.user.email

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["PICTURES.files"]
            names = collection.find({}, {"metadata": 1})
            for name in names:
                response[1].append(name)
        except Exception as err:
            self.status_log("EXCEPTION (get_image_names)", sys.exc_info()[1])
            messages.error(request, f"{type(err)}")
            messages.error(request, traceback.format_exc(), extra_tags='stacktrace')
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get image names due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "PICTURES.files", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    """
    Get all image names that are in use by a device
    """

    def get_device_image_names(self, request):
        response = [status.HTTP_200_OK, []]
        user_email = request.user.email

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["SWI-CFG"]
            names = collection.distinct("SWI.Picture")
            for name in names:
                response[1].append(name)

            collection = self.database["CNTL-CFG"]
            names = collection.distinct("CNTL.Picture")
            for name in names:
                response[1].append(name)

            collection = self.database["OLT-CFG"]
            names = collection.distinct("OLT.Picture")
            for name in names:
                response[1].append(name)

            collection = self.database["ONU-CFG"]
            names = collection.distinct("ONU.Picture")
            for name in names:
                response[1].append(name)
        except Exception as err:
            self.status_log("EXCEPTION (get_device_image_names)", sys.exc_info()[1])
            messages.error(request, f"{type(err)}")
            messages.error(request, traceback.format_exc(), extra_tags='stacktrace')
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get image names due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "PICTURES.files", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    """
    Get first 10 devices that are using the given image name
    """

    def get_devices_from_image(self, request, image_name, device_type):
        response = [status.HTTP_200_OK, []]
        user_email = request.user.email

        try:
            database_manager.get_database(self.database_id)
            if device_type == 'OLT':
                collection = self.database["OLT-CFG"]
                names = collection.find({"OLT.Picture": image_name}, {'_id': 1, 'OLT.Name': 1}).limit(10)
                for name in names:
                    response[1].append({'_id': name['_id'], 'name': name['OLT']['Name']})
            if device_type == 'ONU':
                collection = self.database["ONU-CFG"]
                names = collection.find({"ONU.Picture": image_name}, {'_id': 1, 'ONU.Name': 1}).limit(10)
                for name in names:
                    response[1].append({'_id': name['_id'], 'name': name['ONU']['Name']})
            if device_type == 'CNTL':
                collection = self.database["CNTL-CFG"]
                names = collection.find({"CNTL.Picture": image_name}, {'_id': 1, 'CNTL.Name': 1}).limit(10)
                for name in names:
                    response[1].append({'_id': name['_id'], 'name': name['CNTL']['Name']})
            if device_type == 'SWI':
                collection = self.database["SWI-CFG"]
                names = collection.find({"SWI.Picture": image_name}, {'_id': 1, 'SWI.Name': 1}).limit(10)
                for name in names:
                    response[1].append({'_id': name['_id'], 'name': name['SWI']['Name']})
        except Exception as err:
            self.status_log("EXCEPTION (get_devices_from_image)", sys.exc_info()[1])
            messages.error(request, f"{type(err)}")
            messages.error(request, traceback.format_exc(), extra_tags='stacktrace')
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get device names due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "PICTURES.files", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    """
    Delete CPEs based on ONU ID
    """

    def delete_many_cpes(self, request, cpes, onu_id):
        response = [status.HTTP_200_OK, f"Deleted CPEs from {onu_id}."]
        user_email = request.user.email

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["ONU-CFG"]
            collection.update_one({'_id': onu_id}, {"$set": {"CPE.Delete Mac Addresses": cpes},
                                                    "$inc": {"CPE.Delete Count": 1}}, upsert=True)

        except Exception as err:
            self.status_log("EXCEPTION (delete_many_cpes)", sys.exc_info()[1])
            messages.error(request, f"{type(err)}")
            messages.error(request, traceback.format_exc(), extra_tags='stacktrace')
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not delete CPEs {cpes} due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "ONU-CFG",
                       "deleted": response[0] == status.HTTP_200_OK,
                       "user": user_email}
        self.sys_log(action_data, "delete", response[0] == status.HTTP_200_OK)

        return response

    """
    Get ALL CPEs based on ONU
    """

    def get_all_cpes_for_onu(self, request, onu_id):
        response = [status.HTTP_200_OK, []]
        user_email = request.user.email

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["ONU-CPE-STATE"]
            cpe_doc = collection.find_one({"_id": onu_id})
            if cpe_doc is not None:
                deleteCount = None
                cpe_field = cpe_doc.get('CPE', {})
                if cpe_field:
                    deleteCount = cpe_field.get('Delete Count', None)
                response[1].append(deleteCount)
                for cpe in cpe_doc["CPEs"]:
                    cpef = {"_id": cpe, "CNTL": cpe_doc["CNTL"], "OLT": cpe_doc["OLT"]}
                    cpef.update(cpe_doc['CPEs'][cpe])
                    response[1].append(cpef)
        except Exception as err:
            self.status_log("EXCEPTION (get_all_cpes_for_onu)", sys.exc_info()[1])
            messages.error(request, f"{type(err)}")
            messages.error(request, traceback.format_exc(), extra_tags='stacktrace')
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get CPEs for ONU {onu_id} due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "ONU-CPE-STATE", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    """
    Check if the ONU-CPE-State collection exists
    """

    def check_cpes_exist(self, request):
        response = [status.HTTP_200_OK, []]
        user_email = request.user.email

        try:
            database_manager.get_database(self.database_id)
            all_collections = self.database.list_collection_names()

            if "ONU-CPE-STATE" in all_collections:
                response[1] = True
            else:
                response[1] = False

        except Exception as err:
            self.status_log("EXCEPTION (check_cpes_exist)", sys.exc_info()[1])
            messages.error(request, f"{type(err)}")
            messages.error(request, traceback.format_exc(), extra_tags='stacktrace')
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not check for ONU-CPE-STATE collection due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "ONU-CPE-STATE", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    def get_cpe_state(self, request, cpe_id):
        response = [status.HTTP_200_OK, []]
        user_email = request.user.email

        try:
            collection = self.database["ONU-CPE-STATE"]
            cpe_doc = collection.find_one({"CPEs." + cpe_id: {"$exists": True}})
            cpef = {}
            if cpe_doc is not None:
                # Reformatting to match the old schema to avoid unforeseen front-end problems
                cpef = {"_id": cpe_id, "ONU": {"ID": cpe_doc["_id"]}, "CNTL": cpe_doc["CNTL"], "OLT": cpe_doc["OLT"]}
                cpef.update(cpe_doc['CPEs'][cpe_id])
            response[1] = cpef
        except Exception as err:
            self.status_log("EXCEPTION (get_cpe_state)", sys.exc_info()[1])
            messages.error(request, f"{type(err)}")
            messages.error(request, traceback.format_exc(), extra_tags='stacktrace')
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get cpe state for {cpe_id} due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "ONU-CPE-STATE", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    def get_vcm_state_for_onu(self, request, onu_id):
        response = [status.HTTP_200_OK, {}]
        user_email = request.user.email

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["VCM-STATE"]

            if onu_id != '':
                vcm_state = collection.find_one({"_id": onu_id})
            else:
                vcm_state = collection.find_one({})  # If ONU ID not provided, get first entry (Checking for collection existance)

            response[1] = vcm_state
        except Exception as err:
            self.status_log("EXCEPTION (get_vcm_state_for_onu)", sys.exc_info()[1])
            messages.error(request, f"{type(err)}")
            messages.error(request, traceback.format_exc(), extra_tags='stacktrace')
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get vCM State for ONU {onu_id} due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "VCM-STATE", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    def get_vcm_state_exist(self, request):
        response = [status.HTTP_200_OK, {'Exists': False}]
        user_email = request.user.email

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["VCM-STATE"]

            vcm_state = collection.find_one({})  # If ONU ID not provided, get first entry (Checking for collection existance)

            if vcm_state is not None:
                response[1] = {'Exists': True}
        except Exception as err:
            self.status_log("EXCEPTION (get_vcm_state_exists)", sys.exc_info()[1])
            messages.error(request, f"{type(err)}")
            messages.error(request, traceback.format_exc(), extra_tags='stacktrace')
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get vCM State exists due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "VCM-STATE", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    """
    Get FW Filenames
    """

    def get_olt_fw_filenames(self, request):
        response = [status.HTTP_200_OK, []]
        user_email = request.user.email

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["OLT-FIRMWARE.files"]
            names = collection.find({}, {"metadata": 1, "filename": 1})
            for name in names:
                response[1].append(name)
        except Exception as err:
            self.status_log("EXCEPTION (get_olt_fw_filenames)", sys.exc_info()[1])
            messages.error(request, f"{type(err)}")
            messages.error(request, traceback.format_exc(), extra_tags='stacktrace')
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get OLT firmware filenames due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "OLT-FIRMWARE", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    def get_onu_fw_filenames(self, request):
        response = [status.HTTP_200_OK, []]
        user_email = request.user.email

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["ONU-FIRMWARE.files"]
            names = collection.find({}, {"metadata": 1, "filename": 1})
            for name in names:
                response[1].append(name)
        except Exception as err:
            self.status_log("EXCEPTION (get_onu_fw_filenames)", sys.exc_info()[1])
            messages.error(request, f"{type(err)}")
            messages.error(request, traceback.format_exc(), extra_tags='stacktrace')
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get ONU firmware filenames due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "ONU-FIRMWARE.files", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    """
    Get a device name
    """

    def get_onu_name(self, request, onu_id):
        response = [status.HTTP_200_OK, f"Retrieved ONU {onu_id} name."]
        user_email = request.user.email

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["ONU-CFG"]
            response[1] = collection.find_one({"_id": onu_id}, {"_id": 1, "ONU.Name": 1})
        except Exception as err:
            self.status_log("EXCEPTION (get_onu_name)", sys.exc_info()[1])
            messages.error(request, f"{type(err)}")
            messages.error(request, traceback.format_exc(), extra_tags='stacktrace')
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get ONU {onu_id} name due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "ONU-CFG", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    def get_olt_name(self, request, olt_id):
        response = [status.HTTP_200_OK, f"Retrieved OLT {olt_id} name."]
        user_email = request.user.email

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["OLT-CFG"]
            response[1] = collection.find_one({"_id": olt_id}, {"_id": 1, "OLT.Name": 1})
        except Exception as err:
            self.status_log("EXCEPTION (get_olt_name)", sys.exc_info()[1])
            messages.error(request, f"{type(err)}")
            messages.error(request, traceback.format_exc(), extra_tags='stacktrace')
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get OLT {olt_id} name due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "OLT-CFG", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    def get_cntl_name(self, request, cntl_id):
        response = [status.HTTP_200_OK, f"Retrieved PON Controller {cntl_id} name."]
        user_email = request.user.email

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["CNTL-CFG"]
            response[1] = collection.find_one({"_id": cntl_id}, {"_id": 1, "CNTL.Name": 1})
        except Exception as err:
            self.status_log("EXCEPTION (get_cntl_name)", sys.exc_info()[1])
            messages.error(request, f"{type(err)}")
            messages.error(request, traceback.format_exc(), extra_tags='stacktrace')
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get PON Controller {cntl_id} name due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "CNTL-CFG", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    """
    User methods
    """

    def create_user(self, request, user_data):
        response = [status.HTTP_201_CREATED, "Created new user."]

        try:
            # Set users email as document id
            collection = self.database["USERS"]
            result = collection.insert_one(user_data)
            if not result.acknowledged:
                messages.error(request, "")
                response = [status.HTTP_500_INTERNAL_SERVER_ERROR, "Failed to create new user."]
            if result.inserted_id is None:
                messages.error(request, "Insert ID is None")
                response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                            "Failed to create new user due to Inserted ID is None.::Inserted ID is None."]
        except Exception as err:
            self.status_log("EXCEPTION (create_user)", sys.exc_info()[1])
            messages.error(request, f"{type(err)}")
            messages.error(request, traceback.format_exc(), extra_tags='stacktrace')
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Failed to create new user due to {type(err).__name__}::{sys.exc_info()[1]}"]

        return response

    def authenticate_user(self, request, user_data):
        response = [status.HTTP_200_OK, user_data]

        try:
            collection = self.database["USERS"]
            if collection.find({"_id": user_data["_id"], "password": user_data.password}).count() == 0:
                messages.error(request, "User not found")
                response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                            "Failed to authenticate user due to user not found.::User not found."]
        except Exception as err:
            self.status_log("EXCEPTION (authenticate_user)", sys.exc_info()[1])
            messages.error(request, f"{type(err)}")
            messages.error(request, traceback.format_exc(), extra_tags='stacktrace')
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Failed to authenticate user due to {type(err).__name__}::{sys.exc_info()[1]}"]

        return response

    def user_exists(self, request, email_address):
        response = [status.HTTP_200_OK, "User exists."]

        try:
            collection = self.database["USERS"]
            if collection.find({"_id": email_address}).count() == 0:
                response[1] = "User does not exists."
        except Exception as err:
            self.status_log("EXCEPTION (user_exists)", sys.exc_info()[1])
            messages.error(request, f"{type(err)}")
            messages.error(request, traceback.format_exc(), extra_tags='stacktrace')
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Failed to check if user exists user due to {type(err).__name__}::{sys.exc_info()[1]}"]

        return response

    def get_minimum_password_requirements(self, request, user_email):
        response = [status.HTTP_200_OK, {}]

        try:
            collection = self.user_database["PONMGR-CFG"]
            expiration = collection.find_one({"_id": "Default"}, {"Password Requirements": 1, "_id": 0})

            if expiration is not None:
                response[1] = expiration['Password Requirements']
        except Exception as err:
            messages.error(request, f"{type(err)}")
            messages.error(request, traceback.format_exc(), extra_tags='stacktrace')
            response[0] = status.HTTP_500_INTERNAL_SERVER_ERROR
            self.status_log("EXCEPTION (get_minimum_password_requirements)", sys.exc_info()[1])

        action_data = {"collection": "PONMGR-CFG", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    def set_minimum_password_requirements(self, request, password_requirements_object):
        response = [status.HTTP_200_OK, 'OK', None, {}]
        user_email = request.user.email

        try:
            collection = self.user_database["PONMGR-CFG"]
            update_document = {"$set": {"Password Requirements": password_requirements_object}}
            update_result = collection.update_one(filter={'_id': 'Default'}, update=update_document, upsert=True)
            if update_result.upserted_id == "Default":
                response[0] = status.HTTP_201_CREATED
            else:
                response[0] = status.HTTP_200_OK
            response[2] = update_document
        except Exception as err:
            messages.error(request, f"{type(err)}")
            messages.error(request, traceback.format_exc(), extra_tags='stacktrace')
            response[0] = status.HTTP_500_INTERNAL_SERVER_ERROR
            self.status_log("EXCEPTION (set_minimum_password_requirements)", sys.exc_info()[1])

        action_data = {"collection": "PONMGR-CFG", "old": None, "new": password_requirements_object,
                       "user": user_email}
        self.sys_log(action_data, "write", response[0] == status.HTTP_200_OK or response[0] == status.HTTP_201_CREATED)
        return response

    """
    Return list of all users, each with a list of groups they belong to
    """
    def get_all_users_with_groups(self):
        response = []
        try:
            response = list(self.user_database['auth_user'].aggregate([
                {
                    '$lookup': {
                        'from': 'auth_user_groups',
                        'localField': 'id',
                        'foreignField': 'user_id',
                        'as': 'user_group_relation'
                    }
                }, {
                    '$unwind': {
                        'path': '$user_group_relation',
                        'preserveNullAndEmptyArrays': True
                    }
                }, {
                    '$lookup': {
                        'from': 'auth_group',
                        'localField': 'user_group_relation.group_id',
                        'foreignField': 'id',
                        'as': 'groups_lookup'
                    }
                }, {
                    '$unwind': {
                        'path': '$groups_lookup',
                        'preserveNullAndEmptyArrays': True
                    }
                }, {
                    '$group': {
                        '_id': '$id',
                        'email': {
                            '$first': '$email'
                        },
                        'doc': {
                            '$first': '$$ROOT'
                        },
                        'groups': {
                            '$addToSet': '$groups_lookup.name'
                        }
                    }
                }, {
                    '$replaceWith': {
                        '$mergeObjects': [
                            '$doc', {
                                'groups': '$groups'
                            }
                        ]
                    }
                }, {
                    '$unset': [
                        'password', 'groups_lookup', 'user_group_relation', '_id'
                    ]
                }
            ]))
        except Exception as e:
            self.status_log("EXCEPTION (get_all_users_with_groups)", sys.exc_info()[1])
            response = {}
        return response

    """
    Returns dictionary with all emails as keys and database type as value
    """

    def get_users_database_types(self):
        response = {}
        try:
            db_types_list = list(self.user_database["auth_user"].find({}, {"_id": 0, "email": 1, "database_type": 1}))

            db_types_dict = {}
            for entry in db_types_list:
                if 'email' in entry:
                    db_types_dict[entry['email']] = 'local'
                    if 'database_type' in entry:
                        db_types_dict[entry['email']] = entry['database_type']
            response = db_types_dict
        except Exception as e:
            self.status_log("EXCEPTION (get_users_database_types)", sys.exc_info()[1])
            response = {}
        return response

    def get_user_database_type(self, email):
        response = [status.HTTP_200_OK, 'OK', None, {}]
        try:
            db_types_list = list(self.user_database["auth_user"].find({'email': email}, {"_id": 0, "email": 1, "database_type": 1}))
            user_obj = db_types_list[0]
            if 'database_type' not in user_obj:
                user_obj['database_type'] = 'local'
            response = user_obj
        except Exception as e:
            self.status_log("EXCEPTION (get_users_database_types)", sys.exc_info()[1])
            response = {}
        return response

    def get_user_groups(self, user_email):
        """
        Get the groups/roles the given user is in
        """
        response = []
        try:
            groups_data = list(self.user_database["auth_user"].aggregate([
                {
                    '$match': {
                        'email': user_email
                    }
                }, {
                    '$lookup': {
                        'from': 'auth_user_groups',
                        'localField': 'id',
                        'foreignField': 'user_id',
                        'as': 'user-role relationships'
                    }
                }, {
                    '$unwind': {
                        'path': '$user-role relationships',
                        'preserveNullAndEmptyArrays': False
                    }
                }, {
                    '$lookup': {
                        'from': 'auth_group',
                        'localField': 'user-role relationships.group_id',
                        'foreignField': 'id',
                        'as': 'roles'
                    }
                }, {
                    '$unwind': {
                        'path': '$roles',
                        'preserveNullAndEmptyArrays': False
                    }
                }, {
                    '$group': {
                        '_id': '',
                        'groups': {
                            '$addToSet': '$roles.name'
                        }
                    }
                }
            ]))
            response = groups_data[0]['groups']
        except Exception as err:
            self.status_log(f"EXCEPTION (get_user_groups): {err}", sys.exc_info()[1])
            response = []

        return response

    def get_user_permissions(self, user_email):
        response = []
        try:
            permission_data = list(self.user_database["auth_user"].aggregate([
                {
                    '$match': {
                        'email': user_email
                    }
                }, {
                    '$lookup': {
                        'from': 'auth_user_groups',
                        'localField': 'id',
                        'foreignField': 'user_id',
                        'as': 'user-role relationships'
                    }
                }, {
                    '$unwind': {
                        'path': '$user-role relationships',
                        'preserveNullAndEmptyArrays': False
                    }
                }, {
                    '$lookup': {
                        'from': 'auth_group_permissions',
                        'localField': 'user-role relationships.group_id',
                        'foreignField': 'group_id',
                        'as': 'permissions'
                    }
                }, {
                    '$unwind': {
                        'path': '$permissions',
                        'preserveNullAndEmptyArrays': False
                    }
                }, {
                    '$lookup': {
                        'from': 'auth_permission',
                        'localField': 'permissions.permission_id',
                        'foreignField': 'id',
                        'as': 'permissions'
                    }
                }, {
                    '$unwind': {
                        'path': '$permissions',
                        'preserveNullAndEmptyArrays': False
                    }
                }, {
                    '$group': {
                        '_id': '',
                        'permissions': {
                            '$addToSet': '$permissions.codename'
                        }
                    }
                }
            ]))
            # Checks if global branding permissions are locked.
            if self.logo_setter_unlocked is False:
                permission_data[0]['permissions'] = list(filter(lambda x: "can_update_other_branding" not in x, permission_data[0]['permissions']))

            response = permission_data[0]['permissions']
        except Exception as err:
            self.status_log(f"EXCEPTION (get_user_permissions): {err}", sys.exc_info()[1])
            response = []

        return response

    def get_branding_control_settings(self):
        """
        Checks the custom-controls.json file to see if users of the PDM can change the logo.

            :return true or false
        """
        try:
            if IN_PRODUCTION:
                with open(self.production_branding_controls, 'r', encoding='utf-8') as control_file:
                    control_data = json.load(control_file)
                    can_brand = control_data["enable-app-logos"] == True
            else:
                with open(self.development_branding_controls, 'r', encoding='utf-8') as control_file:
                    control_data = json.load(control_file)
                    can_brand = control_data["enable-app-logos"] == True
        except Exception as e:
            self.status_log("EXCEPTION (get_branding_control_settings)", sys.exc_info()[1])
            can_brand = False

        return can_brand


    def get_all_roles_with_users_permissions_and_timeout(self, user_email):
        response = []
        try:
            pon_mgr_settings = self.user_database['PONMGR-CFG'].find_one('Default')
            groups_with_users = list(self.user_database['auth_group'].aggregate([
                {
                    '$lookup': {
                        'from': 'auth_user_groups',
                        'localField': 'id',
                        'foreignField': 'group_id',
                        'as': 'user_group'
                    }
                }, {
                    '$unwind': {
                        'path': '$user_group',
                        'preserveNullAndEmptyArrays': True
                    }
                }, {
                    '$lookup': {
                        'from': 'auth_user',
                        'localField': 'user_group.user_id',
                        'foreignField': 'id',
                        'as': 'user'
                    }
                }, {
                    '$unwind': {
                        'path': '$user',
                        'preserveNullAndEmptyArrays': True
                    }
                }, {
                    '$group': {
                        '_id': '$id',
                        'timeout': {
                            '$first': '$User Session Expiry Age Timeout'
                        },
                        'override': {
                            '$first': '$User Session Expiry Age Timeout Override'
                        },
                        'users': {
                            '$addToSet': '$user.email'
                        },
                        'name': {
                            '$first': '$name'
                        }
                    }
                }
            ]))

            groups_with_permissions = list(self.user_database['auth_group_permissions'].aggregate([
                {
                    '$lookup': {
                        'from': 'auth_permission',
                        'localField': 'permission_id',
                        'foreignField': 'id',
                        'as': 'permission'
                    }
                }, {
                    '$unwind': {
                        'path': '$permission',
                        'preserveNullAndEmptyArrays': True
                    }
                }, {
                    '$group': {
                        '_id': '$group_id',
                        'permissions': {
                            '$addToSet': '$permission.name'
                        }
                    }
                }
            ]))

            combined_list = groups_with_permissions + groups_with_users
            groups_dict = {}
            for element in combined_list:
                groups_dict[element['_id']] = {**element, **groups_dict.get(element['_id'], {})}

            groups = list(groups_dict.values())

            for group in groups:
                if group['timeout'] == None or group['override'] == False:
                    group['timeout'] = pon_mgr_settings['User Session Expiry Age Timeout']
                if group['override'] == None:
                    group['override'] = False
                group['pk'] = group['_id']
                del group['_id']

            if len(groups) == 0:
                respone = []
            else:
                response = groups
        except Exception as err:
            self.status_log("EXCEPTION (get_all_roles_with_users_permissions_and_timeout)", sys.exc_info()[1])
            response = []

        return response


    """
    Returns all emails of local users for a given role
    """

    def get_all_local_users_per_role(self, user_email, role_id):
        response = []
        try:
            users = list(self.user_database["auth_user_groups"].aggregate([
                {
                    '$match': {
                        'group_id': role_id
                    }
                }, {
                    '$lookup': {
                        'from': 'auth_user',
                        'let': {
                            'user_id': '$user_id'
                        },
                        'pipeline': [
                            {
                                '$match': {
                                    '$expr': {
                                        '$eq': [
                                            '$$user_id', '$id'
                                        ]
                                    },
                                    'database_type': {
                                        '$ne': 'radius'
                                    }
                                }
                            }
                        ],
                        'as': 'user'
                    }
                }, {
                    '$unwind': {
                        'path': '$user',
                        'includeArrayIndex': 'string',
                        'preserveNullAndEmptyArrays': False
                    }
                }, {
                    '$project': {
                        'user._id': 0
                    }
                }, {
                    '$group': {
                        '_id': 0,
                        'users': {
                            '$addToSet': '$user.username'
                        }
                    }
                }
            ]))
            if len(users) == 0:
                respone = []
            else:
                response = users[0]['users']
        except Exception as err:
            self.status_log("EXCEPTION (get_all_local_users_per_role)", sys.exc_info()[1])
            response = []

        return response

    """
    Unattached devices for tree
    """

    def get_all_unattached_controllers(self, request):
        response = [status.HTTP_200_OK, []]
        user_email = request.user.email

        try:
            database_manager.get_database(self.database_id)
            cfg_collections = self.database["CNTL-CFG"]
            cntl_cfg_ids = cfg_collections.distinct("_id", {})
            state_collections = self.database["CNTL-STATE"]

            index = 0
            while index < len(cntl_cfg_ids):
                cntl_state = state_collections.find_one({"_id": cntl_cfg_ids[index]},
                                                        {"_id": 1, "Time": 1, "CNTL.Version": 1})
                if cntl_state is None:
                    cntl_cfg = cfg_collections.find_one({"_id": cntl_cfg_ids[index]}, {"_id": 1, "CNTL.Name": 1})
                    response[1].append({'_id': cntl_cfg_ids[index], 'name': cntl_cfg['CNTL']['Name']})

                index += 1

            val_array = response[1]
            response[1] = sorted(val_array, key=lambda i: (i['name'].lower(), i['_id']))
        except Exception as err:
            self.status_log("EXCEPTION (get_all_unattached_controllers)", sys.exc_info()[1])
            messages.error(request, f"{type(err)}")
            messages.error(request, traceback.format_exc(), extra_tags='stacktrace')
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get unattached PON Controllers list due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "CNTL-STATE, CNTL-CFG", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    # Retrieve an array of all unattached Switches
    def get_all_unattached_switches(self, request):
        response = [status.HTTP_200_OK, []]
        user_email = request.user.email

        try:
            database_manager.get_database(self.database_id)
            cfg_collection = self.database["SWI-CFG"]
            swi_cfg_ids = cfg_collection.distinct("_id", {})
            collection = self.database["OLT-STATE"]
            index = 0
            while index < len(swi_cfg_ids):
                cfg_state = collection.find_one({"Switch.Chassis ID": swi_cfg_ids[index]}, {"_id": 1})
                if cfg_state is None:
                    swi_cfg = cfg_collection.find_one({"_id": swi_cfg_ids[index]})
                    response[1].append({'_id': swi_cfg_ids[index], 'name': swi_cfg['SWI']['Name']})
                index += 1

            val_array = response[1]
            response[1] = sorted(val_array, key=lambda i: (i['name'].lower(), i['_id']))
        except Exception as err:
            self.status_log("EXCEPTION (get_all_unattached_switches)", sys.exc_info()[1])
            messages.error(request, f"{type(err)}")
            messages.error(request, traceback.format_exc(), extra_tags='stacktrace')
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get unattached Switches list due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "OLT-STATE, SWI-CFG", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    # Retrieve an array of all unattached OLTs
    # Pre-Provisioned: INV:!primary, CFG, !STATE, offline
    # Uninventoried w/ no state: !INV, CFG, !STATE, offline
    def get_all_unattached_olts(self, request):
        response = [status.HTTP_200_OK, []]
        user_email = request.user.email

        try:
            database_manager.get_database(self.database_id)
            controller_olts = list(self.database["CNTL-STATE"].aggregate([
                {
                    '$project': {
                        'OLTs': {
                            '$concatArrays': [
                                '$OLTs.Primary',
                                '$OLTs.Primary Missing',
                                '$OLTs.Primary Free',
                                '$OLTs.Secondary',
                                '$OLTs.Secondary Missing',
                                '$OLTs.Secondary Free',
                                '$OLTs.Unspecified',
                                '$OLTs.Unspecified Free'
                            ]
                        }
                    }
                }, {
                    '$unwind': {
                        'path': '$OLTs',
                        'preserveNullAndEmptyArrays': False
                    }
                }, {
                    '$group': {
                        '_id': 'all',
                        'OLTs': {
                            '$addToSet': '$OLTs'
                        }
                    }
                }
            ]))

            if len(controller_olts) > 0:
                controller_olts = controller_olts[0]['OLTs']

            # Get all OLTs from SWI-CFG
            switch_olts = list(self.database["SWI-CFG"].aggregate([
                {
                    '$project': {
                        'OLTs': {
                            '$objectToArray': '$OLTs'
                        }
                    }
                }, {
                    '$unwind': {
                        'path': '$OLTs',
                        'preserveNullAndEmptyArrays': False
                    }
                }, {
                    '$group': {
                        '_id': 'all',
                        'OLTs': {
                            '$addToSet': '$OLTs.k'
                        }
                    }
                }
            ]))

            if len(switch_olts) > 0:
                switch_olts = switch_olts[0]['OLTs']

            # Get all switch IDs
            switch_ids = list(self.database["SWI-CFG"].distinct("_id", {}))

            # Get all OLTs not attached listed in SWI-CFG or CNTL-STATE
            unattached_olts = list(self.database["OLT-CFG"].aggregate([
                {
                    '$match': {
                        '$or': [
                            {
                                '_id': {
                                    '$nin': controller_olts
                                }
                            }, {
                                '_id': {
                                    '$nin': switch_olts
                                }
                            }
                        ]
                    }
                }, {
                    '$group': {
                        '_id': 'OLTs',
                        'OLTs': {
                            '$addToSet': {
                                '_id': '$_id',
                                'Name': '$OLT.Name'
                            }
                        }
                    }
                }
            ]))

            if len(unattached_olts) > 0:
                unattached_olts = unattached_olts[0]['OLTs']

            unattached_olt_ids = []
            for olt in unattached_olts:
                unattached_olt_ids.append(olt['_id'])

            unattached_olt_states = list(self.database["OLT-STATE"].find({"_id": {"$in": unattached_olt_ids}}, {"Switch.Chassis ID": 1}))

            # Check suspected unattached OLTs
            for olt in unattached_olts:
                id = olt["_id"]
                if id != "Default":
                    olt_state = {}
                    for state in unattached_olt_states:
                        if state["_id"] == id:
                            olt_state = state
                    if olt_state and "Switch" in olt_state and olt_state["Switch"]["Chassis ID"] != "":
                        chassis = olt_state["Switch"]["Chassis ID"]
                        unattached_switch = chassis not in switch_ids
                    else:
                        unattached_switch = True

                    if unattached_switch:
                        response[1].append({'_id': id, 'name': olt['Name']})
                else:
                    response[1].append({'_id': id, 'name': olt['Name']})
            val_array = response[1]
            response[1] = sorted(val_array, key=lambda i: (i['name'].lower(), i['_id']))
        except Exception as err:
            self.status_log("EXCEPTION (get_all_unattached_olts)", sys.exc_info()[1])
            messages.error(request, f"{type(err)}")
            messages.error(request, traceback.format_exc(), extra_tags='stacktrace')
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get unattached OLTs list due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "OLT-STATE, OLT-CFG, CNTL-STATE, SWI-CFG", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    # Retrieve an array of all unattached ONUs
    def get_all_unattached_onus(self, request):
        response = [status.HTTP_200_OK, []]
        user_email = request.user.email

        try:
            database_manager.get_database(self.database_id)

            controller_attached_onus = list(self.database['CNTL-STATE'].aggregate([
                {
                    '$addFields': {
                        'result': {
                            '$objectToArray': '$System Status'
                        }
                    }
                }, {
                    '$unwind': {
                        'path': '$result'
                    }
                }, {
                    '$group': {
                        '_id': '',
                        'ONUs': {
                            '$mergeObjects': '$result.v.ONUs'
                        }
                    }
                }, {
                    '$replaceRoot': {
                        'newRoot': '$ONUs'
                    }
                }
            ]))[0]

            # Get a list of ONUs found in OLT-CFG
            olt_onus = list(self.database["OLT-CFG"].aggregate([
                {
                    '$project': {
                        'onus': {
                            '$objectToArray': '$ONUs'
                        }
                    }
                }, {
                    '$project': {
                        '_id': 0,
                        'onu': '$onus.k'
                    }
                }, {
                    '$unwind': {
                        'path': '$onu',
                        'preserveNullAndEmptyArrays': False
                    }
                }, {
                    '$group': {
                        '_id': 'attached_onus',
                        'onus': {
                            '$push': '$onu'
                        }
                    }
                }
            ]))

            if len(olt_onus) > 0:
                olt_onus = olt_onus[0]['onus']

            # Get a list of ONUs found in ONU-CFG but not referenced in OLT-CFG
            unattached_onus = list(self.database["ONU-CFG"].aggregate([
                {
                    '$match': {
                        '_id': {
                            '$nin': olt_onus
                        }
                    }
                }, {
                    '$group': {
                        '_id': '$_id',
                        'Name': {
                            '$first': '$ONU.Name'
                        }
                    }
                }
            ]))

            for onu in unattached_onus:
                id = onu["_id"]
                unattached = True
                if id in controller_attached_onus:
                    unattached = False

                if unattached:
                    response[1].append({'_id': id, 'Name': onu['Name']})

        except Exception as err:
            self.status_log("EXCEPTION (get_all_unattached_onus)", sys.exc_info()[1])
            messages.error(request, f"{type(err)}")
            messages.error(request, traceback.format_exc(), extra_tags='stacktrace')
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get unattached ONUs list due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "OLT-CFG, ONU-CFG, CNTL-STATE", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    def get_all_unattached_onu_configs(self, request):
        response = [status.HTTP_200_OK, []]
        user_email = request.user.email

        try:
            database_manager.get_database(self.database_id)

            configured_controllers = list(self.database["CNTL-STATE"].aggregate([
                {
                    '$addFields': {
                        'result': {
                            '$objectToArray': '$System Status'
                        }
                    }
                }, {
                    '$unwind': {
                        'path': '$result'
                    }
                }, {
                    '$project': {
                        'onu': {
                            '$objectToArray': '$result.v.ONUs'
                        }
                    }
                }, {
                    '$unwind': {
                        'path': '$onu',
                        'preserveNullAndEmptyArrays': False
                    }
                }, {
                    '$group': {
                        '_id': '',
                        'onus': {
                            '$addToSet': '$onu.k'
                        }
                    }
                }
            ]))

            # Get a list of ONUs found in OLT-CFG
            olt_onus = list(self.database["OLT-CFG"].aggregate([
                {
                    '$project': {
                        'onus': {
                            '$objectToArray': '$ONUs'
                        }
                    }
                }, {
                    '$project': {
                        '_id': 0,
                        'onu': '$onus.k'
                    }
                }, {
                    '$unwind': {
                        'path': '$onu',
                        'preserveNullAndEmptyArrays': False
                    }
                }, {
                    '$group': {
                        '_id': 'attached_onus',
                        'onus': {
                            '$push': '$onu'
                        }
                    }
                }
            ]))

            cntl_onus = []
            if len(configured_controllers) > 0 and len(configured_controllers[0]['onus']) > 0:
                cntl_onus = configured_controllers[0]['onus']

            if len(olt_onus) > 0:
                olt_onus = olt_onus[0]['onus']
            olt_onus.extend(cntl_onus)

            # Get a list of ONUs found in ONU-CFG but not referenced in OLT-CFG
            unattached_onus = list(self.database["ONU-CFG"].aggregate([
                {
                    '$match': {
                        '_id': {
                            '$nin': olt_onus
                        }
                    }
                }, {
                    '$project': {
                        'id': 1,
                        'name': '$ONU.Name',
                        'OLT-Service 0': 1,
                        'OLT-Service 1': 1,
                        'OLT-Service 2': 1,
                        'OLT-Service 3': 1,
                        'OLT-Service 4': 1,
                        'OLT-Service 5': 1,
                        'OLT-Service 6': 1,
                        'OLT-Service 7': 1
                    }
                }
            ]))

            response[1] = unattached_onus

        except Exception as err:
            self.status_log("EXCEPTION (get_all_unattached_onus)", sys.exc_info()[1])
            messages.error(request, f"{type(err)}")
            messages.error(request, traceback.format_exc(), extra_tags='stacktrace')
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get unattached ONUs list due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "OLT-CFG, ONU-CFG, CNTL-STATE", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    # Retrieve an array of all onus using a specified sla
    def get_all_sla_onu_usage(self, request, sla):
        response = [status.HTTP_200_OK, []]
        user_email = request.user.email

        try:
            database_manager.get_database(self.database_id)
            onu_state_collection = self.database["ONU-STATE"]

            for i in range(8):
                service = "OLT-Service " + str(i) + ".SLA-CFG.Source"
                onu_cursor = onu_state_collection.find({service: sla},
                                                       {"_id": 1, "OLT.MAC Address": 1, "CNTL.MAC Address": 1})
                for onu in onu_cursor:

                    # Getting switch mac from olt state
                    olt_state_collection = self.database["OLT-STATE"]
                    olt = olt_state_collection.find_one({"_id": onu["OLT"]["MAC Address"]}, {"Switch.Chassis ID": 1})
                    onu["Switch"] = {}
                    onu["Switch"]["Chassis ID"] = olt["Switch"]["Chassis ID"]

                    if onu["_id"] not in response[1]:
                        response[1].append(onu)

        except Exception as err:
            self.status_log("EXCEPTION (get_all_sla_onu_usage)", sys.exc_info()[1])
            messages.error(request, f"{type(err)}")
            messages.error(request, traceback.format_exc(), extra_tags='stacktrace')
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get ONUs using SLA {sla} list due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "ONU-STATE", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    # Retrieve an array of all onus using a specified service config
    def get_all_srv_onu_usage(self, request, srv):
        response = [status.HTTP_200_OK, []]
        user_email = request.user.email

        try:
            database_manager.get_database(self.database_id)
            onu_state_collection = self.database["ONU-STATE"]

            onu_cursor = onu_state_collection.find({"ONU.SRV-CFG": srv},
                                                   {"_id": 1, "OLT.MAC Address": 1, "CNTL.MAC Address": 1})
            for onu in onu_cursor:
                # Getting switch mac from olt state
                olt_state_collection = self.database["OLT-STATE"]
                olt = olt_state_collection.find_one({"_id": onu["OLT"]["MAC Address"]}, {"Switch.Chassis ID": 1})
                onu["Switch"] = {}
                onu["Switch"]["Chassis ID"] = olt["Switch"]["Chassis ID"]

                response[1].append(onu)

        except Exception as err:
            self.status_log("EXCEPTION (get_all_srv_onu_usage)", sys.exc_info()[1])
            messages.error(request, f"{type(err)}")
            messages.error(request, traceback.format_exc(), extra_tags='stacktrace')
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get ONUs using SLA {srv} list due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "ONU-STATE", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    # Retrieve an array of all active CNTL MAC Addresses
    def get_all_controllers(self, request):
        response = [status.HTTP_200_OK, []]
        user_email = request.user.email

        try:
            database_manager.get_database(self.database_id)
            # Get all distinct document ids from collection CNTL_STATE
            collection = self.database["CNTL-STATE"]
            cntl_ids = collection.distinct("_id", {})
            collection = self.database["CNTL-CFG"]
            index = 0
            while index < len(cntl_ids):
                response[1].append({"_id": "", "name": ""})
                response[1][index]["_id"] = cntl_ids[index]
                name = collection.find_one({"_id": cntl_ids[index]}, {"CNTL.Name": 1})
                if name is None:
                    response[1][index]["name"] = ""
                else:
                    response[1][index]["name"] = name["CNTL"]["Name"]
                index += 1

            val_array = response[1]
            response[1] = sorted(val_array, key=lambda i: (i['name'].lower(), i['_id']))
        except Exception as err:
            self.status_log("EXCEPTION (get_all_cntls)", sys.exc_info()[1])
            messages.error(request, f"{type(err)}")
            messages.error(request, traceback.format_exc(), extra_tags='stacktrace')
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get PON Controllers list due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "CNTL-STATE, CNTL-CFG", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    # Retrieve an array of all active CNTLs with their versions
    def get_all_controllers_with_versions(self, request):
        response = [status.HTTP_200_OK, []]
        user_email = request.user.email

        try:
            database_manager.get_database(self.database_id)
            # Get all distinct document ids from collection CNTL_STATE
            collection = self.database["CNTL-CFG"]
            cntl_ids = collection.distinct("_id", {})
            index = 0
            while index < len(cntl_ids):
                response[1].append({"_id": "", "name": "", "version": "", "type": "Controller"})
                response[1][index]["_id"] = cntl_ids[index]
                version = collection.find_one({"_id": cntl_ids[index]}, {"CNTL.CFG Version": 1})
                response[1][index]["version"] = version["CNTL"]["CFG Version"].split("-")[0]
                name = collection.find_one({"_id": cntl_ids[index]}, {"CNTL.Name": 1})
                if name is None:
                    response[1][index]["name"] = ""
                else:
                    response[1][index]["name"] = name["CNTL"]["Name"]
                index += 1
        except Exception as err:
            self.status_log("EXCEPTION (get_all_controllers_with_versions)", sys.exc_info()[1])
            messages.error(request, f"{type(err)}")
            messages.error(request, traceback.format_exc(), extra_tags='stacktrace')
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get PON Controllers list due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "CNTL-STATE, CNTL-CFG", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    # Retrieve an array of all active OLT MAC Addresses
    def get_all_olts(self, request):
        response = [status.HTTP_200_OK, []]
        user_email = request.user.email

        try:
            database_manager.get_database(self.database_id)
            # Get all distinct document ids from collection OLT_STATE
            collection = self.database["OLT-STATE"]
            olt_ids = collection.find({}, {"OLT.PON Mode": 1})
            collection = self.database["OLT-CFG"]
            response[1] = []
            for olt in olt_ids:
                olt_data = {"_id": "", "Name": "", "PON Mode": ""}
                olt_id = olt["_id"]
                olt_data["_id"] = olt_id
                olt_data["PON Mode"] = olt["OLT"]["PON Mode"]
                document = collection.find_one({"_id": olt_id}, {"OLT.Name": 1})
                if document is None:
                    olt_data["Name"] = ""
                else:
                    olt_data["Name"] = document["OLT"]["Name"]
                response[1].append(olt_data)
        except Exception as err:
            self.status_log("EXCEPTION (get_all_olts)", sys.exc_info()[1])
            messages.error(request, f"{type(err)}")
            messages.error(request, traceback.format_exc(), extra_tags='stacktrace')
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get OLTs list due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "OLT-STATE, OLT-CFG", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    # Retrieve an array of OLTs with their versions
    def get_all_olts_with_versions(self, request):
        response = [status.HTTP_200_OK, []]
        user_email = request.user.email

        try:
            database_manager.get_database(self.database_id)
            # Get all distinct document ids from collection OLT_STATE
            collection = self.database["OLT-CFG"]
            olt_ids = collection.distinct("_id", {})
            index = 0
            while index < len(olt_ids):
                response[1].append({"_id": "", "name": "", "version": ""})
                response[1][index]["_id"] = olt_ids[index]
                name = collection.find_one({"_id": olt_ids[index]}, {"OLT.Name": 1})
                if name is None:
                    response[1][index]["name"] = ""
                else:
                    response[1][index]["name"] = name["OLT"]["Name"]
                version = collection.find_one({"_id": olt_ids[index]}, {"CNTL.CFG Version": 1})
                response[1][index]["version"] = version["CNTL"]["CFG Version"].split("-")[0]
                index += 1
        except Exception as err:
            self.status_log("EXCEPTION (get_all_olts_with_versions)", sys.exc_info()[1])
            messages.error(request, f"{type(err)}")
            messages.error(request, traceback.format_exc(), extra_tags='stacktrace')
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get OLTs list due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "OLT-STATE, OLT-CFG", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    # Retrieve an array of all OLTs under the specified Switch
    def get_all_olts_for_switch(self, request, switch_id):
        response = [status.HTTP_200_OK, []]
        user_email = request.user.email

        try:
            database_manager.get_database(self.database_id)
            # Get all distinct document ids from collection OLT_STATE
            collection = self.database["OLT-STATE"]
            olts = collection.find({"Switch.Chassis ID": switch_id},
                                   {"_id": 1, "Switch.Port ID": 1, "CNTL.MAC Address": 1, "Switch.Chassis ID": 1})
            for olt in olts:
                olt_id = olt["_id"]
                port = olt["Switch"]["Port ID"]
                switch = olt["Switch"]["Chassis ID"]
                controller = olt["CNTL"]["MAC Address"]
                online = False

                # Get OLTs Name if it exists
                olt_cfg_coll = self.database["OLT-CFG"]
                name_obj = olt_cfg_coll.find_one({"_id": olt_id}, {"OLT.Name": 1})
                if name_obj is not None and name_obj["OLT"] is not None and name_obj["OLT"]["Name"] is not None:
                    name = name_obj["OLT"]["Name"]
                else:
                    name = ""

                # Check if OLT is online
                cntl_state_coll = self.database["CNTL-STATE"]
                system_status_obj = cntl_state_coll.find_one({"_id": controller}, {"System Status": 1})
                if system_status_obj is not None and "System Status" in system_status_obj and olt_id in \
                        system_status_obj["System Status"].keys():
                    online = True
                response[1].append(
                    {"_id": olt_id, "port": port, "cntl": controller, "name": name, "switch": switch, "online": online})

            val_array = response[1]
            response[1] = sorted(val_array, key=lambda i: (i['port'], i['name'].lower(), i['_id']))
        except Exception as err:
            self.status_log("EXCEPTION (get_all_olts_for_switch)", err)
            messages.error(request, f"{type(err)}")
            messages.error(request, traceback.format_exc(), extra_tags='stacktrace')
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get OLTs for Switch {switch_id} due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "OLT-STATE", "user": user_email}
        self.sys_log(action_data, "get", response[0])

        return response

    # Retrieve an array of all OLTs in the inventory of the specified switch
    def get_all_olts_in_switch_inventory(self, request, switch_id):
        response = [status.HTTP_200_OK, []]
        user_email = request.user.email

        try:
            database_manager.get_database(self.database_id)
            # Get all olts inventoried on switch
            inventoried_olt_macs = []
            collection = self.database["SWI-CFG"]
            olt_mac_list = collection.find({"_id": switch_id}, {"OLTs": 1})
            if olt_mac_list.collection.count_documents({}) > 0 and 'OLTs' in olt_mac_list[0]:
                for olt in olt_mac_list[0]['OLTs'].keys():
                    inventoried_olt_macs.append(olt)

            # Getting state files for all inventoried OLTS
            state_olt_macs = []
            collection = self.database["OLT-STATE"]
            olts = collection.find({"_id": {"$in": inventoried_olt_macs}},
                                   {"_id": 1, "Switch.Port ID": 1, "CNTL.MAC Address": 1, "Switch.Chassis ID": 1})
            for olt in olts:
                port = ""
                switch = ""
                if "Switch" in olt:
                    if "Port ID" in olt["Switch"]:
                        port = olt["Switch"]["Port ID"]

                    if "Chassis ID" in olt["Switch"]:
                        switch = olt["Switch"]["Chassis ID"]

                state_olt_macs.append(olt["_id"])
                olt_id = olt["_id"]
                controller = olt["CNTL"]["MAC Address"]
                online = False

                # Get OLTs Name if it exists
                olt_cfg_coll = self.database["OLT-CFG"]
                name_obj = olt_cfg_coll.find_one({"_id": olt_id}, {"OLT.Name": 1})
                if name_obj is not None and name_obj["OLT"] is not None and name_obj["OLT"]["Name"] is not None:
                    name = name_obj["OLT"]["Name"]
                else:
                    name = ""

                # Check if OLT is online
                cntl_state_coll = self.database["CNTL-STATE"]
                system_status_obj = cntl_state_coll.find_one({"_id": controller}, {"System Status": 1})
                if system_status_obj is not None and "System Status" in system_status_obj and olt_id in \
                        system_status_obj["System Status"].keys():
                    online = True

                response[1].append(
                    {"_id": olt_id, "port": port, "cntl": controller, "name": name, "switch": switch, "online": online})

            # Appending inventoried OLTS that do not have a state file
            for olt in inventoried_olt_macs:
                if olt not in state_olt_macs:
                    olt_id = olt
                    port = ""
                    switch = ""
                    controller = ""
                    online = False

                    # Get OLTs Name if it exists
                    olt_cfg_coll = self.database["OLT-CFG"]
                    name_obj = olt_cfg_coll.find_one({"_id": olt_id}, {"OLT.Name": 1})
                    if name_obj is not None and name_obj["OLT"] is not None and name_obj["OLT"]["Name"] is not None:
                        name = name_obj["OLT"]["Name"]
                    else:
                        name = ""
                    response[1].append({"_id": olt_id, "port": port, "cntl": controller, "name": name, "switch": switch,
                                        "online": online})

            val_array = response[1]
            response[1] = sorted(val_array, key=lambda i: (i['port'], i['name'].lower(), i['_id']))
        except Exception as err:
            self.status_log("EXCEPTION (get_all_olts_for_switch)", sys.exc_info()[1])
            messages.error(request, f"{type(err)}")
            messages.error(request, traceback.format_exc(), extra_tags='stacktrace')
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get OLTs in Switch {switch_id} inventory due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "OLT-STATE", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    def get_all_olts_for_controller(self, request, controller_id):
        response = [status.HTTP_200_OK, []]
        user_email = request.user.email

        try:
            database_manager.get_database(self.database_id)
            # Get all distinct document ids from collection OLT_STATE
            olt_state_coll = self.database["OLT-STATE"]
            if controller_id == "Default":
                olts = olt_state_coll.find({"CNTL.MAC Address": ""}, {"_id": 1})
            else:
                olts = olt_state_coll.find({"CNTL.MAC Address": controller_id},
                                           {"_id": 1, "Switch.Port ID": 1, "Switch.Chassis ID": 1})
            for olt in olts:
                olt_id = olt["_id"]
                port = ''
                switch = ''
                if "Switch" in olt and "Port ID" in olt["Switch"]:
                    port = olt["Switch"]["Port ID"]
                if "Switch" in olt and "Chassis ID" in olt["Switch"]:
                    switch = olt["Switch"]["Chassis ID"]
                # Get OLTs Name if it exists
                olt_cfg_coll = self.database["OLT-CFG"]
                name_obj = olt_cfg_coll.find_one({"_id": olt_id}, {"OLT.Name": 1})
                if name_obj is not None and name_obj["OLT"] is not None and name_obj["OLT"]["Name"] is not None:
                    name = name_obj["OLT"]["Name"]
                else:
                    name = ""
                response[1].append(
                    {"_id": olt_id, "port": port, "switchID": switch, "cntl": controller_id, "name": name})

            val_array = response[1]
            response[1] = sorted(val_array, key=lambda i: (i['port'], i['name'].lower(), i['_id']))
        except Exception as err:
            self.status_log("EXCEPTION (get_all_olts_for_controller)", sys.exc_info()[1])
            messages.error(request, f"{type(err)}")
            messages.error(request, traceback.format_exc(), extra_tags='stacktrace')
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get OLTs under PON Controller {controller_id} due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "OLT-STATE", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    # Retrieve an array of all OLTs under the unknown switch for a given controller
    def get_all_olts_for_unknown_switch(self, request):
        response = [status.HTTP_200_OK, []]
        user_email = request.user.email

        try:
            database_manager.get_database(self.database_id)
            # Get all distinct document ids from collection OLT_STATE
            collection = self.database["OLT-STATE"]
            olts = collection.find({"Switch.Chassis ID": ""}, {"_id": 1})
            for olt in olts:
                olt_id = olt["_id"]
                controller = olt["CNTL"]["MAC Address"]
                # Get OLTs Name if it exists
                olt_cfg_coll = self.database["OLT-CFG"]
                name_obj = olt_cfg_coll.find_one({"_id": olt_id}, {"OLT.Name": 1})
                if name_obj is not None and name_obj["OLT"] is not None and name_obj["OLT"]["Name"] is not None:
                    name = name_obj["OLT"]["Name"]
                else:
                    name = ""
                response[1].append({"_id": olt_id, "cntl": controller, "name": name})

            val_array = response[1]
            response[1] = sorted(val_array, key=lambda i: (i['name'].lower(), i['_id']))
        except Exception as err:
            self.status_log("EXCEPTION (get_all_olts_for_unknown_switch)", sys.exc_info()[1])
            messages.error(request, f"{type(err)}")
            messages.error(request, traceback.format_exc(), extra_tags='stacktrace')
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get OLTs in unknown Switches due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "OLT-STATE", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    # Retrieves a list of all Switch MAC addresses and Names
    def get_all_switches(self, request):
        response = [status.HTTP_200_OK, []]
        user_email = request.user.email

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["OLT-STATE"]
            switch_macs = collection.distinct("Switch.Chassis ID", {})

            collection = self.database["SWI-CFG"]
            for switch in switch_macs:
                data = {"_id": switch, "name": ""}
                name = collection.find_one({"_id": switch}, {"SWI.Name": 1})
                if name is None:
                    data["name"] = ""
                else:
                    data["name"] = name["SWI"]["Name"]
                response[1].append(data)

            val_array = response[1]
            response[1] = sorted(val_array, key=lambda i: (i['name'].lower(), i['_id']))
        except Exception as err:
            self.status_log("EXCEPTION (get_all_switches)", sys.exc_info()[1])
            messages.error(request, f"{type(err)}")
            messages.error(request, traceback.format_exc(), extra_tags='stacktrace')
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get Switches list due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "OLT-STATE, SWI-CFG", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    def get_all_switches_with_versions(self, request):
        response = [status.HTTP_200_OK, []]
        user_email = request.user.email

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["SWI-CFG"]
            switch_macs = collection.distinct("_id", {})

            for switch in switch_macs:
                data = {"_id": switch, "name": "", "version": "", "type": "Switch"}
                version = collection.find_one({"_id": switch}, {"CNTL.CFG Version": 1})
                data["version"] = version["CNTL"]["CFG Version"].split("-")[0]
                name = collection.find_one({"_id": switch}, {"SWI.Name": 1})
                if name is None:
                    data["name"] = ""
                else:
                    data["name"] = name["SWI"]["Name"]
                response[1].append(data)
        except Exception as err:
            self.status_log("EXCEPTION (get_all_switches_with_versions)", sys.exc_info()[1])
            messages.error(request, f"{type(err)}")
            messages.error(request, traceback.format_exc(), extra_tags='stacktrace')
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get Switches list due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "SWI-CFG", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    # Retrieves a list of all Switch MAC addresses and Names for the specified Controller
    def get_all_switches_for_cntl(self, request, cntl_mac_address):
        response = [status.HTTP_200_OK, []]
        user_email = request.user.email
        switch_macs = []

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["OLT-STATE"]
            switch_list = collection.find({"CNTL.MAC Address": cntl_mac_address}, {"Switch.Chassis ID": 1})
            for switch in switch_list:
                if switch["Switch"]["Chassis ID"] != "" and switch["Switch"]["Chassis ID"] not in switch_macs:
                    switch_macs.append(switch["Switch"]["Chassis ID"])

            collection = self.database["SWI-CFG"]
            for switch in switch_macs:
                data = {"_id": switch, "Name": ""}
                name = collection.find_one({"_id": switch}, {"SWI.Name": 1})
                if name is None:
                    data["Name"] = ""
                else:
                    data["Name"] = name["SWI"]["Name"]
                response[1].append(data)
        except Exception as err:
            self.status_log("EXCEPTION (get_all_switches_for_cntl)", sys.exc_info()[1])
            messages.error(request, f"{type(err)}")
            messages.error(request, traceback.format_exc(), extra_tags='stacktrace')
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get Switches list due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "OLT-STATE, SWI-CFG", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    # Retrieve an array of all active ONU MAC Addresses
    def get_all_onus(self, request):
        response = [status.HTTP_200_OK, []]
        user_email = request.user.email

        try:
            database_manager.get_database(self.database_id)
            # Get all distinct document ids from collection ONU_STATE
            collection = self.database["ONU-STATE"]
            onu_ids = collection.distinct("_id", {})
            collection = self.database["ONU-CFG"]
            index = 0
            while index < len(onu_ids):
                response[1].append({"_id": "", "Name": ""})
                response[1][index]["_id"] = onu_ids[index]
                document = collection.find_one({"_id": onu_ids[index]}, {"ONU.Name": 1})
                if document is not None and len(document["ONU"]) != 0:
                    response[1][index]["Name"] = document["ONU"]["Name"]
                index += 1
        except Exception as err:
            self.status_log("EXCEPTION (get_all_onus)", sys.exc_info()[1])
            messages.error(request, f"{type(err)}")
            messages.error(request, traceback.format_exc(), extra_tags='stacktrace')
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get ONUs list due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "ONU-STATE, ONU-CFG", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    # Retrieve all ONU alarm config identifiers
    def get_all_gpon_onu_serial_numbers(self, request):
        response = [status.HTTP_200_OK, []]
        user_email = request.user.email

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["ONU-STATE"]
            onu_cursor = collection.find({"ONU.PON Mode": "GPON"}, {"_id": 1})
            for doc in onu_cursor:
                response[1].append(doc['_id'])
        except Exception as err:
            self.status_log("EXCEPTION (get_all_gpon_onu_serial_numbers)", sys.exc_info()[1])
            messages.error(request, f"{type(err)}")
            messages.error(request, traceback.format_exc(), extra_tags='stacktrace')
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get all GPON ONU Serial Numbers due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "ONU-STATE", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    # Retrieves a list of ONUs with their versions, names and IDs
    def get_all_onus_with_versions(self, request):
        response = [status.HTTP_200_OK, []]
        user_email = request.user.email

        try:
            database_manager.get_database(self.database_id)
            # Get all distinct document ids from collection ONU_STATE
            collection = self.database["ONU-CFG"]
            onu_ids = collection.distinct("_id", {})
            index = 0
            while index < len(onu_ids):
                response[1].append({"_id": "", "name": ""})
                response[1][index]["_id"] = onu_ids[index]
                version = collection.find_one({"_id": onu_ids[index]}, {"CNTL.CFG Version": 1})
                response[1][index]["version"] = version["CNTL"]["CFG Version"].split("-")[0]
                document = collection.find_one({"_id": onu_ids[index]}, {"ONU.Name": 1})
                if document is not None and len(document["ONU"]) != 0:
                    response[1][index]["name"] = document["ONU"]["Name"]
                index += 1
        except Exception as err:
            self.status_log("EXCEPTION (get_all_onus_with_versions)", sys.exc_info()[1])
            messages.error(request, f"{type(err)}")
            messages.error(request, traceback.format_exc(), extra_tags='stacktrace')
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get ONUs list due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "ONU-CFG", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    # Retrieves all unspecified ONUs
    def get_all_unspecified_onus(self, request):
        response = [status.HTTP_200_OK, []]
        user_email = request.user.email

        try:
            database_manager.get_database(self.database_id)
            # Get all ONU CFG ids that are not found in an OLT STATE or CFG
            collection = self.database["ONU-CFG"]
            onus = collection.find({}, {"_id": 1, "ONU.Name": 1})
            collection = self.database["OLT-STATE"]
            olts_active = list(collection.find({}, {"ONUs": 1, "_id": 0}))
            collection = self.database["OLT-CFG"]
            olts_inactive = list(collection.find({}, {"ONUs": 1, "_id": 0}))
            for onu in onus:
                if onu["_id"] != "Default":
                    is_unspec = True
                    for olt_onu in olts_active:
                        if onu["_id"] in list(olt_onu["ONUs"].keys()):
                            is_unspec = False
                            break
                    for olt_onu in olts_inactive:
                        if onu["_id"] in list(olt_onu["ONUs"].keys()):
                            is_unspec = False
                            break
                    if is_unspec:
                        response[1].append({"_id": onu["_id"], "Name": onu["ONU"]["Name"]})
        except Exception as err:
            self.status_log("EXCEPTION (get_all_unspecified_onus)", sys.exc_info()[1])
            messages.error(request, f"{type(err)}")
            messages.error(request, traceback.format_exc(), extra_tags='stacktrace')
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get unspecified ONUs list due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "ONU-CFG, OLT-STATE, OLT-CFG", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    # Retrieve an array of all ONUs under the specified OLT
    def get_all_onus_for_olt(self, request, olt_id):
        response = [status.HTTP_200_OK, []]
        user_email = request.user.email

        try:
            database_manager.get_database(self.database_id)
            # Get all distinct document ids from collection OLT_STATE
            collection = self.database["OLT-STATE"]
            onus = collection.find({"_id": olt_id}, {"ONUs": 1})
            for onu in onus:
                response[1].append(onu)
        except Exception as err:
            self.status_log("EXCEPTION (get_all_onus_for_olt)", sys.exc_info()[1])
            messages.error(request, f"{type(err)}")
            messages.error(request, traceback.format_exc(), extra_tags='stacktrace')
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get ONUs for OLT {olt_id} list due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "OLT-STATE", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    # Retrieve ONUs from OLTs under CNTL-STATE that matches the one passed in
    def get_all_onus_under_olt(self, request, olt_id):
        response = [status.HTTP_200_OK, []]
        user_email = request.user.email

        try:
            database_manager.get_database(self.database_id)

            # Time check for controller
            cntl_time_cutoff_r21 = str(datetime.datetime.utcnow() - datetime.timedelta(minutes=6))
            cntl_time_cutoff_r22 = str(datetime.datetime.utcnow() - datetime.timedelta(minutes=2))

            cntl_state = self.database["CNTL-STATE"]
            onus_obj = list(cntl_state.aggregate([{
                '$addFields': {
                    'olderThanR220': {
                        '$cond': [
                            {
                                '$or': [
                                    {
                                        '$eq': [
                                            {
                                                '$indexOfCP': [
                                                    '$CNTL.Version', '2.1.'
                                                ]
                                            }, 1
                                        ]
                                    }, {
                                        '$eq': [
                                            {
                                                '$indexOfCP': [
                                                    '$CNTL.Version', '2.0'
                                                ]
                                            }, 1
                                        ]
                                    }, {
                                        '$eq': [
                                            {
                                                '$indexOfCP': [
                                                    '$CNTL.Version', '1.'
                                                ]
                                            }, 1
                                        ]
                                    }
                                ]
                            }, True, False
                        ]
                    }
                }
            }, {
                '$addFields': {
                    'status': {
                        '$cond': [
                            {
                                '$or': [
                                    {
                                        '$and': [
                                            {
                                                '$eq': [
                                                    '$olderThanR220', False
                                                ]
                                            }, {
                                                '$gte': [
                                                    '$Time', cntl_time_cutoff_r22
                                                ]
                                            }
                                        ]
                                    }, {
                                        '$and': [
                                            {
                                                '$ne': [
                                                    '$olderThanR220', False
                                                ]
                                            }, {
                                                '$gte': [
                                                    '$Time', cntl_time_cutoff_r21
                                                ]
                                            }
                                        ]
                                    }
                                ]
                            }, 'online', 'offline'
                        ]
                    }
                }
            }, {
                '$project': {
                    '_id': 0,
                    'System Status': {
                        '$objectToArray': '$System Status'
                    },
                    'cntl status': '$status'
                }
            }, {
                '$unwind': {
                    'path': '$System Status',
                    'preserveNullAndEmptyArrays': False
                }
            }, {
                '$match': {
                    'System Status.k': olt_id
                }
            }, {
                '$project': {
                    'ONUs': {
                        '$objectToArray': '$System Status.v.ONUs'
                    },
                    'cntl status': '$cntl status'
                }
            }, {
                '$unwind': {
                    'path': '$ONUs',
                    'preserveNullAndEmptyArrays': False
                }
            }, {
                '$lookup': {
                    'from': 'ONU-CFG',
                    'localField': 'ONUs.k',
                    'foreignField': '_id',
                    'as': 'name'
                }
            }, {
                '$addFields': {
                    'name': {
                        '$arrayElemAt': [
                            '$name.ONU.Name', 0
                        ]
                    }
                }
            }, {
                '$addFields': {
                    'onuArray': [
                        {
                            'k': '$ONUs.k',
                            'v': {
                                'state': '$ONUs.v',
                                'name': '$name',
                                'cntl status': "$cntl status"
                            }
                        }
                    ]
                }
            }, {
                '$addFields': {
                    'onu': {
                        '$arrayToObject': '$onuArray'
                    }
                }
            }, {
                '$group': {
                    '_id': '',
                    'cntlOfflineOnus': {
                        '$mergeObjects': {
                            '$cond': [
                                {
                                    '$eq': [
                                        '$cntl status', 'offline'
                                    ]
                                }, '$onu', '$$REMOVE'
                            ]
                        }
                    },
                    'cntlOnlineOnus': {
                        '$mergeObjects': {
                            '$cond': [
                                {
                                    '$eq': [
                                        '$cntl status', 'online'
                                    ]
                                }, '$onu', '$$REMOVE'
                            ]
                        }
                    }
                }
            }]))[0]

            cntl_online_onus = onus_obj['cntlOnlineOnus']
            onus = onus_obj['cntlOfflineOnus']

            for onu_key in cntl_online_onus:
                onus[onu_key] = cntl_online_onus[onu_key]

            # Check if controller is online for ONU state
            for onu_key in onus:
                state = "Offline"
                if onus[onu_key]["state"] != "" and onus[onu_key]["cntl status"] == "online":
                        state = onus[onu_key]["state"]

                response[1].append({"_id": onu_key, "state": state, "name": onus[onu_key]["name"]})

            val_array = response[1]
            response[1] = sorted(val_array, key=lambda i: (i['name'].lower(), i['_id']))
        except Exception as err:
            self.status_log("EXCEPTION (get_all_onus_under_olt)", sys.exc_info()[1])
            messages.error(request, f"{type(err)}")
            messages.error(request, traceback.format_exc(), extra_tags='stacktrace')
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get ONUs under OLT {olt_id} due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "CNTL-STATE", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    # Retrieve an object formatted for ONU inventory report
    def get_onu_inventory(self, request, switch_id):
        response = [status.HTTP_200_OK, []]
        user_email = request.user.email

        try:
            state_collection = self.database["ONU-STATE"]
            olt_collection = self.database["OLT-STATE"]

            switch_region_pop = self.get_switch_region_pop(user_email, switch_id)
            region = switch_region_pop[1]['Region']
            pop = switch_region_pop[1]['POP']

            # check if GPON and EPON mode are present for all ONUs in ONU state collection
            gpon = state_collection.find_one({'ONU.PON Mode': 'GPON'})
            epon = state_collection.find_one({'ONU.PON Mode': 'EPON'})

            # for OLTs with no switch info
            if switch_id == "":
                olts = list(olt_collection.aggregate([
                    {
                        '$match': {
                            'Switch.Chassis ID': {
                                '$exists': False
                            }
                        }
                    }, {
                        '$project': {
                            '_id': 1
                        }
                    }
                ]))
                olt_ids = [x['_id'] for x in olts]
            else:
                olts = list(olt_collection.find({'Switch.Chassis ID': switch_id}, {'_id': 1}))
                olt_ids = [x['_id'] for x in olts]

            if epon and not gpon:  # epon only
                onu_inventory = list(state_collection.aggregate([
                    {
                        '$match': {
                            'OLT.MAC Address': {'$in': olt_ids}
                        }
                    }, {
                        '$project': {
                            'CNTL.MAC Address': 1,
                            'OLT.MAC Address': 1,
                            'STATE-ONU': '$ONU',
                            'Time': 1
                        }
                    }, {
                        '$lookup': {
                            'from': 'ONU-CFG',
                            'localField': '_id',
                            'foreignField': '_id',
                            'as': 'CFG'
                        }
                    }, {
                        '$addFields': {
                            'Online_Duration': {
                                '$subtract': [{'$toDate': '$Time'}, {'$toDate': '$STATE-ONU.Online Time'}]
                            }
                        }
                    }, {
                        '$addFields': {
                            'days_long': {
                                '$divide': ['$Online_Duration', 86400000]
                            }
                        }
                    }, {
                        '$addFields': {
                            'days': {
                                '$trunc': ['$days_long']
                            },
                            'hours_long': {
                                '$multiply': [
                                    {
                                        '$subtract': ['$days_long', {'$trunc': ['$days_long']}]
                                    }, 24
                                ]
                            }
                        }
                    }, {
                        '$addFields': {
                            'hours': {
                                '$trunc': ['$hours_long']
                            },
                            'minutes': {
                                '$trunc': [
                                    {
                                        '$multiply': [
                                            {
                                                '$subtract': ['$hours_long', {'$trunc': ['$hours_long']}]
                                            }, 60
                                        ]
                                    }
                                ]
                            }
                        }
                    }, {
                        '$project': {
                            '_id': 0,
                            '[ONU-CFG][ONU][Address]': {'$arrayElemAt': ['$CFG.ONU.Address', 0]},
                            '[ONU-CFG][ONU][Location]': {'$arrayElemAt': ['$CFG.ONU.Location', 0]},
                            '[ONU-CFG][ONU][Name]': {'$arrayElemAt': ['$CFG.ONU.Name', 0]},
                            '[ONU-CFG][_id]': '$_id',
                            'Region': region,
                            'POP': pop,
                            '[ONU-CFG][ONU][Create Date]': {'$arrayElemAt': ['$CFG.ONU.Create Date', 0]},
                            '[ONU-STATE][ONU][Online Time]': '$STATE-ONU.Online Time',
                            'Online Duration': {
                                '$concat': [
                                    {
                                        '$toString': '$days'
                                    }, 'd, ', {
                                        '$toString': '$hours'
                                    }, 'h, ', {
                                        '$toString': '$minutes'
                                    }, 'm'
                                ]
                            },
                            '[ONU-STATE][CNTL][MAC Address]': {'$ifNull': ['$CNTL.MAC Address', '']},
                            '[ONU-STATE][OLT][MAC Address]': {'$ifNull': ['$OLT.MAC Address', '']},
                            '[ONU-STATE][ONU][Boot Version]': {'$ifNull': ['$STATE-ONU.Boot Version', '']},
                            '[ONU-STATE][ONU][Chip Model]': {'$ifNull': ['$STATE-ONU.Chip Model', '']},
                            '[ONU-STATE][ONU][Chip Version]': {'$ifNull': ['$STATE-ONU.Chip Version', '']},
                            '[ONU-STATE][ONU][DPoE Version]': {'$ifNull': ['$STATE-ONU.DPoE Version', '']},
                            '[ONU-STATE][ONU][FW Version]': {'$ifNull': ['$STATE-ONU.FW Version', '']},
                            '[ONU-STATE][ONU][JEDEC ID]': {'$ifNull': ['$STATE-ONU.JEDEC ID', '']},
                            '[ONU-STATE][ONU][Manufacturer]': {'$ifNull': ['$STATE-ONU.Manufacturer', '']},
                            '[ONU-STATE][ONU][Model]': {'$ifNull': ['$STATE-ONU.Model', '']},
                            '[ONU-STATE][ONU][Serial Number]': {'$ifNull': ['$STATE-ONU.Serial Number', '']},
                            '[ONU-STATE][ONU][Software Bundle]': {'$ifNull': ['$STATE-ONU.Software Bundle', '']},
                            '[ONU-STATE][ONU][Vendor]': {'$ifNull': ['$STATE-ONU.Vendor', '']}
                        }
                    }
                ]))
            elif gpon and not epon:  # gpon only
                onu_inventory = list(state_collection.aggregate([
                    {
                        '$match': {
                            'OLT.MAC Address': {'$in': olt_ids}
                        }
                    }, {
                        '$project': {
                            'CNTL.MAC Address': 1,
                            'OLT.MAC Address': 1,
                            'STATE-ONU': '$ONU',
                            'Time': 1
                        }
                    }, {
                        '$lookup': {
                            'from': 'ONU-CFG',
                            'localField': '_id',
                            'foreignField': '_id',
                            'as': 'CFG'
                        }
                    }, {
                        '$addFields': {
                            'Online_Duration': {
                                '$subtract': [{'$toDate': '$Time'}, {'$toDate': '$STATE-ONU.Online Time'}]
                            }
                        }
                    }, {
                        '$addFields': {
                            'days_long': {
                                '$divide': ['$Online_Duration', 86400000]
                            }
                        }
                    }, {
                        '$addFields': {
                            'days': {
                                '$trunc': ['$days_long']
                            },
                            'hours_long': {
                                '$multiply': [
                                    {
                                        '$subtract': ['$days_long', {'$trunc': ['$days_long']}]
                                    }, 24
                                ]
                            }
                        }
                    }, {
                        '$addFields': {
                            'hours': {
                                '$trunc': ['$hours_long']
                            },
                            'minutes': {
                                '$trunc': [
                                    {
                                        '$multiply': [
                                            {
                                                '$subtract': ['$hours_long', {'$trunc': ['$hours_long']}]
                                            }, 60
                                        ]
                                    }
                                ]
                            }
                        }
                    }, {
                        '$project': {
                            '_id': 0,
                            '[ONU-CFG][_id]': '$_id',
                            '[ONU-CFG][ONU][Name]': {'$arrayElemAt': ['$CFG.ONU.Name', 0]},
                            '[ONU-CFG][ONU][Address]': {'$arrayElemAt': ['$CFG.ONU.Address', 0]},
                            '[ONU-CFG][ONU][Location]': {'$arrayElemAt': ['$CFG.ONU.Location', 0]},
                            'Region': region,
                            'POP': pop,
                            '[ONU-CFG][ONU][Create Date]': {'$arrayElemAt': ['$CFG.ONU.Create Date', 0]},
                            '[ONU-STATE][ONU][Online Time]': '$STATE-ONU.Online Time',
                            'Online Duration': {
                                '$concat': [
                                    {
                                        '$toString': '$days'
                                    }, 'd, ', {
                                        '$toString': '$hours'
                                    }, 'h, ', {
                                        '$toString': '$minutes'
                                    }, 'm'
                                ]
                            },
                            '[ONU-STATE][CNTL][MAC Address]': {'$ifNull': ['$CNTL.MAC Address', '']},
                            '[ONU-STATE][OLT][MAC Address]': {'$ifNull': ['$OLT.MAC Address', '']},
                            '[ONU-STATE][ONU][Equipment ID]': {'$ifNull': ['$STATE-ONU.Equipment ID', '']},
                            '[ONU-STATE][ONU][FW Version]': {'$ifNull': ['$STATE-ONU.FW Version', '']},
                            '[ONU-STATE][ONU][HW Version]': {'$ifNull': ['$STATE-ONU.HW Version', '']},
                            '[ONU-STATE][ONU][Host MAC Address]': {'$ifNull': ['$STATE-ONU.Host MAC Address', '']},
                            '[ONU-STATE][ONU][Manufacturer]': {'$ifNull': ['$STATE-ONU.Manufacturer', '']},
                            '[ONU-STATE][ONU][Model]': {'$ifNull': ['$STATE-ONU.Model', '']},
                            '[ONU-STATE][ONU][Registration ID]': {'$ifNull': ['$STATE-ONU.Registration ID', '']},
                            '[ONU-STATE][ONU][Serial Number]': {'$ifNull': ['$STATE-ONU.Serial Number', '']},
                            '[ONU-STATE][ONU][Vendor]': {'$ifNull': ['$STATE-ONU.Vendor', '']}
                        }
                    }
                ]))
            else:  # both epon and gpon
                onu_inventory = list(state_collection.aggregate([
                    {
                        '$match': {
                            'OLT.MAC Address': {'$in': olt_ids}
                        }
                    }, {
                        '$project': {
                            'CNTL.MAC Address': 1,
                            'OLT.MAC Address': 1,
                            'STATE-ONU': '$ONU',
                            'Time': 1
                        }
                    }, {
                        '$lookup': {
                            'from': 'ONU-CFG',
                            'localField': '_id',
                            'foreignField': '_id',
                            'as': 'CFG'
                        }
                    }, {
                        '$addFields': {
                            'Online_Duration': {
                                '$subtract': [{'$toDate': '$Time'}, {'$toDate': '$STATE-ONU.Online Time'}]
                            }
                        }
                    }, {
                        '$addFields': {
                            'days_long': {
                                '$divide': ['$Online_Duration', 86400000]
                            }
                        }
                    }, {
                        '$addFields': {
                            'days': {
                                '$trunc': ['$days_long']
                            },
                            'hours_long': {
                                '$multiply': [
                                    {
                                        '$subtract': ['$days_long', {'$trunc': ['$days_long']}]
                                    }, 24
                                ]
                            }
                        }
                    }, {
                        '$addFields': {
                            'hours': {
                                '$trunc': ['$hours_long']
                            },
                            'minutes': {
                                '$trunc': [
                                    {
                                        '$multiply': [
                                            {
                                                '$subtract': ['$hours_long', {'$trunc': ['$hours_long']}]
                                            }, 60
                                        ]
                                    }
                                ]
                            }
                        }
                    }, {
                        '$project': {
                            '_id': 0,
                            '[ONU-CFG][_id]': '$_id',
                            '[ONU-CFG][ONU][Name]': {'$arrayElemAt': ['$CFG.ONU.Name', 0]},
                            '[ONU-CFG][ONU][Address]': {'$arrayElemAt': ['$CFG.ONU.Address', 0]},
                            '[ONU-CFG][ONU][Location]': {'$arrayElemAt': ['$CFG.ONU.Location', 0]},
                            'Region': region,
                            'POP': pop,
                            '[ONU-CFG][ONU][Create Date]': {'$arrayElemAt': ['$CFG.ONU.Create Date', 0]},
                            '[ONU-STATE][ONU][Online Time]': '$STATE-ONU.Online Time',
                            'Online Duration': {
                                '$concat': [
                                    {
                                        '$toString': '$days'
                                    }, 'd, ', {
                                        '$toString': '$hours'
                                    }, 'h, ', {
                                        '$toString': '$minutes'
                                    }, 'm'
                                ]
                            },
                            '[ONU-STATE][CNTL][MAC Address]': {'$ifNull': ['$CNTL.MAC Address', '']},
                            '[ONU-STATE][OLT][MAC Address]': {'$ifNull': ['$OLT.MAC Address', '']},
                            '[ONU-STATE][ONU][Equipment ID]': {'$ifNull': ['$STATE-ONU.Equipment ID', '']},
                            '[ONU-STATE][ONU][FW Version]': {'$ifNull': ['$STATE-ONU.FW Version', '']},
                            '[ONU-STATE][ONU][HW Version]': {'$ifNull': ['$STATE-ONU.HW Version', '']},
                            '[ONU-STATE][ONU][Boot Version]': {'$ifNull': ['$STATE-ONU.Boot Version', '']},
                            '[ONU-STATE][ONU][Chip Model]': {'$ifNull': ['$STATE-ONU.Chip Model', '']},
                            '[ONU-STATE][ONU][Chip Version]': {'$ifNull': ['$STATE-ONU.Chip Version', '']},
                            '[ONU-STATE][ONU][DPoE Version]': {'$ifNull': ['$STATE-ONU.DPoE Version', '']},
                            '[ONU-STATE][ONU][Host MAC Address]': {'$ifNull': ['$STATE-ONU.Host MAC Address', '']},
                            '[ONU-STATE][ONU][JEDEC ID]': {'$ifNull': ['$STATE-ONU.JEDEC ID', '']},
                            '[ONU-STATE][ONU][Manufacturer]': {'$ifNull': ['$STATE-ONU.Manufacturer', '']},
                            '[ONU-STATE][ONU][Model]': {'$ifNull': ['$STATE-ONU.Model', '']},
                            '[ONU-STATE][ONU][Registration ID]': {'$ifNull': ['$STATE-ONU.Registration ID', '']},
                            '[ONU-STATE][ONU][Software Bundle]': {'$ifNull': ['$STATE-ONU.Software Bundle', '']},
                            '[ONU-STATE][ONU][Serial Number]': {'$ifNull': ['$STATE-ONU.Serial Number', '']},
                            '[ONU-STATE][ONU][Vendor]': {'$ifNull': ['$STATE-ONU.Vendor', '']}
                        }
                    }
                ]))
            response[1] = onu_inventory
        except Exception as err:
            self.status_log("EXCEPTION (get_onu_inventory)", sys.exc_info()[1])
            messages.error(request, f"{type(err)}")
            messages.error(request, traceback.format_exc(), extra_tags='stacktrace')
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get ONU inventory due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "ONU-STATE", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    # Retrieve all ONU labels
    def get_all_onu_labels(self, request):
        response = [status.HTTP_200_OK, []]
        user_email = request.user.email

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["ONU-CFG"]
            distinct_labels = sorted(collection.distinct("ONU.Labels", {})[0:20], key=str.lower)
            response[1] = distinct_labels
        except Exception as err:
            self.status_log("EXCEPTION (get_all_onu_labels)", sys.exc_info()[1])
            messages.error(request, f"{type(err)}")
            messages.error(request, traceback.format_exc(), extra_tags='stacktrace')
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get ONU labels due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "ONU-CFG", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    # Retrieve all OLT labels
    def get_all_olt_labels(self, request):
        response = [status.HTTP_200_OK, []]
        user_email = request.user.email

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["OLT-CFG"]
            distinct_labels = sorted(collection.distinct("OLT.Labels", {})[0:20], key=str.lower)
            response[1] = distinct_labels
        except Exception as err:
            self.status_log("EXCEPTION (get_all_olt_labels)", sys.exc_info()[1])
            messages.error(request, f"{type(err)}")
            messages.error(request, traceback.format_exc(), extra_tags='stacktrace')
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get OLT labels due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "OLT-CFG", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    # Retrieve all CNTL labels
    def get_all_cntl_labels(self, request):
        response = [status.HTTP_200_OK, []]
        user_email = request.user.email

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["CNTL-CFG"]
            distinct_labels = sorted(collection.distinct("CNTL.Labels", {})[0:20], key=str.lower)
            response[1] = distinct_labels
        except Exception as err:
            self.status_log("EXCEPTION (get_all_cntl_labels)", sys.exc_info()[1])
            messages.error(request, f"{type(err)}")
            messages.error(request, traceback.format_exc(), extra_tags='stacktrace')
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get CNTL labels due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "CNTL-CFG", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    # Retrieve all SWI labels
    def get_all_switch_labels(self, request):
        response = [status.HTTP_200_OK, []]
        user_email = request.user.email

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["SWI-CFG"]
            distinct_labels = sorted(collection.distinct("SWI.Labels", {})[0:20], key=str.lower)
            response[1] = distinct_labels
        except Exception as err:
            self.status_log("EXCEPTION (get_all_switch_labels)", sys.exc_info()[1])
            messages.error(request, f"{type(err)}")
            messages.error(request, traceback.format_exc(), extra_tags='stacktrace')
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get SWI labels due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "SWI-CFG", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    # Get all Regions and associated POPs currently configured
    def get_all_switch_regions_pops(self, request):
        response = [status.HTTP_200_OK, {}]
        user_email = request.user.email

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["SWI-CFG"]
            distinct_regions = sorted(collection.distinct("SWI.Region", {}), key=str.lower)
            for region in distinct_regions:
                if region != '':
                    distinct_pops = sorted(collection.distinct("SWI.POP", {"SWI.Region": region}), key=str.lower)
                    response[1][region] = distinct_pops
        except Exception as err:
            self.status_log("EXCEPTION (get_all_switch_regions_pops)", sys.exc_info()[1])
            messages.error(request, f"{type(err)}")
            messages.error(request, traceback.format_exc(), extra_tags='stacktrace')
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get SWI Regions and POPs due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "SWI-CFG", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    # Get the region and pop associated with a given switch MAC Address
    def get_switch_region_pop(self, user_email, mac_address):
        response = [status.HTTP_200_OK, {}]

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["SWI-CFG"]
            swi_cfg = collection.find_one({"_id": mac_address}, {"_id": 1, "SWI.Region": 1, "SWI.POP": 1})

            if swi_cfg is not None and 'Region' in swi_cfg['SWI'] and 'POP' in swi_cfg['SWI']:
                response[1]['Region'] = swi_cfg['SWI']['Region']
                response[1]['POP'] = swi_cfg['SWI']['POP']
            else:
                response[1]['Region'] = ''
                response[1]['POP'] = ''
        except Exception as err:
            self.status_log("EXCEPTION (get_switch_region_pop)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get SWI Region and POP due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "SWI-CFG", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    # Retrieve PON AUTO config
    def get_pon_auto_config(self, pon_auto_id):
        response = [status.HTTP_200_OK, [], None, {}]

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["AUTO-CFG"]
            response[1] = collection.find_one({"_id": pon_auto_id})
        except Exception as err:
            self.status_log("EXCEPTION (get_pon_auto_config)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get PON Automation config due to {type(err).__name__}::{sys.exc_info()[1]}", None,
                        None]

        return response

    # Retrieve PON AUTO config
    def update_pon_auto_config(self, pon_auto_id, new_config):
        response = [status.HTTP_200_OK, None, None, None]

        try:
            database_manager.get_database(self.database_id)
            if new_config and 'data' in new_config and '_id' in new_config['data']:
                collection = self.database["AUTO-CFG"]
                response[2] = new_config['data']
                response[3] = collection.find_one_and_replace(filter={"_id": pon_auto_id},
                                                              replacement=new_config['data'],
                                                              return_document=ReturnDocument.BEFORE, upsert=True)
            else:
                response = [status.HTTP_400_BAD_REQUEST,
                            "Request body must be of format '{{ data: <AUTO-CFG document> }}'", None, None]
        except Exception as err:
            self.status_log("EXCEPTION (update_pon_auto_config)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not update PON Automation config due to {type(err).__name__}::{sys.exc_info()[1]}", None,
                        None]

        return response

    # Retrieve PON AUTO state
    def get_pon_auto_state(self, pon_auto_id):
        response = [status.HTTP_200_OK, []]

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["AUTO-STATE"]
            response[1] = collection.find_one({"_id": pon_auto_id})
        except Exception as err:
            self.status_log("EXCEPTION (get_pon_auto_state)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get PON Automation state due to {type(err).__name__}::{sys.exc_info()[1]}"]

        return response

    # Retrieve PON AUTO state count from AUTO-STATE collection
    def get_pon_auto_state_count(self):
        response = [status.HTTP_200_OK, []]

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["AUTO-STATE"]
            response[1] = len(list(collection.find({}, {"_id": 1})))
        except Exception as err:
            self.status_log("EXCEPTION (get_pon_auto_state_count)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get PON Automation state count due to {type(err).__name__}::{sys.exc_info()[1]}"]

        return response

    # Retrive PON AUTO states and whether they're online or not
    def get_pon_auto_status(self, user_email):
        response = [status.HTTP_200_OK, []]

        try:
            database_manager.get_database(self.database_id)
            automation_states = list(self.database["AUTO-STATE"].find({}, {"Time": 1, "AUTO.Automation Pipeline Disabled": 1}).sort("Time", pymongo.DESCENDING))
            for state in automation_states:
                online = False
                if "Time" in state:
                    online = self.automation_time_is_online(state["Time"])
                state["Online"] = online

                automation_pipeline_disabled = False
                if "AUTO" in state and "Automation Pipeline Disabled" in state["AUTO"]:
                    automation_pipeline_disabled = state["AUTO"]["Automation Pipeline Disabled"]
                del state["AUTO"]
                state["Automation Pipeline Disabled"] = automation_pipeline_disabled

                response[1].append(state)

        except Exception as err:
            self.status_log("EXCEPTION (get_pon_auto_status)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get PON Automation status due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "AUTO-STATE", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)
        return response

    """
    Get tree data
    """

    # Retrieve all unique PON AUTO services from AUTO-STATE collection
    def get_pon_auto_services_for_tree(self):
        response = [status.HTTP_200_OK, []]

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["AUTO-STATE"]
            pon_auto_services = list(collection.aggregate([
                {
                    "$lookup": {
                        "from": "AUTO-CFG",
                        "localField": "_id",
                        "foreignField": "_id",
                        "as": "CFG"
                    }
                },
                {
                    "$addFields": {
                        "Name": {
                            "$arrayElemAt": ["$CFG.AUTO.Name", 0]
                        }
                    }
                },
                {
                    "$project": {
                        "_id": 1,
                        "Name": 1,
                        "Time": 1,
                        "AUTO.CFG Version": 1
                    }
                }
            ]))

            for service in pon_auto_services:
                response[1].append({
                    "_id": service["_id"],
                    "name": service["Name"],
                    "offline": not self.automation_time_is_online(service["Time"])
                })

            response[1] = sorted(response[1], key=lambda i: (i['name'].lower(), i['_id']))
        except Exception as err:
            self.status_log("EXCEPTION (get_pon_auto_services_for_tree)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get PON Automation services list due to {type(err).__name__}::{sys.exc_info()[1]}"]

        return response

    # Retrieve all unique MGMT LANs from CNTL-STATEs
    def get_mgmt_lans_for_tree(self, user_email):
        response = [status.HTTP_200_OK, []]

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["CNTL-STATE"]
            lan_resp = collection.distinct("MGMT LAN.Name", {})

            index = 0
            while index < len(lan_resp):
                response[1].append({"name": ""})
                # Get State data
                response[1][index]["name"] = lan_resp[index]
                index += 1

            val_array = response[1]
            response[1] = sorted(val_array, key=lambda i: (i['name'].lower()))
        except Exception as err:
            self.status_log("EXCEPTION (get_mgmt_lans_for_tree)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get PON Controllers list due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "CNTL-STATE", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    # Retrieve an array of controllers and associated tree data
    def get_cntls_for_tree(self, user_email, mgmt_lan):
        response = [status.HTTP_200_OK, []]

        try:
            database_manager.get_database(self.database_id)
            # Get all distinct document ids from collection CNTL_STATE
            collection = self.database.get_collection("CNTL-STATE")
            controllers = list(collection.aggregate([
                {
                    "$match": {
                        "MGMT LAN.Name": mgmt_lan
                    }
                },
                {
                    "$project": {
                        "_id": 1, "MGMT LAN.Name": 1, "CNTL.Version": 1, "CNTL.Pause": 1, "Time": 1,
                        "Alarm.0-EMERG": 1, "Alarm.1-ALERT": 1, "Alarm.2-CRIT": 1, "Alarm.3-ERROR": 1,
                        "Alarm.4-WARNING": 1
                    }
                },
                {
                    "$lookup": {
                        "from": "CNTL-CFG",
                        "let": {"stateId": "$_id"},
                        "pipeline": [
                            {
                                "$match": {
                                    "$expr": {"$eq": ["$_id", "$$stateId"]}
                                }
                            },
                            {
                                "$project": {
                                    "CNTL.Name": 1
                                }
                            }
                        ],
                        "as": "CNTL-CFG"
                    }
                },
                {
                    "$lookup": {
                        "from": "CNTL-ALARM-HIST-STATE",
                        "let": {"stateId": "$_id"},
                        "pipeline": [
                            {"$match": {"$expr": {"$eq": ["$_id", "$$stateId"]}}},
                            {
                                "$project": {
                                    "Alarms": {
                                        "$filter": {
                                            "input": "$Alarms",
                                            "cond": {"$and": [{"$eq": ["$$this.Active State", True]},
                                                              {"$eq": ["$$this.Ack State", False]}]}
                                        }
                                    }
                                }
                            }
                        ], "as": "CNTL-ALARM-HIST-STATE"
                    }
                },
                {
                    '$lookup': {
                        'from': 'CNTL-AUTO-STATE',
                        "let": {"stateId": "$_id"},
                        'pipeline': [
                            {
                                '$match': {
                                    '$expr': {
                                        '$eq': [
                                            '$_id', '$$stateId'
                                        ]
                                    }
                                }
                            }, {
                                '$project': {
                                    'Enable': '$AUTO.Enable'
                                }
                            }
                        ],
                        'as': 'CNTL-AUTO'
                    }
                },
                {
                    '$lookup': {
                        'from': 'AUTO-STATE',
                        'as': 'AUTO-Time',
                        'pipeline': [
                            {
                                '$project': {
                                    'Time': 1
                                }
                            }
                        ]
                    }
                }
            ]))

            for controller in controllers:
                alarm_history_supported = False

                # If using PON Controller major release version of 3 or greater than ALARM-HIST-STATE is supported
                if controller and controller["CNTL"] and get_nested_value(controller, ["CNTL", "Version"]) is not None and \
                        (len(get_nested_value(controller, ["CNTL", "Version"])) > 2) and int(
                    controller["CNTL"]["Version"][1:2]) >= 3:
                    alarm_history_supported = True

                id = controller["_id"]
                name = get_nested_value(controller, ["CNTL-CFG", 0, "CNTL", "Name"], "")
                offline = self.controller_time_is_online(controller["Time"], controller["CNTL"]["Version"]) == False
                paused = controller["CNTL"]["Pause"]
                mgmt_lan = controller["MGMT LAN"]["Name"]

                auto_disabled = False
                if "CNTL-AUTO" in controller:
                    if len(controller["CNTL-AUTO"]) > 0 and "Enable" in controller["CNTL-AUTO"][0]:
                        auto_disabled = not controller["CNTL-AUTO"][0]["Enable"]

                auto_offline = True
                if "AUTO-Time" in controller and len(controller["AUTO-Time"]) > 0 and "Time" in controller["AUTO-Time"][0]:
                    auto_offline = not self.automation_time_is_online(controller["AUTO-Time"][0]["Time"])

                if alarm_history_supported:
                    emerg_alarms = []
                    alert_alarms = []
                    crit_alarms = []
                    error_alarms = []
                    warning_alarms = []

                    if 'CNTL-ALARM-HIST-STATE' in controller and len(controller['CNTL-ALARM-HIST-STATE']) > 0 and \
                            controller['CNTL-ALARM-HIST-STATE'][0]['Alarms']:
                        for alarm_history in controller['CNTL-ALARM-HIST-STATE'][0]['Alarms']:
                            alarm_text = alarm_history['Text']
                            if alarm_history['Severity'] == "0-EMERG":
                                emerg_alarms.append(alarm_text)
                            elif alarm_history['Severity'] == "1-ALERT":
                                alert_alarms.append(alarm_text)
                            elif alarm_history['Severity'] == "2-CRIT":
                                crit_alarms.append(alarm_text)
                            elif alarm_history['Severity'] == "3-ERROR":
                                error_alarms.append(alarm_text)
                            elif alarm_history['Severity'] == "4-WARNING":
                                warning_alarms.append(alarm_text)

                    errors = {
                        "EMERG": emerg_alarms,
                        "ALERT": alert_alarms,
                        "CRIT": crit_alarms,
                        "ERROR": error_alarms
                    }
                    warnings = warning_alarms
                else:
                    errors = {
                        "EMERG": controller["Alarm"]["0-EMERG"],
                        "ALERT": controller["Alarm"]["1-ALERT"],
                        "CRIT": controller["Alarm"]["2-CRIT"],
                        "ERROR": controller["Alarm"]["3-ERROR"]
                    }
                    warnings = controller["Alarm"]["4-WARNING"]

                response[1].append({"_id": id,
                                    "name": name,
                                    "offline": offline,
                                    "auto_offline": auto_offline,
                                    "cntl_auto_disabled": auto_disabled,
                                    "paused": paused,
                                    "mgmt_lan": mgmt_lan,
                                    "errors": errors,
                                    "warnings": warnings})

            val_array = response[1]
            response[1] = sorted(val_array, key=lambda i: (i['name'].lower(), i['_id']))
        except Exception as err:
            self.status_log("EXCEPTION (get_cntls_for_tree)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get PON Controllers list due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "CNTL-STATE, CNTL-CFG", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    # Retrieve regions for tree
    def get_regions_for_tree(self, user_email):
        response = [status.HTTP_200_OK, []]

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["OLT-STATE"]
            switch_macs = collection.distinct("Switch.Chassis ID", {})
            unassigned_region_added = False

            collection = self.database["SWI-CFG"]

            switch_cfgs = collection.find({"_id": {"$in": switch_macs}}, {"_id": 1, "SWI.Region": 1, "SWI.POP": 1, "OLTs": 1})
            all_inventoried_olts = []

            for cfg in switch_cfgs:
                if (
                        ('Region' in cfg['SWI'] and 'POP' in cfg['SWI']) and
                        (cfg['SWI']['Region'].strip() != '' and cfg['SWI']['POP'].strip() != '')
                ):
                    if cfg['SWI']['Region'] not in [value for elem in response[1] for value in elem.values()]:
                        data = {"_id": cfg['SWI']['Region']}
                        response[1].append(data)
                # WARN: The space following Unassigned is intentional. This is to distinguish the special Unassigned heirarchy from a manually created Unassigned Region
                elif 'Unassigned ' not in [value for elem in response[1] for value in
                                           elem.values()]:  # Only add Unassigned node if not already added
                    data = {"_id": 'Unassigned '}
                    response[1].append(data)
                    unassigned_region_added = True

                if 'OLTs' in cfg:
                    all_inventoried_olts.extend(cfg['OLTs'])

            if not unassigned_region_added:
                # Add unknown switch object if there's an uninventoried olt without a switch
                collection = self.database["OLT-STATE"]
                olts_without_switch = list(collection.find({'Switch': {}, '_id': {'$nin': all_inventoried_olts}}, {'_id': 1}))

                if olts_without_switch is not None and len(olts_without_switch) > 0:
                    olts_without_switch = [olt['_id'] for olt in olts_without_switch]
                    for olt in all_inventoried_olts:
                        if olt in olts_without_switch:
                            olts_without_switch = {key: val for key, val in olts_without_switch if val != olt}

                    if len(olts_without_switch) > 0:
                        data = {"_id": 'Unassigned '}
                        response[1].append(data)

        except Exception as err:
            self.status_log("EXCEPTION (get_regions_for_tree)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get Regions list due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "SWI-CFG", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    # Retrieve regions for tree
    def get_pops_under_region_for_tree(self, user_email, region):
        response = [status.HTTP_200_OK, []]

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["OLT-STATE"]
            switch_macs = collection.distinct("Switch.Chassis ID", {})

            collection = self.database["SWI-CFG"]
            distinct_pops = sorted(collection.distinct("SWI.POP", {"_id": {"$in": switch_macs}, "SWI.Region": region}),
                                   key=str.lower)

            for pop in distinct_pops:
                if pop.strip() != '':
                    data = {"_id": pop}
                    response[1].append(data)

        except Exception as err:
            self.status_log("EXCEPTION (get_pops_under_region_for_tree)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get POPs list due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "SWI-CFG", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    # Get All switches under a pop for the tree
    # POPs can have the same name under different Regions, so region is also required
    def get_switches_under_pop_for_tree(self, user_email, region, pop):
        response = [status.HTTP_200_OK, []]

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["OLT-STATE"]
            switch_macs = collection.distinct("Switch.Chassis ID", {})
            switch_automation_states_list = list(self.database["SWI-AUTO-STATE"].find({"_id": {"$in": switch_macs}}, {"_id": 1, "AUTO.Enable": 1, "Time": 1}))
            switch_automation_states = {}
            auto_offline = True
            original_region = region
            original_pop = pop

            if len(switch_automation_states_list) > 0:
                for switch_auto_state in switch_automation_states_list:
                    if switch_auto_state["_id"] != None:
                        switch_automation_states[switch_auto_state["_id"]] = switch_auto_state
                        if auto_offline:
                            auto_offline = not self.automation_time_is_online(switch_auto_state["Time"])

            all_inventoried_olts = []
            collection = self.database["SWI-CFG"]

            for switch in switch_macs:
                data = {"_id": switch, "name": "", "auto_offline": auto_offline, "switch_auto_disabled": True}
                switch_cfg_data = collection.find_one({"_id": switch}, {"SWI.Name": 1, "SWI.Region": 1, "SWI.POP": 1, "OLTs": 1})

                if switch in switch_automation_states:
                    data["switch_auto_disabled"] = not get_nested_value(switch_automation_states, [switch, "AUTO", "Enable"], False)

                if switch_cfg_data is not None:
                    # WARN: The space following Unassigned is intentional. This is to distinguish the special Unassigned heirarchy from a manually created Unassigned Region
                    # If region and pop are 'Unassigned ', zero out region and pop fields to find all unassigned values in DB.
                    # API call does not work properly if you try to use blank values in URL path parameters
                    if region == 'Unassigned ' and pop == 'Unassigned ':
                        region = ''
                        pop = ''
                    # Assigned Switches
                    if region != '' and pop != '':
                        if 'Region' in switch_cfg_data["SWI"] and 'POP' in switch_cfg_data["SWI"] and \
                                switch_cfg_data["SWI"]["Region"] == region and switch_cfg_data["SWI"]["POP"] == pop:
                            if switch_cfg_data is None:
                                data["name"] = ""
                            else:
                                data["name"] = switch_cfg_data["SWI"]["Name"]
                            response[1].append(data)
                    else:  # Unassigned Switches
                        if 'Region' not in switch_cfg_data["SWI"] or 'POP' not in switch_cfg_data["SWI"] or \
                                switch_cfg_data["SWI"]["Region"].strip() == '' or switch_cfg_data["SWI"][
                            "POP"].strip() == '':
                            if switch_cfg_data is None:
                                data["name"] = ""
                            else:
                                data["name"] = switch_cfg_data["SWI"]["Name"]
                            response[1].append(data)

                    if 'OLTs' in switch_cfg_data:
                        all_inventoried_olts.extend(switch_cfg_data['OLTs'])
                    

            olts_without_switch = None
            if original_region == 'Unassigned ' and original_pop == 'Unassigned ':
                collection = self.database["OLT-STATE"]
                olts_without_switch = list(collection.find({'Switch': {}, '_id': {'$nin': all_inventoried_olts}}, {'_id': 1}))

            if olts_without_switch != None and len(olts_without_switch) > 0:
                response[1].append({'_id': 'Unknown Switch', 'name': ''})

            val_array = response[1]
            response[1] = sorted(val_array, key=lambda i: (i['name'].lower(), i['_id']))

        except Exception as err:
            self.status_log("EXCEPTION (get_switches_under_pop_for_tree)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get Switches list due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "OLT-STATE, SWI-CFG", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    def get_olts_under_switch_for_tree(self, user_email, switch_id):
        response = [status.HTTP_200_OK, []]
        isOffline = False
        alarm_history_supported = False
        system_status = ''

        try:
            database_manager.get_database(self.database_id)
            # Get all distinct document ids from collection OLT_STATE
            collection = self.database["OLT-STATE"]
            if switch_id == "Default":
                olts = collection.find({"Switch.Chassis ID": ""}, {"_id": 1})

            else:
                if switch_id == 'Unknown Switch':
                    match_dict = {'Switch': {}}
                    inventory_match_doc = {}
                else:
                    match_dict = {'$or': [
                        {"Switch.Chassis ID": switch_id},
                        {"Switch": {}},
                    ]}
                    inventory_match_doc = {"_id": switch_id}

                olts = list(collection.aggregate([
                    {
                        "$match": match_dict
                    },
                    {
                        "$project": {
                            "_id": 1, "CNTL.MAC Address": 1, "CNTL.Version": 1, "Switch.Port ID": 1,
                            "Switch.Chassis ID": 1,
                            "Protection.Peer": 1, "Protection.Status": 1, "ONUs": 1, "ONU States": 1,
                            "OLT.Reset Count": 1,
                            "OLT.PON Mode": 1,
                            "NNI.Supported Speeds": 1,
                            "Alarm.0-EMERG": 1, "Alarm.1-ALERT": 1, "Alarm.2-CRIT": 1, "Alarm.3-ERROR": 1,
                            "Alarm.4-WARNING": 1,
                            "OLTs": 1
                        }
                    },
                    {
                        "$lookup": {
                            "from": "OLT-CFG",
                            "let": {"stateId": "$_id"},
                            "pipeline": [
                                {
                                    "$match": {
                                        "$expr": {"$eq": ["$_id", "$$stateId"]}
                                    }
                                },
                                {
                                    "$project": {
                                        "OLT.Reset Count": 1, "OLT.Name": 1
                                    }
                                }
                            ],
                            "as": "oltCfg"
                        }
                    },
                    {
                        "$lookup": {
                            "from": "OLT-ALARM-HIST-STATE",
                            "let": {"stateId": "$_id"},
                            "pipeline": [
                                {"$match": {"$expr": {"$eq": ["$_id", "$$stateId"]}}},
                                {
                                    "$project": {
                                        "Alarms": {
                                            "$filter": {
                                                "input": "$Alarms",
                                                "cond": {"$and": [{"$eq": ["$$this.Active State", True]},
                                                                  {"$eq": ["$$this.Ack State", False]}]}
                                            }
                                        }
                                    }
                                }
                            ], "as": "OLT-ALARM-HIST-STATE"
                        }
                    },
                    {
                        '$lookup': {
                            'from': 'CNTL-AUTO-STATE',
                            "let": {"stateId": "$CNTL.MAC Address"},
                            'pipeline': [
                                {
                                    '$match': {
                                        '$expr': {
                                            '$eq': [
                                                '$_id', '$$stateId'
                                            ]
                                        }
                                    }
                                }, {
                                    '$project': {
                                        'Enable': '$AUTO.Enable'
                                    }
                                }
                            ],
                            'as': 'CNTL-AUTO'
                        }
                    },
                    {
                        '$lookup': {
                            'from': 'AUTO-STATE',
                            'as': 'AUTO-Time',
                            'pipeline': [
                                {
                                    '$project': {
                                        'Time': 1
                                    }
                                }
                            ]
                        }
                    },
                    {
                        '$lookup': {
                            'from': 'OLT-AUTO-STATE',
                            "let": {"stateId": "$_id"},
                            'pipeline': [
                                {
                                    '$match': {
                                        '$expr': {
                                            '$eq': [
                                                '$_id', '$$stateId'
                                            ]
                                        }
                                    }
                                }, {
                                    '$project': {
                                        'Enable': '$AUTO.Enable'
                                    }
                                }
                            ],
                            'as': 'OLT-AUTO'
                        }
                    }
                ]))

            if switch_id == "Unknown Switch":
                ifOffline = False
                pass

            # Get inventoried OLTs
            inventoried_olts = list(self.database["SWI-CFG"].aggregate([
                {
                    '$match': inventory_match_doc
                }, {
                    '$addFields': {
                        'olts': {
                            '$objectToArray': '$OLTs'
                        }
                    }
                }, {
                    '$unwind': '$olts'
                }, {
                    '$group': {
                        '_id': 'all_onus',
                        'inventoried_olts': {
                            '$addToSet': '$olts.k'
                        }
                    }
                }
            ]))

            if len(inventoried_olts) > 0:
                inventoried_olts = inventoried_olts[0]['inventoried_olts']

            # If using PON Controller major release version of 3 or greater than ALARM-HIST-STATE is supported
            if olts and len(olts) > 0 and olts[0]["CNTL"] and olts[0]["CNTL"]["Version"] and int(
                    olts[0]["CNTL"]["Version"][1:2]) >= 3:
                alarm_history_supported = True

            auto_offline = True
            olt_ids = []
            all_olts = []
            for olt_data in olts:
                if switch_id == 'Unknown Switch':  # Unknown Switch should only contain un-inventoried switches
                    if olt_data["_id"] not in inventoried_olts:
                        olt_ids.append(olt_data["_id"])
                        all_olts.append(olt_data)
                else:  # Show OLT without OLT-STATE[Switch] data only if it is inventoried on that switch
                    if (olt_data["_id"] in inventoried_olts and olt_data['Switch'] == {}) or olt_data['Switch'] != {}:
                        olt_ids.append(olt_data["_id"])
                        all_olts.append(olt_data)
                if auto_offline == True and "AUTO-Time" in olt_data and len(olt_data["AUTO-Time"]) > 0 and "Time" in \
                        olt_data["AUTO-Time"][0]:
                    auto_offline = not self.automation_time_is_online(olt_data["AUTO-Time"][0]["Time"])

            # Get pre-provisioned OLTs
            prepro_olts = list(self.database["SWI-CFG"].aggregate([
                {
                    '$match': {
                        '_id': switch_id
                    }
                }, {
                    '$addFields': {
                        'olts': {
                            '$objectToArray': '$OLTs'
                        }
                    }
                }, {'$unwind': '$olts'}, {
                    '$project': {
                        'k': '$olts.k',
                        'v': '$olts.v'
                    }
                }, {
                    '$match': {
                        '$expr': {
                            # Only keep IDs of OLTs that were not found in OLT-STATE
                            '$not': {
                                '$in': ['$k', olt_ids]
                            }
                        }
                    }
                }, {
                    '$lookup': {
                        'from': 'OLT-CFG',
                        'let': {
                            'cfgId': '$k'
                        },
                        'pipeline': [
                            {
                                '$match': {
                                    '$expr': {
                                        '$eq': ['$_id', '$$cfgId']
                                    }
                                }
                            }
                        ],
                        'as': 'oltCfg'
                    }
                }, {
                    '$project': {
                        '_id': 1,
                        'k': 1,
                        'v': 1,
                        'oltCfg': {
                            '$cond': {
                                'if': {
                                    '$gte': [{'$size': '$oltCfg'}, 1]
                                },
                                'then': {
                                    '$arrayElemAt': ['$oltCfg', 0]
                                },
                                'else': {}
                            }
                        }
                    }
                }
            ]))

            # get CNTL states
            cntl_states = list(self.database["CNTL-STATE"].find(projection={"_id": 1, "Time": 1, "System Status": 1, "CNTL.Version": 1}))

            for olt in all_olts:
                reg_onus = 0
                system_status = ''
                # Get OLTs Name if it exists
                if olt['oltCfg'] and olt['oltCfg'][0]:
                    olt_cfg = olt['oltCfg'][0]
                    if olt_cfg["OLT"]["Name"] is not None:
                        name = olt_cfg["OLT"]["Name"]
                    if olt_cfg["OLT"]["Reset Count"] is not None:
                        reset_count_config = olt_cfg["OLT"]["Reset Count"]
                else:
                    name = ""
                    reset_count_config = 99999999

                # Get Offline status
                controller_id = olt["CNTL"]["MAC Address"]
                cntl_state = None
                # Find Controller state for this OLT
                for state in cntl_states:
                    if state["_id"] == controller_id:
                        cntl_state = state

                isOffline = False
                if cntl_state is not None and "Time" in cntl_state:
                    if self.controller_time_is_online(cntl_state["Time"], cntl_state["CNTL"]["Version"]):
                        if olt is not None and olt["_id"] in cntl_state["System Status"]:
                            if cntl_state["System Status"][olt["_id"]]["OLT State"] == "Unspecified" or \
                                    cntl_state["System Status"][olt["_id"]]["OLT State"] == "Primary" or \
                                    cntl_state["System Status"][olt["_id"]]["OLT State"] == "Secondary":
                                isOffline = False
                                if olt is not None and "ONU States" in olt:
                                    if "Registered" in olt["ONU States"] and "Unspecified" in olt["ONU States"]:
                                        reg_onus = len(olt["ONU States"]["Registered"]) + len(
                                            olt["ONU States"]["Unspecified"])

                                # Determine OLT Protection Active/Standby Status
                                if 'Protection' in olt and 'Peer' in olt['Protection'] and len(
                                        olt['Protection']['Peer']) > 0:
                                    system_status = olt['Protection']['Status']

                            else:
                                isOffline = True
                        else:
                            isOffline = True
                    else:
                        isOffline = True
                else:
                    isOffline = True

                # Test if OLT is resetting if it is online
                if not isOffline:
                    isResetting = olt["OLT"]["Reset Count"] != reset_count_config
                    if isResetting:
                        system_status = "Pending Reset"

                if alarm_history_supported:
                    emerg_alarms = []
                    alert_alarms = []
                    crit_alarms = []
                    error_alarms = []
                    warning_alarms = []

                    if 'OLT-ALARM-HIST-STATE' in olt and len(olt['OLT-ALARM-HIST-STATE']) > 0 and \
                            olt['OLT-ALARM-HIST-STATE'][0]['Alarms']:
                        for alarm_history in olt['OLT-ALARM-HIST-STATE'][0]['Alarms']:
                            alarm_text = alarm_history['Text']
                            if alarm_history['Severity'] == "0-EMERG":
                                emerg_alarms.append(alarm_text)
                            elif alarm_history['Severity'] == "1-ALERT":
                                alert_alarms.append(alarm_text)
                            elif alarm_history['Severity'] == "2-CRIT":
                                crit_alarms.append(alarm_text)
                            elif alarm_history['Severity'] == "3-ERROR":
                                error_alarms.append(alarm_text)
                            elif alarm_history['Severity'] == "4-WARNING":
                                warning_alarms.append(alarm_text)

                    errors = {
                        "EMERG": emerg_alarms,
                        "ALERT": alert_alarms,
                        "CRIT": crit_alarms,
                        "ERROR": error_alarms
                    }
                    warnings = warning_alarms
                else:
                    errors = {
                        "EMERG": olt["Alarm"]["0-EMERG"],
                        "ALERT": olt["Alarm"]["1-ALERT"],
                        "CRIT": olt["Alarm"]["2-CRIT"],
                        "ERROR": olt["Alarm"]["3-ERROR"]
                    }
                    warnings = olt["Alarm"]["4-WARNING"]

                port = ''

                switch = ''

                if "Switch" in olt and "Port ID" in olt["Switch"]:
                    port = olt["Switch"]["Port ID"]
                if "Switch" in olt and "Chassis ID" in olt["Switch"]:
                    switch = olt["Switch"]["Chassis ID"]

                # Get OLT Automation Enable
                olt_auto_disabled = True
                cntl_auto_disabled = True

                if "CNTL-AUTO" in olt:
                    if len(olt["CNTL-AUTO"]) > 0 and "Enable" in olt["CNTL-AUTO"][0]:
                        cntl_auto_disabled = not olt["CNTL-AUTO"][0]["Enable"]

                if "OLT-AUTO" in olt:
                    if len(olt["OLT-AUTO"]) > 0 and "Enable" in olt["OLT-AUTO"][0]:
                        olt_auto_disabled = not olt["OLT-AUTO"][0]["Enable"]

                # Assign correct PON Standard
                pon_standard = ""
                if "NNI" in olt and "Supported Speeds" in olt["NNI"]:
                    supported_speeds = olt["NNI"]["Supported Speeds"]
                    if "25Gbps" in supported_speeds:
                        pon_standard = "25GS"
                    elif "10Gbps" in supported_speeds:
                        if "OLT" in olt and "PON Mode" in olt["OLT"]:
                            if olt["OLT"]["PON Mode"] == "EPON":
                                pon_standard = "10GE"
                            else:
                                pon_standard = "XGS"

                response[1].append({"_id": olt["_id"],
                                    "port": port,
                                    "cntl": olt["CNTL"]["MAC Address"],
                                    "name": name,
                                    "offline": isOffline,
                                    "auto_offline": auto_offline,
                                    "cntl_auto_disabled": cntl_auto_disabled,
                                    "olt_auto_disabled": olt_auto_disabled,
                                    "total_onus": len(olt["ONUs"]),
                                    "reg_onus": reg_onus,
                                    "switch": switch,
                                    "state": system_status,
                                    "errors": errors,
                                    "warnings": warnings,
                                    "pon_standard": pon_standard})

            for olt in prepro_olts:
                olt_id = olt["k"]
                if olt_id != "Default" and olt['oltCfg']:
                    system_status = "Prepovisioned"
                    olt_cfg = olt['oltCfg']
                    name = ''
                    errors = {
                        "EMERG": [],
                        "ALERT": [],
                        "CRIT": [],
                        "ERROR": []
                    }
                    warnings = []
                    if olt_cfg:
                        name = olt_cfg["OLT"]["Name"]
                    response[1].append({"_id": olt_id,
                                        "port": olt["v"]["Port ID"],
                                        "cntl": '',
                                        "name": name,
                                        "offline": True,
                                        "total_onus": 0,
                                        "reg_onus": 0,
                                        "switch": switch_id,
                                        "state": system_status,
                                        "errors": errors,
                                        "warnings": warnings,
                                        "pon_standard": ''})

            val_array = response[1]
            response[1] = sorted(val_array, key=lambda i: (i['port'], i['name'].lower(), i['_id']))
        except Exception as err:
            self.status_log("EXCEPTION (get_olts_under_switch_for_tree)", err)
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get OLTs for Switch {switch_id} due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "OLT-STATE, OLT-CFG",
                       "returned": response[1],
                       "user": user_email}
        self.sys_log(action_data, "get", response[0])

        return response

    def get_olts_under_controller_for_tree(self, user_email, controller_id):
        response = [status.HTTP_200_OK, []]
        isOffline = False
        alarm_history_supported = False
        system_status = ''

        try:
            database_manager.get_database(self.database_id)
            # Get all distinct document ids from collection OLT_STATE
            olt_state_coll = self.database["OLT-STATE"]
            if controller_id == "Default":
                state_olts = olt_state_coll.find({"CNTL.MAC Address": ""}, {"_id": 1})
            else:
                state_olts = list(olt_state_coll.aggregate([
                    {
                        "$match": {
                            "CNTL.MAC Address": controller_id
                        }
                    },
                    {
                        "$project": {
                            "_id": 1, "CNTL.MAC Address": 1, "CNTL.Version": 1, "Switch.Port ID": 1,
                            "Switch.Chassis ID": 1,
                            "Protection.Peer": 1, "Protection.Status": 1, "ONUs": 1, "ONU States": 1,
                            "OLT.Reset Count": 1,
                            "OLT.PON Mode": 1,
                            "NNI.Supported Speeds": 1,
                            "Alarm.0-EMERG": 1, "Alarm.1-ALERT": 1, "Alarm.2-CRIT": 1, "Alarm.3-ERROR": 1,
                            "Alarm.4-WARNING": 1
                        }
                    },
                    {
                        "$lookup": {
                            "from": "OLT-CFG",
                            "let": {"stateId": "$_id"},
                            "pipeline": [
                                {
                                    "$match": {
                                        "$expr": {"$eq": ["$_id", "$$stateId"]}
                                    }
                                },
                                {
                                    "$project": {
                                        "OLT.Reset Count": 1, "OLT.Name": 1
                                    }
                                }
                            ],
                            "as": "oltCfg"
                        }
                    },
                    {
                        "$lookup": {
                            "from": "OLT-ALARM-HIST-STATE",
                            "let": {"stateId": "$_id"},
                            "pipeline": [
                                {"$match": {"$expr": {"$eq": ["$_id", "$$stateId"]}}},
                                {
                                    "$project": {
                                        "Alarms": {
                                            "$filter": {
                                                "input": "$Alarms",
                                                "cond": {"$and": [{"$eq": ["$$this.Active State", True]},
                                                                  {"$eq": ["$$this.Ack State", False]}]}
                                            }
                                        }
                                    }
                                }
                            ], "as": "OLT-ALARM-HIST-STATE"
                        }
                    },
                    {
                        '$lookup': {
                            'from': 'CNTL-AUTO-STATE',
                            "let": {"stateId": "$CNTL.MAC Address"},
                            'pipeline': [
                                {
                                    '$match': {
                                        '$expr': {
                                            '$eq': [
                                                '$_id', '$$stateId'
                                            ]
                                        }
                                    }
                                }, {
                                    '$project': {
                                        'Enable': '$AUTO.Enable'
                                    }
                                }
                            ],
                            'as': 'CNTL-AUTO'
                        }
                    },
                    {
                        '$lookup': {
                            'from': 'AUTO-STATE',
                            'as': 'AUTO-Time',
                            'pipeline': [
                                {
                                    '$project': {
                                        'Time': 1
                                    }
                                }
                            ]
                        }
                    },
                    {
                        '$lookup': {
                            'from': 'OLT-AUTO-STATE',
                            "let": {"stateId": "$_id"},
                            'pipeline': [
                                {
                                    '$match': {
                                        '$expr': {
                                            '$eq': [
                                                '$_id', '$$stateId'
                                            ]
                                        }
                                    }
                                }, {
                                    '$project': {
                                        'Enable': '$AUTO.Enable'
                                    }
                                }
                            ],
                            'as': 'OLT-AUTO'
                        }
                    }
                ]))
            # If using PON Controller major release version of 3 or greater than ALARM-HIST-STATE is supported
            if state_olts and len(state_olts) > 0 and state_olts[0]["CNTL"] and state_olts[0]["CNTL"][
                "Version"] and int(state_olts[0]["CNTL"]["Version"][1:2]) >= 3:
                alarm_history_supported = True

            cntl_state = self.database["CNTL-STATE"].find_one({"_id": controller_id},
                                                              {"Time": 1, "System Status": 1, "OLTs": 1,
                                                               "CNTL.Version": 1})

            state_ids = []
            all_olts = []
            for olt in state_olts:
                state_ids.append(olt["_id"])
                all_olts.append(olt)

            inventoried_olts_without_state = list(self.database["CNTL-CFG"].aggregate([
                {"$match": {"_id": controller_id}},
                {"$project": {"OLTs.Primary": 1}},
                {"$lookup": {
                    "from": "OLT-CFG",
                    "let": {"primaryInventories": "$OLTs.Primary"},
                    "pipeline": [{"$match": {"$expr": {
                        "$and": [{"$in": ["$_id", "$$primaryInventories"]}, {"$not": {"$in": ["$_id", state_ids]}}]
                    }}},
                        {"$project": {"_id": 1, "ONUs": 1, "OLT.Name": 1, "OLT.Reset Count": 1}}
                    ], "as": "oltCfg"
                }
                },
                {"$unwind": "$oltCfg"},
                {"$project": {"_id": "$oltCfg._id", "CNTL.MAC Address": "$_id", "OLT": "$oltCfg.OLT", "ONUs": "$oltCfg.ONUs"}}
            ]))

            cntl_auto_disabled = True
            auto_offline = True

            for olt_data in all_olts:
                # Automation enable for parent devices.
                if cntl_auto_disabled == True and "CNTL-AUTO" in olt_data:
                    if len(olt_data["CNTL-AUTO"]) > 0 and len(olt_data["CNTL-AUTO"]) > 0 and "Enable" in olt_data["CNTL-AUTO"][0]:
                        cntl_auto_disabled = not olt_data["CNTL-AUTO"][0]["Enable"]
                if auto_offline == True and "AUTO-Time" in olt_data and len(olt_data["AUTO-Time"]) > 0 and "Time" in olt_data["AUTO-Time"][0]:
                    auto_offline = not self.automation_time_is_online(olt_data["AUTO-Time"][0]["Time"])

            # compile a list of olt consisting of stateolt and inventoried_olts_without_state olts
            all_olts.extend(inventoried_olts_without_state)

            for olt in all_olts:
                isUnattached = False
                reg_onus = 0
                system_status = ''
                # Get OLTs Name if it exists
                if "oltCfg" in olt:
                    if len(olt["oltCfg"]) > 0 and "OLT" in olt["oltCfg"][0]:
                        #### IF YOU GET HERE, THEN YOU HAVE STATE && CFG
                        if "Name" in olt['oltCfg'][0]["OLT"]:
                            name = olt['oltCfg'][0]["OLT"]["Name"]
                        if "Reset Count" in olt['oltCfg'][0]["OLT"]:
                            reset_count_config = olt['oltCfg'][0]["OLT"]["Reset Count"]
                    else:
                        #### IF YOU GET HERE, THEN YOU HAVE STATE BUT NOT CFG
                        name = ""
                        reset_count_config = 99999999
                else:
                    #### IF YOU GET HERE, THEN YOU HAVE CFG BUT NO STATE
                    system_status = 'Preprovisioned'

                    if "OLT" in olt:
                        if 'Name' in olt["OLT"]:
                            name = olt["OLT"]["Name"]
                        # These olts have no state and therefore a comparison is not made btw state and cfg
                        reset_count_config = -1

                        # Add fields that are missing in CFG as ''
                        olt['Switch'] = {}
                        olt["Switch"]["Port ID"] = ''
                        olt["Switch"]["Chassis ID"] = ''
                        olt["Alarm"] = {}
                        olt["Alarm"]["0-EMERG"] = ''
                        olt["Alarm"]["1-ALERT"] = ''
                        olt["Alarm"]["2-CRIT"] = ''
                        olt["Alarm"]["3-ERROR"] = ''
                        olt["Alarm"]["4-WARNING"] = ''

                # Getting Offline status
                isOffline = False
                if cntl_state is not None and "Time" in cntl_state:
                    if self.controller_time_is_online(cntl_state["Time"], cntl_state["CNTL"]["Version"]):
                        if olt is not None and olt["_id"] in cntl_state["System Status"]:
                            if cntl_state["System Status"][olt["_id"]]["OLT State"] == "Unspecified" or \
                                    cntl_state["System Status"][olt["_id"]]["OLT State"] == "Primary" or \
                                    cntl_state["System Status"][olt["_id"]]["OLT State"] == "Secondary":
                                isOffline = False
                                if olt is not None and "ONU States" in olt:
                                    if "Registered" in olt["ONU States"] and "Unspecified" in olt["ONU States"]:
                                        reg_onus = len(olt["ONU States"]["Registered"]) + len(
                                            olt["ONU States"]["Unspecified"])

                                # Determine OLT Protection Active/Standby Status
                                if 'Protection' in olt and 'Peer' in olt['Protection'] and len(
                                        olt['Protection']['Peer']) > 0:
                                    system_status = olt['Protection']['Status']
                            else:
                                isOffline = True
                        else:
                            # Check if OLT is inventoried, if not then its considered unattached
                            if olt is not None and olt["_id"] in cntl_state["OLTs"]["Primary"] or olt["_id"] in \
                                    cntl_state["OLTs"]["Primary Missing"] or olt["_id"] in cntl_state["OLTs"][
                                "Primary Free"] or \
                                    olt["_id"] in cntl_state["OLTs"]["Secondary"] or olt["_id"] in \
                                    cntl_state["OLTs"]["Secondary Free"] or olt["_id"] in cntl_state["OLTs"][
                                "Secondary Missing"] or olt["_id"] in cntl_state["OLTs"]["Secondary Others"]:
                                isOffline = True
                            else:
                                isUnattached = True
                    else:
                        isOffline = True
                else:
                    isOffline = True

                # Get OLT Automation Enable
                olt_auto_disabled = True
                if "OLT-AUTO" in olt and len(olt["OLT-AUTO"]) > 0:
                    if "Enable" in olt["OLT-AUTO"][0]:
                        olt_auto_disabled = not olt["OLT-AUTO"][0]["Enable"]

                if not isUnattached:
                    # Test if OLT is resetting if it is online
                    if not isOffline:
                        isResetting = olt["OLT"]["Reset Count"] != reset_count_config
                        if reset_count_config == -1:
                            isResetting = False
                        if isResetting:
                            system_status = "Pending Reset"

                    if alarm_history_supported:
                        emerg_alarms = []
                        alert_alarms = []
                        crit_alarms = []
                        error_alarms = []
                        warning_alarms = []

                        if 'OLT-ALARM-HIST-STATE' in olt and len(olt['OLT-ALARM-HIST-STATE']) > 0 and \
                                olt['OLT-ALARM-HIST-STATE'][0]['Alarms']:
                            for alarm_history in olt['OLT-ALARM-HIST-STATE'][0]['Alarms']:
                                alarm_text = alarm_history['Text']
                                if alarm_history['Severity'] == "0-EMERG":
                                    emerg_alarms.append(alarm_text)
                                elif alarm_history['Severity'] == "1-ALERT":
                                    alert_alarms.append(alarm_text)
                                elif alarm_history['Severity'] == "2-CRIT":
                                    crit_alarms.append(alarm_text)
                                elif alarm_history['Severity'] == "3-ERROR":
                                    error_alarms.append(alarm_text)
                                elif alarm_history['Severity'] == "4-WARNING":
                                    warning_alarms.append(alarm_text)

                        errors = {
                            "EMERG": emerg_alarms,
                            "ALERT": alert_alarms,
                            "CRIT": crit_alarms,
                            "ERROR": error_alarms
                        }
                        warnings = warning_alarms
                    else:
                        errors = {
                            "EMERG": olt["Alarm"]["0-EMERG"],
                            "ALERT": olt["Alarm"]["1-ALERT"],
                            "CRIT": olt["Alarm"]["2-CRIT"],
                            "ERROR": olt["Alarm"]["3-ERROR"]
                        }
                        warnings = olt["Alarm"]["4-WARNING"]

                    switch_port = ""
                    switch_chassis = ""
                    if "Switch" in olt:
                        if "Port ID" in olt["Switch"]:
                            switch_port = olt["Switch"]["Port ID"]
                        if "Chassis ID" in olt["Switch"]:
                            switch_chassis = olt["Switch"]["Chassis ID"]

                    # Assign correct PON Standard
                    pon_standard = ""
                    if "NNI" in olt and "Supported Speeds" in olt["NNI"]:
                        supported_speeds = olt["NNI"]["Supported Speeds"]
                        if "25Gbps" in supported_speeds:
                            pon_standard = "25GS"
                        elif "10Gbps" in supported_speeds:
                            if "OLT" in olt and "PON Mode" in olt["OLT"]:
                                if olt["OLT"]["PON Mode"] == "EPON":
                                    pon_standard = "10GE"
                                else:
                                    pon_standard = "XGS"

                    response[1].append({"_id": olt["_id"],
                                        "port": switch_port,
                                        "cntl": olt["CNTL"]["MAC Address"],
                                        "name": name,
                                        "offline": isOffline,
                                        "auto_offline": auto_offline,
                                        "cntl_auto_disabled": cntl_auto_disabled,
                                        "olt_auto_disabled": olt_auto_disabled,
                                        "total_onus": len(olt["ONUs"]),
                                        "reg_onus": reg_onus,
                                        "switch": switch_chassis,
                                        "state": system_status,
                                        "errors": errors,
                                        "warnings": warnings,
                                        "pon_standard": pon_standard})
            val_array = response[1]
            response[1] = sorted(val_array, key=lambda i: (i['name'].lower(), i['_id']))
        except Exception as err:
            self.status_log("EXCEPTION (get_olts_under_controller_for_tree)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get OLTs for Controller {controller_id} due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "OLT-STATE, OLT-CFG",
                       "returned": response[1],
                       "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    def get_onus_under_olt_for_tree(self, user_email, olt_id):
        response = [status.HTTP_200_OK, []]
        alarm_history_supported = False

        try:
            database_manager.get_database(self.database_id)
            # Get System Status from the controller of the given OLT
            system_status_docs = list(self.database["CNTL-STATE"].find(
                {'$or': [{f"System Status.{olt_id}.OLT State": "Primary"},
                         {f"System Status.{olt_id}.OLT State": "Secondary"},
                         {f"System Status.{olt_id}.OLT State": "Unspecified"}]},
                {f"System Status.{olt_id}": 1, "Time": 1, "_id": 1, "CNTL.Version": 1}
            ).sort("Time", pymongo.DESCENDING).limit(1))

            system_status_doc = None
            onu_ids = []
            if len(system_status_docs) > 0:
                system_status_doc = system_status_docs[0]
                onu_ids = list(system_status_doc["System Status"][olt_id]["ONUs"].keys())
                match_document = {
                    "_id": {"$in": onu_ids}
                }
            else:
                match_document = {
                    "OLT.MAC Address": olt_id
                }

            special_ids = list(self.database["OLT-STATE"].aggregate([
                {
                    "$match": {
                        "_id": olt_id
                    }
                }, {
                    "$project": {
                        "_id": 0,
                        "onus": {
                            "$objectToArray": "$ONUs"
                        }
                    }
                }, {
                    "$project": {
                        "special_onu_ids": {
                            "$reduce": {
                                "input": "$onus.k",
                                "initialValue": {
                                    "$arrayElemAt": ["$onus.k", 0]
                                },
                                "in": {
                                    "$setDifference": ["$onus.k", onu_ids]
                                }
                            }
                        }
                    }
                }
            ]))

            special_onu_ids = []
            if len(special_ids) > 0:
                if "special_onu_ids" in special_ids[0]:
                    special_onu_ids = special_ids[0]["special_onu_ids"]

            onu_cfg_coll = self.database["ONU-CFG"]
            onus = list(onu_cfg_coll.aggregate([
                {
                    "$match": {
                        "$or": [
                            {
                                "_id": {
                                    "$in": special_onu_ids
                                }
                            }, match_document
                        ]
                    }
                },
                {
                    "$project": {
                        "ONU.Name": 1, "ONU.Reset Count": 1
                    }
                },
                {
                    "$lookup": {
                        "from": "OLT-STATE",
                        "let": {"oltId": olt_id},
                        "pipeline": [
                            {"$match": {"$expr": {"$eq": ["$_id", "$$oltId"]}}},
                            {"$project": {"_id": 1, "ONU States": 1, "OLT.PON Mode": 1, "NNI.Supported Speeds": 1}}
                        ],
                        "as": "OLT-STATE"
                    }
                },
                {
                    "$lookup": {
                        "from": "ONU-STATE",
                        "let": {"onuId": "$_id"},
                        "pipeline": [
                            {"$match": {"$expr": {"$eq": ["$_id", "$$onuId"]}}},
                            {"$project": {"_id": 1, "CNTL.MAC Address": 1, "CNTL.Version": 1, "OLT.MAC Address": 1,
                                          "ONU.Reset Count": 1,
                                          "Alarm.0-EMERG": 1, "Alarm.1-ALERT": 1, "Alarm.2-CRIT": 1, "Alarm.3-ERROR": 1,
                                          "Alarm.4-WARNING": 1}}
                        ],
                        "as": "ONU-STATE"
                    }
                },
                {
                    "$lookup": {
                        "from": "ONU-ALARM-HIST-STATE",
                        "let": {"onuId": "$_id"},
                        "pipeline": [
                            {"$match": {"$expr": {"$eq": ["$_id", "$$onuId"]}}},
                            {
                                "$project": {
                                    "Alarms": {
                                        "$filter": {
                                            "input": "$Alarms",
                                            "cond": {"$and": [{"$eq": ["$$this.Active State", True]},
                                                              {"$eq": ["$$this.Ack State", False]}]}
                                        }
                                    }
                                }
                            }
                        ], "as": "ONU-ALARM-HIST-STATE"
                    }
                },
                {
                    '$lookup': {
                        'from': 'CNTL-AUTO-STATE',
                        "let": {"stateId": {'$arrayElemAt': ["$ONU-STATE", 0]}},
                        'pipeline': [
                            {
                                '$match': {
                                    '$expr': {
                                        '$eq': [
                                            '$_id', '$$stateId.CNTL.MAC Address'
                                        ]
                                    }
                                }
                            }, {
                                '$project': {
                                    'Enable': '$AUTO.Enable'
                                }
                            }
                        ],
                        'as': 'CNTL-AUTO'
                    }
                },
                {
                    '$lookup': {
                        'from': 'AUTO-STATE',
                        'as': 'AUTO-Time',
                        'pipeline': [
                            {
                                '$project': {
                                    'Time': 1
                                }
                            }
                        ]
                    }
                },
                {
                    '$lookup': {
                        'from': 'OLT-AUTO-STATE',
                        "let": {"stateId": {'$arrayElemAt': ["$ONU-STATE", 0]}},
                        'pipeline': [
                            {
                                '$match': {
                                    '$expr': {
                                        '$eq': [
                                            '$_id', '$$stateId.OLT.MAC Address'
                                        ]
                                    }
                                }
                            }, {
                                '$project': {
                                    'Enable': '$AUTO.Enable'
                                }
                            }
                        ],
                        'as': 'OLT-AUTO'
                    }
                },
                {
                    '$lookup': {
                        'from': 'ONU-AUTO-STATE',
                        "let": {"stateId": "$_id"},
                        'pipeline': [
                            {
                                '$match': {
                                    '$expr': {
                                        '$eq': [
                                            '$_id', '$$stateId'
                                        ]
                                    }
                                }
                            }, {
                                '$project': {
                                    'Enable': '$AUTO.Enable'
                                }
                            }
                        ],
                        'as': 'ONU-AUTO'
                    }
                }
            ]))

            # If using PON Controller major release version of 3 or greater than ALARM-HIST-STATE is supported
            if onus and len(onus) > 0 and get_nested_value(onus[0], ["ONU-STATE", 0, "CNTL", "Version"]) is not None and len(get_nested_value(onus[0], ["ONU-STATE", 0, "CNTL", "Version"])) > 2 and int(onus[0]["ONU-STATE"][0]["CNTL"]["Version"][1:2]) >= 3:
                alarm_history_supported = True

            cntl_auto_disabled = True
            olt_auto_disabled = True
            auto_offline = True

            for onu_data in onus:
                # Automation enable for parent devices.
                if cntl_auto_disabled == True and "CNTL-AUTO" in onu_data:
                    if len(onu_data["CNTL-AUTO"]) > 0 and "Enable" in onu_data["CNTL-AUTO"][0]:
                        cntl_auto_disabled = not onu_data["CNTL-AUTO"][0]["Enable"]
                if olt_auto_disabled == True and "OLT-AUTO" in onu_data:
                    if len(onu_data["OLT-AUTO"]) > 0 and "Enable" in onu_data["OLT-AUTO"][0]:
                        olt_auto_disabled = not onu_data["OLT-AUTO"][0]["Enable"]
                if auto_offline == True and "AUTO-Time" in onu_data and len(onu_data["AUTO-Time"]) > 0 and "Time" in \
                        onu_data["AUTO-Time"][0]:
                    auto_offline = not self.automation_time_is_online(onu_data["AUTO-Time"][0]["Time"])

            for onu_data in onus:
                onu_id = onu_data['_id']
                cntl_id = get_nested_value(onu_data, ["ONU-STATE", 0, "CNTL", "MAC Address"], "")
                name = onu_data["ONU"]["Name"]
                system_status = "Offline"
                is_offline = True
                is_resetting = False
                config_reset_count = onu_data["ONU"]["Reset Count"]
                state_reset_count = get_nested_value(onu_data, ["ONU-STATE", 0, "ONU", "Reset Count"], -1)

                # Check that CNTL is online.
                if system_status_doc is not None and "Time" in system_status_doc and "CNTL" in system_status_doc and "Version" in \
                        system_status_doc["CNTL"] and self.controller_time_is_online(system_status_doc["Time"],
                                                                                     system_status_doc["CNTL"][
                                                                                         "Version"]):
                    if olt_id in system_status_doc["System Status"]:
                        if onu_id in system_status_doc["System Status"][olt_id]["ONUs"]:
                            system_status = system_status_doc["System Status"][olt_id]["ONUs"][onu_id]

                            # If ONU Status is Deregistered in CNTL-STATE, we must verify this is correct by checking OLT-STATE
                            if system_status == "Deregistered":
                                olt_state = get_nested_value(onu_data, ["OLT-STATE", 0])
                                if olt_state:
                                    # Setting ONU status based on OLT-State ONU States
                                    if onu_id in olt_state["ONU States"]["Disabled"]:
                                        system_status = "Disabled"
                                    elif onu_id in olt_state["ONU States"]["Disallowed Admin"]:
                                        system_status = "Disallowed Admin"
                                    elif onu_id in olt_state["ONU States"]["Disallowed Error"]:
                                        system_status = "Disallowed Error"
                                    elif onu_id in olt_state["ONU States"]["Dying Gasp"]:
                                        system_status = "Dying Gasp"
                                    elif onu_id in olt_state["ONU States"]["Unprovisioned"]:
                                        system_status = "Unprovisioned"

                            if system_status_doc["System Status"][olt_id]["ONUs"][onu_id] == "Unspecified" or \
                                    system_status_doc["System Status"][olt_id]["ONUs"][onu_id] == "Registered" or \
                                    system_status_doc["System Status"][olt_id]["ONUs"][onu_id] == "FW Upgrade":
                                is_offline = False

                # Check if the ONU is resetting if online
                if not is_offline:
                    if config_reset_count >= 0:
                        is_resetting = state_reset_count != config_reset_count

                    if is_resetting:
                        system_status = "Pending Reset"

                onu_state = get_nested_value(onu_data, ["ONU-STATE", 0])
                olt_state = get_nested_value(onu_data, ["OLT-STATE", 0])
                if olt_state and 'ONU States' in olt_state and 'Deregistered' in olt_state['ONU States']:
                    # The device is 'Preprovisioned' If the ONU has no ONU-STATE data, and the ONU State is 'Deregistered'
                    if onu_state is None and onu_id in olt_state['ONU States']['Deregistered']:
                        system_status = "Preprovisioned"

                # Get OLT Automation Enable
                onu_auto_disabled = True
                if "ONU-AUTO" in onu_data:
                    if len(onu_data["ONU-AUTO"]) > 0 and "Enable" in onu_data["ONU-AUTO"][0]:
                        onu_auto_disabled = not onu_data["ONU-AUTO"][0]["Enable"]

                if alarm_history_supported:
                    emerg_alarms = []
                    alert_alarms = []
                    crit_alarms = []
                    error_alarms = []
                    warning_alarms = []

                    if 'ONU-ALARM-HIST-STATE' in onu_data and len(onu_data['ONU-ALARM-HIST-STATE']) > 0 and \
                            onu_data['ONU-ALARM-HIST-STATE'][0]['Alarms']:
                        for alarm_history in onu_data['ONU-ALARM-HIST-STATE'][0]['Alarms']:
                            alarm_text = alarm_history['Text']
                            if alarm_history['Severity'] == "0-EMERG":
                                emerg_alarms.append(alarm_text)
                            elif alarm_history['Severity'] == "1-ALERT":
                                alert_alarms.append(alarm_text)
                            elif alarm_history['Severity'] == "2-CRIT":
                                crit_alarms.append(alarm_text)
                            elif alarm_history['Severity'] == "3-ERROR":
                                error_alarms.append(alarm_text)
                            elif alarm_history['Severity'] == "4-WARNING":
                                warning_alarms.append(alarm_text)

                    errors = {
                        "EMERG": emerg_alarms,
                        "ALERT": alert_alarms,
                        "CRIT": crit_alarms,
                        "ERROR": error_alarms
                    }
                    warnings = warning_alarms
                else:
                    errors = {
                        "EMERG": get_nested_value(onu_data, ["ONU-STATE", 0, "Alarm", "0-EMERG"], []),
                        "ALERT": get_nested_value(onu_data, ["ONU-STATE", 0, "Alarm", "1-ALERT"], []),
                        "CRIT": get_nested_value(onu_data, ["ONU-STATE", 0, "Alarm", "2-CRIT"], []),
                        "ERROR": get_nested_value(onu_data, ["ONU-STATE", 0, "Alarm", "3-ERROR"], [])
                    }
                    warnings = get_nested_value(onu_data, ["ONU-STATE", 0, "Alarm", "4-WARNING"], [])

                # Assign correct PON Standard
                olt_state = get_nested_value(onu_data, ["OLT-STATE", 0])
                pon_standard = ""
                if "NNI" in olt_state and "Supported Speeds" in olt_state["NNI"]:
                    supported_speeds = olt_state["NNI"]["Supported Speeds"]
                    if "25Gbps" in supported_speeds:
                        pon_standard = "25GS"
                    elif "10Gbps" in supported_speeds:
                        if "OLT" in olt_state and "PON Mode" in olt_state["OLT"]:
                            if olt_state["OLT"]["PON Mode"] == "EPON":
                                pon_standard = "10GE"
                            else:
                                pon_standard = "XGS"

                response[1].append({"_id": onu_id,
                                    "cntl": cntl_id,
                                    "name": name,
                                    "offline": is_offline,
                                    "auto_offline": auto_offline,
                                    "cntl_auto_disabled": cntl_auto_disabled,
                                    "olt_auto_disabled": olt_auto_disabled,
                                    "onu_auto_disabled": onu_auto_disabled,
                                    "state": system_status,
                                    "errors": errors,
                                    "warnings": warnings,
                                    "pon_standard": pon_standard})

            val_array = response[1]
            response[1] = sorted(val_array, key=lambda i: (i['name'].lower(), i['_id']))
        except Exception as err:
            self.status_log("EXCEPTION (get_onus_under_olt_for_tree)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get ONUs for OLT {olt_id} list due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "OLT-STATE, ONU-STATE, ONU-CFG",
                       "returned": response[1],
                       "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    def get_onu_configs_for_olt(self, user_email, olt_id):
        response = [status.HTTP_200_OK, []]

        try:
            database_manager.get_database(self.database_id)
            # Get all distinct document ids from collection OLT_STATE
            olt_state_coll = self.database["OLT-STATE"]
            olt = olt_state_coll.find_one({"_id": olt_id}, {"ONUs": 1})

            if olt is not None and "ONUs" in olt:
                onu_list = list(olt["ONUs"])
                if onu_list and len(onu_list) > 0:
                    onu_cfg_coll = self.database["ONU-CFG"]
                    response[1] = onu_cfg_coll.find({"_id": {"$in": onu_list}})
                    response[1] = sorted(response[1], key=lambda i: (i['ONU']['Name'].lower(), i['_id']))

        except Exception as err:
            self.status_log("EXCEPTION (get_onu_configs_for_olt)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get ONUs for OLT {olt_id} list due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "OLT-STATE, ONU-CFG",
                       "returned": response[1],
                       "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    def get_onu_states_for_olt(self, user_email, olt_id):
        response = [status.HTTP_200_OK, []]

        try:
            database_manager.get_database(self.database_id)
            # Get all distinct document ids from collection OLT_STATE
            olt_state_coll = self.database["OLT-STATE"]
            olt = olt_state_coll.find_one({"_id": olt_id}, {"ONUs": 1})
            onu_state_coll = self.database["ONU-STATE"]
            if olt is not None and "ONUs" in olt:
                for onu in olt["ONUs"]:
                    onu_state = onu_state_coll.find_one({"_id": onu})
                    if onu_state is not None:
                        response[1].append(onu_state)

            response[1] = sorted(response[1], key=lambda i: (i['_id']))
        except Exception as err:
            self.status_log("EXCEPTION (get_onu_configs_for_olt)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get ONUs for OLT {olt_id} list due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "OLT-STATE, ONU-STATE",
                       "returned": response[1],
                       "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    def search_tree_device(self, user_email, device_type, filter_str):
        response = [status.HTTP_200_OK, []]

        try:
            database_manager.get_database(self.database_id)
            if device_type == "Controller":
                collection = self.database.get_collection("CNTL-CFG")
                controllers = collection.find({"$or": [
                    {"_id": {"$regex": filter_str, "$options": 'i'}},
                    {"CNTL.Name": {"$regex": filter_str, "$options": 'i'}}
                ]},
                    {"_id": 1, "CNTL.Name": 1}
                )

                for controller in controllers:
                    response[1].append({"_id": controller["_id"], "name": controller["CNTL"]["Name"]})
            elif device_type == 'Switch':
                collection = self.database.get_collection("SWI-CFG")
                switches = collection.find({"$or": [
                    {"_id": {"$regex": filter_str, "$options": 'i'}},
                    {"SWI.Name": {"$regex": filter_str, "$options": 'i'}}
                ]},
                    {"_id": 1, "SWI.Name": 1}
                )

                for switch in switches:
                    response[1].append({"_id": switch["_id"], "name": switch["SWI"]["Name"]})
            elif device_type == 'OLT':
                collection = self.database.get_collection("OLT-CFG")
                olts = collection.find({"$or": [
                    {"_id": {"$regex": filter_str, "$options": 'i'}},
                    {"OLT.Name": {"$regex": filter_str, "$options": 'i'}}
                ]},
                    {"_id": 1, "OLT.Name": 1}
                )

                for olt in olts:
                    response[1].append({"_id": olt["_id"], "name": olt["OLT"]["Name"]})
            elif device_type == 'Port':
                collection = self.database.get_collection("OLT-STATE")
                olts = collection.find({"Switch.Port ID": {"$regex": filter_str, "$options": 'i'}},
                                       {"_id": 1, "Switch.Port ID": 1}
                                       )

                for olt in olts:
                    response[1].append({"_id": olt["_id"], "name": olt["Switch"]["Port ID"]})
            elif device_type == 'ONU':
                collection = self.database.get_collection("ONU-CFG")
                onus = collection.find({"$or": [
                    {"_id": {"$regex": filter_str, "$options": 'i'}},
                    {"ONU.Name": {"$regex": filter_str, "$options": 'i'}}
                ]},
                    {"_id": 1, "ONU.Name": 1}
                )

                for onu in onus:
                    response[1].append({"_id": onu["_id"], "name": onu["ONU"]["Name"]})
        except Exception as err:
            self.status_log("EXCEPTION (search_tree_device)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get filtered tree device list due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "OLT-STATE, ONU-STATE, SWI-CFG, CNTL-STATE",
                       "returned": response[1],
                       "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    """
    Get all states
    """

    # Retrieve an array of all active CNTL states
    def get_all_cntl_states(self, user_email):
        response = [status.HTTP_200_OK, []]

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["CNTL-STATE"]
            cntls_cursor = collection.find()
            for state in cntls_cursor:
                response[1].append(state)
        except Exception as err:
            self.status_log("EXCEPTION (get_all_cntl_states)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get PON Controller states due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "CNTL-STATE", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    # Retrieve an array of all active OLT states
    def get_all_olt_states(self, user_email):
        response = [status.HTTP_200_OK, []]

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["OLT-STATE"]
            olts_cursor = collection.find()
            for state in olts_cursor:
                response[1].append(state)
        except Exception as err:
            self.status_log("EXCEPTION (get_all_olt_states)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get OLT states due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "OLT-STATE", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    # Retrieve an array of all active ONU states
    def get_all_onu_states(self, user_email):
        response = [status.HTTP_200_OK, []]

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["ONU-STATE"]
            onus_cursor = collection.find()
            for state in onus_cursor:
                response[1].append(state)
        except Exception as err:
            self.status_log("EXCEPTION (get_all_onu_states)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get ONU states due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "ONU-STATE", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    """
    Get all alarms
    """

    # Retrieve an array of all controller alarms
    def get_all_cntl_alarms(self, user_email):
        response = [status.HTTP_200_OK, []]

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["CNTL-STATE"]
            response[1] = list(collection.aggregate([
                {
                    "$lookup": {
                        "from": "CNTL-ALARM-HIST-STATE",
                        "let": {"cntlId": "$_id"},
                        "pipeline": [
                            {"$match": {"$expr": {"$eq": ["$_id", "$$cntlId"]}}},
                            {
                                "$project": {
                                    "Alarms": {
                                        "$filter": {
                                            "input": "$Alarms",
                                            "cond": {"$and": [{"$eq": ["$$this.Active State", True]},
                                                              {"$eq": ["$$this.Ack State", False]}]}
                                        }
                                    }
                                }
                            }, {
                                '$unwind': {
                                    'path': '$Alarms',
                                    'preserveNullAndEmptyArrays': False
                                }
                            }, {
                                '$group': {
                                    '_id': '_id',
                                    'Alarms': {
                                        '$push': {
                                            'Text': '$Alarms.Text',
                                            'Severity': '$Alarms.Severity',
                                            'Source': '$Alarms.Source',
                                            'Object Type': '$Alarms.Object Type',
                                            'Alarm Type': '$Alarms.Alarm Type'
                                        }
                                    }
                                }
                            }
                        ],
                        "as": "alarmHistory"
                    }
                },
                {
                    "$addFields": {
                        "majorVersion": {
                            '$cond': {
                                'if': {
                                    '$cond': {
                                        'if': {
                                            '$ifNull': [
                                                '$CNTL.Version', False
                                            ]
                                        },
                                        'then': {
                                            '$gte': [
                                                {
                                                    '$strLenCP': '$CNTL.Version'
                                                }, 2
                                            ]
                                        },
                                        'else': False
                                    }
                                },
                                'then': {
                                    '$toInt': {
                                        '$substr': [
                                            '$CNTL.Version', 1, 1
                                        ]
                                    }
                                },
                                'else': 0
                            }
                        },
                        "CNTL-ALARM-HIST-STATE": {
                            "$arrayElemAt": ["$alarmHistory.Alarms", 0]
                        }
                    }
                },
                {
                    '$match': {
                        'majorVersion': {
                            '$ne': 0
                        }
                    }
                },
                {
                    "$lookup": {
                        "from": "CNTL-CFG",
                        "let": {"cntlId": "$_id"},
                        "pipeline": [
                            {"$match": {"$and": [{"$expr": {"$eq": ["$_id", "$$cntlId"]}},
                                                 {"$expr": {"$ne": ["$CNTL.Name", '']}}]}},
                            {
                                "$project": {
                                    "_id": 0,
                                    "Name": "$CNTL.Name",
                                }
                            }
                        ],
                        "as": "Name"
                    }
                },
                {
                    "$project": {
                        "Name": "$Name.Name",
                        "Time": 1,
                        "majorVersion": 1,
                        "Alarm": 1,
                        "CNTL-ALARM-HIST-STATE": 1
                    }
                },
                {
                    "$project": {
                        "Name": 1,
                        "CNTL.MAC Address": 1,
                        "OLT.MAC Address": 1,
                        "Time": 1,
                        "Alarm": {
                            "$cond": {
                                "if": {"$gte": ["$majorVersion", 3]},
                                "then": "$$REMOVE",
                                "else": "$Alarm"
                            }
                        },
                        "CNTL-ALARM-HIST-STATE": {
                            "$cond": {
                                "if": {"$gte": ["$majorVersion", 3]},
                                "then": "$CNTL-ALARM-HIST-STATE",
                                "else": "$$REMOVE"
                            }
                        }
                    }
                }
            ]))
        except Exception as err:
            self.status_log("EXCEPTION (get_all_cntl_alarms)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get all PON Controller alarms due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "CNTL-STATE", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    # Retrieve an array of all OLT alarms
    def get_all_olt_alarms(self, user_email):
        response = [status.HTTP_200_OK, []]

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["OLT-STATE"]
            response[1] = list(collection.aggregate([
                {
                    "$lookup": {
                        "from": "OLT-ALARM-HIST-STATE",
                        "let": {"oltId": "$_id"},
                        "pipeline": [
                            {"$match": {"$expr": {"$eq": ["$_id", "$$oltId"]}}},
                            {
                                "$project": {
                                    "Alarms": {
                                        "$filter": {
                                            "input": "$Alarms",
                                            "cond": {"$and": [{"$eq": ["$$this.Active State", True]},
                                                              {"$eq": ["$$this.Ack State", False]}]}
                                        }
                                    }
                                }
                            }, {
                                '$unwind': {
                                    'path': '$Alarms',
                                    'preserveNullAndEmptyArrays': False
                                }
                            }, {
                                '$group': {
                                    '_id': '_id',
                                    'Alarms': {
                                        '$push': {
                                            'Text': '$Alarms.Text',
                                            'Severity': '$Alarms.Severity',
                                            'Source': '$Alarms.Source',
                                            'Object Type': '$Alarms.Object Type',
                                            'Alarm Type': '$Alarms.Alarm Type'
                                        }
                                    }
                                }
                            }
                        ],
                        "as": "alarmHistory"
                    }
                },
                {
                    "$addFields": {
                        "majorVersion": {
                            '$cond': {
                                'if': {
                                    '$cond': {
                                        'if': {
                                            '$ifNull': [
                                                '$CNTL.Version', False
                                            ]
                                        },
                                        'then': {
                                            '$gte': [
                                                {
                                                    '$strLenCP': '$CNTL.Version'
                                                }, 2
                                            ]
                                        },
                                        'else': False
                                    }
                                },
                                'then': {
                                    '$toInt': {
                                        '$substr': [
                                            '$CNTL.Version', 1, 1
                                        ]
                                    }
                                },
                                'else': 0
                            }
                        },
                        "OLT-ALARM-HIST-STATE": {
                            "$arrayElemAt": ["$alarmHistory.Alarms", 0]
                        }
                    }
                },
                {
                    '$match': {
                        'majorVersion': {
                            '$ne': 0
                        }
                    }
                },
                {
                    "$lookup": {
                        "from": "OLT-CFG",
                        "let": {"oltId": "$_id"},
                        "pipeline": [
                            {"$match": {"$and": [{"$expr": {"$eq": ["$_id", "$$oltId"]}},
                                                 {"$expr": {"$ne": ["$OLT.Name", '']}}]}},
                            {
                                "$project": {
                                    "_id": 0,
                                    "Name": "$OLT.Name",
                                }
                            }
                        ],
                        "as": "Name"
                    }
                },
                {
                    "$project": {
                        "Name": "$Name.Name",
                        "CNTL.MAC Address": 1,
                        "OLT.MAC Address": 1,
                        "Time": 1,
                        "majorVersion": 1,
                        "Alarm": 1,
                        "OLT-ALARM-HIST-STATE": 1
                    }
                },
                {
                    "$project": {
                        "Name": 1,
                        "CNTL.MAC Address": 1,
                        "Time": 1,
                        "Alarm": {
                            "$cond": {
                                "if": {"$gte": ["$majorVersion", 3]},
                                "then": "$$REMOVE",
                                "else": "$Alarm"
                            }
                        },
                        "OLT-ALARM-HIST-STATE": {
                            "$cond": {
                                "if": {"$gte": ["$majorVersion", 3]},
                                "then": "$OLT-ALARM-HIST-STATE",
                                "else": "$$REMOVE"
                            }
                        }
                    }
                }
            ]))
        except Exception as err:
            self.status_log("EXCEPTION (get_all_olt_alarms)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get all OLT alarms due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "OLT-STATE", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    # Retrieve and array of all ONU alarms
    def get_all_onu_alarms(self, user_email, attribute):
        response = [status.HTTP_200_OK, []]

        try:
            database_manager.get_database(self.database_id)
            collection = self.database.get_collection("ONU-STATE")
            response[1] = list(collection.aggregate([
                {
                    '$lookup': {
                        'from': 'ONU-ALARM-HIST-STATE',
                        'let': {
                            'onuId': '$_id'
                        },
                        'pipeline': [
                            {
                                '$match': {
                                    '$expr': {
                                        '$eq': [
                                            '$_id', '$$onuId'
                                        ]
                                    }
                                }
                            }, {
                                '$project': {
                                    'Alarms': {
                                        '$filter': {
                                            'input': '$Alarms',
                                            'cond': {
                                                '$and': [
                                                    {
                                                        '$eq': [
                                                            '$$this.Active State', True
                                                        ]
                                                    }, {
                                                        '$eq': [
                                                            '$$this.Ack State', False
                                                        ]
                                                    }
                                                ]
                                            }
                                        }
                                    }
                                }
                            }, {
                                '$unwind': {
                                    'path': '$Alarms',
                                    'preserveNullAndEmptyArrays': False
                                }
                            }, {
                                '$group': {
                                    '_id': '_id',
                                    'Alarms': {
                                        '$push': {
                                            'Text': '$Alarms.Text',
                                            'Severity': '$Alarms.Severity',
                                            'Source': '$Alarms.Source',
                                            'Object Type': '$Alarms.Object Type',
                                            'Alarm Type': '$Alarms.Alarm Type'
                                        }
                                    }
                                }
                            }
                        ],
                        'as': 'alarmHistory'
                    }
                },
                {
                    '$addFields': {
                        'majorVersion': {
                            '$cond': {
                                'if': {
                                    '$cond': {
                                        'if': {
                                            '$ifNull': [
                                                '$CNTL.Version', False
                                            ]
                                        },
                                        'then': {
                                            '$gte': [
                                                {
                                                    '$strLenCP': '$CNTL.Version'
                                                }, 2
                                            ]
                                        },
                                        'else': False
                                    }
                                },
                                'then': {
                                    '$toInt': {
                                        '$substr': [
                                            '$CNTL.Version', 1, 1
                                        ]
                                    }
                                },
                                'else': 0
                            }
                        },
                        'ONU-ALARM-HIST-STATE': {
                            '$arrayElemAt': [
                                '$alarmHistory.Alarms', 0
                            ]
                        },
                    }
                },
                {
                    '$match': {
                        'majorVersion': {
                            '$ne': 0
                        }
                    }
                },
                {
                    "$lookup": {
                        "from": "ONU-CFG",
                        "let": {"onuId": "$_id"},
                        "pipeline": [
                            {"$match": { "$and": [{"$expr": {"$eq": ["$_id", "$$onuId"]}}, {"$expr": {"$ne": ["$ONU.Name", '']}}]}},
                            {
                                "$project": {
                                    "_id": 0,
                                    "Name": "$ONU.Name",
                                }
                            }
                        ],
                        "as": "Name"
                    }
                },
                {
                    '$project': {
                        'Name': "$Name.Name",
                        'CNTL.MAC Address': 1,
                        'OLT.MAC Address': 1,
                        'Time': 1,
                        'majorVersion': 1,
                        'Alarm': 1,
                        'ONU-ALARM-HIST-STATE': 1
                    }
                }, {
                    '$project': {
                        'Name': 1,
                        'CNTL.MAC Address': 1,
                        'OLT.MAC Address': 1,
                        'Time': 1,
                        'Alarm': {
                            '$cond': {
                                'if': {
                                    '$gte': [
                                        '$majorVersion', 3
                                    ]
                                },
                                'then': '$$REMOVE',
                                'else': '$Alarm'
                            }
                        },
                        'ONU-ALARM-HIST-STATE': {
                            '$cond': {
                                'if': {
                                    '$gte': [
                                        '$majorVersion', 3
                                    ]
                                },
                                'then': '$ONU-ALARM-HIST-STATE',
                                'else': '$$REMOVE'
                            }
                        }
                    }
                }
            ]))
        except Exception as err:
            self.status_log("EXCEPTION (get_all_onu_alarms)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get all ONU alarms due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "ONU-STATE", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    """
    Get all ONU states for a specified OLT
    """

    def get_all_onu_states_for_olt(self, olt_id, user_email):
        response = [status.HTTP_200_OK, []]

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["ONU-STATE"]
            query = {"OLT.MAC Address": olt_id}
            onu_states_cursor = collection.find(query)
            for state in onu_states_cursor:
                response[1].append(state)
        except Exception as err:
            self.status_log("EXCEPTION (get_all_onu_states_for_olt)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get ONU states for OLT {olt_id} due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "ONU-STATE", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    """
    Allowing ONU Registration
    """

    # Checking if ONU is disallowed on any OLT AND if there is a different Reg Allow ONU pending on the given OLT
    def get_onu_registration_allow(self, onu_id, user_email):
        response = [status.HTTP_200_OK, {"is_disallowed": False, "can_allow": True}]

        try:
            database_manager.get_database(self.database_id)
            # Retrieve all OLT states where the given ONU is disallowed
            olt_state_collection = self.database["OLT-STATE"]
            olt_states_cursor = olt_state_collection.find({"ONU States.Disallowed Error": onu_id},
                                                          {"_id": 1, "OLT.Reg Allow ONU": 1, "OLT.Reg Allow Count": 1})
            # If the ONU is disallowed in 1 or more OLTs, then continue
            if olt_states_cursor.collection.count_documents({}) > 0:
                response[1]['is_disallowed'] = True
                olt_config_collection = self.database["OLT-CFG"]

                # Check if the OLT the ONU is disallowed on is currently in the process of allowing an ONU already
                for olt in olt_states_cursor:
                    olt_cfg = olt_config_collection.find_one({"_id": olt["_id"]}, {"_id": 1, "OLT.Reg Allow ONU": 1,
                                                                                   "OLT.Reg Allow Count": 1})
                    if olt_cfg is not None:
                        if not (olt_cfg["OLT"]["Reg Allow ONU"] == olt["OLT"]["Reg Allow ONU"] and olt_cfg["OLT"][
                            "Reg Allow Count"] == olt["OLT"]["Reg Allow Count"]):
                            response[1]['can_allow'] = False

        except Exception as err:
            self.status_log("EXCEPTION (get_onu_registration_allow)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get OLTs {onu_id} is disallowed on due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "OLT-STATE", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    # Checking if ONU is disallowed on any OLT AND if there is a different Reg Allow ONU pending on the given OLT
    def put_onu_registration_allow(self, onu_id, user_email):
        response = [status.HTTP_200_OK, [], None, {}]

        try:
            database_manager.get_database(self.database_id)
            # Retrieve all OLT states where the given ONU is disallowed
            olt_state_collection = self.database["OLT-STATE"]
            olt_states_cursor = olt_state_collection.find({"ONU States.Disallowed Error": onu_id},
                                                          {"_id": 1, "OLT.Reg Allow ONU": 1, "OLT.Reg Allow Count": 1})
            # If the ONU is disallowed in 1 or more OLTs, then continue
            if olt_states_cursor.collection.count_documents({}) > 0:
                olt_config_collection = self.database["OLT-CFG"]

                new_doc = {}
                # Update OLT config reg allow ONU with onu mac and increment Reg Allow count, if there is a config file to update
                for olt in olt_states_cursor:
                    update_document = {"$set": {"OLT.Reg Allow ONU": onu_id},
                                       "$inc": {"OLT.Reg Allow Count": 1, "OLT.CFG Change Count": 1}}
                    olt_config_collection.update_one({"_id": olt["_id"]}, update_document)
                    new_doc[olt["_id"]] = update_document
                response[2] = new_doc
        except Exception as err:
            self.status_log("EXCEPTION (put_onu_registration_allow)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not clear OLTs {onu_id} is disallowed on due to {type(err).__name__}::{sys.exc_info()[1]}",
                        None, None]

        action_data = {"collection": "OLT-CONFIG", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    """
    Get state of the specified device
    """

    # Retrieve the state of the specified CNTL
    def get_cntl_state(self, controller_id, user_email):
        response = [status.HTTP_200_OK, []]

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["CNTL-STATE"]
            response[1] = collection.find_one({"_id": controller_id})
        except Exception as err:
            self.status_log("EXCEPTION (get_cntl_state)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get PON Controller {controller_id} state due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "CNTL-STATE", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    # Retrieve the auth state of the specified CNTL
    def get_cntl_auth_state(self, controller_id, user_email):
        response = [status.HTTP_200_OK, []]

        try:
            database_manager.get_database(self.database_id)
            if "CNTL-AUTH-STATE" in self.database.list_collection_names():
                collection = self.database["CNTL-AUTH-STATE"]
                response[1] = collection.find_one({"_id": controller_id})
            else:
                response[1] = None
        except Exception as err:
            self.status_log("EXCEPTION (get_cntl_auth_state)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get PON Controller {controller_id} authentication state due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "CNTL-AUTH-STATE", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    # Retrieve the state of the specified Switch
    def get_switch_state(self, switch_id, user_email):
        response = [status.HTTP_200_OK, []]

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["OLT-STATE"]
            cursor = collection.find({"Switch.Chassis ID": switch_id})
            if cursor.count() > 0:
                response[1] = cursor.sort("Time", pymongo.DESCENDING)[0]
        except Exception as err:
            self.status_log("EXCEPTION (get_switch_state)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get Switch {switch_id} state due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "OLT-STATE", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    # Retrieve the state of the specified OLT
    def get_olt_state(self, olt_id, user_email):
        response = [status.HTTP_200_OK, []]

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["OLT-STATE"]
            response[1] = collection.find_one({"_id": olt_id})
        except Exception as err:
            self.status_log("EXCEPTION (get_olt_state)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get OLT {olt_id} state due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "OLT-STATE", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    # Retrieve the state of the specified ONU
    def get_onu_state(self, onu_id, user_email):
        response = [status.HTTP_200_OK, []]

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["ONU-STATE"]
            response[1] = collection.find_one({"_id": onu_id})
        except Exception as err:
            self.status_log("EXCEPTION (get_onu_state)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get ONU {onu_id} state due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "ONU-STATE", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response


    """
    Get all EPON ONU
    """
    def get_epon_onus(self, user_email):
        response = [status.HTTP_200_OK, []]

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["ONU-STATE"]

            response[1] = list(collection.find({"ONU.PON Mode": {"$eq": "EPON"}}))

        except Exception as err:
            self.status_log("EXCEPTION (get_selected_onu_states)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get EPON ONUs due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "ONU-STATE", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response


    """
    Get all configs
    """

    # Retrieve all controller configs
    def get_all_cntl_cfgs(self, user_email):
        response = [status.HTTP_200_OK, []]

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["CNTL-CFG"]
            query = {"_id": {'$regex': '^((?!Default).)*$'}}
            cntl_cfgs_cursor = collection.find(query)
            for cfg in cntl_cfgs_cursor:
                response[1].append(cfg)
        except Exception as err:
            self.status_log("EXCEPTION (get_all_cntl_cfgs)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get all PON Controller configurations due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "CNTL-CFG", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    # Retrieve all olt configs
    def get_all_olt_cfgs(self, user_email):
        response = [status.HTTP_200_OK, []]

        try:
            database_manager.get_database(self.database_id)
            query = {"_id": {'$regex': '^((?!Default).)*$'}}
            collection = self.database["OLT-CFG"]
            olt_cfgs_cursor = collection.find(query)
            for cfg in olt_cfgs_cursor:
                response[1].append(cfg)
        except Exception as err:
            self.status_log("EXCEPTION (get_all_olt_cfgs)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get all OLT configurations due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "OLT-CFG", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    # Retrieve all onu configs
    def get_all_onu_cfgs(self, user_email):
        response = [status.HTTP_200_OK, []]

        try:
            database_manager.get_database(self.database_id)
            query = {"_id": {'$regex': '^((?!Default).)*$'}}
            collection = self.database["ONU-CFG"]
            onu_cfgs_cursor = collection.find(query)
            for cfg in onu_cfgs_cursor:
                response[1].append(cfg)
        except Exception as err:
            self.status_log("EXCEPTION (get_all_onu_cfgs)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get all ONU configurations due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "ONU-CFG", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    """
    Get output logs for the specified device
    """

    # Retrieve the logs of the specified CNTL since the specified time
    def get_cntl_log(self, controller_id, since_utc_time, user_email):
        response = [status.HTTP_200_OK, []]

        try:
            database_manager.get_database(self.database_id)
            # Get logs from collection CNTL-LOG
            collection_name = "SYSLOG-CNTL-{}".format(controller_id.replace(":", ""))
            collection = self.database[collection_name]
            cntl_log_cursor = collection.find({"time": {"$gt": since_utc_time}, "device ID": controller_id},
                                              {"_id": 0, "device ID": 0}).limit(10000)
            for log in cntl_log_cursor:
                response[1].append(log)
        except Exception as err:
            self.status_log("EXCEPTION (get_cntl_log)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get PON Controller {controller_id} logs due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "SYSLOG-CNTL-{}".format(controller_id.replace(":", "")), "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    # Retrieve the logs of the specified OLT since the specified time
    def get_olt_log(self, olt_id, since_utc_time, user_email):
        response = [status.HTTP_200_OK, []]

        try:
            database_manager.get_database(self.database_id)
            # Get logs from collection OLT-LOG
            collection_name = "SYSLOG-OLT-{}".format(olt_id.replace(":", ""))
            collection = self.database[collection_name]
            olt_log_cursor = collection.find({"time": {"$gt": since_utc_time}, "device ID": olt_id},
                                             {"_id": 0, "device ID": 0}).limit(10000)
            for log in olt_log_cursor:
                response[1].append(log)
        except Exception as err:
            self.status_log("EXCEPTION (get_olt_log)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get OLT {olt_id} logs due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "SYSLOG-OLT-{}".format(olt_id.replace(":", "")), "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    # Retrieve the logs of the specified ONU since the specified time
    def get_onu_log(self, onu_id, since_utc_time, user_email):
        response = [status.HTTP_200_OK, []]

        try:
            database_manager.get_database(self.database_id)
            # Get logs from collection ONU-LOG
            collection_name = "SYSLOG-ONU-{}".format(onu_id.replace(":", ""))
            collection = self.database[collection_name]
            onu_log_cursor = collection.find({"time": {"$gt": since_utc_time}, "device ID": onu_id},
                                             {"_id": 0, "device ID": 0}).limit(10000)
            for log in onu_log_cursor:
                response[1].append(log)
        except Exception as err:
            self.status_log("EXCEPTION (get_onu_log)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get ONU {onu_id} logs due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "SYSLOG-ONU-{}".format(onu_id.replace(":", "")), "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    """
    Get output stats for the specified device
    """

    # Retrieve the stats of the specified CNTL
    def get_cntl_stats(self, controller_id, user_email, since_utc_time, to_utc_time=None):
        response = [status.HTTP_200_OK, []]

        try:
            database_manager.get_database(self.database_id)
            # Get logs from collection CNTL_<mac_address>_STATS
            collection_name = "STATS-CNTL-{}".format(controller_id.replace(":", ""))
            collection = self.database[collection_name]
            if to_utc_time:
                cntl_stats_cursor = collection.find({"_id": {"$gte": since_utc_time, "$lte": to_utc_time}}).limit(
                    10000)
            else:
                cntl_stats_cursor = collection.find({"_id": {"$gt": since_utc_time}}).limit(10000)
            for stats in cntl_stats_cursor:
                stats["mac_address"] = controller_id
                response[1].append(stats)
        except Exception as err:
            self.status_log("EXCEPTION (get_cntl_stats)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get PON Controller {controller_id} statistics due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "STATS-CNTL-{}".format(controller_id.replace(":", "")), "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    # Retrieve the stats of the specified OLT
    def get_olt_stats(self, olt_id, user_email, since_utc_time, to_utc_time=None):
        response = [status.HTTP_200_OK, []]

        try:
            database_manager.get_database(self.database_id)
            # Get logs from collection OLT_<mac_address>_STATS
            collection_name = "STATS-OLT-{}".format(olt_id.replace(":", ""))
            collection = self.database[collection_name]
            if to_utc_time:
                olt_stats_cursor = collection.find({"_id": {"$gte": since_utc_time, "$lte": to_utc_time}}).limit(10000)
            else:
                olt_stats_cursor = collection.find({"_id": {"$gt": since_utc_time}}).limit(10000)
            for stats in olt_stats_cursor:
                stats["mac_address"] = olt_id
                response[1].append(stats)
        except Exception as err:
            self.status_log("EXCEPTION (get_olt_stats)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get OLT {olt_id} statistics due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "STATS-OLT-{}".format(olt_id.replace(":", "")), "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    # Retrieve the stats of the specified ONU
    def get_onu_stats(self, onu_id, user_email, since_utc_time, to_utc_time=None):
        response = [status.HTTP_200_OK, []]

        try:
            database_manager.get_database(self.database_id)
            # Get stats from collection ONU_<mac_address>_STATS
            collection_name = "STATS-ONU-{}".format(onu_id.replace(":", ""))
            collection = self.database[collection_name]
            if to_utc_time:
                onu_stats_cursor = collection.find({"_id": {"$gte": since_utc_time, "$lte": to_utc_time}}).limit(10000)
            else:
                onu_stats_cursor = collection.find({"_id": {"$gt": since_utc_time}}).limit(10000)
            for stats in onu_stats_cursor:
                stats["mac_address"] = onu_id
                response[1].append(stats)
        except Exception as err:
            self.status_log("EXCEPTION (get_onu_stats)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get ONU {onu_id} statistics due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "STATS-ONU-{}".format(onu_id.replace(":", "")), "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    def get_many_onu_stats(self, onus, user_email, since_utc_time, to_utc_time=None):
        response = [status.HTTP_200_OK, []]

        try:
            database_manager.get_database(self.database_id)

            # Angular httpclient passes arrays back as strings so this puts them back into a list
            onu_ids = onus.split(",")

            collection = self.database["STATS-ONU"]
            if to_utc_time:
                response[1] = list(collection.find({
                    "$and": [
                        {"device ID": {"$in": onu_ids}},
                        {"valid": True},
                        {"_id": {"$gte": since_utc_time, "$lte": to_utc_time}},
                    ]
                }).limit(10000))
            else:
                response[1] = list(collection.find({
                    "$and": [
                        {"device ID": {"$in": onu_ids}},
                        {"valid": True},
                        {"_id": {"$gte": since_utc_time}},
                    ]
                }).limit(10000))

        except Exception as err:
            self.status_log("EXCEPTION (get_many_onu_stats)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get ONU {onu_id} statistics due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "STATS-ONU", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    def get_many_distinct_onu_stats(self, onus, stats, user_email, since_utc_time, to_utc_time=None):
        response = [status.HTTP_200_OK, []]

        try:
            database_manager.get_database(self.database_id)

            # Angular httpclient passes arrays back as strings so this puts them back into a list
            onu_ids = onus.split(",")
            selected_stats = stats.split(",")
            stat_dict = {
                "Time": 1,
                "device ID": 1
            }
            for stat in selected_stats:
                stat_dict.update({stat: 1})

            collection = self.database["STATS-ONU"]
            if to_utc_time:
                response[1] = list(collection.aggregate([
                    {
                        "$match": {
                            "$and": [
                                {"device ID": {"$in": onu_ids}},
                                {"valid": True},
                                {"_id": {"$gte": since_utc_time, "$lte": to_utc_time}},
                            ]
                        }
                    },
                    {"$project": stat_dict}
                ]))
            else:
                response[1] = list(collection.aggregate([
                    {
                        "$match": {
                            "$and": [
                                {"device ID": {"$in": onu_ids}},
                                {"valid": True},
                                {"_id": {"$gte": since_utc_time}},
                            ]
                        }
                    },
                    {"$project": stat_dict}
                ]))

        except Exception as err:
            self.status_log("EXCEPTION (get_many_distinct_onu_stats)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get ONU {onus} statistics due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "STATS-ONU", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    def get_onu_stats_totals(self, onus, user_email, since_utc_time, to_utc_time=None):
        response = [status.HTTP_200_OK, []]

        try:
            database_manager.get_database(self.database_id)
            statDocs = []
            allGroups = {}
            ommitSum = ['asic', 'laser', 'xcvr', 'rx optical level', 'rx optical level idle', 'tx optical level', 'equalization delay',
                        'fiber distance', 'one way delay', 'interval_end_time'];

            # Angular httpclient passes arrays back as strings so this puts them back into a list
            onu_ids = onus.split(",")

            collection = self.database["STATS-ONU"]
            if to_utc_time:
                statDocs = list(collection.find({
                    "$and": [
                        {"device ID": {"$in": onu_ids}},
                        {"valid": True},
                        {"_id": {"$gte": since_utc_time, "$lte": to_utc_time}},
                    ]
                }).limit(10000))
            else:
                statDocs = list(collection.find({
                    "$and": [
                        {"device ID": {"$in": onu_ids}},
                        {"valid": True},
                        {"_id": {"$gte": since_utc_time}},
                    ]
                }).limit(10000))

            ranges = {}
            # Calculate the totals for each stat
            for entry in statDocs:
                groups = entry.keys()
                allGroups['Time'] = entry['_id']
                for group in groups:
                    if group != 'Automation' and type(entry[group]) is dict and len(entry[group].keys()) > 0:
                        if group not in allGroups:
                            allGroups[group] = {}
                        for k, v in entry[group].items():
                            if k != "control_block" and k != "Learned Addresses":
                                if k in allGroups[group]:
                                    # If the stat is a gauge, we find the range and display it as a string
                                    if k.lower() in ommitSum:
                                        # populate ranges dictionary
                                        if group not in ranges:
                                            ranges[group] = {}
                                        if k.lower() not in ranges[group]:
                                            ranges[group][k.lower()] = {}
                                        # find min and max values for each group-key pairing
                                        range_keys = ranges[group][k.lower()].keys()
                                        if "min" not in range_keys or v < ranges[group][k.lower()]["min"]:
                                            ranges[group][k.lower()]["min"] = v
                                        if "max" not in range_keys or v > ranges[group][k.lower()]["max"]:
                                            ranges[group][k.lower()]["max"] = v
                                        allGroups[group][k] = f'[{ranges[group][k.lower()]["min"]}...' \
                                                              f'{ranges[group][k.lower()]["max"]}]'
                                    else:
                                        value = allGroups[group][k]
                                        value += v
                                        allGroups[group][k] = value
                                else:
                                    allGroups[group].update({k: v})

            response[1] = allGroups

        except Exception as err:
            self.status_log("EXCEPTION (get_onu_stats_totals)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get ONU {onu_id} statistics due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "STATS-ONU", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    """
    Get all output stats
    """

    # Retrieve all olt stats
    def get_all_olt_stats(self, request, since_utc_time):
        response = [status.HTTP_200_OK, []]
        user_email = request.user.email
        olt_macs = self.get_all_olts(request)[1]

        try:
            database_manager.get_database(self.database_id)
            for mac in olt_macs:
                olt_id = mac["_id"]
                response[1][olt_id] = []
                collection_name = "STATS-OLT-{}".format(olt_id.replace(":", ""))
                collection = self.database[collection_name]
                olt_stats_cursor = collection.find({"_id": {"$gt": since_utc_time}}).limit(10000)
                for document in olt_stats_cursor:
                    response[1][olt_id].append(document)
        except Exception as err:
            self.status_log("EXCEPTION (get_all_olt_stats)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get all OLT statistics due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "STATS-OLT-<ALL>", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    # Retrieve all onu stats
    def get_all_onu_stats(self, request, since_utc_time):
        response = [status.HTTP_200_OK, []]
        user_email = request.user.email
        onu_macs = self.get_all_onus(request)[1]

        try:
            database_manager.get_database(self.database_id)
            for mac in onu_macs:
                response[1][mac] = []
                collection_name = "STATS-ONU-{}".format(mac.replace(":", ""))
                collection = self.database[collection_name]
                onu_stats_cursor = collection.find({"_id": {"$gt": since_utc_time}}).limit(10000)
                for document in onu_stats_cursor:
                    response[1][mac].append(document)
        except Exception as err:
            self.status_log("EXCEPTION (get_all_onu_stats)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get all ONU statistics due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "STATS-ONU-<ALL>", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    """
    Get all alarm configs
    """

    # Retrieve all controller alarm configs
    def get_all_cntl_alarms_cfgs(self, user_email):
        response = [status.HTTP_200_OK, []]

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["CNTL-ALARM-CFG"]
            cntl_alarms_cfgs_cursor = collection.find()
            for cfg in cntl_alarms_cfgs_cursor:
                response[1].append(cfg)
        except Exception as err:
            self.status_log("EXCEPTION (get_all_cntl_alarms_cfgs)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get all PON Controller alarm configurations due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "CNTL-ALARM-CFG", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    # Retrieve all controller alarm config identifiers
    def get_all_cntl_alarms_ids(self, user_email):
        response = [status.HTTP_200_OK, []]

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["CNTL-ALARM-CFG"]
            cntl_alarms_cfgs_cursor = collection.find({}, {"_id": 1})
            for cfg in cntl_alarms_cfgs_cursor:
                response[1].append(cfg['_id'])
        except Exception as err:
            self.status_log("EXCEPTION (get_all_cntl_alarms_cfgs)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get all PON Controller alarm configurations due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "CNTL-ALARM-CFG", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    # Retrieve all olt alarm configs
    def get_all_olt_alarms_cfgs(self, user_email):
        response = [status.HTTP_200_OK, []]

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["OLT-ALARM-CFG"]
            olt_alarms_cfgs_cursor = collection.find()
            for cfg in olt_alarms_cfgs_cursor:
                response[1].append(cfg)
        except Exception as err:
            self.status_log("EXCEPTION (get_all_olt_alarms_cfgs)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get all OLT alarm configurations due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "OLT-ALARM-CFG", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    # Retrieve all OLT alarm config identifiers
    def get_all_olt_alarms_ids(self, user_email):
        response = [status.HTTP_200_OK, []]

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["OLT-ALARM-CFG"]
            olt_alarms_cfgs_cursor = collection.find({}, {"_id": 1})
            for cfg in olt_alarms_cfgs_cursor:
                response[1].append(cfg['_id'])
        except Exception as err:
            self.status_log("EXCEPTION (get_all_olt_alarms_cfgs)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get all OLT alarm configurations due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "OLT-ALARM-CFG", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    # Retrieve all onu alarm configs
    def get_all_onu_alarms_cfgs(self, user_email):
        response = [status.HTTP_200_OK, []]

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["ONU-ALARM-CFG"]
            onu_alarms_cfgs_cursor = collection.find()
            for cfg in onu_alarms_cfgs_cursor:
                response[1].append(cfg)
        except Exception as err:
            self.status_log("EXCEPTION (get_all_onu_alarms_cfgs)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get all ONU alarm configurations due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "ONU-ALARM-CFG", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    # Retrieve all ONU alarm config identifiers
    def get_all_onu_alarms_ids(self, user_email):
        response = [status.HTTP_200_OK, []]

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["ONU-ALARM-CFG"]
            onu_alarms_cfgs_cursor = collection.find({}, {"_id": 1})
            for cfg in onu_alarms_cfgs_cursor:
                response[1].append(cfg['_id'])
        except Exception as err:
            self.status_log("EXCEPTION (get_all_onu_alarms_cfgs)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get all ONU alarm configurations due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "ONU-ALARM-CFG", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    # Update all Management LAN Names
    def update_mgmt_lan_names(self, cntl_id, new_name, user_email):
        old = ""
        response = [status.HTTP_200_OK, f"PON Controller Management LAN names were updated."]

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["CNTL-STATE"]
            doc = collection.find_one({"_id": cntl_id})
            # doc will be None if this is a pre-provisioned device
            if doc is not None:
                old = doc["MGMT LAN"]["Name"]
                devices = doc["MGMT LAN Devices"]
                collection = self.database["CNTL-CFG"]
                for device in devices:
                    if devices[device]["Device Type"] == "Tibit ME":
                        collection.update_one({'_id': device}, {"$set": {"MGMT LAN.Name": new_name},
                                                                "$inc": {"CNTL.CFG Change Count": 1}}, upsert=True)
            else:
                response = [status.HTTP_200_OK,
                            f"No STATE document for this device. Management LAN names cannot be updated."]
        except Exception as err:
            self.status_log("EXCEPTION (update_mgmt_lan_names)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not update PON Controller Management LAN names due to {type(err).__name__}::{sys.exc_info()[1]}"]
            old = "ERROR"

        action_data = {"collection": "CNTL-STATE, CNTL-CFG",
                       "old": old, "new": new_name, "user": user_email}
        self.sys_log(action_data, "write", response[0] == status.HTTP_200_OK)

        return response

    """
    Sla configs
    """

    # Retrieve all sla config identifiers
    def get_all_sla_ids(self, user_email):
        response = [status.HTTP_200_OK, []]

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["SLA-CFG"]
            sla_cfgs_cursor = collection.find({}, {"_id": 1})
            for cfg in sla_cfgs_cursor:
                response[1].append(cfg['_id'])
        except Exception as err:
            self.status_log("EXCEPTION (get_all_sla_ids)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get all sla configuration ids due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "SLA-CFG", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    # Retrieve all service config identifiers
    def get_all_service_ids(self, user_email):
        response = [status.HTTP_200_OK, []]

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["SRV-CFG"]
            srv_cfgs_cursor = collection.find({}, {"_id": 1})
            for cfg in srv_cfgs_cursor:
                response[1].append(cfg['_id'])
        except Exception as err:
            self.status_log("EXCEPTION (get_all_service_ids)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get all service configuration ids due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "SRV-CFG", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    # Retrieve all service config identifiers
    def get_all_downstream_map_ids(self, user_email):
        response = [status.HTTP_200_OK, []]

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["DS-MAP-CFG"]
            ds_map_cfgs_cursor = collection.find({}, {"_id": 1})
            for cfg in ds_map_cfgs_cursor:
                response[1].append(cfg['_id'])
        except Exception as err:
            self.status_log("EXCEPTION (get_all_downstream_map_ids)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get all Downstream map configuration ids due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "DS-MAP-CFG", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    # Retrieve all OMCI service config identifiers
    def get_all_omci_service_ids(self, user_email):
        response = [status.HTTP_200_OK, []]

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["SRV-CFG"]
            srv_cfgs_cursor = collection.find({"OMCI": {'$exists': True}}, {"_id": 1})
            for cfg in srv_cfgs_cursor:
                response[1].append(cfg['_id'])
        except Exception as err:
            self.status_log("EXCEPTION (get_all_service_ids)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get all omci service configuration ids due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "SRV-CFG", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    # Save Logos to file system under /app/images/ by overwriting old filename
    # Also used to write to footer-text.txt for footer info.
    def upload_logo_image(self, data_dict, user_email):
        response = [status.HTTP_200_OK, f"Picture {data_dict['filename']} was uploaded.",
                    data_dict['filename'], None]

        try:
            database_manager.get_database(self.database_id)
            file_data = json.dumps(data_dict['val'])
            if data_dict['filename'] == "footer-text.txt":
                try:
                    text_file = open(self.production_footer_text + "footer-text.txt", "w")
                    text_file.write(file_data.strip('""'))
                    text_file.close()
                except FileNotFoundError:
                    text_file = open(self.development_footer_text + "footer-text.txt", "w")
                    text_file.write(file_data.strip('""'))
                    text_file.close()
            else:
                try:
                    with open(self.production_logo_directory + data_dict['filename'], 'wb') as file:
                        file.write(base64.b64decode(file_data.split(",")[1]))
                        file.close()
                except FileNotFoundError:
                    with open(self.development_logo_directory + data_dict['filename'], 'wb') as file:
                        file.write(base64.b64decode(file_data.split(",")[1]))
                        file.close()

        except Exception as err:
            self.status_log("EXCEPTION (upload_logo_image)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not upload picture {data_dict['filename']} due to {type(err).__name__}::{sys.exc_info()[1]}",
                        None, None]

        action_data = {"collection": "NONE", "old": data_dict['filename'], "new": data_dict,
                       "user": user_email}
        self.sys_log(action_data, "write", response[0] == status.HTTP_200_OK)

        return response

    """
    Download files
    """

    # Put an image into the database
    def download_image(self, filename, user_email):
        response = [status.HTTP_200_OK, ""]

        try:
            database_manager.get_database(self.database_id)
            fs = gridfs.GridFSBucket(db=self.database, bucket_name="PICTURES")
            down = fs.find({"_id": os.path.splitext(filename)[0]})
            if down.count() > 0:
                for file in down:
                    response[1] = base64.b64encode(file.read())
            down.close()
        except Exception as err:
            self.status_log("EXCEPTION (download_image)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not download picture {filename} due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "PICTURES", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    def download_olt_debug_state(self, olt_id, user_email):
        """Returns a JSON file of the specified OLTs document from the the OLT-DEBUG-STATE collection.
        :param olt_id The ID of the OLT to get the debug state for
        :param user_email The email of the user requesting the OLT debug state"""
        response = [status.HTTP_200_OK, ""]
        collection_name = "OLT-DEBUG-STATE"

        try:
            database_manager.get_database(self.database_id)
            if collection_name in self.database.list_collection_names():
                collection = self.database[collection_name]
                olt_debug_state = collection.find_one({"_id": olt_id})
                if olt_debug_state is not None:
                    response[1] = olt_debug_state
                else:
                    response = [status.HTTP_404_NOT_FOUND,
                                f"The OLT {olt_id} does not have an {collection_name} entry. Click 'Generate' to create one."]
            else:
                response = [status.HTTP_404_NOT_FOUND,
                            f"The {collection_name} collection does not exist in the database. Click 'Generate' to create it."]
        except Exception as err:
            self.status_log("EXCEPTION (download_olt_debug_state)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not download OLT {olt_id} Debug State due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": collection_name, "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    # Delete the specified Picture file
    def delete_picture(self, filename, user_email):
        response = [status.HTTP_200_OK, f"Picture {filename} was deleted."]

        try:
            database_manager.get_database(self.database_id)
            file_id = os.path.splitext(filename)[0]
            fs = gridfs.GridFSBucket(db=self.database, bucket_name="PICTURES")
            metadata = self.database["PICTURES.files"].find_one({"_id": file_id}, {"_id": 0, "metadata.Device Type": 1})
            fs.delete(file_id)

            # Remove from defaults
            if metadata["metadata"]["Device Type"] == "CNTL":
                relative_path = "pictures/controllers/"
            elif metadata["metadata"]["Device Type"] == "SWI":
                relative_path = "pictures/switches/"
            elif metadata["metadata"]["Device Type"] == "OLT":
                relative_path = "pictures/olts/"
            else:
                relative_path = "pictures/onus/"

            if os.path.exists(self.database_seed_base_dir + relative_path + filename):
                os.remove(self.database_seed_base_dir + relative_path + filename)

            # Remove from metadata
            with open(self.database_seed_base_dir + "metadata.json", 'r') as file:
                json_data = json.load(file)
                if relative_path + filename in json_data:
                    json_data.pop(relative_path + filename)
            with open(self.database_seed_base_dir + "metadata.json", 'w') as file:
                file.write(json.dumps(json_data, sort_keys=True))
        except Exception as err:
            self.status_log("EXCEPTION (delete_picture)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not delete Picture {filename} due to {type(err).__name__}:{sys.exc_info()[1]}"]

        action_data = {"collection": "PICTURES",
                       "deleted": response[0] == status.HTTP_200_OK, "user": user_email}
        self.sys_log(action_data, "delete", response[0] == status.HTTP_200_OK)

        return response

    """
    Modify File Metadata
    """

    def update_image(self, file_id, metadata, user_email):
        response = [status.HTTP_200_OK, status.HTTP_200_OK]

        try:
            database_manager.get_database(self.database_id)
            old = ""
            collection = self.database['PICTURES.files']
            file = collection.find_one({'_id': file_id})
            if file is not None:
                old = file['metadata']

            collection.update_one({'_id': file_id}, {"$set": {"metadata": metadata}})
        except Exception as err:
            self.status_log("EXCEPTION (update_image)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not update Picture {file_id} due to {type(err).__name__}:{sys.exc_info()[1]}"]

        action_data = {"collection": "PICTURES", "old": old, "new": metadata, "user": user_email}
        self.sys_log(action_data, "write", response[0] == status.HTTP_200_OK)

        return response

    """
    Get Firmware/Picture Information
    """

    # Retrieves information about the OLT firmware files
    def get_all_olt_firmwares_info(self, user_email):
        response = [status.HTTP_200_OK, []]

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["OLT-FIRMWARE.files"]
            firmwares = collection.find({}).sort('_id')
            for file in firmwares:
                response[1].append(file)
        except Exception as err:
            self.status_log("EXCEPTION (get_all_olt_firmwares_info)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get all OLT firmware information due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "OLT-FIRMWARE.files", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    # Retrieves information about the OLT firmware files
    def get_all_onu_firmwares_info(self, user_email):
        response = [status.HTTP_200_OK, []]

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["ONU-FIRMWARE.files"]
            firmwares = collection.find({})
            for file in firmwares:
                response[1].append(file)
        except Exception as err:
            self.status_log("EXCEPTION (get_all_onu_firmwares_info)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get all ONU firmware information due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "ONU-FIRMWARE.files", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    # Retrieves the list of firmwares and its versions assigned to each ONU
    def get_first_assigned_onu_firmware(self, user_email, firmware, limit=1):
        response = [status.HTTP_200_OK, []]

        try:
            database_manager.get_database(self.database_id)
            limit = int(limit)
            if limit < 0:
                raise ValueError
            collection = self.database["ONU-CFG"]
            firmware_info = collection.find(
                {"ONU.FW Bank Files": {"$in": [firmware]}},
                {"_id": 1, "ONU.Name": 1},
            ).limit(limit)
            for fi in firmware_info:
                response[1].append(fi)
        except Exception as err:
            self.status_log("EXCEPTION (get_all_onu_firmwares)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get all ONU firmware due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "ONU-CFG", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    # Retrieves the list of firmwares and its versions assigned to each ONU
    def get_first_assigned_onu_sla(self, user_email, sla):
        response = [status.HTTP_200_OK, []]

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["ONU-CFG"]
            sla_info = collection.find(
                {"$or": [{"OLT-Service 0.SLA-CFG": sla}, {"OLT-Service 1.SLA-CFG": sla}, {"OLT-Service 2.SLA-CFG": sla},
                         {"OLT-Service 3.SLA-CFG": sla}, {"OLT-Service 4.SLA-CFG": sla}, {"OLT-Service 5.SLA-CFG": sla},
                         {"OLT-Service 6.SLA-CFG": sla}, {"OLT-Service 7.SLA-CFG": sla}]},
                {"_id": 1}
            ).limit(1)
            for fi in sla_info:
                response[1].append(fi)
        except Exception as err:
            self.status_log("EXCEPTION (get_all_assigned_onu_slas)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get all ONU SLAs due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "ONU-CFG", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    # Retrieves the list of firmwares and its versions assigned to each ONU
    def get_first_assigned_olt_firmware(self, user_email, firmware, limit=1):
        response = [status.HTTP_200_OK, []]

        try:
            database_manager.get_database(self.database_id)
            limit = int(limit)
            if limit < 0:
                raise ValueError
            collection = self.database["OLT-CFG"]
            firmware_info = collection.find(
                {"OLT.FW Bank Files": {"$in": [firmware]}},
                {"_id": 1, "OLT.Name": 1},
            ).limit(limit)
            for fi in firmware_info:
                response[1].append(fi)
        except Exception as err:
            self.status_log("EXCEPTION (get_all_olt_firmwares)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get all OLT firmware due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "OLT-CFG", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    # Retrieves information about the Picture files
    def get_all_pictures_info(self, user_email):
        response = [status.HTTP_200_OK, []]

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["PICTURES.files"]
            firmwares = collection.find({})
            for file in firmwares:
                response[1].append(file)
        except Exception as err:
            self.status_log("EXCEPTION (get_all_pictures_info)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get all ONU firmware information due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "PICTURES.files", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    """
    Get Compatibility Data
    """

    # Get list of all ONU/ONT manufacturers and models
    def get_onu_compatibility_data(self, user_email):
        response = [status.HTTP_200_OK, {"Vendors": [], "Models": []}]

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["ONU-STATE"]
            vendors = collection.distinct("ONU.Vendor")
            models = collection.distinct("ONU.Model")
            collection = self.database["ONU-FIRMWARE.files"]
            vendors_from_firmwares = collection.distinct("metadata.Compatible Manufacturer")
            models_from_firmwares = collection.distinct("metadata.Compatible Model")

            # Combine Model array and values
            models.extend(model for model in models_from_firmwares if model not in models)
            # Combine Vendor array and values
            vendors.extend(vendor for vendor in vendors_from_firmwares if vendor not in vendors)

            response[1]['Vendors'] = vendors
            response[1]['Models'] = models
        except Exception as err:
            self.status_log("EXCEPTION (get_onu_compatibility_data)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get ONU compatibility data due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "ONU-STATE, ONU-FIRMWARE.files", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    # Get list of all OLT models
    def get_olt_compatibility_data(self, user_email):
        response = [status.HTTP_200_OK, {"Models": []}]

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["OLT-STATE"]
            models = collection.distinct("OLT.ASIC")
            collection = self.database["OLT-FIRMWARE.files"]
            models_from_firmwares = collection.distinct("metadata.Compatible Model")

            # Combine Model array and values
            models.extend(model for model in models_from_firmwares if model not in models)
            response[1]['Models'] = models
        except Exception as err:
            self.status_log("EXCEPTION (get_olt_compatibility_data)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get OLT compatibility data due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "OLT-STATE", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    # Get sys logs for specific user
    def get_sys_logs_for_user(self, user_email, since_utc_time, email):
        response = [status.HTTP_200_OK, []]

        try:
            database_manager.get_database(self.database_id)
            collection = self.user_database["SYSLOG-ACTIONS"]
            logs = collection.find({"Time": {"$gt": since_utc_time}, "User": user_email}, no_cursor_timeout=True)
            response[1] = list(logs)
        except Exception as err:
            self.status_log("EXCEPTION (get_sys_logs_for_user)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get user system logs due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "SYSLOG-ACTIONS", "user": email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    # Get sys logs for specific user
    def get_sys_log_returned(self, log_id, user_email):
        response = [status.HTTP_200_OK, []]

        try:
            database_manager.get_database(self.database_id)
            collection = self.user_database["SYSLOG-ACTIONS"]
            response[1] = collection.find_one({'_id': log_id}, {"New": 1})
        except Exception as err:
            self.status_log("EXCEPTION (get_sys_log_returned)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get user system logs return data due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "SYSLOG-ACTIONS", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    # Log change user permission
    def change_user_permission(self, user_email):
        action_data = {"collection": "auth_user", "user": user_email}
        self.sys_log(action_data, "get", True)
        return

    # Log delete user
    def delete_user(self, user_email):
        action_data = {"collection": "auth_user", "user": user_email}
        self.sys_log(action_data, "get", True)
        return

    # Log change user last name
    def change_user_firstname(self, user_email):
        action_data = {"collection": "auth_user", "user": user_email}
        self.sys_log(action_data, "get", True)
        return

    # Log change user first name
    def change_user_lastname(self, user_email):
        action_data = {"collection": "auth_user", "user": user_email}
        self.sys_log(action_data, "get", True)
        return

    # Log change user password
    def change_user_password(self, user_email):
        action_data = {"collection": "auth_user", "user": user_email}
        self.sys_log(action_data, "get", True)
        return

    # Log change user password
    def create_user_log(self, user_created, user_email):
        action_data = {"collection": "auth_user", "old": {},
                       "new": user_created, "user": user_email}
        self.sys_log(action_data, "write", True)
        return

    """
    MIB Information
    """

    # Retrieve an array of all ONU MIB CUR States
    def get_onu_mib_cur_states(self, user_email):
        response = [status.HTTP_200_OK, []]

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["ONU-MIB-CUR-STATE"]
            cntls_cursor = collection.find()
            for state in cntls_cursor:
                response[1].append(state)
        except Exception as err:
            self.status_log("EXCEPTION (get_onu_mib_cur_states)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get all MIB Current states due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "ONU-MIB-CUR-STATE", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    # Retrieve ONU MIB CUR State for the specified ONU
    def get_onu_mib_cur_state(self, onu_id, user_email):
        response = [status.HTTP_200_OK, ""]

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["ONU-MIB-CUR-STATE"]
            response[1] = collection.find_one({"_id": onu_id})
        except Exception as err:
            self.status_log("EXCEPTION (get_onu_mib_cur_state)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get ONU {onu_id} MIB Current state due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "ONU-MIB-CUR-STATE", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    # Retrieve ONU MIB CUR States with Ids matching list of ONU Ids
    def get_many_mib_current_states(self, onus, user_email):
        response = [status.HTTP_200_OK, []]

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["ONU-MIB-CUR-STATE"]

            onu_ids = onus.split(",")
            response[1] = list(collection.find({"_id": {"$in": onu_ids}}).limit(10000))

        except Exception as err:
            self.status_log("EXCEPTION (get_many_mib_current_states)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get many MIB current states due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "MIB-CUR-STATES", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    # Retrieves the ONUs firmware upgrade status and percentage
    def get_onu_upgrade_status(self, onu_id, user_email):
        response = [status.HTTP_200_OK, {}]
        try:
            database_manager.get_database(self.database_id)
            onu_status = self.database["ONU-STATE"].find_one(
                {"_id": onu_id, "ONU.FW Upgrade Status.Status": {"$exists": True}})
            if onu_status is not None:
                cntl_status = self.database["CNTL-STATE"].find_one({"_id": onu_status["CNTL"]["MAC Address"]},
                                                                   {"_id": 0, "ONU.FW Upgrades": 1})
                if cntl_status is not None:
                    response[1] = {
                        "CNTL": cntl_status["ONU"]["FW Upgrades"],
                        "ONU": onu_status["ONU"]["FW Upgrade Status"]
                    }
        except Exception as err:
            self.status_log("EXCEPTION (get_onu_upgrade_status)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get ONU {onu_id} upgrade status due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "ONU-STATE", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    # Retrieves the ONUs firmware upgrade statuses and percentages for all ONUs under the OLT
    def get_onu_upgrade_statuses_for_olt(self, olt_id, user_email):
        response = [status.HTTP_200_OK, []]

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["ONU-STATE"]
            onus = collection.find({"OLT.MAC Address": olt_id}, {"_id": 1, "ONU.FW Upgrade Status": 1})
            olt_status = self.database["OLT-STATE"].find_one({"_id": olt_id}, {"_id": 0, "ONU.FW Upgrades": 1})
            for onu in onus:
                if 'Status' in onu["ONU"]["FW Upgrade Status"] and onu["ONU"]["FW Upgrade Status"][
                    'Status'] == 'Downloading' and onu['_id'] not in olt_status['ONU']['FW Upgrades']:
                    response[1].append({"_id": onu["_id"], "Upgrade Status": {'Status': 'Failed'}})
                else:
                    response[1].append({"_id": onu["_id"], "Upgrade Status": onu["ONU"]["FW Upgrade Status"]})
        except Exception as err:
            self.status_log("EXCEPTION (get_onu_upgrade_statuses_for_olt)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get ONU upgrade statuses for OLT {olt_id} due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "ONU-STATE", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    # Retrieves the ONUs firmware upgrade statuses and percentages for all ONUs under the Controller
    def get_onu_upgrade_statuses_for_controller(self, cntl_id, user_email):
        response = [status.HTTP_200_OK, []]

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["ONU-STATE"]
            onus = collection.find({"CNTL.MAC Address": cntl_id},
                                   {"_id": 1, "OLT.MAC Address": 1, "ONU.FW Upgrade Status": 1})
            cntl_status = self.database["CNTL-STATE"].find_one({"_id": cntl_id}, {"_id": 0, "ONU.FW Upgrades": 1})
            for onu in onus:
                if 'Status' in onu["ONU"]["FW Upgrade Status"] and onu["ONU"]["FW Upgrade Status"][
                    'Status'] == 'Downloading' and onu['_id'] not in cntl_status['ONU']['FW Upgrades']:
                    response[1].append({"_id": onu["_id"], "OLT ID": onu["OLT"]["MAC Address"],
                                        "Upgrade Status": {'Status': 'Failed'}})
                else:
                    response[1].append({"_id": onu["_id"], "OLT ID": onu["OLT"]["MAC Address"],
                                        "Upgrade Status": onu["ONU"]["FW Upgrade Status"]})
        except Exception as err:
            self.status_log("EXCEPTION (get_onu_upgrade_statuses_for_controller)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get ONU upgrade statuses for PON Controller {cntl_id} due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "ONU-STATE", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    # Retrieve an array of all ONU MIB RST States
    def get_onu_mib_rst_states(self, user_email):
        response = [status.HTTP_200_OK, []]

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["ONU-MIB-RST-STATE"]
            cntls_cursor = collection.find()
            for state in cntls_cursor:
                response[1].append(state)
        except Exception as err:
            self.status_log("EXCEPTION (get_onu_mib_rst_states)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get all MIB Reset states due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "ONU-MIB-RST-STATE", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    # Retrieve ONU MIB RST State for the specified ONU
    def get_onu_mib_rst_state(self, onu_id, user_email):
        response = [status.HTTP_200_OK, ""]

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["ONU-MIB-RST-STATE"]
            response[1] = collection.find_one({"_id": onu_id})
        except Exception as err:
            self.status_log("EXCEPTION (get_onu_mib_rst_state)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get ONU {onu_id} MIB Reset state due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "ONU-MIB-RST-STATE", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    def get_custom_device_query(self, collection, attribute, operator, value, user_email):
        response = [status.HTTP_200_OK, []]
        num_value = None

        try:
            try:
                num_value = int(value)
            except ValueError:
                num_value = float(value)
        except ValueError:
            if value.lower() == 'false':
                value = False
            elif value.lower() == 'true':
                value = True

        try:
            database_manager.get_database(self.database_id)
            collection = self.database[collection]
            devices = []
            if operator == "==":
                if num_value is not None:
                    devices = collection.find({attribute: {"$in": [value, num_value]}}, {attribute: 1})
                else:
                    devices = collection.find({attribute: {"$eq": value}}, {attribute: 1})
            elif operator == "!=":
                if num_value is not None:
                    devices = collection.find({attribute: {"$ne": [value, num_value]}}, {attribute: 1})
                else:
                    devices = collection.find({attribute: {"$ne": value}}, {attribute: 1})
            elif operator == "<=":
                devices = collection.find({attribute: {"$lte": num_value}}, {attribute: 1})
            elif operator == "<":
                devices = collection.find({attribute: {"$lt": num_value}}, {attribute: 1})
            elif operator == ">=":
                devices = collection.find({attribute: {"$gte": num_value}}, {attribute: 1})
            elif operator == ">":
                devices = collection.find({attribute: {"$gt": num_value}}, {attribute: 1})
            elif operator == 'like':
                devices = collection.find({attribute: {"$regex": value, "$options": 'i'}}, {attribute: 1})
            elif operator == 'all':
                devices = collection.find({attribute: {"$exists": True}}, {attribute: 1})

            for dev in devices:
                response[1].append(dev)
        except Exception as err:
            self.status_log("EXCEPTION (get_custom_device_query)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get custom query due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": collection, "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    # Retrieves all the data for the test loop on the switch thermal test page
    def get_switch_thermal_test_data(self, switch_id, user_email):
        response = [status.HTTP_200_OK, []]
        stats = []

        try:
            database_manager.get_database(self.database_id)
            olt_state_collection = self.database["OLT-STATE"]
            onu_state_collection = self.database["ONU-STATE"]

            olts = olt_state_collection.find({"Switch.Chassis ID": switch_id}, {"_id": 1})

            for item in olts:
                olt = item["_id"]
                formatted_olt = olt.replace(":", "")
                stats_collection = self.database["STATS-OLT-{}".format(formatted_olt)]
                # stats data to return
                cursor = stats_collection.find().sort([('Time', -1)]).limit(8)
                # There will only ever be one item in cursor. Not sure of another way to write this.
                for x in cursor:
                    if "OLT-NNI" in x:
                        stats = x
                        break

                state = olt_state_collection.find_one({"_id": olt})
                bps = 0
                for onu in state["ONUs"]:
                    onu_state = onu_state_collection.find_one({"_id": onu})
                    if onu_state is not None:
                        for index in range(0, 8):
                            if f"OLT-PON Service {index}" in onu_state["STATS"]:
                                if "RX Rate bps" in onu_state["STATS"][f"OLT-PON Service {index}"]:
                                    bps += onu_state["STATS"][f"OLT-PON Service {index}"]["RX Rate bps"]
                                if "TX Rate bps" in onu_state["STATS"][f"OLT-PON Service {index}"]:
                                    bps += onu_state["STATS"][f"OLT-PON Service {index}"]["TX Rate bps"]

                response[1].append({"olt": olt, "state": state, "stats": stats, "traffic": bps})

        except Exception as err:
            self.status_log("EXCEPTION (get_switch_thermal_test_data)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get thermal test data due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "OLT-STATE, ONU-STATE", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    def get_device_upgrade_hierarchy(self, user_email):
        response = [status.HTTP_200_OK, {'controllers': {}, 'switches': {}}]

        try:
            database_manager.get_database(self.database_id)
            # Collect all OLTs and ONUs beneath each Controller using Controllers System Status field
            cntl_ids = list(set().union(*[
                self.database['CNTL-STATE'].distinct('_id'),
                self.database['CNTL-CFG'].distinct('_id')
            ]))
            for cntl in cntl_ids:
                system_status = self.database['CNTL-STATE'].find_one({'_id': cntl}, {'System Status': 1, '_id': 0})
                if system_status is not None:
                    response[1]['controllers'][cntl] = {'olts': {}}
                    for olt in system_status['System Status'].keys():
                        response[1]['controllers'][cntl]['olts'][olt] = {'onus': []}
                        for onu in system_status['System Status'][olt]['ONUs'].keys():
                            response[1]['controllers'][cntl]['olts'][olt]['onus'].append(onu)

            # Get Switch Ids from OLT States then use Controller tree from above for OUNs
            switch_ids = self.database['OLT-STATE'].distinct("Switch.Chassis ID")
            for switch in switch_ids:
                response[1]['switches'][switch] = {'olts': {}}
                olts = self.database['OLT-STATE'].find({"Switch.Chassis ID": switch}, {"_id": 1, "CNTL.MAC Address": 1})
                for olt in olts:
                    if "CNTL" in olt and "MAC Address" in olt["CNTL"]:
                        cntl_mac = olt["CNTL"]["MAC Address"]
                        if cntl_mac in response[1]['controllers']:
                            response[1]['switches'][switch]['olts'] = response[1]['controllers'][cntl_mac]['olts']
                        else:
                            response[1]['switches'][switch]['olts'] = {}
        except Exception as err:
            self.status_log("EXCEPTION (get_device_upgrade_hierarchy)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get device upgrade tree due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "CNTL-STATE, OLT-STATE, ONU-STATE, CNTL-CFG, SWI-CFG, OLT-CFG, ONU-CFG",
                       "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    # Upgrades the specified config files with the latest fields
    def upgrade_configurations_versions(self, new_config_info, user_email):
        response = [status.HTTP_200_OK, "Configuration versions were updated.", None, {}]
        all_collections = {}

        try:
            database_manager.get_database(self.database_id)
            new_data = {}

            # Get list of all collections and all document ids to be updated
            for data in new_config_info:
                collection_name = data["collection"]
                version = data["newVersion"]

                # PON Auto and documents managed by PON Controller have their CFG Versions is different places
                if collection_name in ['CNTL-AUTO-CFG', 'SWI-AUTO-CFG', 'OLT-AUTO-CFG', 'ONU-AUTO-CFG', 'AUTO-CFG']:
                    update_document = {"$set": {"AUTO.CFG Version": version}}
                else:
                    update_document = {"$set": {"CNTL.CFG Version": version}}

                if collection_name in ["CNTL-CFG", "SWI-CFG", "OLT-CFG", "ONU-CFG"]:
                    device_type = str(collection_name).split("-")[0].upper()
                    update_document["$inc"] = {f"{device_type}.CFG Change Count": 1}

                if collection_name not in all_collections:
                    # Loop through the new attributes to place in that document
                    for field in data["newFields"]:
                        document = list(field.keys())
                        value = list(field.values())
                        update_document["$set"][document[0]] = value[0]

                    all_collections[collection_name] = {
                        "ids": [],
                        "updateDoc": update_document
                    }

                all_collections[collection_name]["ids"].append(data["device"])

            # Preform the updates
            for collection_name, update_data in all_collections.items():
                collection = self.database[collection_name]
                collection.update_many(filter={"_id": {"$in": update_data["ids"]}}, update=update_data["updateDoc"],
                                       upsert=True)
                new_data[collection_name] = {
                    "affected": update_data["ids"],
                    "updates": update_data["updateDoc"]
                }

            response[2] = new_data
        except Exception as err:
            self.status_log("EXCEPTION (upgrade_configurations_versions)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not update configuration versions due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": all_collections,
                       "returned": response[1], "user": user_email}
        self.sys_log(action_data, "put", response[0] == status.HTTP_200_OK)

        return response

    """
    Utility functions
    """

    # Log the user action to the system
    # Deprecated as of R3.0.0. Logic removed as all logging is now handled by PonManagerLogHandler
    def sys_log(self, data, type, success):
        return True

    """
    Tibit Settings
    """

    def get_session_key_expiry_date(self, session_key):
        response = [status.HTTP_200_OK, ""]

        try:
            collection = self.user_database["auth_group"]
            expiry_date = list(collection.find({'session_key': session_key}))
            if len(expiry_date) > 0:
                return expiry_date[0]['expire_date']
            else:
                return 0
        except Exception as err:
            return 0

    # Get user session expiry age from user database
    def get_user_session_expiry(self, user_email):
        response = [status.HTTP_200_OK, ""]

        try:
            expiration_amounts = list(self.user_database["auth_user"].aggregate([
                {
                    '$match': {
                        'email': user_email
                    }
                }, {
                    '$lookup': {
                        'from': 'auth_user_groups',
                        'localField': 'id',
                        'foreignField': 'user_id',
                        'as': 'roles'
                    }
                }, {
                    '$unwind': {
                        'path': '$roles',
                        'includeArrayIndex': 'string',
                        'preserveNullAndEmptyArrays': False
                    }
                }, {
                    '$lookup': {
                        'from': 'auth_group',
                        'localField': 'roles.group_id',
                        'foreignField': 'id',
                        'as': 'group'
                    }
                }, {
                    '$project': {
                        'Timeout': {
                            '$arrayElemAt': [
                                '$group.User Session Expiry Age Timeout', 0
                            ]
                        },
                        'Override': {
                            '$arrayElemAt': [
                                '$group.User Session Expiry Age Timeout Override', 0
                            ]
                        }
                    }
                }, {
                    '$sort': {
                        'Timeout': -1
                    }
                }
            ]))
            expiration = None

            if len(expiration_amounts) > 0:
                for exp_dict in expiration_amounts:
                    if "Timeout" in exp_dict:
                        expiration = exp_dict["Timeout"]
                        break

            if expiration is None:
                collection = self.user_database["PONMGR-CFG"]
                expiration = collection.find_one({"_id": "Default"})["User Session Expiry Age Timeout"]

            if expiration is not None:
                response[1] = expiration

        except Exception as err:
            self.status_log("EXCEPTION (get_user_session_expiry)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get user session expiry age due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "PONMGR-CFG", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

        # Get user session expiry age from user database

    def set_role_session_expiry(self, role_id, override, timeout, user_email):
        response = [status.HTTP_200_OK, "OK", None, {}]

        try:
            collection = self.user_database["auth_group"]
            old_document = collection.find_one({"id": role_id})
            update_document = {"$set": {"User Session Expiry Age Timeout": timeout, "User Session Expiry Age Timeout Override": override}}
            update_result = collection.update_one(filter={'id': role_id}, update=update_document, upsert=True)
            if update_result.upserted_id == "Default":
                response[0] = status.HTTP_201_CREATED
            else:
                response[0] = status.HTTP_200_OK
            response[2] = update_document
        except Exception as err:
            self.status_log("EXCEPTION (set_user_session_expiry)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not set user session expiry age due to {type(err).__name__}::{sys.exc_info()[1]}", None,
                        None]

        action_data = {"collection": "PONMGR-CFG",
                       "old": old_document, "new": update_document, "user": user_email}
        self.sys_log(action_data, "write", response[0] == status.HTTP_200_OK or response[0] == status.HTTP_201_CREATED)

        return response

    def get_role_session_expiry(self, id):
        response = [status.HTTP_200_OK, ""]
        try:
            collection = self.user_database["auth_group"]
            role_info = collection.find_one({"id": id})
            if role_info and "User Session Expiry Age Timeout" in role_info:
                timeout = role_info["User Session Expiry Age Timeout"]
                timeout_override = role_info["User Session Expiry Age Timeout Override"]
                response[1] = {
                    "timeout": timeout,
                    "override": timeout_override
                }
            # Missing fields, use global
            else:
                collection = self.user_database["PONMGR-CFG"]
                settings = collection.find_one({'_id': 'Default'})
                timeout = settings["User Session Expiry Age Timeout"]
                response[1] = {
                    "timeout": timeout,
                    "override": False
                }
        except Exception as err:
            self.status_log("EXCEPTION (get_role_session_expiry)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get role session expiry age due to {type(err).__name__}::{sys.exc_info()[1]}", None,
                        None]
        return response

    def get_global_session_time(self):
        response = [status.HTTP_200_OK, ""]
        try:
            collection = self.user_database["PONMGR-CFG"]
            settings = collection.find_one({'_id': 'Default'})
            if settings:
                timeout = settings["User Session Expiry Age Timeout"]
                response[1] = {
                    "timeout": timeout,
                }
        except Exception as err:
            self.status_log("EXCEPTION (get_global_session_expiry)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get global session expiry age due to {type(err).__name__}::{sys.exc_info()[1]}", None,
                        None]
        return response

    def set_global_session_time(self, timeout, user_email):
        response = [status.HTTP_200_OK, "OK", None, {}]

        try:
            collection = self.user_database["PONMGR-CFG"]
            update_document = {"$set": {"User Session Expiry Age Timeout": timeout}}
            update_result = collection.update_one(filter={'_id': 'Default'}, update=update_document, upsert=True)

            # Set for all role timeouts that aren't overriden
            collection = self.user_database["auth_group"]
            group_update_document = {"$set": {"User Session Expiry Age Timeout": timeout, "User Session Expiry Age Timeout Override": False}}
            collection.update_many(filter={'User Session Expiry Age Timeout Override': {'$ne': True}}, update=group_update_document, upsert=True)

            if update_result.upserted_id == "Default":
                response[0] = status.HTTP_201_CREATED
            else:
                response[0] = status.HTTP_200_OK
            response[2] = update_document
        except Exception as err:
            self.status_log("EXCEPTION (set_user_session_expiry)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not set user session expiry age due to {type(err).__name__}::{sys.exc_info()[1]}", None,
                        None]

        action_data = {"collection": "PONMGR-CFG",
                       "old": None, "new": timeout, "user": user_email}
        self.sys_log(action_data, "write", response[0] == status.HTTP_200_OK or response[0] == status.HTTP_201_CREATED)

        return response

    def get_all_session_expiry(self):
        response = [status.HTTP_200_OK, "OK", None, {}]
        try:
            collection = self.user_database["auth_group"]
            roles = list(collection.find({}, {"User Session Expiry Age Timeout": 1, "name": 1}))
            timeoout_dictionary = {}
            global_timeout = None

            for role in roles:
                if "User Session Expiry Age Timeout" in role:
                    timeoout_dictionary[role['name']] = role["User Session Expiry Age Timeout"]
                else:
                    if global_timeout is None:
                        global_timeout = self.user_database["PONMGR-CFG"].find_one({'_id': 'Default'})["User Session Expiry Age Timeout"]
                    timeoout_dictionary[role['name']] = global_timeout
            response[1] = timeoout_dictionary
        except Exception as err:
            self.status_log("EXCEPTION (get_role_session_expiry)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get role session expiry age due to {type(err).__name__}::{sys.exc_info()[1]}", None,
                        None]
        return response

    def get_all_user_sessions(self, user_email, session_id):
        response = [status.HTTP_200_OK, "OK", None, {}]
        try:
            collection = self.user_database["auth_user"]
            users = list(collection.find({}, {"email": 1, "id": 1}))
            collection = self.user_database["django_session"]
            sessions_encoded = list(collection.find())
            sessions_decoded = []

            for session in sessions_encoded:
                encoded_string = session['session_data']
                try:
                    session_data = signing.loads(force_str(encoded_string), salt='django.contrib.sessions.SessionStore', serializer=BSONSerializer)
                except BadSignature:
                    session_data = {'bad_signature': True}
                dict = {**session_data, **session}
                email = ''
                for user in users:
                    if '_auth_user_id' in dict and user['id'] == int(dict['_auth_user_id']):
                        email = user['email']
                        break
                dict['email'] = email
                system = ''
                browser = ''
                if 'user agent' in dict:
                    agent_parsed_list = httpagentparser.simple_detect(dict['user agent'])
                    if len(agent_parsed_list) >= 2:
                        system = agent_parsed_list[0]
                        browser = agent_parsed_list[1]
                dict['system'] = system
                dict['browser'] = browser
                dict['current'] = session['session_key'] == session_id
                del dict['session_key']
                del dict['session_data']
                dict['_id'] = str(dict['_id'])

                sessions_decoded.append(dict)
            response[1] = sessions_decoded
        except Exception as err:
            self.status_log("EXCEPTION (get_all_user_sessions)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get user sessions due to {type(err).__name__}::{sys.exc_info()[1]}", None,
                        None]
        return response

    # Accepts a list of user sessions to delete
    def delete_user_sessions(self, sessions):
        response = [status.HTTP_200_OK, "OK", None, {}]
        try:
            mongo_ids = []
            for session in sessions:
                if '_id' in session:
                    mongo_ids.append(ObjectId(session['_id']))
            collection = self.user_database["django_session"]
            collection.delete_many({"_id": {"$in": mongo_ids}})
        except Exception as err:
            self.status_log("EXCEPTION (delete_user_sessions)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not delete user sessions due to {type(err).__name__}::{sys.exc_info()[1]}", None,
                        None]
        return response

    # Get locale timestamp format from user database
    def get_locale_timestamp_format(self, user_email):
        response = [status.HTTP_200_OK, ""]

        try:
            collection = self.user_database["PONMGR-CFG"]
            expiration = collection.find_one({"_id": "Default"})

            if expiration is not None:
                response[1] = expiration
        except Exception as err:
            self.status_log("EXCEPTION (get_locale_timestamp_format)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get locale timestamp format due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "PONMGR-CFG", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    # Get user session expiry age from user database
    def set_locale_timestamp_format(self, locale_timestamp, user_email):
        response = [status.HTTP_200_OK, "OK", None, {}]

        try:
            collection = self.user_database["PONMGR-CFG"]
            update_document = {"$set": {"Locale Timestamp Format": locale_timestamp}}
            update_result = collection.update_one(filter={'_id': 'Default'}, update=update_document, upsert=True)
            if update_result.upserted_id == "Default":
                response[0] = status.HTTP_201_CREATED
            else:
                response[0] = status.HTTP_200_OK
            response[2] = update_document
        except Exception as err:
            self.status_log("EXCEPTION (set_locale_timestamp_format)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not set locale_timestamp_format due to {type(err).__name__}::{sys.exc_info()[1]}", None,
                        None]

        action_data = {"collection": "PONMGR-CFG",
                       "old": None, "new": locale_timestamp, "user": user_email}
        self.sys_log(action_data, "write", response[0] == status.HTTP_200_OK or response[0] == status.HTTP_201_CREATED)

        return response

    # Get open street location preferences from user database
    def get_location_pref(self, user_email):
        response = [status.HTTP_200_OK, ""]

        try:
            collection = self.user_database["PONMGR-CFG"]
            expiration = collection.find_one({"_id": "Default"})

            if expiration is not None:
                response[1] = expiration
        except Exception as err:
            self.status_log("EXCEPTION (get_location_pref)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get open street location preferences due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "PONMGR-CFG", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    # Set open street location preferences from user database
    def set_location_pref(self, location_pref, user_email):
        response = [status.HTTP_200_OK, "OK", None, {}]

        try:
            collection = self.user_database["PONMGR-CFG"]
            update_document = {"$set": {"Enable Open Street Location": location_pref}}
            update_result = collection.update_one(filter={'_id': 'Default'}, update=update_document, upsert=True)
            if update_result.upserted_id == "Default":
                response[0] = status.HTTP_201_CREATED
            else:
                response[0] = status.HTTP_200_OK
            response[2] = update_document
        except Exception as err:
            self.status_log("EXCEPTION (set_location_pref)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not set location_pref due to {type(err).__name__}::{sys.exc_info()[1]}", None,
                        None]

        action_data = {"collection": "PONMGR-CFG",
                       "old": None, "new": location_pref, "user": user_email}
        self.sys_log(action_data, "write", response[0] == status.HTTP_200_OK or response[0] == status.HTTP_201_CREATED)

        return response

    # Get pon mode for the system from user database
    def get_pon_mode(self, user_email):
        response = [status.HTTP_200_OK, ""]

        try:
            collection = self.user_database["PONMGR-CFG"]
            default_olt_settings = collection.find_one({"_id": "Default"})

            if default_olt_settings is not None:
                response[1] = default_olt_settings
        except Exception as err:
            self.status_log("EXCEPTION (get_pon_mode)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get pon mode due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "PONMGR-CFG", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    def set_pon_mode(self, pon_mode, user_email):
        response = [status.HTTP_200_OK, "OK", None, {}]

        try:
            collection = self.user_database["PONMGR-CFG"]
            old = collection.find_one({'_id': 'Default'})
            update_document = {"$set": {"PON Mode": pon_mode}}
            update_result = collection.update_one(filter={'_id': 'Default'}, update=update_document, upsert=True)
            if update_result.upserted_id == "Default":
                response[0] = status.HTTP_201_CREATED
            else:
                response[0] = status.HTTP_200_OK
            response[2] = update_document
        except Exception as err:
            self.status_log("EXCEPTION (set_pon_mode)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not set pon_mode due to {type(err).__name__}::{sys.exc_info()[1]}", None, None]

        action_data = {"collection": "PONMGR-CFG",
                       "old": old, "new": pon_mode, "user": user_email}
        self.sys_log(action_data, "write",
                     response[0] == status.HTTP_200_OK or response[0] == status.HTTP_201_CREATED)

        return response

    def get_radius_settings(self, user_email):
        response = [status.HTTP_200_OK, "OK", None, {}]
        try:
            collection = self.user_database["PONMGR-CFG"]
            radius_settings = collection.find_one({"_id": "Default"}, {"Radius": 1})
            if radius_settings is not None:
                response[1] = radius_settings
        except Exception as err:
            self.status_log("EXCEPTION (get_radius_settings)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get radius settings due to {type(err).__name__}::{sys.exc_info()[1]}", None, None]

        action_data = {"collection": "PONMGR-CFG", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    def set_radius_settings(self, radius_settings, user_email):
        response = [status.HTTP_200_OK, "OK", None, {}]
        try:
            collection = self.user_database["PONMGR-CFG"]
            old = collection.find_one({'_id': 'Default'}, {'Radius': 1})
            update_document = {"$set": {"Radius": radius_settings}}
            update_result = collection.update_one(filter={'_id': 'Default'}, update=update_document, upsert=True)
        except Exception as err:
            self.status_log("EXCEPTION (set_radius_settings)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not set radius settings due to {type(err).__name__}::{sys.exc_info()[1]}", None, None]

        action_data = {"collection": "PONMGR-CFG",
                       "old": old, "new": update_result, "user": user_email}
        self.sys_log(action_data, "write",
                     response[0] == status.HTTP_200_OK or response[0] == status.HTTP_201_CREATED)
        return response

    # Utility Functions
    # Print message to the user
    @staticmethod
    def status_log(msg_type, message):
        print("PON MANAGER {} - {}".format(msg_type, message))

    @staticmethod
    def is_int(value):
        try:
            int(value)
            return True
        except ValueError:
            return False

    # Returns current time - given time
    @staticmethod
    def get_time_duration(timestamp, second_timestamp=None):
        time = datetime.datetime.strptime(timestamp, '%Y-%m-%d %H:%M:%S.%f')
        # time_delta = datetime.datetime.utcnow() - time
        time_delta_string = ""
        if second_timestamp:
            time2 = datetime.datetime.strptime(second_timestamp, '%Y-%m-%d %H:%M:%S.%f')
            time_delta = time2 - time
        else:
            time_delta = datetime.datetime.utcnow() - time

        if time_delta.days >= 1:
            time_delta_string += str(time_delta.days) + "d, "
        if time_delta.seconds >= 3600:
            time_delta_string += str(time_delta.seconds // 3600) + "h, "
        if time_delta.seconds >= 60:
            time_delta_string += str((time_delta.seconds % 3600) // 60) + "m"
        return time_delta_string

    @staticmethod
    def controller_time_is_online(cntl_state_timestamp, cntl_state_version):
        seconds_in_day = 60 * 60 * 24
        cntl_time = datetime.datetime.strptime(cntl_state_timestamp, '%Y-%m-%d %H:%M:%S.%f')
        time_delta = datetime.datetime.utcnow() - cntl_time
        max_time_delta = 6

        controller_version = cntl_state_version[1:].split("-")[0]
        controller_version_numbers = controller_version.split(".")

        if len(controller_version_numbers) > 2 and (int(controller_version_numbers[0]) >= 3 or (
                int(controller_version_numbers[0]) == 2 and int(controller_version_numbers[1]) >= 2)):
            max_time_delta = 2

        return divmod(time_delta.days * seconds_in_day + time_delta.seconds, 60)[0] < max_time_delta

    @staticmethod
    def automation_time_is_online(auto_state_timestamp):
        seconds_in_day = 60 * 60 * 24
        auto_time = datetime.datetime.strptime(auto_state_timestamp, '%Y-%m-%d %H:%M:%S.%f')
        time_delta = datetime.datetime.utcnow() - auto_time
        max_time_delta = 3

        return divmod(time_delta.days * seconds_in_day + time_delta.seconds, 60)[0] < max_time_delta

    """
    Angular 2 new endpoints
    """

    def make_bulk_change(self, changes, user_email):
        response = [status.HTTP_200_OK, {}, None, {}]
        changes = changes['changesObj']
        attribute = changes['attribute']
        value = changes['value']
        failed_devices = []
        do_change = False
        coll_prefix = ''
        collection = {}

        try:
            database_manager.get_database(self.database_id)
            # First checks the type of device to save to
            if changes['collection'] == 'ONU Configurations':
                collection = self.database['ONU-CFG']
                coll_prefix = 'ONU'
                fields_to_increment = ['ONU.Reset Count', 'ONU.Reset BER Count', 'ONU.FW Upgrade Abort Clear Count', 'Alarm History.Ack Count', 'Alarm History.Purge Count']

                # Special case that needs special back-end validation
                if attribute == 'ONU.FW Bank Files' or attribute == 'ONU.FW Bank Versions':
                    onu_state_collection = self.database['ONU-STATE']

                    for onu_id in changes['devices']:
                        fw_bank_files = ['', '']
                        fw_bank_versions = ['', '']
                        partial_success = False
                        save_fw = False
                        onu_state = onu_state_collection.find_one({"_id": onu_id}, {"ONU.Vendor": 1, "ONU.Equipment ID": 1})

                        # Check if the ONU has a state file and the state has a listed Vendor
                        if onu_state:
                            vendor_name = onu_state['ONU']['Vendor']
                            vendor_name_normalized = vendor_name[0:4].upper()
                            equipment_id = ''
                            if 'Equipment ID' in onu_state['ONU']:
                                equipment_id = onu_state['ONU']['Equipment ID']

                            # Checks if user is configuring each bank. If they are and one fails it never saves.
                            # Check if user is configuring FW Bank 0
                            if 'fw0' in value:
                                if value['fw0'][0]['_id'] == '':
                                    fw_bank_files[0] = ''
                                    fw_bank_versions[0] = ''
                                    save_fw = True
                                # Ensure Manufacturer in the FW metadata matches that of the ONUs state Vendor
                                elif vendor_name == value['fw0'][0]['metadata']['Compatible Manufacturer'] or (isinstance(value['fw0'][0]['metadata']['Compatible Manufacturer'], str) and vendor_name_normalized == value['fw0'][0]['metadata']['Compatible Manufacturer'][0:4].upper()):
                                    if vendor_name == 'CIEN':  # Ciena ONUs MUST check model as well as Vendor
                                        normalized_compatible_models = map(lambda model: model.replace('_', '-'), value['fw0'][0]['metadata']['Compatible Model'])
                                        normalized_equipment_id = equipment_id.replace('_', '-')

                                        if normalized_equipment_id in normalized_compatible_models:
                                            fw_bank_files[0] = (value['fw0'][0]['_id'])
                                            fw_bank_versions[0] = (value['fw0'][0]['metadata']['Version'])
                                            save_fw = True
                                        else:
                                            save_fw = False
                                    else:  # All other ONUs
                                        fw_bank_files[0] = (value['fw0'][0]['_id'])
                                        fw_bank_versions[0] = (value['fw0'][0]['metadata']['Version'])
                                        save_fw = True
                                else:
                                    save_fw = False
                            else:
                                save_fw = True
                            # Check if user is configuring FW Bank 1
                            if 'fw1' in value and save_fw != False:
                                if value['fw1'][0]['_id'] == '':
                                    fw_bank_files[1] = ''
                                    fw_bank_versions[1] = ''
                                    save_fw = True
                                # Ensure Manufacturer in the FW metadata matches that of the ONUs state Vendor
                                elif vendor_name == value['fw1'][0]['metadata']['Compatible Manufacturer'] or (isinstance(value['fw0'][0]['metadata']['Compatible Manufacturer'], str) and vendor_name_normalized == value['fw0'][0]['metadata']['Compatible Manufacturer'][0:4].upper()):
                                    if vendor_name == 'CIEN':  # Ciena ONUs MUST check model as well as Vendor
                                        normalized_compatible_models = map(lambda model: model.replace('_', '-'), value['fw1'][0]['metadata']['Compatible Model'])
                                        normalized_equipment_id = equipment_id.replace('_', '-')

                                        if normalized_equipment_id in normalized_compatible_models:
                                            fw_bank_files[1] = (value['fw1'][0]['_id'])
                                            fw_bank_versions[1] = (value['fw1'][0]['metadata']['Version'])
                                            save_fw = True
                                        else:
                                            save_fw = False
                                    else:  # All other ONUs
                                        fw_bank_files[1] = (value['fw1'][0]['_id'])
                                        fw_bank_versions[1] = (value['fw1'][0]['metadata']['Version'])
                                        save_fw = True
                                else:
                                    save_fw = False
                        else:
                            # If the ONU doesn't have a state, the FW is saved since there is no validation check possible
                            if 'fw0' in value:
                                fw_bank_files[0] = (value['fw0'][0]['_id'])
                                fw_bank_versions[0] = (value['fw0'][0]['metadata']['Version'])
                                save_fw = True

                            # Check if user is configuring FW Bank 1
                            if 'fw1' in value:
                                fw_bank_files[1] = value['fw1'][0]['_id']
                                fw_bank_versions[1] = value['fw1'][0]['metadata']['Version']
                                save_fw = True

                        if save_fw:
                            # Save FW Bank Version to DB
                            update_result = collection.update_one({'_id': onu_id},
                                                                  {"$set": {
                                                                      'ONU.FW Bank Versions': fw_bank_versions,
                                                                      'ONU.FW Bank Files': fw_bank_files
                                                                  },
                                                                      "$inc": {f"{coll_prefix}.CFG Change Count": 1}
                                                                  })
                            if update_result.matched_count <= 0:
                                failed_devices.append(onu_id)
                        else:
                            failed_devices.append(onu_id)
                # End FW change

                # Check and increment reset count
                elif attribute in fields_to_increment:
                    for device_id in changes['devices']:
                        update_result = collection.update_one({'_id': device_id}, {"$inc": {attribute: 1}})
                        if update_result.matched_count <= 0:
                            failed_devices.append(device_id)

                # All other changes
                else:
                    do_change = True

            elif changes['collection'] == 'OLT Configurations':
                collection = self.database['OLT-CFG']
                coll_prefix = 'OLT'
                fields_to_increment = ['OLT.Reset Count', 'Alarm History.Ack Count', 'Alarm History.Purge Count']

                # Special case that needs special back-end validation
                if attribute == 'OLT.FW Bank Files' or attribute == 'OLT.FW Bank Versions':
                    # For each file, get version from OLT-FIRMWARE.files and save the arrays of filenames and versions
                    fw_collection = self.database["OLT-FIRMWARE.files"]
                    version_array = []
                    filename_array = []

                    for version in value:
                        version_array.append(version)
                        if version != '':
                            fw_file = fw_collection.find_one({'metadata.Version': version}, {"filename": 1})
                            if fw_file is not None:
                                filename = fw_file['filename']
                                filename_array.append(filename)
                            else:
                                do_change = False
                                failed_devices = changes['devices']
                                break
                        else:
                            filename_array.append('')

                    # Save OLT FW Version and Filename to the db
                    for olt_id in changes['devices']:
                        update_result = collection.update_one({'_id': olt_id},
                                                              {"$set": {
                                                                  'OLT.FW Bank Versions': version_array,
                                                                  'OLT.FW Bank Files': filename_array
                                                              },
                                                                  "$inc": {f"{coll_prefix}.CFG Change Count": 1}
                                                              })
                        if update_result.matched_count <= 0:
                            failed_devices.append(olt_id)
                # End FW change

                # Checks individual devices for pon mode then determines if value is within limits
                elif attribute == 'NNI.Max Frame Size':
                    for device_id in changes['devices']:
                        pon_mode = fw_file = collection.find_one({'_id': device_id}, {'OLT.PON Mode': 1})
                        if pon_mode is not None:
                            pon_mode = pon_mode['OLT']['PON Mode']
                            if pon_mode == 'GPON':
                                if value <= 9600:
                                    update_result = collection.update_one(
                                        {'_id': device_id}, {"$set": {attribute: value},
                                                             "$inc": {f"{coll_prefix}.CFG Change Count": 1}})
                                    if update_result.matched_count <= 0:
                                        failed_devices.append(device_id)
                                else:
                                    failed_devices.append(device_id)
                            else:
                                if value <= 12500:
                                    update_result = collection.update_one(
                                        {'_id': device_id}, {"$set": {attribute: value},
                                                             "$inc": {f"{coll_prefix}.CFG Change Count": 1}})
                                    if update_result.matched_count <= 0:
                                        failed_devices.append(device_id)
                                else:
                                    failed_devices.append(device_id)
                        else:
                            failed_devices.append(device_id)
                # Check and increment reset count
                elif attribute in fields_to_increment:
                    for device_id in changes['devices']:
                        update_result = collection.update_one({'_id': device_id}, {"$inc": {attribute: 1}})
                        if update_result.matched_count <= 0:
                            failed_devices.append(device_id)
                else:
                    do_change = True

            # No special cases for controllers or switches so those just save
            elif changes['collection'] == 'Controller Configurations':
                collection = self.database['CNTL-CFG']
                coll_prefix = 'CNTL'
                fields_to_increment = ['Alarm History.Ack Count', 'Alarm History.Purge Count']

                if attribute in fields_to_increment:
                    for device_id in changes['devices']:
                        update_result = collection.update_one({'_id': device_id}, {"$inc": {attribute: 1}})
                        if update_result.matched_count <= 0:
                            failed_devices.append(device_id)
                else:
                    do_change = True

            elif changes['collection'] == 'Switch Configurations' or changes['collection'] == 'Switch/Router Configurations':
                collection = self.database['SWI-CFG']
                coll_prefix = 'SWI'
                do_change = True

            if do_change:
                # Potential for a field to be changed between the update and find
                # Only way to do an update many and determine which documents were actually updated (not just count)
                collection.update_many({"_id": {"$in": changes['devices']}},
                                       {"$set": {attribute: value}, "$inc": {f"{coll_prefix}.CFG Change Count": 1}})
                updated_devices = collection.find({"_id": {"$in": changes['devices']}}, {attribute: 1})

                for device in updated_devices:
                    result_value = device
                    keys = str(attribute).split(".")
                    for key in keys:
                        result_value = result_value[key]

                    if result_value != value:
                        failed_devices.append(device['_id'])

            response[1] = {'failed': failed_devices, 'newVal': value}
            response[2] = " ".join([collection.name, attribute, "=", str(value)])
        except Exception as err:
            self.status_log("EXCEPTION (make_bulk_change)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not make bulk change due to {type(err).__name__}::{sys.exc_info()[1]}", None, {}]

        action_data = {"collection": "ONU-CFG, OLT-CFG, CNTL-CFG, SWI-CFG", "user": user_email}
        if len(failed_devices) > 0:
            response[0] = status.HTTP_207_MULTI_STATUS
            self.sys_log(action_data, "post", response[0] == status.HTTP_207_MULTI_STATUS)
        else:
            response[0] = status.HTTP_200_OK
            self.sys_log(action_data, "post", response[0] == status.HTTP_200_OK)

        return response

    def get_controller_counts(self, user_email):
        response = [status.HTTP_200_OK, {"online": [], "offline": [], "pre-provisioned": []}]
        online_count = 0
        offline_count = 0
        pre_pro_count = 0

        try:
            database_manager.get_database(self.database_id)

            count_prepro = list(self.database["CNTL-CFG"].aggregate([
                {
                    "$lookup": {
                        "from": "CNTL-STATE",
                        "localField": "_id",
                        "foreignField": "_id",
                        "as": "matched"
                    }
                },
                {
                    "$match": {
                        "$and": [{
                            "matched": {"$eq": []},
                            # This performs the function to get the ids from config not in state
                            "_id": {"$ne": "Default"}
                        }]
                    }
                },
                {
                    "$count": "prepro_count"
                }
            ]))
            pre_pro_count = get_nested_value(count_prepro, [0, "prepro_count"], 0)

            controller_states = self.database["CNTL-STATE"].find({}, {"System Status": 1, "Time": 1, "CNTL.Version": 1})
            for controller in controller_states:
                is_cntl_online = self.controller_time_is_online(controller["Time"], controller["CNTL"]["Version"])
                if not is_cntl_online:
                    offline_count += 1
                else:
                    online_count += 1

            response[1] = [{"name": "online", "value": online_count},
                           {"name": "offline", "value": offline_count},
                           {"name": "pre-provisioned", "value": pre_pro_count}]
        except Exception as err:
            self.status_log("EXCEPTION (get_controller_counts)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get PON Controller counts due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "CNTL-CFG, CNTL-STATE", "returned": response[1], "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    def get_olt_counts(self, user_email):
        response = [status.HTTP_200_OK, {}]
        online_count = 0
        offline_count = 0
        pre_pro_count = 0
        unattached_count = 0

        try:
            database_manager.get_database(self.database_id)
            olt_state_count = self.database["OLT-STATE"].count_documents({})  # number of documents in olt-state

            cntl_time_cutoff_r21 = str(datetime.datetime.utcnow() - datetime.timedelta(minutes=6))
            cntl_time_cutoff_r22 = str(datetime.datetime.utcnow() - datetime.timedelta(minutes=2))

            olt_counts = list(self.database["CNTL-STATE"].aggregate([
                {  # olderThanR220 field set to true if cntl is R221 or older
                    '$addFields': {
                        'olderThanR220': {
                            '$cond': [
                                {
                                    '$or': [
                                        {
                                            '$eq': [
                                                {
                                                    '$indexOfCP': [
                                                        '$CNTL.Version', '2.1.'
                                                    ]
                                                }, 1
                                            ]
                                        }, {
                                            '$eq': [
                                                {
                                                    '$indexOfCP': [
                                                        '$CNTL.Version', '2.0'
                                                    ]
                                                }, 1
                                            ]
                                        }, {
                                            '$eq': [
                                                {
                                                    '$indexOfCP': [
                                                        '$CNTL.Version', '1.'
                                                    ]
                                                }, 1
                                            ]
                                        }
                                    ]
                                }, True, False
                            ]
                        }
                    }
                }, {  # add controller status as online or offline
                    '$addFields': {
                        'status': {
                            '$cond': [
                                {
                                    '$or': [
                                        {
                                            '$and': [
                                                {
                                                    '$eq': [
                                                        '$olderThanR220', False
                                                    ]
                                                }, {
                                                    '$gte': [
                                                        '$Time', cntl_time_cutoff_r22
                                                    ]
                                                }
                                            ]
                                        }, {
                                            '$and': [
                                                {
                                                    '$ne': [
                                                        '$olderThanR220', False
                                                    ]
                                                }, {
                                                    '$gte': [
                                                        '$Time', cntl_time_cutoff_r21
                                                    ]
                                                }
                                            ]
                                        }
                                    ]
                                }, {
                                    'status': 'online'
                                }, {
                                    'status': 'offline'
                                }
                            ]
                        }
                    }
                }, {  # System Status object to array
                    '$project': {
                        '_id': 0,
                        'System Status': {
                            '$objectToArray': '$System Status'
                        },
                        'Status': '$status.status'
                    }
                }, {  # Each olt in System Status to their own document
                    '$unwind': {
                        'path': '$System Status',
                        'preserveNullAndEmptyArrays': False
                    }
                }, {  # Project to only have olt id, olt state, controller state
                    '$project': {
                        'olt': '$System Status.k',
                        'state': '$System Status.v.OLT State',
                        'cntl': '$Status'
                    }
                }, {  # online: olt online and cntl online; else offline
                    '$group': {
                        '_id': 0,
                        'online': {
                            '$addToSet': {
                                '$cond': [
                                    {
                                        '$and': [
                                            {
                                                '$or': [
                                                    {
                                                        '$eq': [
                                                            '$state', 'Primary'
                                                        ]
                                                    }, {
                                                        '$eq': [
                                                            '$state', 'Secondary'
                                                        ]
                                                    }, {
                                                        '$eq': [
                                                            '$state', 'Unspecified'
                                                        ]
                                                    }
                                                ]
                                            }, {
                                                '$eq': [
                                                    '$cntl', 'online'
                                                ]
                                            }
                                        ]
                                    }, '$olt', '$$REMOVE'
                                ]
                            }
                        },
                        'offline': {
                            '$addToSet': {
                                '$cond': [
                                    {
                                        '$and': [
                                            {
                                                '$ne': [
                                                    '$state', 'Unspecified'
                                                ]
                                            }, {
                                                '$ne': [
                                                    '$state', 'Primary'
                                                ]
                                            }, {
                                                '$ne': [
                                                    '$state', 'Secondary'
                                                ]
                                            }, {
                                                '$eq': [
                                                    '$cntl', 'online'
                                                ]
                                            }
                                        ]
                                    }, '$olt', {
                                        '$cond': [
                                            {
                                                '$eq': [
                                                    '$cntl', 'offline'
                                                ]
                                            }, '$olt', '$$REMOVE'
                                        ]
                                    }
                                ]
                            }
                        }
                    }
                }, {  # online count: online; offline count: offline - (offline intersect online)
                    '$project': {
                        'online count': {
                            '$size': '$online'
                        },
                        'offline count': {
                            '$subtract': [
                                {
                                    '$size': '$offline'
                                }, {
                                    '$size': {
                                        '$setIntersection': [
                                            '$offline', '$online'
                                        ]
                                    }
                                }
                            ]
                        }
                    }
                }
            ]))

            # Find pre-provisioned by getting OLT-CFG ids that don't have a state.
            agg_prepro = list(self.database["OLT-CFG"].aggregate([
                {
                    "$lookup": {
                        "from": "OLT-STATE",
                        "localField": "_id",
                        "foreignField": "_id",
                        "as": "matched"
                    }
                },
                {
                    "$match": {
                        "$and": [{
                            "matched": {"$eq": []},
                            # This performs the function to get the ids from config not in state
                            "_id": {"$ne": "Default"}
                        }]
                    }
                },
                {
                    "$count": "prepro_count"
                }
            ]))
            pre_pro_count = get_nested_value(agg_prepro, [0, "prepro_count"], 0)
            online_count = get_nested_value(olt_counts, [0, "online count"], 0)
            offline_count = get_nested_value(olt_counts, [0, "offline count"], 0)

            if (offline_count + online_count) < olt_state_count:
                unattached_count = olt_state_count - online_count - offline_count

            response[1] = [{"name": "online", "value": online_count},
                           {"name": "offline", "value": offline_count},
                           {"name": "unattached", "value": unattached_count},
                           {"name": "pre-provisioned", "value": pre_pro_count}]
        except Exception as err:
            self.status_log("EXCEPTION (get_olt_counts)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get OLT counts due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "OLT-CFG, OLT-STATE", "returned": response[1], "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    def get_onu_counts(self, user_email):
        response = [status.HTTP_200_OK, {}]
        online_count = 0
        offline_count = 0
        pre_pro_count = 0
        unattached_count = 0
        onu_state_count = 0

        try:
            database_manager.get_database(self.database_id)

            onu_state_count = self.database["ONU-STATE"].count_documents({})  # number of documents in onu-state

            cntl_time_cutoff_r21 = str(datetime.datetime.utcnow() - datetime.timedelta(minutes=6))
            cntl_time_cutoff_r22 = str(datetime.datetime.utcnow() - datetime.timedelta(minutes=2))

            onu_counts = list(self.database["CNTL-STATE"].aggregate([
                {  # add olderThanR220 field if cntl is R221 or older
                    '$addFields': {
                        'olderThanR220': {
                            '$cond': [
                                {
                                    '$or': [
                                        {
                                            '$eq': [
                                                {
                                                    '$indexOfCP': [
                                                        '$CNTL.Version', '2.1.'
                                                    ]
                                                }, 1
                                            ]
                                        }, {
                                            '$eq': [
                                                {
                                                    '$indexOfCP': [
                                                        '$CNTL.Version', '2.0'
                                                    ]
                                                }, 1
                                            ]
                                        }, {
                                            '$eq': [
                                                {
                                                    '$indexOfCP': [
                                                        '$CNTL.Version', '1.'
                                                    ]
                                                }, 1
                                            ]
                                        }
                                    ]
                                }, True, False
                            ]
                        }
                    }
                }, {  # add status online or offline for each controller
                    '$addFields': {
                        'status': {
                            '$cond': [
                                {
                                    '$or': [
                                        {
                                            '$and': [
                                                {
                                                    '$eq': [
                                                        '$olderThanR220', False
                                                    ]
                                                }, {
                                                    '$gte': [
                                                        '$Time', cntl_time_cutoff_r22
                                                    ]
                                                }
                                            ]
                                        }, {
                                            '$and': [
                                                {
                                                    '$ne': [
                                                        '$olderThanR220', False
                                                    ]
                                                }, {
                                                    '$gte': [
                                                        '$Time', cntl_time_cutoff_r21
                                                    ]
                                                }
                                            ]
                                        }
                                    ]
                                }, {
                                    'status': 'online'
                                }, {
                                    'status': 'offline'
                                }
                            ]
                        }
                    }
                }, {  # project only system status and cntl status. System status obj to array
                    '$project': {
                        '_id': 0,
                        'System Status': {
                            '$objectToArray': '$System Status'
                        },
                        'Status': '$status.status'
                    }
                }, {  # every olt in system status array becomes its own object
                    '$unwind': {
                        'path': '$System Status',
                        'preserveNullAndEmptyArrays': False
                    }
                }, {  # onu objects turned into array of onus
                    '$project': {
                        'onu': {
                            '$objectToArray': '$System Status.v.ONUs'
                        },
                        'cntl status': '$Status'
                    }
                }, {  # separate documents for each onu
                    '$unwind': {
                        'path': '$onu',
                        'preserveNullAndEmptyArrays': False
                    }
                }, {  # if online cntl and registered/unspecified/Fw upgrade add to online set. Else add to offline set
                    '$group': {
                        '_id': 0,
                        'online': {
                            '$addToSet': {
                                '$cond': [
                                    {
                                        '$and': [
                                            {
                                                '$or': [
                                                    {
                                                        '$eq': [
                                                            '$onu.v', 'Registered'
                                                        ]
                                                    }, {
                                                        '$eq': [
                                                            '$onu.v', 'Unspecified'
                                                        ]
                                                    }, {
                                                        '$eq': [
                                                            '$onu.v', 'FW Upgrade'
                                                        ]
                                                    }
                                                ]
                                            }, {
                                                '$eq': [
                                                    '$cntl status', 'online'
                                                ]
                                            }
                                        ]
                                    }, '$onu.k', '$$REMOVE'
                                ]
                            }
                        },
                        'offline': {
                            '$addToSet': {
                                '$cond': [
                                    {
                                        '$and': [
                                            {
                                                '$ne': [
                                                    '$onu.v', 'Registered'
                                                ]
                                            }, {
                                                '$ne': [
                                                    '$onu.v', 'Unspecified'
                                                ]
                                            }, {
                                                '$eq': [
                                                    '$cntl status', 'online'
                                                ]
                                            }, {
                                                '$ne': [
                                                    '$onu.v', 'FW Upgrade'
                                                ]
                                            }
                                        ]
                                    }, '$onu.k', {
                                        '$cond': [
                                            {
                                                '$eq': [
                                                    '$cntl status', 'offline'
                                                ]
                                            }, '$onu.k', '$$REMOVE'
                                        ]
                                    }
                                ]
                            }
                        }
                    }
                }, {  # online count = count(online) offline count = count(offline - offline intersect online)
                    '$project': {
                        'online count': {
                            '$size': '$online'
                        },
                        'offline count': {
                            '$subtract': [
                                {
                                    '$size': '$offline'
                                }, {
                                    '$size': {
                                        '$setIntersection': [
                                            '$offline', '$online'
                                        ]
                                    }
                                }
                            ]
                        },
                        'offline list': '$offline'
                    }
                }
            ]))

            list_pre_pro = list(self.database["ONU-CFG"].aggregate([
                {
                    "$lookup": {
                        "from": "ONU-STATE",
                        "localField": "_id",
                        "foreignField": "_id",
                        "as": "matched"
                    }
                },
                {
                    "$match": {
                        "$and": [{
                            "matched": {"$eq": []},
                            # This performs the function to get the ids from config not in state
                            "_id": {"$ne": "Default"}
                        }]
                    }
                },
                {
                    '$project': {
                        '_id': 1
                    }
                }
            ]))
            online_count = get_nested_value(onu_counts, [0, "online count"], 0)
            pre_pro_count = len(list_pre_pro)
            offline_list = get_nested_value(onu_counts, [0, "offline list"], 0)
            # remove pre-provisioned ONUs attached to an OLT from the offline count
            dereg_pre_pro_count = len([onu['_id'] for onu in list_pre_pro if onu['_id'] in offline_list])
            offline_count = get_nested_value(onu_counts, [0, "offline count"], 0) - dereg_pre_pro_count
            if (offline_count + online_count) < onu_state_count:
                unattached_count = onu_state_count - online_count - offline_count
            response[1] = [{"name": "online", "value": online_count},
                           {"name": "offline", "value": offline_count},
                           {"name": "unattached", "value": unattached_count},
                           {"name": "pre-provisioned", "value": pre_pro_count}]
        except Exception as err:
            self.status_log("EXCEPTION (get_onu_counts)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get ONU counts due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "ONU-CFG, ONU-STATE", "returned": response[1], "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    def get_online_onus_for_controller(self, controller_id, user_email):
        response = [status.HTTP_200_OK, []]

        try:
            database_manager.get_database(self.database_id)
            cntl_state = self.database["CNTL-STATE"].find_one({"_id": controller_id},
                                                              {"System Status": 1, "_id": 0, "Time": 1,
                                                               "CNTL.Version": 1})
            if cntl_state is not None and "Time" in cntl_state and self.controller_time_is_online(cntl_state["Time"],
                                                                                                  cntl_state["CNTL"][
                                                                                                      "Version"]):
                for olt in cntl_state["System Status"]:
                    for onu in cntl_state["System Status"][olt]["ONUs"]:
                        if cntl_state["System Status"][olt]["ONUs"][onu] == "Unspecified" or \
                                cntl_state["System Status"][olt]["ONUs"][onu] == "Registered":
                            response[1].append(onu)
        except Exception as err:
            self.status_log("EXCEPTION (get_online_onus_for_controller)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get ONUs under PON Controller {controller_id} "
                        f"due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "ONU-STATE", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    # Retrieve all controller config ids
    def get_all_cntl_cfg_ids(self, user_email):
        response = [status.HTTP_200_OK, []]

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["CNTL-CFG"]
            cfg_ids_cursor = collection.find({}, {"_id": 1})
            for id in cfg_ids_cursor:
                response[1].append(id['_id'])
        except Exception as err:
            self.status_log("EXCEPTION (get_all_cntl_cfg_ids)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get all PON Controller configuration IDs due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "CNTL-CFG", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    # Retrieve all switch config ids
    def get_all_switch_cfg_ids(self, user_email):
        response = [status.HTTP_200_OK, []]

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["SWI-CFG"]
            cfg_ids_cursor = collection.find({}, {"_id": 1})
            for id in cfg_ids_cursor:
                response[1].append(id['_id'])
        except Exception as err:
            self.status_log("EXCEPTION (get_all_swi_cfg_ids)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get all Switch configuration IDs due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "SWI-CFG", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    # Retrieve all olt config ids
    def get_all_olt_cfg_ids(self, user_email):
        response = [status.HTTP_200_OK, []]

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["OLT-CFG"]
            cfg_ids_cursor = collection.find({}, {"_id": 1})
            for id in cfg_ids_cursor:
                response[1].append(id['_id'])
        except Exception as err:
            self.status_log("EXCEPTION (get_all_olt_cfg_ids)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get all OLT configuration IDs due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "OLT-CFG", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    # Returns the ids of all offline OLTs based on a user provided amount of minutes
    def get_offline_olts(self, user_email, minutes):
        response = [status.HTTP_200_OK, []]

        try:
            if minutes == 'undefined' or minutes == '':
                minutes = '0'
            additional_minutes = int(minutes)
            database_manager.get_database(self.database_id)
            cntl_time_cutoff_r21 = str(datetime.datetime.utcnow() - datetime.timedelta(minutes=6))
            cntl_time_cutoff_r22 = str(datetime.datetime.utcnow() - datetime.timedelta(minutes=2))

            olt_counts = list(self.database["CNTL-STATE"].aggregate([
                {  # olderThanR220 field set to true if cntl is R221 or older
                    '$addFields': {
                        'olderThanR220': {
                            '$cond': [
                                {
                                    '$or': [
                                        {
                                            '$eq': [
                                                {
                                                    '$indexOfCP': [
                                                        '$CNTL.Version', '2.1.'
                                                    ]
                                                }, 1
                                            ]
                                        }, {
                                            '$eq': [
                                                {
                                                    '$indexOfCP': [
                                                        '$CNTL.Version', '2.0'
                                                    ]
                                                }, 1
                                            ]
                                        }, {
                                            '$eq': [
                                                {
                                                    '$indexOfCP': [
                                                        '$CNTL.Version', '1.'
                                                    ]
                                                }, 1
                                            ]
                                        }
                                    ]
                                }, True, False
                            ]
                        }
                    }
                }, {  # add controller status as online or offline
                    '$addFields': {
                        'status': {
                            '$cond': [
                                {
                                    '$or': [
                                        {
                                            '$and': [
                                                {
                                                    '$eq': [
                                                        '$olderThanR220', False
                                                    ]
                                                }, {
                                                    '$gte': [
                                                        '$Time', cntl_time_cutoff_r22
                                                    ]
                                                }
                                            ]
                                        }, {
                                            '$and': [
                                                {
                                                    '$ne': [
                                                        '$olderThanR220', False
                                                    ]
                                                }, {
                                                    '$gte': [
                                                        '$Time', cntl_time_cutoff_r21
                                                    ]
                                                }
                                            ]
                                        }
                                    ]
                                }, {
                                    'status': 'online'
                                }, {
                                    'status': 'offline'
                                }
                            ]
                        }
                    }
                }, {  # System Status object to array
                    '$project': {
                        '_id': 0,
                        'System Status': {
                            '$objectToArray': '$System Status'
                        },
                        'Status': '$status.status'
                    }
                }, {  # Each olt in System Status to their own document
                    '$unwind': {
                        'path': '$System Status',
                        'preserveNullAndEmptyArrays': False
                    }
                }, {  # Project to only have olt id, olt state, controller state
                    '$project': {
                        'olt': '$System Status.k',
                        'state': '$System Status.v.OLT State',
                        'cntl': '$Status'
                    }
                }, {  # online: olt online and cntl online; else offline
                    '$group': {
                        '_id': 0,
                        'online': {
                            '$addToSet': {
                                '$cond': [
                                    {
                                        '$and': [
                                            {
                                                '$or': [
                                                    {
                                                        '$eq': [
                                                            '$state', 'Primary'
                                                        ]
                                                    }, {
                                                        '$eq': [
                                                            '$state', 'Secondary'
                                                        ]
                                                    }, {
                                                        '$eq': [
                                                            '$state', 'Unspecified'
                                                        ]
                                                    }
                                                ]
                                            }, {
                                                '$eq': [
                                                    '$cntl', 'online'
                                                ]
                                            }
                                        ]
                                    }, '$olt', '$$REMOVE'
                                ]
                            }
                        },
                        'offline': {
                            '$addToSet': {
                                '$cond': [
                                    {
                                        '$and': [
                                            {
                                                '$ne': [
                                                    '$state', 'Unspecified'
                                                ]
                                            }, {
                                                '$ne': [
                                                    '$state', 'Primary'
                                                ]
                                            }, {
                                                '$ne': [
                                                    '$state', 'Secondary'
                                                ]
                                            }, {
                                                '$eq': [
                                                    '$cntl', 'online'
                                                ]
                                            }
                                        ]
                                    }, '$olt', {
                                        '$cond': [
                                            {
                                                '$eq': [
                                                    '$cntl', 'offline'
                                                ]
                                            }, '$olt', '$$REMOVE'
                                        ]
                                    }
                                ]
                            }
                        }
                    }
                }, {
                    '$project': {
                        'online list': '$online',
                        'offline list': '$offline'
                    }
                }
            ]))

            list_pre_pro = list(self.database["OLT-CFG"].aggregate([
                {
                    "$lookup": {
                        "from": "OLT-STATE",
                        "localField": "_id",
                        "foreignField": "_id",
                        "as": "matched"
                    }
                },
                {
                    "$match": {
                        "$and": [{
                            "matched": {"$eq": []},
                            # This performs the function to get the ids from config not in state
                            "_id": {"$ne": "Default"}
                        }]
                    }
                },
                {
                    '$project': {
                        '_id': 1
                    }
                }
            ]))

            pre_pro_onus = [doc['_id'] for doc in list_pre_pro]

            # Use the additional_minutes to get only the onus that have been offline for a certain amount of time.
            x_minutes_ago = str(datetime.datetime.utcnow() - datetime.timedelta(minutes=(additional_minutes + 2)))
            req_offline_time_olts = self.database['OLT-STATE'].find({'Time': {'$gt': x_minutes_ago}})
            req_time_ids = [doc['_id'] for doc in req_offline_time_olts]

            offline_list = get_nested_value(olt_counts, [0, "offline list"], 0)
            offline_list = [olt for olt in offline_list if olt not in pre_pro_onus and olt not in req_time_ids]

            response[1] = offline_list

        except Exception as err:
            self.status_log("EXCEPTION (get_offline_olts)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get all offline OLT IDs due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "CNTL-CFG", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)
        return response

    def get_offline_onus(self, user_email, minutes):
        response = [status.HTTP_200_OK, []]

        try:
            if minutes == 'undefined' or minutes == '':
                minutes = '0'
            additional_minutes = int(minutes)
            database_manager.get_database(self.database_id)
            cntl_time_cutoff_r21 = str(datetime.datetime.utcnow() - datetime.timedelta(minutes=6))
            cntl_time_cutoff_r22 = str(datetime.datetime.utcnow() - datetime.timedelta(minutes=2))

            onu_counts = list(self.database["CNTL-STATE"].aggregate([
                {  # add olderThanR220 field if cntl is R221 or older
                    '$addFields': {
                        'olderThanR220': {
                            '$cond': [
                                {
                                    '$or': [
                                        {
                                            '$eq': [
                                                {
                                                    '$indexOfCP': [
                                                        '$CNTL.Version', '2.1.'
                                                    ]
                                                }, 1
                                            ]
                                        }, {
                                            '$eq': [
                                                {
                                                    '$indexOfCP': [
                                                        '$CNTL.Version', '2.0'
                                                    ]
                                                }, 1
                                            ]
                                        }, {
                                            '$eq': [
                                                {
                                                    '$indexOfCP': [
                                                        '$CNTL.Version', '1.'
                                                    ]
                                                }, 1
                                            ]
                                        }
                                    ]
                                }, True, False
                            ]
                        }
                    }
                }, {  # add status online or offline for each controller
                    '$addFields': {
                        'status': {
                            '$cond': [
                                {
                                    '$or': [
                                        {
                                            '$and': [
                                                {
                                                    '$eq': [
                                                        '$olderThanR220', False
                                                    ]
                                                }, {
                                                    '$gte': [
                                                        '$Time', cntl_time_cutoff_r22
                                                    ]
                                                }
                                            ]
                                        }, {
                                            '$and': [
                                                {
                                                    '$ne': [
                                                        '$olderThanR220', False
                                                    ]
                                                }, {
                                                    '$gte': [
                                                        '$Time', cntl_time_cutoff_r21
                                                    ]
                                                }
                                            ]
                                        }
                                    ]
                                }, {
                                    'status': 'online'
                                }, {
                                    'status': 'offline'
                                }
                            ]
                        }
                    }
                }, {  # project only system status and cntl status. System status obj to array
                    '$project': {
                        '_id': 0,
                        'System Status': {
                            '$objectToArray': '$System Status'
                        },
                        'Status': '$status.status'
                    }
                }, {  # every olt in system status array becomes its own object
                    '$unwind': {
                        'path': '$System Status',
                        'preserveNullAndEmptyArrays': False
                    }
                }, {  # onu objects turned into array of onus
                    '$project': {
                        'onu': {
                            '$objectToArray': '$System Status.v.ONUs'
                        },
                        'cntl status': '$Status'
                    }
                }, {  # separate documents for each onu
                    '$unwind': {
                        'path': '$onu',
                        'preserveNullAndEmptyArrays': False
                    }
                }, {  # if online cntl and registered/unspecified/Fw upgrade add to online set. Else add to offline set
                    '$group': {
                        '_id': 0,
                        'online': {
                            '$addToSet': {
                                '$cond': [
                                    {
                                        '$and': [
                                            {
                                                '$or': [
                                                    {
                                                        '$eq': [
                                                            '$onu.v', 'Registered'
                                                        ]
                                                    }, {
                                                        '$eq': [
                                                            '$onu.v', 'Unspecified'
                                                        ]
                                                    }, {
                                                        '$eq': [
                                                            '$onu.v', 'FW Upgrade'
                                                        ]
                                                    }
                                                ]
                                            }, {
                                                '$eq': [
                                                    '$cntl status', 'online'
                                                ]
                                            }
                                        ]
                                    }, '$onu.k', '$$REMOVE'
                                ]
                            }
                        },
                        'offline': {
                            '$addToSet': {
                                '$cond': [
                                    {
                                        '$and': [
                                            {
                                                '$ne': [
                                                    '$onu.v', 'Registered'
                                                ]
                                            }, {
                                                '$ne': [
                                                    '$onu.v', 'Unspecified'
                                                ]
                                            }, {
                                                '$eq': [
                                                    '$cntl status', 'online'
                                                ]
                                            }, {
                                                '$ne': [
                                                    '$onu.v', 'FW Upgrade'
                                                ]
                                            }
                                        ]
                                    }, '$onu.k', {
                                        '$cond': [
                                            {
                                                '$eq': [
                                                    '$cntl status', 'offline'
                                                ]
                                            }, '$onu.k', '$$REMOVE'
                                        ]
                                    }
                                ]
                            }
                        }
                    }
                }, {
                    '$project': {
                        'online list': '$online',
                        'offline list': '$offline'
                    }
                }
            ]))

            list_pre_pro = list(self.database["ONU-CFG"].aggregate([
                {
                    "$lookup": {
                        "from": "ONU-STATE",
                        "localField": "_id",
                        "foreignField": "_id",
                        "as": "matched"
                    }
                },
                {
                    "$match": {
                        "$and": [{
                            "matched": {"$eq": []},
                            # This performs the function to get the ids from config not in state
                            "_id": {"$ne": "Default"}
                        }]
                    }
                },
                {
                    '$project': {
                        '_id': 1
                    }
                }
            ]))

            offline_list = get_nested_value(onu_counts, [0, "offline list"], 0)
            online_list = get_nested_value(onu_counts, [0, "online list"], 0)
            # Remove pre-provisioned ONUs attached to an OLT from the offline count
            dereg_pre_pro = [onu['_id'] for onu in list_pre_pro if onu['_id'] in offline_list]
            # Remove any ids that are in the dereg_pre_pro list.
            offline_onus = [onu for onu in offline_list if onu not in dereg_pre_pro]
            all_onu_states = self.database["ONU-STATE"].find({}, {'_id': 1})
            onu_states_list = [doc['_id'] for doc in all_onu_states]

            # Remove unattached from offline list
            unattached_onus = [onu for onu in onu_states_list if onu not in offline_onus and onu not in online_list]
            offline_onus = [onu for onu in offline_onus if onu not in unattached_onus]

            # Use the additional_minutes to get only the onus that have been offline for a certain amount of time.
            x_minutes_ago = str(datetime.datetime.utcnow() - datetime.timedelta(minutes=(additional_minutes + 2)))
            req_offline_time_olts = self.database['ONU-STATE'].find({'Time': {'$gt': x_minutes_ago}})
            req_time_ids = [doc['_id'] for doc in req_offline_time_olts]
            offline_onus = [onu for onu in offline_onus if onu not in req_time_ids]

            response[1] = offline_onus

        except Exception as err:
            self.status_log("EXCEPTION (get_offline_onus)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get all offline ONUs due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "CNTL-CFG, ONU-CFG, ONU-STATE", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    def get_pre_provisioned_onus(self, user_email):
        response = [status.HTTP_200_OK, []]
        try:
            database_manager.get_database(self.database_id)
            list_pre_pro = list(self.database["ONU-CFG"].aggregate([
                {
                    "$lookup": {
                        "from": "ONU-STATE",
                        "localField": "_id",
                        "foreignField": "_id",
                        "as": "matched"
                    }
                },
                {
                    "$match": {
                        "$and": [{
                            "matched": {"$eq": []},
                            # This performs the function to get the ids from config not in state
                            "_id": {"$ne": "Default"}
                        }]
                    }
                },
                {
                    '$project': {
                        '_id': 1
                    }
                }
            ]))

            pre_pro_onus = [doc['_id'] for doc in list_pre_pro]
            response[1] = pre_pro_onus

        except Exception as err:
            self.status_log("EXCEPTION (get_pre_provisioned_onus)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get Pre-Provisioned ONUs due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "ONU-STATE, ONU-CFG", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    def get_pre_provisioned_olts(self, user_email):
        response = [status.HTTP_200_OK, {}]
        try:
            database_manager.get_database(self.database_id)

            # Find pre-provisioned by getting OLT-CFG ids that don't have a state.
            list_pre_pro = list(self.database["OLT-CFG"].aggregate([
                {
                    "$lookup": {
                        "from": "OLT-STATE",
                        "localField": "_id",
                        "foreignField": "_id",
                        "as": "matched"
                    }
                },
                {
                    "$match": {
                        "$and": [{
                            "matched": {"$eq": []},
                            # This performs the function to get the ids from config not in state
                            "_id": {"$ne": "Default"}
                        }]
                    }
                },
                {
                    '$project': {
                        '_id': 1
                    }
                }
            ]))

            pre_pro_olts = [doc['_id'] for doc in list_pre_pro]
            response[1] = pre_pro_olts

        except Exception as err:
            self.status_log("EXCEPTION (get_pre_provisioned_olts)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get Pre-Provisioned OLTs due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "OLT-CFG, OLT-STATE", "returned": response[1], "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    # Retrieve available OLTs to be selected as a Protection Peer
    def get_all_available_peer_olt_ids(self, user_email, olt_id):
        response = [status.HTTP_200_OK, []]
        check_casc_olt = []
        cascading_groups = []

        try:
            database_manager.get_database(self.database_id)
            cntl_cfg_collection = self.database["CNTL-CFG"]

            # Getting controller that the requesting OLT is inventoried on
            managing_cntl_inventory = list(cntl_cfg_collection.aggregate([
                {"$match": {"OLTs.Primary": olt_id}}]))

            # Get All OLT IDs that are on same controller inventory as requesting OLT, and have the field...
            # ...Protection.Peer set to A string of length 0
            if len(managing_cntl_inventory) > 0:
                all_inventoried_olts = managing_cntl_inventory[0]['OLTs']['Primary']

                olt_cfg_collection = self.database["OLT-CFG"]
                available_peer_olt_ids_cursor = olt_cfg_collection.find(
                    {"_id": {"$in": all_inventoried_olts}, "Protection.Peer": ''}, {'_id': 1})

                for id in available_peer_olt_ids_cursor:
                    # Not allowed to provision a protection peer as yourself
                    if id['_id'] != olt_id:
                        check_casc_olt.append(id['_id'])

                # Find all OLT MAC Addresses under Cascading Groups
                cascading_groups_cursor = cntl_cfg_collection.aggregate([
                    {
                        '$project': {
                            'groups': {
                                '$objectToArray': '$Cascading.Groups'
                            }
                        }
                    }, {
                        '$unwind': {
                            'path': '$groups',
                            'preserveNullAndEmptyArrays': False
                        }
                    }, {
                        '$unwind': {
                            'path': '$groups.v.Members',
                            'includeArrayIndex': 'string',
                            'preserveNullAndEmptyArrays': False
                        }
                    }, {
                        '$group': {
                            '_id': '_id',
                            'Members': {
                                '$addToSet': '$groups.v.Members'
                            }
                        }
                    }
                ])

                # Assign cascading group OLTs
                for group in cascading_groups_cursor:
                    cascading_groups = group['Members']

                # Not allowed to add OLTs that are in a Cascading Group
                if len(cascading_groups) > 0:
                    for member in cascading_groups:
                        if member not in check_casc_olt:
                            response[1].append(member)

                # If no Cascading Groups exist, return available OLTs
                else:
                    for olt in check_casc_olt:
                        response[1].append(olt)

        except Exception as err:
            self.status_log("EXCEPTION (get_all_available_peer_olt_ids)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get all available Protection Peer OLT IDs due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "OLT-CFG", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    # Set PON Protection Automatic switchover to be Armed or Disarmed
    def put_protection_automatic_switchover(self, user_email, mac_address, auto_switchover_status):
        response = [status.HTTP_200_OK, f"OLT {mac_address} PON Protection Automatic Switchover status was updated.",
                    None, {}]

        # Must have peer configured for PON Controller to pickup action
        try:
            database_manager.get_database(self.database_id)
            olt_cfg_collection = self.database['OLT-CFG']
            olt = olt_cfg_collection.find_one({"_id": mac_address}, {"_id": 1, "Protection": 1})
            if olt is not None:
                olt_update_document = {"$set": {"Protection.Watch": auto_switchover_status},
                                       "$inc": {"Protection.Watch Count": 1, "OLT.CFG Change Count": 1}}
                peer_update_document = {"$set": {"Protection.Watch": auto_switchover_status},
                                        "$inc": {"Protection.Watch Count": 1, "OLT.CFG Change Count": 1}}
                olt_cfg_collection.update_one(filter={"_id": mac_address}, update=olt_update_document)
                olt_cfg_collection.update_one(filter={"_id": olt["Protection"]["Peer"]}, update=peer_update_document)
                # Compare old to new values for logging changes
                new_doc = {
                    mac_address: olt_update_document,
                    olt["Protection"]["Peer"]: peer_update_document
                }
                response[2] = new_doc
        except Exception as err:
            self.status_log("EXCEPTION (put_protection_automatic_switchover)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not update OLT {mac_address} PON Protection Automatic Switchover status due to {type(err).__name__}::{sys.exc_info()[1]}",
                        None, None]

        action_data = {"collection": "OLT-CFG",
                       "old": "", "new": auto_switchover_status, "user": user_email}
        self.sys_log(action_data, "write", response[0] == status.HTTP_200_OK or response[0] == status.HTTP_201_CREATED)

        return response

    # Set PON Protection Forced switchover to be Armed or Disarmed
    def put_protection_forced_switchover(self, user_email, mac_address):
        response = [status.HTTP_200_OK, f"OLT {mac_address} PON Protection Forced Switchover status was updated.", None,
                    {}]

        # Must have peer configured and Protection.Watch set to Disabled for PON Controller to pickup action
        try:
            database_manager.get_database(self.database_id)
            olt_cfg_collection = self.database['OLT-CFG']
            update_document = {"$inc": {"Protection.Switchover Count": 1, "OLT.CFG Change Count": 1}}
            olt_cfg_collection.update_one({"_id": mac_address}, update_document, upsert=True)
            response[2] = update_document
        except Exception as err:
            self.status_log("EXCEPTION (put_protection_forced_switchover)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not update OLT {mac_address} PON Protection Forced Switchover status due to {type(err).__name__}::{sys.exc_info()[1]}",
                        None, None]

        action_data = {"collection": "OLT-CFG",
                       "old": "", "new": "", "user": user_email}
        self.sys_log(action_data, "write", response[0] == status.HTTP_200_OK or response[0] == status.HTTP_201_CREATED)

        return response

    # Set PON Protection Forced switchover to be Armed or Disarmed
    def put_protection_cfg(self, user_email, mac_address, peer, other_configs):
        response = [status.HTTP_200_OK, f"OLTs {mac_address} and {peer} PON Protection Configurations were updated.",
                    None, {}]

        try:
            database_manager.get_database(self.database_id)
            # Update current OLT config
            olt_cfg_collection = self.database['OLT-CFG']
            selected_olt_config = olt_cfg_collection.find_one({"_id": mac_address}, {"_id": 1, "Protection": 1})

            # Changes to documents
            selected_olt_values = {"Protection.Peer": peer}
            peer_olt_values = {
                "Protection.Peer": mac_address,
                "Protection.Watch": selected_olt_config["Protection"]['Watch'],
            }
            freeing_old_olt_peer_values = {
                "Protection.Peer": '',
                "Protection.Protection.Watch": 'Disabled',
            }

            # Other changes to documents
            if other_configs is not None and other_configs != {}:
                selected_olt_values.update({
                    "Protection.ONU Fail Limit": other_configs['Protection.ONU Fail Limit'],
                    "Protection.NNI RX LOS Pulse": other_configs['Protection.NNI RX LOS Pulse'],
                })
                peer_olt_values.update({
                    "Protection.ONU Fail Limit": other_configs['Protection.ONU Fail Limit'],
                    "Protection.NNI RX LOS Pulse": other_configs['Protection.NNI RX LOS Pulse'],
                })

            olt_cfg_collection.update_one({"_id": mac_address}, {"$set": selected_olt_values, "$inc": {"OLT.CFG Change Count": 1}})

            # Setting config of paired OLT peer
            if peer != '':
                olt_cfg_collection.update_one({"_id": peer}, {"$set": peer_olt_values, "$inc": {"OLT.CFG Change Count": 1}})

            # If removing peer or changing to a different peer, we want to free up the partner peer
            if peer == '' or peer != selected_olt_config['Protection']['Peer']:
                olt_cfg_collection.update_one({"_id": selected_olt_config['Protection']['Peer']}, {"$set": freeing_old_olt_peer_values, "$inc": {"OLT.CFG Change Count": 1}})

            response[2] = {
                mac_address: selected_olt_values,
                peer: peer_olt_values,
                selected_olt_config['Protection']['Peer']: freeing_old_olt_peer_values
            }
        except Exception as err:
            self.status_log("EXCEPTION (put_protection_forced_switchover)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not update OLTs {mac_address} and {peer} PON Protection configurations due to {type(err).__name__}::{sys.exc_info()[1]}",
                        None, None]

        action_data = {"collection": "OLT-CFG",
                       "old": "", "new": "", "user": user_email}
        self.sys_log(action_data, "write", response[0] == status.HTTP_200_OK or response[0] == status.HTTP_201_CREATED)

        return response

    # Retrieve all onu config ids
    def get_all_onu_cfg_ids(self, user_email):
        response = [status.HTTP_200_OK, []]

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["ONU-CFG"]
            cfg_ids_cursor = collection.find({}, {"_id": 1})
            for id in cfg_ids_cursor:
                response[1].append(id['_id'])
        except Exception as err:
            self.status_log("EXCEPTION (get_all_onu_cfg_ids)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get all ONU configuration IDs due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "ONU-CFG", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    """
    ======================================================================
    =========================== PON Automation ===========================
    ======================================================================
    """

    # Retrieve automation configs for ONUs under the OLT
    def get_onu_automation_cfgs_for_olt(self, olt_id, user_email):
        response = [status.HTTP_200_OK, []]

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["OLT-STATE"]
            response[1] = list(collection.aggregate([
                {
                    '$match': {
                        '_id': olt_id
                    }
                }, {
                    '$project': {
                        'ONUs': {
                            '$concatArrays': [
                                '$ONU States.Registered', '$ONU States.Unspecified', '$ONU States.Unprovisioned'
                            ]
                        }
                    }
                }, {
                    '$lookup': {
                        'from': 'ONU-AUTO-CFG',
                        'localField': 'ONUs',
                        'foreignField': '_id',
                        'as': 'results'
                    }
                }, {
                    '$project': {
                        'ONU Auto CFGs': '$results',
                        '_id': 0
                    }
                }
            ]))[0]["ONU Auto CFGs"]
        except Exception as err:
            self.status_log("EXCEPTION (get_onu_automation_cfgs_for_olt)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get onu automation configs for olt due to {type(err).__name__}::{sys.exc_info()[1]}"]

        return response

    # Retrieve automation states for ONUs under the OLT
    def get_onu_automation_states_for_olt(self, olt_id, projection, user_email):
        response = [status.HTTP_200_OK, []]

        try:
            database_manager.get_database(self.database_id)
            olt_state_coll = self.database["OLT-STATE"]
            olt_state = olt_state_coll.find_one({"_id": olt_id})
            onu_ids = []
            if olt_state is not None and "ONU States" in olt_state:
                olt_onu_states = olt_state["ONU States"]
                if "Registered" in olt_onu_states:
                    onu_ids += olt_onu_states["Registered"]
                if "Unspecified" in olt_onu_states:
                    onu_ids += olt_onu_states["Unspecified"]
                if "Unprovisioned" in olt_onu_states:
                    onu_ids += olt_onu_states["Unprovisioned"]

            onu_auto_state_coll = self.database["ONU-AUTO-STATE"]
            response[1] = list(onu_auto_state_coll.find({"_id": {"$in": onu_ids}}, projection))
        except Exception as err:
            self.status_log("EXCEPTION (get_automation_state)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get ONU Automation States for OLT {olt_id} due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "ONU-AUTO-STATE", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    # Retrieve specified Automation config
    def get_automation_state(self, device_type, device_id, user_email):
        response = [status.HTTP_200_OK, []]
        device_type = str(device_type).upper()

        try:
            database_manager.get_database(self.database_id)
            collection = self.database[f"{device_type}-AUTO-STATE"]
            document = collection.find_one({"_id": device_id})

            if document is not None:
                response[1] = document
            else:
                response[1] = "Not found"
        except Exception as err:
            self.status_log("EXCEPTION (get_automation_state)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get {device_type} {device_id} Automation State due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": f"{device_type}-AUTO-CFG", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    # Retrieve specified Automation config
    def get_automation_cfg(self, device_type, template_id, user_email):
        response = [status.HTTP_200_OK, []]
        device_type = str(device_type).upper()

        try:
            database_manager.get_database(self.database_id)
            collection = self.database[f"{device_type}-AUTO-CFG"]
            document = collection.find_one({"_id": template_id})

            if document is not None:
                response[1] = document
            else:
                response[1] = "Not found"
        except Exception as err:
            self.status_log("EXCEPTION (get_automation_cfg)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get {device_type} Automation configuration due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": f"{device_type}-AUTO-CFG", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    # Update specified Automation config
    def update_automation_cfg(self, device_type, template_id, new_document, user_email):
        response = [status.HTTP_200_OK, ["Document updated"]]
        old = None
        device_type = str(device_type).upper()

        try:
            database_manager.get_database(self.database_id)
            collection = self.database[f"{device_type}-AUTO-CFG"]
            old = collection.find_one({'id': template_id})
            if "_id" in new_document:
                del new_document['_id']

            updated = collection.replace_one({'_id': template_id}, new_document, upsert=True)
            if updated.upserted_id == template_id:
                response[0] = status.HTTP_201_CREATED
            elif updated.matched_count == 1:
                response[0] = status.HTTP_200_OK
            else:
                response[0] = status.HTTP_500_INTERNAL_SERVER_ERROR
                response[1] = f"Could not update {template_id} {device_type} Automation. No documents were modified."
        except Exception as err:
            self.status_log("EXCEPTION (update_automation_cfg)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not update {device_type} Automation configuration due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": f"{device_type}-AUTO-CFG", "user": user_email, "old": old, "new": new_document}
        self.sys_log(action_data, "put", response[0] == status.HTTP_200_OK)

        return response

    # Delete the specified automation configuration document
    def delete_automation_cfg(self, device_type, template_id, user_email):
        response = [status.HTTP_200_OK, f"{template_id} ONU Automation was deleted."]
        device_type = str(device_type).upper()

        try:
            database_manager.get_database(self.database_id)
            collection = self.database[f"{device_type.upper()}-AUTO-CFG"]
            collection.delete_one({'_id': template_id})
        except Exception as err:
            self.status_log("EXCEPTION (delete_automation_cfg)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not {device_type} Automation Configuration {template_id} due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": f"{device_type}-AUTO-CFG", "deleted": response[0] == status.HTTP_200_OK,
                       "user": user_email}
        self.sys_log(action_data, "delete", response[0] == status.HTTP_200_OK)

        return response

    # Delete the specified reference from the Global mappings for the given device type
    def delete_automation_reference(self, device_type, automation_step, reference_name, user_email):
        response = [status.HTTP_200_OK, f"All ONU  data was deleted."]
        device_type = str(device_type).upper()

        try:
            database_manager.get_database(self.database_id)
            collection = self.database[f"{device_type}-AUTO-CFG"]
            if automation_step.upper() == "IDENTIFY":
                # Controller template uses Version as key, others use Description
                if device_type == "CNTL":
                    collection.update_one({'_id': "Global"},
                                          {"$pull": {"IDENTIFY.Mapping": {"[CNTL-STATE][CNTL][Version]": reference_name}}})
                else:
                    collection.update_one({'_id': "Global"},
                                          {"$pull": {"IDENTIFY.Mapping": {"Description": reference_name}}})
            elif automation_step.upper() == "SERVICE_PKG" and device_type == "ONU":
                collection.update_one({'_id': "Global"}, {"$unset": {
                    f"SERVICE.PACKAGES.{reference_name}": "",
                    f"AUTH.{reference_name}": "",
                    f"VLAN.{reference_name}": "",
                    f"SLA.{reference_name}": "",
                    f"DHCP.{reference_name}": ""
                }, })
            else:
                collection.update_one({'_id': "Global"}, {"$unset": {f"{automation_step.upper()}.{reference_name}": ""}})
        except Exception as err:
            self.status_log("EXCEPTION (delete_automation_reference)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not delete {device_type} Automation {automation_step.upper()} {reference_name} due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": f"{device_type}-AUTO-CFG", "deleted": response[0] == status.HTTP_200_OK,
                       "user": user_email}
        self.sys_log(action_data, "delete", response[0] == status.HTTP_200_OK)

        return response

    # Retrieve all values of specific state fields
    def get_distinct_onu_state_identifiers(self, user_email):
        response = [status.HTTP_200_OK,
                    {"ONU.Manufacturer": [], "ONU.Model": [], "ONU.Vendor": [], "ONU.HW Version": [],
                     "ONU.Equipment ID": []}]

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["ONU-STATE"]
            response[1]["ONU.Manufacturer"] = collection.distinct("ONU.Manufacturer")
            response[1]["ONU.Model"] = collection.distinct("ONU.Model")
            response[1]["ONU.Vendor"] = collection.distinct("ONU.Vendor")
            response[1]["ONU.HW Version"] = collection.distinct("ONU.HW Version")
            response[1]["ONU.Equipment ID"] = collection.distinct("ONU.Equipment ID")
        except Exception as err:
            self.status_log("EXCEPTION (get_distinct_onu_state_identifiers)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get distinct ONU state fields due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "ONU-STATE", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    # Retrieve all values of specific state fields
    def get_distinct_olt_state_identifiers(self, user_email):
        response = [status.HTTP_200_OK, {"OLT.Manufacturer": [], "OLT.Model": [], "OLT.Vendor": []}]

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["OLT-STATE"]
            response[1]["OLT.Manufacturer"] = collection.distinct("OLT.Manufacturer")
            response[1]["OLT.Model"] = collection.distinct("OLT.Model")
            response[1]["OLT.Vendor"] = collection.distinct("OLT.Vendor")
        except Exception as err:
            self.status_log("EXCEPTION (get_distinct_olt_state_identifiers)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get distinct OLT state fields due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "OLT-STATE", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    def get_switch_automation_counts(self, user_email):
        response = [status.HTTP_200_OK, {}]

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["SWI-AUTO-STATE"]
            response[1] = list(collection.aggregate([
                {
                    '$project': {
                        'status.Id': '$_id',
                        'status.Task': '$AUTO.TASK',
                        'status.Status': '$AUTO.STATUS',
                        'status.Enable': '$AUTO.Enable'
                    }
                }, {
                    '$group': {
                        '_id': 0,
                        'data': {
                            '$addToSet': '$status'
                        }
                    }
                }, {
                    '$project': {
                        '_id': 0,
                        'data': 1,
                        'DONE': {
                            '$size': {
                                '$filter': {
                                    'input': '$data',
                                    'as': 'index',
                                    'cond': {
                                        '$and': [
                                            {
                                                '$eq': [
                                                    '$$index.Status', 'DONE'
                                                ]
                                            }, {
                                                '$eq': [
                                                    '$$index.Enable', True
                                                ]
                                            }
                                        ]
                                    }
                                }
                            }
                        },
                        'ERROR': {
                            '$size': {
                                '$filter': {
                                    'input': '$data',
                                    'as': 'index',
                                    'cond': {
                                        '$and': [
                                            {
                                                '$eq': [
                                                    '$$index.Status', 'ERROR'
                                                ]
                                            }, {
                                                '$eq': [
                                                    '$$index.Enable', True
                                                ]
                                            }
                                        ]
                                    }
                                }
                            }
                        },
                        'WAITING FOR DEVICE': {
                            '$size': {
                                '$filter': {
                                    'input': '$data',
                                    'as': 'index',
                                    'cond': {
                                        '$and': [
                                            {
                                                '$eq': [
                                                    '$$index.Status', 'WAITING FOR DEVICE'
                                                ]
                                            }, {
                                                '$eq': [
                                                    '$$index.Enable', True
                                                ]
                                            }
                                        ]
                                    }
                                }
                            }
                        },
                        'WAITING FOR INPUT': {
                            '$size': {
                                '$filter': {
                                    'input': '$data',
                                    'as': 'index',
                                    'cond': {
                                        '$and': [
                                            {
                                                '$eq': [
                                                    '$$index.Status', 'WAITING FOR INPUT'
                                                ]
                                            }, {
                                                '$eq': [
                                                    '$$index.Enable', True
                                                ]
                                            }
                                        ]
                                    }
                                }
                            }
                        },
                        'ENABLED': {
                            '$size': {
                                '$filter': {
                                    'input': '$data',
                                    'as': 'index',
                                    'cond': {
                                        '$eq': [
                                            '$$index.Enable', True
                                        ]
                                    }
                                }
                            }
                        },
                        'DISABLED': {
                            '$size': {
                                '$filter': {
                                    'input': '$data',
                                    'as': 'index',
                                    'cond': {
                                        '$eq': [
                                            '$$index.Enable', False
                                        ]
                                    }
                                }
                            }
                        }
                    }
                }
            ]))[0]
        except Exception as err:
            self.status_log("EXCEPTION (get_switch_automation_counts)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get switch automation counts due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "AUTO-STATE", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    def get_cntl_automation_counts(self, user_email):
        response = [status.HTTP_200_OK, {}]

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["CNTL-AUTO-STATE"]
            response[1] = list(collection.aggregate([
                {
                    '$project': {
                        'status.Id': '$_id',
                        'status.Task': '$AUTO.TASK',
                        'status.Status': '$AUTO.STATUS',
                        'status.Enable': '$AUTO.Enable'
                    }
                }, {
                    '$group': {
                        '_id': 0,
                        'data': {
                            '$addToSet': '$status'
                        }
                    }
                }, {
                    '$project': {
                        '_id': 0,
                        'data': 1,
                        'DONE': {
                            '$size': {
                                '$filter': {
                                    'input': '$data',
                                    'as': 'index',
                                    'cond': {
                                        '$and': [
                                            {
                                                '$eq': [
                                                    '$$index.Status', 'DONE'
                                                ]
                                            }, {
                                                '$eq': [
                                                    '$$index.Enable', True
                                                ]
                                            }
                                        ]
                                    }
                                }
                            }
                        },
                        'ERROR': {
                            '$size': {
                                '$filter': {
                                    'input': '$data',
                                    'as': 'index',
                                    'cond': {
                                        '$and': [
                                            {
                                                '$eq': [
                                                    '$$index.Status', 'ERROR'
                                                ]
                                            }, {
                                                '$eq': [
                                                    '$$index.Enable', True
                                                ]
                                            }
                                        ]
                                    }
                                }
                            }
                        },
                        'WAITING FOR DEVICE': {
                            '$size': {
                                '$filter': {
                                    'input': '$data',
                                    'as': 'index',
                                    'cond': {
                                        '$and': [
                                            {
                                                '$eq': [
                                                    '$$index.Status', 'WAITING FOR DEVICE'
                                                ]
                                            }, {
                                                '$eq': [
                                                    '$$index.Enable', True
                                                ]
                                            }
                                        ]
                                    }
                                }
                            }
                        },
                        'WAITING FOR INPUT': {
                            '$size': {
                                '$filter': {
                                    'input': '$data',
                                    'as': 'index',
                                    'cond': {
                                        '$and': [
                                            {
                                                '$eq': [
                                                    '$$index.Status', 'WAITING FOR INPUT'
                                                ]
                                            }, {
                                                '$eq': [
                                                    '$$index.Enable', True
                                                ]
                                            }
                                        ]
                                    }
                                }
                            }
                        },
                        'ENABLED': {
                            '$size': {
                                '$filter': {
                                    'input': '$data',
                                    'as': 'index',
                                    'cond': {
                                        '$eq': [
                                            '$$index.Enable', True
                                        ]
                                    }
                                }
                            }
                        },
                        'DISABLED': {
                            '$size': {
                                '$filter': {
                                    'input': '$data',
                                    'as': 'index',
                                    'cond': {
                                        '$eq': [
                                            '$$index.Enable', False
                                        ]
                                    }
                                }
                            }
                        }
                    }
                }
            ]))[0]
        except Exception as err:
            self.status_log("EXCEPTION (get_cntl_automation_counts)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get controller automation counts due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "AUTO-STATE", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    def get_olt_automation_counts(self, user_email):
        response = [status.HTTP_200_OK, {}]

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["OLT-AUTO-STATE"]
            counts = list(collection.aggregate([
                {
                    '$project': {
                        'status.Id': '$_id',
                        'status.Task': '$AUTO.TASK',
                        'status.Status': '$AUTO.STATUS',
                        'status.Enable': '$AUTO.Enable'
                    }
                }, {
                    '$lookup': {
                        'from': 'OLT-STATE',
                        'let': {
                            'id': '$status.Id'
                        },
                        'pipeline': [
                            {
                                '$match': {
                                    '$expr': {
                                        '$eq': [
                                            '$_id', '$$id'
                                        ]
                                    }
                                }
                            },
                            {
                                '$project': {
                                    '_id': 0,
                                    'CNTL.MAC Address': 1,
                                    'Switch.Chassis ID': 1
                                }
                            }
                        ],
                        'as': 'OLT-STATE'
                    }
                }, {
                    '$project': {
                        'status.Id': 1,
                        'status.Task': 1,
                        'status.Status': 1,
                        'status.Enable': 1,
                        'status.Parents': {'$arrayElemAt': ['$OLT-STATE', 0]}
                    }
                }, {
                    '$group': {
                        '_id': 0,
                        'data': {
                            '$addToSet': '$status'
                        }
                    }
                }, {
                    '$project': {
                        '_id': 0,
                        'data': 1,
                        'DONE': {
                            '$size': {
                                '$filter': {
                                    'input': '$data',
                                    'as': 'index',
                                    'cond': {
                                        '$and': [
                                            {
                                                '$eq': [
                                                    '$$index.Status', 'DONE'
                                                ]
                                            }, {
                                                '$eq': [
                                                    '$$index.Enable', True
                                                ]
                                            }
                                        ]
                                    }
                                }
                            }
                        },
                        'ERROR': {
                            '$size': {
                                '$filter': {
                                    'input': '$data',
                                    'as': 'index',
                                    'cond': {
                                        '$and': [
                                            {
                                                '$eq': [
                                                    '$$index.Status', 'ERROR'
                                                ]
                                            }, {
                                                '$eq': [
                                                    '$$index.Enable', True
                                                ]
                                            }
                                        ]
                                    }
                                }
                            }
                        },
                        'WAITING FOR DEVICE': {
                            '$size': {
                                '$filter': {
                                    'input': '$data',
                                    'as': 'index',
                                    'cond': {
                                        '$and': [
                                            {
                                                '$eq': [
                                                    '$$index.Status', 'WAITING FOR DEVICE'
                                                ]
                                            }, {
                                                '$eq': [
                                                    '$$index.Enable', True
                                                ]
                                            }
                                        ]
                                    }
                                }
                            }
                        },
                        'WAITING FOR INPUT': {
                            '$size': {
                                '$filter': {
                                    'input': '$data',
                                    'as': 'index',
                                    'cond': {
                                        '$and': [
                                            {
                                                '$eq': [
                                                    '$$index.Task', 'WAITING FOR INPUT'
                                                ]
                                            }, {
                                                '$eq': [
                                                    '$$index.Enable', True
                                                ]
                                            }
                                        ]
                                    }
                                }
                            }
                        },
                        'ENABLED': {
                            '$size': {
                                '$filter': {
                                    'input': '$data',
                                    'as': 'index',
                                    'cond': {
                                        '$eq': [
                                            '$$index.Enable', True
                                        ]
                                    }
                                }
                            }
                        },
                        'DISABLED': {
                            '$size': {
                                '$filter': {
                                    'input': '$data',
                                    'as': 'index',
                                    'cond': {
                                        '$eq': [
                                            '$$index.Enable', False
                                        ]
                                    }
                                }
                            }
                        }
                    }
                }
            ]))
            response[1] = counts[0]
        except Exception as err:
            self.status_log("EXCEPTION (get_olt_automation_counts)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get olt automation counts due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "AUTO-STATE", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    def get_onu_automation_counts(self, user_email):
        response = [status.HTTP_200_OK, {}]

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["ONU-AUTO-STATE"]
            data = list(collection.aggregate([
                {
                    '$project': {
                        '_id': 0,
                        'Id': '$_id',
                        'Task': '$AUTO.TASK',
                        'Status': '$AUTO.STATUS',
                        'Enable': '$AUTO.Enable',
                    }
                }
            ]))
            counts = list(collection.aggregate([
                {
                    '$project': {
                        'status.Id': '$_id',
                        'status.Task': '$AUTO.TASK',
                        'status.Status': '$AUTO.STATUS',
                        'status.Enable': '$AUTO.Enable'
                    }
                }, {
                    '$group': {
                        '_id': 0,
                        'DONE': {
                            '$sum': {
                                '$cond': [
                                    {
                                        '$and': [
                                            {
                                                '$eq': [
                                                    '$status.Status', 'DONE'
                                                ]
                                            }, {
                                                '$eq': [
                                                    '$status.Enable', True
                                                ]
                                            }
                                        ]
                                    }, 1, 0
                                ]
                            }
                        },
                        'ERROR': {
                            '$sum': {
                                '$cond': [
                                    {
                                        '$and': [
                                            {
                                                '$eq': [
                                                    '$status.Status', 'ERROR'
                                                ]
                                            }, {
                                                '$eq': [
                                                    '$status.Enable', True
                                                ]
                                            }
                                        ]
                                    }, 1, 0
                                ]
                            }
                        },
                        'WAITING FOR DEVICE': {
                            '$sum': {
                                '$cond': [
                                    {
                                        '$and': [
                                            {
                                                '$eq': [
                                                    '$status.Status', 'WAITING FOR DEVICE'
                                                ]
                                            }, {
                                                '$eq': [
                                                    '$status.Enable', True
                                                ]
                                            }
                                        ]
                                    }, 1, 0
                                ]
                            }
                        },
                        'WAITING FOR INPUT': {
                            '$sum': {
                                '$cond': [
                                    {
                                        '$and': [
                                            {
                                                '$eq': [
                                                    '$status.Status', 'WAITING FOR INPUT'
                                                ]
                                            }, {
                                                '$eq': [
                                                    '$status.Enable', True
                                                ]
                                            }
                                        ]
                                    }, 1, 0
                                ]
                            }
                        },
                        'ENABLED': {
                            '$sum': {
                                '$cond': [
                                    {
                                        '$eq': [
                                            '$status.Enable', True
                                        ]
                                    }, 1, 0
                                ]
                            }
                        },
                        'DISABLED': {
                            '$sum': {
                                '$cond': [
                                    {
                                        '$eq': [
                                            '$status.Enable', False
                                        ]
                                    }, 1, 0
                                ]
                            }
                        }
                    }
                }
            ]))
            counts = counts[0]
            counts["data"] = data
            response[1] = counts
        except Exception as err:
            self.status_log("EXCEPTION (get_onu_automation_counts)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get switch automation counts due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "AUTO-STATE", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    def get_onu_vcm_counts(self, user_email):
        response = [status.HTTP_200_OK, {}]

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["VCM-STATE"]
            counts = list(collection.aggregate([
                {
                    '$project': {
                        'status.Id': '$_id',
                        'status.Status': '$VCM.State',
                    }
                }, {
                    '$group': {
                        '_id': 0,
                        'Operational': {
                            '$sum': {
                                '$cond': [
                                    {
                                        '$eq': [
                                            '$status.Status', 'Operational'
                                        ]
                                    }, 1, 0
                                ]
                            }
                        },
                        'Registering': {
                            '$sum': {
                                '$cond': [
                                    {
                                        '$eq': [
                                            '$status.Status', 'Start Registration'
                                        ]
                                    }, 1, 0
                                ]
                            }
                        },
                        'TFTP': {
                            '$sum': {
                                '$cond': [
                                    {
                                        '$eq': [
                                            '$status.Status', 'Start TFTP'
                                        ]
                                    }, 1, 0
                                ]
                            }
                        },
                        'DHCP': {
                            '$sum': {
                                '$cond': [
                                    {
                                        '$eq': [
                                            '$status.Status', 'Start DHCP'
                                        ]
                                    }, 1, 0
                                ]
                            }
                        },
                        'Ranging': {
                            '$sum': {
                                '$cond': [
                                    {
                                        '$or': [
                                            {
                                                '$eq': [
                                                    '$status.Status', 'Ranging'
                                                ]
                                            },
                                            {
                                                '$eq': [
                                                    '$status.Status', 'Ranging Complete'
                                                ]
                                            }
                                        ]
                                    }, 1, 0
                                ]
                            }
                        },
                        'Reset': {
                            '$sum': {
                                '$cond': [
                                    {
                                        '$eq': [
                                            '$status.Status', 'Reset'
                                        ]
                                    }, 1, 0
                                ]
                            }
                        },
                        'Invalid': {
                            '$sum': {
                                '$cond': [
                                    {
                                        '$eq': [
                                            '$status.Status', 'Invalid Config'
                                        ]
                                    }, 1, 0
                                ]
                            }
                        },
                        'Offline': {
                            '$sum': {
                                '$cond': [
                                    {
                                        '$eq': [
                                            '$status.Status', 'Offline'
                                        ]
                                    }, 1, 0
                                ]
                            }
                        },
                        'Disabled': {
                            '$sum': {
                                '$cond': [
                                    {
                                        '$eq': [
                                            '$status.Status', 'Disabled'
                                        ]
                                    }, 1, 0
                                ]
                            }
                        }
                    }
                }
            ]))
            if counts:
                counts = counts[0]
                response[1] = counts
        except Exception as err:
            self.status_log("EXCEPTION (get_onu_vcm_counts)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get vCM counts due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "VCM-STATE", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    """
    ======================================================================
    ============================== Grafana ===============================
    ======================================================================
    """

    # Retrieve all grafana links for OLTs
    def get_grafana_olt_links(self, user_email):
        response = [status.HTTP_200_OK, []]

        try:
            mydb = database_manager.get_database(self.database_id)
            collection = self.database["OLT-DASH-CFG"]
            all_collections = mydb.list_collection_names();

            if "OLT-DASH-CFG" in all_collections:
                links_cursor = collection.find()
                for link in links_cursor:
                    response[1].append(link)

            else:
                response[1] = None
        except Exception as err:
            self.status_log("EXCEPTION (get_grafana_links)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get OLT Grafana links due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "OLT-DASH-CFG", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    # Save or Delete Grafana links for OLTs
    def put_grafana_olt_links(self, links, user_email):
        response = [status.HTTP_200_OK, []]

        try:
            database_manager.get_database(self.database_id)
            collection = self.database['OLT-DASH-CFG']

            for link in links:
                id = link["_id"]
                if 'delete' in link:
                    collection.delete_one({"_id": id})
                else:
                    collection.replace_one({"_id": id}, link, upsert=True);

        except Exception as err:
            self.status_log("EXCEPTION (put_grafana_links)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not update OLT-DASH-CFG status due to {type(err).__name__}::{sys.exc_info()[1]}",
                        None, None]

        action_data = {"collection": "OLT-DASH-CFG", "user": user_email}
        self.sys_log(action_data, "put", response[0] == status.HTTP_200_OK or response[0] == status.HTTP_201_CREATED)

        return response

    # Retrieve all grafana links for ONUs
    def get_grafana_onu_links(self, user_email):
        response = [status.HTTP_200_OK, []]

        try:
            mydb = database_manager.get_database(self.database_id)
            collection = self.database["ONU-DASH-CFG"]
            all_collections = mydb.list_collection_names();

            if "ONU-DASH-CFG" in all_collections:
                links_cursor = collection.find()
                for link in links_cursor:
                    response[1].append(link)

            else:
                response[1] = None
        except Exception as err:
            self.status_log("EXCEPTION (get_grafana_links)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get ONU Grafana links due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "ONU-DASH-CFG", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    # Save or Delete Grafana links for ONUs
    def put_grafana_onu_links(self, links, user_email):
        response = [status.HTTP_200_OK, []]

        try:
            database_manager.get_database(self.database_id)
            collection = self.database['ONU-DASH-CFG']

            for link in links:
                id = link["_id"]
                if 'delete' in link:
                    collection.delete_one({"_id": id})
                else:
                    collection.replace_one({"_id": id}, link, upsert=True);

        except Exception as err:
            self.status_log("EXCEPTION (put_grafana_links)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not update ONU-DASH-CFG status due to {type(err).__name__}::{sys.exc_info()[1]}",
                        None, None]

        action_data = {"collection": "ONU-DASH-CFG", "user": user_email}
        self.sys_log(action_data, "put",
                     response[0] == status.HTTP_200_OK or response[0] == status.HTTP_201_CREATED)

        return response

    """
        ======================================================================
        ========================== Device Summary ============================
        ======================================================================
    """

    # retrieves online/offline status, state, and time stats (ex: online duration)
    def get_onu_summary(self, user_email, onu_id):
        response = [status.HTTP_200_OK, []]
        try:
            system_status = "Offline"
            is_offline = True
            is_resetting = False
            onu_data = self.database["ONU-STATE"].find_one({"_id": onu_id})
            vcm_data = self.database["VCM-STATE"].find_one({"_id": onu_id})

            if onu_data is not None:
                config_reset_count = onu_data["ONU"]["Reset Count"]
                state_reset_count = onu_data["ONU"]["Reset Count"]
                olt_id = onu_data["OLT"]["MAC Address"]
                olt_state = self.database["OLT-STATE"].find_one({"_id": olt_id})
                cntl_id = onu_data["CNTL"]["MAC Address"]
                cntl_state = self.database["CNTL-STATE"].find_one({"_id": cntl_id})
            else:
                config_reset_count = None
                state_reset_count = None
                olt_id = None
                olt_state = None
                cntl_state = None
                cntl_id = None

            # Check that CNTL is online.
            if cntl_state is not None and "Time" in cntl_state and "CNTL" in cntl_state and "Version" in \
                    cntl_state["CNTL"] and self.controller_time_is_online(cntl_state["Time"], cntl_state["CNTL"]["Version"]):
                if olt_id in cntl_state["System Status"]:
                    if onu_id in cntl_state["System Status"][olt_id]["ONUs"]:
                        system_status = cntl_state["System Status"][olt_id]["ONUs"][onu_id]

                        # If ONU Status is Deregistered in CNTL-STATE, we must verify this is correct by checking OLT-STATE
                        if system_status == "Deregistered":
                            olt_state = get_nested_value(onu_data, ["OLT-STATE", 0])
                            if olt_state:
                                # Setting ONU status based on OLT-State ONU States
                                if onu_data in olt_state["ONU States"]["Disabled"]:
                                    system_status = "Disabled"
                                elif onu_data in olt_state["ONU States"]["Disallowed Admin"]:
                                    system_status = "Disallowed Admin"
                                elif onu_data in olt_state["ONU States"]["Disallowed Error"]:
                                    system_status = "Disallowed Error"
                                elif onu_data in olt_state["ONU States"]["Dying Gasp"]:
                                    system_status = "Dying Gasp"
                                elif onu_data in olt_state["ONU States"]["Unprovisioned"]:
                                    system_status = "Unprovisioned"

                        if cntl_state["System Status"][olt_id]["ONUs"][onu_id] == "Unspecified" or \
                                cntl_state["System Status"][olt_id]["ONUs"][onu_id] == "Registered" or \
                                cntl_state["System Status"][olt_id]["ONUs"][onu_id] == "FW Upgrade":
                            is_offline = False

            # Check if the ONU is resetting if online
            if not is_offline:
                if config_reset_count is not None and config_reset_count >= 0:
                    is_resetting = state_reset_count != config_reset_count
                else:
                    is_resetting = ''

                if is_resetting:
                    system_status = "Pending Reset"

            # ONU Duration
            if onu_data is not None:
                last_online = onu_data["ONU"]["Online Time"]
                last_offline = onu_data["Time"]
                uptime_duration = self.get_time_duration(last_online, last_offline)
                downtime_duration = self.get_time_duration(last_offline)
                last_status_update = onu_data["Time"]

            else:
                last_online = ''
                last_offline = ''
                uptime_duration = ''
                downtime_duration = ''
                last_status_update = ''

            # vCM Duration
            if vcm_data is not None and "VCM" in vcm_data:
                vcm_last_online = vcm_data["VCM"]["Operational Time"]
                vcm_last_offline = vcm_data["Time"]
                vcm_downtime_duration = self.get_time_duration(vcm_last_offline)

                # Set duration to empty if Operational Time is empty
                if vcm_last_online == "":
                    vcm_uptime_duration = ""
                else:
                    vcm_uptime_duration = self.get_time_duration(vcm_last_online, vcm_last_offline)

                if 'Lease Expiry Time' in vcm_data['DHCP'] and vcm_data['DHCP']['Lease Expiry Time']:
                    # Determine if lease has already expired or not by timedelta sign
                    vcm_lease_remaining_unformatted = str((datetime.datetime.strptime(vcm_data['DHCP']['Lease Expiry Time'], '%Y-%m-%d %H:%M:%S.%f') - datetime.datetime.now()))
                    if vcm_lease_remaining_unformatted.startswith('-'):  # If expired, get time since expiry
                        vcm_lease_remaining_sign = '-'
                        vcm_lease_remaining = self.get_time_duration(vcm_data['DHCP']['Lease Expiry Time'])
                    else:  # If not expired, get time duration until expiry
                        vcm_lease_remaining_sign = '+'
                        vcm_lease_remaining = self.get_time_duration(str(datetime.datetime.strptime(str(datetime.datetime.utcnow()), '%Y-%m-%d %H:%M:%S.%f')), vcm_data['DHCP']['Lease Expiry Time'])

                else:
                    vcm_lease_remaining = ""
                    vcm_lease_remaining_sign = '+'
            else:
                vcm_last_online = ""
                vcm_last_offline = ""
                vcm_uptime_duration = ""
                vcm_downtime_duration = ""
                vcm_lease_remaining = ""
                vcm_lease_remaining_sign = '+'

            response[1] = {"_id": onu_id,
                           "online": not is_offline,
                           "last_satus_update": last_status_update,
                           "last_online": last_online,
                           "last_offline": last_offline,
                           "uptime_duration": uptime_duration,
                           "downtime_duration": downtime_duration,
                           "state": system_status,
                           "state_collection": onu_data,
                           "vcm": {
                               "last_online": vcm_last_online,
                               "last_offline": vcm_last_offline,
                               "uptime_duration": vcm_uptime_duration,
                               "downtime_duration": vcm_downtime_duration,
                               "lease_remaining": {
                                   "time_delta": vcm_lease_remaining,
                                   "time_delta_sign": vcm_lease_remaining_sign
                               },
                               "vcm_collection": vcm_data
                           }}
        except Exception as err:
            self.status_log("EXCEPTION (get_onu_summary)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get ONU Summary for {onu_id} due to {type(err).__name__}::{sys.exc_info()[1]}"]

        return response

    # retrieves online/offline status, state, and time stats (ex: online duration)
    def get_olt_summary(self, user_email, olt_id):
        response = [status.HTTP_200_OK, []]
        try:
            database_manager.get_database(self.database_id)
            olt_state = self.database["OLT-STATE"].find_one({"_id": olt_id})
            if olt_state is not None:
                cntl_id = olt_state["CNTL"]["MAC Address"]
                cntl_state = self.database["CNTL-STATE"].find_one({"_id": cntl_id})
            else:
                cntl_state = None
            isOffline = False

            if cntl_state is not None and "Time" in cntl_state:
                if self.controller_time_is_online(cntl_state["Time"], cntl_state["CNTL"]["Version"]):
                    if olt_state is not None and olt_state["_id"] in cntl_state["System Status"]:
                        if cntl_state["System Status"][olt_state["_id"]]["OLT State"] == "Unspecified" or \
                                cntl_state["System Status"][olt_state["_id"]]["OLT State"] == "Primary" or \
                                cntl_state["System Status"][olt_state["_id"]]["OLT State"] == "Secondary":
                            isOffline = False
                        else:
                            isOffline = True
                    else:
                        isOffline = True
                else:
                    isOffline = True
            else:
                isOffline = True

            if olt_state is not None and "Time" in olt_state:
                last_status_update = olt_state["Time"]
                last_online = olt_state["OLT"]["Online Time"]
                last_offline = olt_state["Time"]
                uptime_duration = self.get_time_duration(last_online, last_offline)
                downtime_duration = self.get_time_duration(last_offline)
                uptime = olt_state["OLT"]["Uptime"]
            else:
                last_status_update = ''
                last_online = ''
                last_offline = ''
                uptime_duration = ''
                downtime_duration = ''
                uptime = ''

            response[1] = {"_id": olt_id,
                           "online": not isOffline,
                           "last_status_update": last_status_update,
                           "last_online": last_online,
                           "last_offline": last_offline,
                           "uptime_duration": uptime_duration,
                           "uptime": uptime,
                           "downtime_duration": downtime_duration,
                           "state_collection": olt_state}

        except Exception as err:
            self.status_log("EXCEPTION (get_olt_summary)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get OLT Summary for {olt_id} due to {type(err).__name__}::{sys.exc_info()[1]}"]
        return response

    # retrieves online/offline status and state
    def get_cntl_summary(self, user_email, cntl_id):
        response = [status.HTTP_200_OK, []]

        try:
            database_manager.get_database(self.database_id)
            cntl_state = self.database["CNTL-STATE"].find_one({"_id": cntl_id})
            online = self.controller_time_is_online(cntl_state["Time"], cntl_state["CNTL"]["Version"])
            last_status_update = cntl_state["Time"]
            paused = cntl_state["CNTL"]["Pause"]
            response[1] = {"online": online, "paused": paused, "last_status_update": last_status_update, "state_collection": cntl_state}

        except Exception as err:
            self.status_log("EXCEPTION (get_cntl_summary)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get CNTL Summary for {cntl_id} due to {type(err).__name__}::{sys.exc_info()[1]}"]
        return response

    """
        ======================================================================
        ========================== Search Filters ============================
        ======================================================================
    """

    # Retrieve all Search filters
    def get_search_filters(self, device, user_email):
        response = [status.HTTP_200_OK, []]

        try:
            device_collection = ""
            if device == 'onu':
                device_collection = 'ONU-SEARCH-CFG'
            elif device == 'olt':
                device_collection = 'OLT-SEARCH-CFG'
            elif device == 'cntl':
                device_collection = 'CNTL-SEARCH-CFG'
            elif device == 'switch':
                device_collection = 'SWI-SEARCH-CFG'
            elif device == 'cpes':
                device_collection = 'CPE-SEARCH-CFG'

            mydb = database_manager.get_database(self.database_id)
            collection = self.database[device_collection]
            all_collections = mydb.list_collection_names()

            if collection.name in all_collections:
                response[1] = list(collection.find())

            else:
                response[1] = None

        except Exception as err:
            self.status_log("EXCEPTION (get_search_filters)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get Search filters due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "SEARCH-FILTERS", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    # Save or Delete Search filters
    def put_search_filters(self, filters, device, user_email):
        response = [status.HTTP_200_OK, []]

        try:
            device_collection = ""
            if device == 'onu':
                device_collection = 'ONU-SEARCH-CFG'
            elif device == 'olt':
                device_collection = 'OLT-SEARCH-CFG'
            elif device == 'cntl':
                device_collection = 'CNTL-SEARCH-CFG'
            elif device == 'switch':
                device_collection = 'SWI-SEARCH-CFG'
            elif device == 'cpes':
                device_collection = 'CPE-SEARCH-CFG'

            database_manager.get_database(self.database_id)
            collection = self.database[device_collection]

            id = filters["_id"]
            if 'delete' in filters:
                collection.delete_one({"_id": id})
            else:
                collection.replace_one({"_id": id}, filters, upsert=True);

        except Exception as err:
            self.status_log("EXCEPTION (put_search_filters)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not update SEARCH-FILTERS status due to {type(err).__name__}::{sys.exc_info()[1]}",
                        None, None]

        action_data = {"collection": "SEARCH-FILTERS", "user": user_email}
        self.sys_log(action_data, "put",
                     response[0] == status.HTTP_200_OK or response[0] == status.HTTP_201_CREATED)

        return response

    """
        ======================================================================
        ========================== Replace Device ============================
        ======================================================================
    """

    def put_replace_onu(self, old_onu_id, new_onu_id, user_email):
        response = [status.HTTP_200_OK, [], None, {}]

        try:
            database_manager.get_database(self.database_id)
            # Retrieve all OLT configs where the given ONU id is found
            olt_cfg_collection = self.database['OLT-CFG']
            # Find OLT cfgs that do reference the old ONU and do not reference the new ONU
            olt_cfg_cursor = olt_cfg_collection.find({f'ONUs.{old_onu_id}': {'$exists': True}})
            for olt in olt_cfg_cursor:
                # Add new ONU as a copy of old ONU
                update_document = [{'$unset': f'ONUs.{old_onu_id}'},
                                   {'$set': {f'ONUs.{new_onu_id}': olt['ONUs'][old_onu_id]}}]
                olt_cfg_collection.update_one(filter={'_id': olt['_id']}, update=update_document, upsert=False)

        except Exception as err:
            self.status_log("EXCEPTION (put_replace_onu)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not replace ONU {old_onu_id} with ONU {new_onu_id} due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "OLT-CFG", "user": user_email}
        self.sys_log(action_data, "put", response[0] == status.HTTP_200_OK)

        return response

    def put_replace_olt(self, old_olt_id, new_olt_id, user_email):
        response = [status.HTTP_200_OK, [], None, {}]

        try:
            database_manager.get_database(self.database_id)

            # Add to Switch inventories
            swi_cfg_collection = self.database['SWI-CFG']
            # Find SWI cfgs that do reference the old OLT and do not reference the new OLT
            swi_cfg_cursor = swi_cfg_collection.find({'$and': [{f'OLTs.{old_olt_id}': {'$exists': True}},
                                                               {f'OLTs.{new_olt_id}': {'$exists': False}}]})
            for swi in swi_cfg_cursor:
                # Add new OLT as a copy of old OLT
                update_document = {'$set': {f'OLTs.{new_olt_id}': swi['OLTs'][old_olt_id]}}
                swi_cfg_collection.update_one(filter={'_id': swi['_id']}, update=update_document, upsert=False)

            # Add to PON Controller inventories
            cntl_cfg_collection = self.database['CNTL-CFG']
            # Find CNTL cfgs that do reference the old OLT and do not reference the new OLT
            cntl_cfg_cursor = cntl_cfg_collection.find(
                {'$and': [
                    {'$or': [
                        {'OLTs.Primary': old_olt_id},
                        {'OLTs.Secondary': old_olt_id},
                        {'OLTs.Excluded': old_olt_id},
                    ]},
                    {'$nor': [
                        {'OLTs.Primary': new_olt_id},
                        {'OLTs.Secondary': new_olt_id},
                        {'OLTs.Excluded': new_olt_id},
                    ]}
                ]})
            for cntl in cntl_cfg_cursor:
                olt_inventory = cntl['OLTs']
                if old_olt_id in cntl['OLTs']['Primary']:
                    olt_inventory['Primary'].append(new_olt_id)
                if old_olt_id in cntl['OLTs']['Secondary']:
                    olt_inventory['Secondary'].append(new_olt_id)
                if old_olt_id in cntl['OLTs']['Excluded']:
                    olt_inventory['Excluded'].append(new_olt_id)
                update_document = {'$set': {'OLTs': olt_inventory}}
                cntl_cfg_collection.update_one(filter={'_id': cntl['_id']}, update=update_document, upsert=False)

            # Update ONU MAC Addresses
            onu_cfg_collection = self.database['ONU-CFG']
            filter_push_document = {'$and': [{'OLT.MAC Address': old_olt_id},
                                             {'OLT.MAC Address': {'$not': {
                                                 '$eq': new_olt_id}}}]}  # Find ONU configs that only reference the old OLT
            onu_cfg_collection.update_many(filter=filter_push_document,
                                           update={'$push': {'OLT.MAC Address': new_olt_id}},
                                           upsert=False)  # Add new OLT ID to list of MAC Addresses

        except Exception as err:
            self.status_log("EXCEPTION (put_replace_olt)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not replace OLT {old_olt_id} with OLT {new_olt_id} due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "SWI-CFG, CNTL-CFG, ONU-CFG", "user": user_email}
        self.sys_log(action_data, "put", response[0] == status.HTTP_200_OK)

        return response

    """
        ======================================================================
        ========================= Monitoring Alarms ==========================
        ======================================================================
    """

    def build_alarm_data(self, device, alarm_configs, alarm_states, device_type, alarm_severities=[]):
        alarm_histories = []
        device_id = device['_id']
        for alarm in device['Alarms']:
            # Don't need to check severity for non ONU devices
            if alarm['Severity'] in alarm_severities or device_type != 'ONU':
                time = alarm['Time Last Cleared']
                state = 'Cleared'
                if alarm['Active State'] == True:
                    time = alarm['Time Last Raised']
                    state = 'Raised'

                alarm_id = alarm['Source'] + ':' + alarm['Alarm Type']

                if alarm['Source'] == 'STATE' or alarm['Source'] == 'STATS' or alarm['Source'] == 'GPON':
                    alarm_id = alarm['Source'] + ':' + alarm['Object Type'] + ':' + alarm['Alarm Type']

                instance = alarm['Object Type'] + ':' + alarm['Object Instance']
                if alarm['Source'] == 'STATE' or alarm['Source'] == 'STATS':
                    instance = ''

                ack = 'No'
                if alarm['Ack State'] == True:
                    ack = 'Yes'

                config_id = alarm['Alarm ID']
                cfg_alarms = get_nested_value(alarm_configs, [device_id, 'Alarm History', 'Alarms'], {})
                comments = ''
                comment_operator = ''
                for comment in cfg_alarms:
                    if comment['Alarm ID'] == config_id:
                        comments = comment['Comment']
                        comment_operator = ""
                        if 'Comment Operator' in comment:
                            comment_operator = comment['Comment Operator']

                device_label = 'CNTL' if device_type == "Controller" else device_type
                alarm_hist = {
                    'configId': config_id,
                    'deviceType': device_type,
                    'deviceName': get_nested_value(alarm_configs, [device_id, device_label, 'Name'], ''),
                    'deviceId': device_id,
                    'time': time,
                    'state': state,
                    'severity': alarm['Severity'],
                    'alarmID': alarm_id,
                    'instance': instance,
                    'raisedCount': alarm['Raised Count'],
                    'ack': ack,
                    'lastAckTime': alarm['Time Last Ack'],
                    'lastAckUser': alarm['Operator Ack'],
                    'comments': comments,
                    'commentOperator': comment_operator,
                    'lastRaisedTime': alarm['Time Last Raised'],
                    'lastClearedTime': alarm['Time Last Cleared'],
                    'text': alarm['Text'],
                    'configAckCount': get_nested_value(alarm_configs,
                                                       [device_id, 'Alarm History', 'Ack Count'], 0),
                    'stateAckCount': get_nested_value(alarm_states,
                                                      [device_id, 'Alarm History', 'Ack Count'], 0),
                    'configPurgeCount': get_nested_value(alarm_configs,
                                                         [device_id, 'Alarm History', 'Purge Count'], 0),
                    'statePurgeCount': get_nested_value(alarm_states,
                                                        [device_id, 'Alarm History', 'Purge Count'], 0),
                    'newComment': False
                }
                alarm_histories.append(alarm_hist)
        return alarm_histories

    def get_all_cntl_alarms_hist(self, user_email):
        response = [status.HTTP_200_OK, {}]

        try:
            database_manager.get_database(self.database_id)

            # Get CNTL Config as dictionary
            alarm_config_collection = self.database["CNTL-CFG"]
            alarm_config_cursor = list(alarm_config_collection.find({}, {"_id": 1, "CNTL.Name": 1, "Alarm History": 1}))
            alarm_configs_dict = {item['_id']: item for item in alarm_config_cursor}

            # Get CNTL State as dictionary
            alarm_state_collection = self.database["CNTL-STATE"]
            alarm_state_cursor = list(alarm_state_collection.find({}, {"_id": 1, "Alarm History": 1}))
            alarm_states_dict = {item['_id']: item for item in alarm_state_cursor}

            # Get CNTL alarm histories
            alarm_histories_collection = self.database["CNTL-ALARM-HIST-STATE"]
            alarm_histories_cursor = alarm_histories_collection.find({})

            alarm_histories = []
            for device in alarm_histories_cursor:
                alarm_hist = self.build_alarm_data(device, alarm_configs_dict, alarm_states_dict, 'Controller')
                alarm_histories.extend(alarm_hist)

            alarm_histories.sort(key=lambda alarm: alarm['time'], reverse=True)
            alarm_histories.sort(key=lambda alarm: alarm['severity'], reverse=False)
            response[1] = alarm_histories
        except Exception as err:
            self.status_log("EXCEPTION (get_all_cntl_alarms_hist)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get PON Controller alarm history table data due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "CNTL-STATE,CNTL-CFG,CNTL-ALARM-HIST-STATE", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    def get_all_olt_alarms_hist(self, user_email):
        response = [status.HTTP_200_OK, {}]

        try:
            database_manager.get_database(self.database_id)

            # Get OLT Config as dictionary
            alarm_config_collection = self.database["OLT-CFG"]
            alarm_config_cursor = list(alarm_config_collection.find({}, {"_id": 1, "OLT.Name": 1, "Alarm History": 1}))
            alarm_configs_dict = {item['_id']: item for item in alarm_config_cursor}

            # Get OLT State as dictionary
            alarm_state_collection = self.database["OLT-STATE"]
            alarm_state_cursor = list(alarm_state_collection.find({}, {"_id": 1, "Alarm History": 1}))
            alarm_states_dict = {item['_id']: item for item in alarm_state_cursor}

            # Get OLT alarm histories
            alarm_histories_collection = self.database["OLT-ALARM-HIST-STATE"]
            alarm_histories_cursor = alarm_histories_collection.find({})

            alarm_histories = []
            for device in alarm_histories_cursor:
                alarm_hist = self.build_alarm_data(device, alarm_configs_dict, alarm_states_dict, 'OLT')
                alarm_histories.extend(alarm_hist)

            alarm_histories.sort(key=lambda alarm: alarm['time'], reverse=True)
            alarm_histories.sort(key=lambda alarm: alarm['severity'], reverse=False)
            response[1] = alarm_histories
        except Exception as err:
            self.status_log("EXCEPTION (get_all_olt_alarms_hist)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get OLT alarm history table data due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "OLT-STATE,OLT-CFG,OLT-ALARM-HIST-STATE", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    def get_all_onu_alarms_hist(self, user_email):
        response = [status.HTTP_200_OK, {}]

        try:
            database_manager.get_database(self.database_id)
            all_alarm_histories = []
            alarm_limit = 20000
            alarm_count = 0

            # Get ONU Config as dictionary
            alarm_config_collection = self.database["ONU-CFG"]
            alarm_config_cursor = list(alarm_config_collection.find({}, {"_id": 1, "ONU.Name": 1, "Alarm History": 1}))
            alarm_configs_dict = {item['_id']: item for item in alarm_config_cursor}

            # Get ONU State as dictionary
            alarm_state_collection = self.database["ONU-STATE"]
            alarm_state_cursor = list(alarm_state_collection.find({}, {"_id": 1, "Alarm History": 1}))
            alarm_states_dict = {item['_id']: item for item in alarm_state_cursor}

            # Get ONU alarm histories
            alarm_histories_collection = self.database["ONU-ALARM-HIST-STATE"]

            alarm_severities_high = ["0-EMERG", "1-ALERT"]
            alarm_histories_cursor_high = alarm_histories_collection.find(
                {'Alarms.Severity': {'$in': alarm_severities_high}},
                {"Alarms": {"$filter": {"input": "$Alarms", "as": "item",
                                        "cond": {"$in": ["$$item.Severity", alarm_severities_high]}}}})

            for device in alarm_histories_cursor_high:
                if alarm_count < alarm_limit:
                    alarm_histories = self.build_alarm_data(device, alarm_configs_dict, alarm_states_dict, 'ONU',
                                                            alarm_severities_high)
                    all_alarm_histories.extend(alarm_histories)
                    alarm_count = len(all_alarm_histories)

            if alarm_count < alarm_limit:
                alarm_severities_med = ["2-CRIT", "3-ERROR", "4-WARNING"]
                alarm_histories_cursor_med = alarm_histories_collection.find(
                    {'Alarms.Severity': {'$in': alarm_severities_med}},
                    {"Alarms": {"$filter": {"input": "$Alarms", "as": "item",
                                            "cond": {"$in": ["$$item.Severity", alarm_severities_med]}}}})
                for device in alarm_histories_cursor_med:
                    if alarm_count < alarm_limit:
                        alarm_histories = self.build_alarm_data(device, alarm_configs_dict, alarm_states_dict, 'ONU',
                                                                alarm_severities_med)
                        all_alarm_histories.extend(alarm_histories)
                        alarm_count = len(all_alarm_histories)

            if alarm_count < alarm_limit:
                alarm_severities_low = ["5-NOTICE", "6-INFO"]
                alarm_histories_cursor_low = alarm_histories_collection.find(
                    {'Alarms.Severity': {'$in': alarm_severities_low}},
                    {"Alarms": {"$filter": {"input": "$Alarms", "as": "item",
                                            "cond": {"$in": ["$$item.Severity", alarm_severities_low]}}}})
                for device in alarm_histories_cursor_low:
                    if alarm_count < alarm_limit:
                        alarm_histories = self.build_alarm_data(device, alarm_configs_dict, alarm_states_dict, 'ONU',
                                                                alarm_severities_low)
                        all_alarm_histories.extend(alarm_histories)
                        alarm_count = len(all_alarm_histories)

            all_alarm_histories.sort(key=lambda alarm: alarm['time'], reverse=True)
            all_alarm_histories.sort(key=lambda alarm: alarm['severity'], reverse=False)
            response[1] = all_alarm_histories
        except Exception as err:
            self.status_log("EXCEPTION (get_all_onu_alarms_hist)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get ONU alarm history table data due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "ONU-STATE,ONU-CFG,ONU-ALARM-HIST-STATE", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    def get_cntl_alarm_hist_count(self, user_email):
        response = [status.HTTP_200_OK, {}]

        try:
            database_manager.get_database(self.database_id)

            # Get CNTL alarm histories
            alarm_histories_collection = self.database["CNTL-ALARM-HIST-STATE"]
            alarm_histories_cursor = alarm_histories_collection.find({}, {"_id": 0, "Alarms": 1})
            alarm_counts = [
                {'name': 'Emergency', 'value': 0},
                {'name': 'Alert', 'value': 0},
                {'name': 'Critical', 'value': 0},
                {'name': 'Error', 'value': 0},
                {'name': 'Warning', 'value': 0},
                {'name': 'Notice', 'value': 0},
                {'name': 'Info', 'value': 0},
                {'name': 'Debug', 'value': 0}
            ]
            for alarms in alarm_histories_cursor:
                for alarm in alarms['Alarms']:
                    severity = int(alarm['Severity'][0])
                    alarm_counts[severity]['value'] = alarm_counts[severity]['value'] + 1

            response[1] = alarm_counts
        except Exception as err:
            self.status_log("EXCEPTION (get_cntl_alarm_hist_count)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get PON Controller alarm history counts due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "CNTL-ALARM-HIST-STATE", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    def get_olt_alarm_hist_count(self, user_email):
        response = [status.HTTP_200_OK, {}]

        try:
            database_manager.get_database(self.database_id)

            # Get OLT alarm histories
            alarm_histories_collection = self.database["OLT-ALARM-HIST-STATE"]
            alarm_histories_cursor = alarm_histories_collection.find({}, {"_id": 0, "Alarms": 1})
            alarm_counts = [
                {'name': 'Emergency', 'value': 0},
                {'name': 'Alert', 'value': 0},
                {'name': 'Critical', 'value': 0},
                {'name': 'Error', 'value': 0},
                {'name': 'Warning', 'value': 0},
                {'name': 'Notice', 'value': 0},
                {'name': 'Info', 'value': 0},
                {'name': 'Debug', 'value': 0}
            ]

            for alarms in alarm_histories_cursor:
                for alarm in alarms['Alarms']:
                    severity = int(alarm['Severity'][0])
                    alarm_counts[severity]['value'] = alarm_counts[severity]['value'] + 1

            response[1] = alarm_counts
        except Exception as err:
            self.status_log("EXCEPTION (get_olt_alarm_hist_count)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get OLT alarm history counts due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "OLT-ALARM-HIST-STATE", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    def get_onu_alarm_hist_count(self, user_email):
        response = [status.HTTP_200_OK, {}]

        try:
            database_manager.get_database(self.database_id)

            # Get ONU alarm histories
            alarm_histories_collection = self.database["ONU-ALARM-HIST-STATE"]
            alarm_histories_cursor = alarm_histories_collection.find({}, {"_id": 0, "Alarms": 1})
            alarm_counts = [
                {'name': 'Emergency', 'value': 0},
                {'name': 'Alert', 'value': 0},
                {'name': 'Critical', 'value': 0},
                {'name': 'Error', 'value': 0},
                {'name': 'Warning', 'value': 0},
                {'name': 'Notice', 'value': 0},
                {'name': 'Info', 'value': 0},
                {'name': 'Debug', 'value': 0}
            ]

            for alarms in alarm_histories_cursor:
                for alarm in alarms['Alarms']:
                    severity = int(alarm['Severity'][0])
                    alarm_counts[severity]['value'] = alarm_counts[severity]['value'] + 1

            response[1] = alarm_counts
        except Exception as err:
            self.status_log("EXCEPTION (get_onu_alarm_hist_count)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get ONU alarm history counts due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "ONU-ALARM-HIST-STATE", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    def bulk_comment_alarms(self, user_email, alarms, device_type):
        response = [status.HTTP_200_OK, {}]
        alarms = alarms['Alarms']

        try:
            database_manager.get_database(self.database_id)
            collection = self.database[device_type + "-CFG"]
            update_docs = []
            for alarm in alarms:
                alarm_data = alarms[alarm]
                update_doc = {"$set": {"Alarm History.Alarms": alarm_data}}
                update_docs.append(UpdateOne({'_id': alarm}, update_doc, upsert=False))

            if len(update_docs) > 0:
                collection.bulk_write(update_docs)

        except Exception as err:
            self.status_log("EXCEPTION (bulk_comment_alarms)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not bulk comment alarms data due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "ONU-CFG, OLT-CFG, CNTL-CFG", "user": user_email}
        self.sys_log(action_data, "put", response[0] == status.HTTP_200_OK)

        return response

    def bulk_purge_alarms(self, user_email, alarms, device_type):
        response = [status.HTTP_200_OK, {}]

        all_device_alarms = alarms['Alarms']

        try:
            database_manager.get_database(self.database_id)

            device_display_type = device_type
            if device_type == 'CNTL':
                device_display_type = 'Controller'

            all_alarms = list(filter(lambda alarm: alarm['deviceType'] == device_display_type, all_device_alarms))

            alarms_dict = {}
            for item in all_alarms:
                if item['deviceId'] in alarms_dict:
                    alarms_dict[item['deviceId']].append(item)
                else:
                    alarms_dict[item['deviceId']] = [item]

            alarm_docs = []
            collection = self.database[device_type + "-CFG"]
            # Get all unique device ids from alarms
            device_ids = set([alarm['deviceId'] for alarm in all_alarms if 'deviceId' in alarm])
            for device in device_ids:
                onu_alarms = alarms_dict[device]
                alarm_ids = []
                for alarm in onu_alarms:
                    alarm_ids.append(alarm['configId'])

                doc = {
                    "$set": {"Alarm History.Alarm IDs": alarm_ids},
                    "$inc": {"Alarm History.Purge Count": 1},
                    "$pull": {"Alarm History.Alarms": {"Alarm ID": {'$in': alarm_ids}}}
                }
                alarm_docs.append(UpdateOne({'_id': device}, doc, upsert=False))

            if len(alarm_docs) > 0:
                collection.bulk_write(alarm_docs)

        except Exception as err:
            self.status_log("EXCEPTION (bulk_purge_alarms)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not purge alarms data due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "ONU-CFG, OLT-CFG, CNTL-CFG", "user": user_email}
        self.sys_log(action_data, "put", response[0] == status.HTTP_200_OK)

        return response

    def bulk_acknowledge_alarms(self, user_email, alarms, device_type):
        response = [status.HTTP_200_OK, {}]

        all_device_alarms = alarms['Alarms']
        ack_operator = alarms['Ack Operator']

        try:
            database_manager.get_database(self.database_id)

            device_display_type = device_type
            if device_type == 'CNTL':
                device_display_type = 'Controller'

            all_alarms = list(filter(lambda alarm: alarm['deviceType'] == device_display_type, all_device_alarms))

            alarms_dict = {}
            for item in all_alarms:
                if item['deviceId'] in alarms_dict:
                    alarms_dict[item['deviceId']].append(item)
                else:
                    alarms_dict[item['deviceId']] = [item]

            alarm_docs = []
            collection = self.database[device_type + "-CFG"]
            # Get all unique device ids from alarms
            device_ids = set([alarm['deviceId'] for alarm in all_alarms if 'deviceId' in alarm])
            for device in device_ids:
                onu_alarms = alarms_dict[device]
                alarm_ids = []
                alarms = []
                old_alarms = []
                for alarm in onu_alarms:
                    # If new comment, set the comment operator
                    comment_operator = ""
                    if 'commentOperator' in alarm:
                        comment_operator = alarm['commentOperator']
                    if alarm['newComment']:
                        comment_operator = ack_operator

                    alarm_ids.append(alarm['configId'])
                    old_alarms.append({
                        'Alarm ID': alarm['configId']
                    })
                    alarms.append({
                        'Alarm ID': alarm['configId'],
                        'Comment': alarm['comments'],
                        'Comment Operator': comment_operator
                    })

                pull_doc = {
                    "$pull": {
                        "Alarm History.Alarms": {"Alarm ID": {"$in": alarm_ids}}
                    }
                }

                push_doc = {
                    "$set": {"Alarm History.Alarm IDs": alarm_ids,
                             "Alarm History.Ack Operator": ack_operator,
                    }, "$push": {
                        "Alarm History.Alarms": {"$each": alarms}
                    },
                    "$inc": {
                        "Alarm History.Ack Count": 1
                    }
                }
                alarm_docs.append(UpdateOne({'_id': device}, pull_doc, upsert=False))
                alarm_docs.append(UpdateOne({'_id': device}, push_doc, upsert=False))

            if len(alarm_docs) > 0:
                collection.bulk_write(alarm_docs)

        except Exception as err:
            self.status_log("EXCEPTION (bulk_acknowledge_alarms)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not bulk acknowledge alarms data due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "ONU-CFG, OLT-CFG, CNTL-CFG", "user": user_email}
        self.sys_log(action_data, "put", response[0] == status.HTTP_200_OK)

        return response

    """
        ======================================================================
        =============================== Tasks ================================
        ======================================================================
    """

    def get_task_validation_status(self, user_email, device_ids, device_type, fw_bank_pointer, fw_versions, fw_files):
        response = [status.HTTP_200_OK, []]

        try:
            # Format params
            device_ids = device_ids.split(',')
            device_type = device_type.lower()
            fw_versions = fw_versions.split(',')
            fw_files = fw_files.split(',')
            fw_bank_pointer = int(fw_bank_pointer)

            db = database_manager.get_database(self.database_id)

            device_collection = ""
            device_projection = {}
            if device_type == 'onu':
                device_collection = 'ONU-STATE'
                device_projection = {
                    'ONU.FW Bank Ptr': 1,
                    'ONU.FW Bank Versions': 1,
                    'ONU.Vendor': 1,
                    'ONU.Equipment ID': 1
                }

            elif device_type == 'olt':
                device_collection = 'OLT-STATE'
                device_projection = {
                    'OLT.FW Bank Ptr': 1,
                    'OLT.FW Bank Versions': 1
                }

            # If ONU jobs, check for errors
            firmware_files = []
            if device_type == 'onu':
                firmware_collection = 'ONU-FIRMWARE.files'
                # Get all firmware files
                collection = self.database[firmware_collection]
                firmware_cursor = collection.find({'filename': {'$in': fw_files}},
                                                     projection={'metadata.Compatible Manufacturer': 1, 'metadata.Compatible Model': 1})
                firmware_files = list(firmware_cursor)

            # Get all device states
            collection = self.database[device_collection]
            device_cursor = collection.find({'_id': {'$in': device_ids}}, projection=device_projection)

            # Validate each device
            warnings = []
            errors = []
            for device in device_cursor:
                device_type_reference = device_type.upper()
                # Check if Active bank is changing
                if get_nested_value(device, [device_type_reference, "FW Bank Ptr"]) != fw_bank_pointer:
                    if device['_id'] not in warnings:
                        warnings.append(device['_id'])
                # Firmware version is changing in the active bank
                if get_nested_value(device, [device_type_reference, "FW Bank Versions", fw_bank_pointer]) != fw_versions[fw_bank_pointer]:
                    if device['_id'] not in warnings:
                        warnings.append(device['_id'])

                # Only check for errors on ONUs
                if device_type == 'onu':
                    for firmware in firmware_files:
                        # Check for vendor conflict
                        if get_nested_value(device, [device_type_reference, "Vendor"]) != firmware['metadata']['Compatible Manufacturer']:
                            if device['_id'] not in errors:
                                errors.append(device['_id'])
                            break

                        # Check for model conflict for CIEN ONUs
                        if get_nested_value(device, [device_type_reference, "Vendor"]) == 'CIEN' or get_nested_value(device, [device_type_reference, "Vendor"]) == 'CINA':
                            if get_nested_value(device, [device_type_reference, "PON Mode"]) == 'EPON':
                                onu_model = get_nested_value(device, [device_type_reference, "Model"])
                            else:
                                onu_model = get_nested_value(device, [device_type_reference, "Equipment ID"])

                            # FW Model may use '_' or '-'. Normalize all '_' to '-'
                            if onu_model:
                                onu_model = onu_model.replace('_', '-')
                            for model in firmware['metadata']['Compatible Model']:
                                firmware_model = model.replace('_', '-')
                                if onu_model != firmware_model:
                                    if device['_id'] not in errors:
                                        errors.append(device['_id'])
                                    break

            response[1] = {'warnings': warnings, 'errors': errors}

        except Exception as err:
            self.status_log("EXCEPTION (get_task_validation_status)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get task validation status due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "OLT-STATE, OLT-FIRMWARE.files, ONU-STATE, ONU-FIRMWARE.files", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response


    def get_task_detailed_states_by_page(self, user_email, task_id, filter_value, sort_active, sort_direction,
                                           page_index, page_size):
        response = [status.HTTP_200_OK, []]

        try:
            # Format params
            current_page = int(page_index)
            page_size = int(page_size)

            # Get the number of documents to skip
            skip_count = page_size * (current_page - 1)

            db = database_manager.get_database(self.database_id)
            collection = 'AUTO-TASK-DETAILED-STATE'
            collection = self.database[collection]

            # Get one page of detailed states
            filter_doc = {'Task ID': task_id}
            if filter_value != '':
                filter_doc = {
                    'Task ID': task_id,
                    '$or': [
                        {'Device ID': {'$regex': re.compile(r"(?i)" + filter_value)}},
                        {'Task.Status': {'$regex': re.compile(r"(?i)" + filter_value)}},
                    ]
                }

            sort_value = 1
            if sort_direction == 'asc':
                sort_value = -1
            elif sort_direction == 'desc':
                sort_value = 1

            sort_doc = {}
            if sort_direction != '':
                if sort_active == "Status":
                    sort_doc = {'Task.Status': sort_value}
                if sort_active == "Device ID":
                    sort_doc = {'_id': sort_value}

            detailed_states_cursor = collection.find(filter_doc, skip=skip_count, limit=page_size, sort=sort_doc)
            detailed_states = list(detailed_states_cursor)

            # Get total count of detailed states
            total_records = collection.count_documents(filter_doc)

            response[1] = {
                'data': detailed_states,
                'totalRecords': total_records
            }

        except Exception as err:
            self.status_log("EXCEPTION (get_task_detailed_states_by_page)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get detailed states page due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "AUTO-TASK-DETAILED-STATE", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response


    def get_all_pon_auto_enabled_tasks(self, user_email):
        response = [status.HTTP_200_OK, []]

        try:
            database_manager.get_database(self.database_id)
            auto_collections = ['SWI-AUTO-STATE', 'CNTL-AUTO-STATE', 'OLT-AUTO-STATE', 'ONU-AUTO-STATE'];

            response[1] = {'SWI': [], 'CNTL': [], 'OLT': [], 'ONU': []}
            for collection_name in auto_collections:
                collection = self.database[collection_name]
                global_doc = collection.find_one({'_id': 'Global'})
                enabled_tasks = []
                if global_doc:
                    for item in global_doc:
                        auto_task = global_doc[item]
                        # Check if auto_task is an object
                        if not isinstance(auto_task, str):
                            if get_nested_value(auto_task, ["AUTO", "Skip"]) is not None \
                                    and get_nested_value(auto_task, ["AUTO", "Skip"]) is False:
                                enabled_tasks.append(item)

                device_type = collection_name.replace('-AUTO-STATE', '')
                response[1][device_type] = enabled_tasks

        except Exception as err:
            self.status_log("EXCEPTION (get_task_detailed_states_by_page)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get PON Automation enabled tasks due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "SWI-AUTO-STATE, CNTL-AUTO-STATE, OLT-AUTO-STATE, ONU-AUTO-STATE", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response
