#!/usr/bin/env bash
#--------------------------------------------------------------------------#
# Copyright (C) 2015 - 2021 by Tibit Communications, Inc.                  #
# All rights reserved.                                                     #
#                                                                          #
#    _______ ____  _ ______                                                #
#   /_  __(_) __ )(_)_  __/                                                #
#    / / / / __  / / / /                                                   #
#   / / / / /_/ / / / /                                                    #
#  /_/ /_/_____/_/ /_/                                                     #
#                                                                          #
#--------------------------------------------------------------------------#

# Revision 2.0
VERSION=2.0

helpFunction()
{
   echo "Version: $VERSION
Usage for GPON: $0 -o <onu_id>
Usage for EPON: $0 -o <onu_mac> 
    optional: -h <database_host> -n <database_name> -d <database_port> -u <username> -p <password> -t <tlsCAFILE> -f <filename>
    defaults: -h 127.0.0.1 -n tibit_pon_controller -d 27017 -f debug-<onu_id>.tgz

  With MongoDB Authentication and/or TLS/SSL, use the following options:
    -u <username>
    -p <password>  (if omitted, script will prompt for password)
    -t <tlsCAFILE>
    -a <authenticationDatabase>  (Default: tibit_users)

  Example: $0 -u pdmPonManager -t /etc/ssl/rootCA.pem -o TBIT00000001

  Version: $VERSION
   "
   exit 1 # Exit script after printing help
}

printVersion()
# Prints version of script and exits
{
  echo "Version: $VERSION"
  exit 1
}

while getopts o:h:n:f:d:t:u:p:a:hv opt
do
  case "${opt}" in
    o) ONU_MAC=${OPTARG};;
    h) DATABASE_HOST=${OPTARG};;
    n) DATABASE_NAME=${OPTARG};;
    f) FILENAME=${OPTARG};;
    d) DATABASE_PORT=${OPTARG};;
    t) TLSCAFILE=${OPTARG};;
    u) USERNAME=${OPTARG};;
    p) PASSWORD=${OPTARG};;
    a) AUTHDATABASE=${OPTARG};;
    v) printVersion;;
    h) helpFunction;;
  esac
done

if [ -z "${ONU_MAC}" ]; then
    echo "No ONU ID specified."
    helpFunction
fi

# Verify the following commands exist

if ! command -v mongo &> /dev/null
then
    echo "mongo shell command not found.  Please install with \"sudo apt-get install mongodb-org-shell\""
    exit 1
fi

ONU_ID=${ONU_MAC//:}

if [ -z "${DATABASE_HOST}" ]; then
   DATABASE_HOST=127.0.0.1
fi
if [ -z "${DATABASE_NAME}" ]; then
   DATABASE_NAME=tibit_pon_controller
fi
if [ -z "${DATABASE_PORT}" ]; then
   DATABASE_PORT=27017
fi
if [ -z "${FILENAME}" ]; then
   FILENAME=debug-${ONU_ID}-$(date -u +%m%d%y-%H%M)
fi
mkdir ${FILENAME}
if [ -z "${AUTHDATABASE}" ]; then
   AUTHDATABASE=tibit_users
fi

#if [  -n "${TLSCAFILE}" ] && [[ "${DATABASE_HOST}"  =~ ^[0-9]+\.[0-9]+\.[0-9]+\.[0-9]+$ ]]; then
#    echo "With tls, the mongo shell command only accepts domain name for the host (i.e. mongodb.domain.net)"
#    helpFunction
#fi
if [ -n "${USERNAME}" ] && [ -z "${PASSWORD}" ]; then
    read -s -p 'Enter password for access to MongoDB: ' PASSWORD
    echo
fi

# Check version of MongoDB Shell
MONGO_SHELL=$(mongo --version | grep MongoDB | awk '{ print $4 }')
echo "MongoDB shell version ${MONGO_SHELL}" 2>&1 | tee -a ${FILENAME}/command.log
if [[ "${MONGO_SHELL}" =~ ^v4\.0.* ]]; then
    if [ -z "${USERNAME}" ] && [ -z "${TLSCAFILE}" ]; then
        SECUREOPTIONS=""
    elif [ -n "${USERNAME}" ] && [ -z "${TLSCAFILE}" ]; then
        SECUREOPTIONS="-u ${USERNAME} -p ${PASSWORD} --authenticationDatabase=${AUTHDATABASE}"
    elif [ -n "${TLSCAFILE}" ] && [ -z "${USERNAME}" ]; then
        SECUREOPTIONS="--ssl --sslCAFile=${TLSCAFILE}"
    else
        SECUREOPTIONS="--ssl --sslCAFile=${TLSCAFILE} -u ${USERNAME} -p ${PASSWORD} --authenticationDatabase=${AUTHDATABASE}"
    fi
else
    if [ -z "${USERNAME}" ] && [ -z "${TLSCAFILE}" ]; then
        SECUREOPTIONS=""
    elif [ -n "${USERNAME}" ] && [ -z "${TLSCAFILE}" ]; then
        SECUREOPTIONS="-u ${USERNAME} -p ${PASSWORD} --authenticationDatabase=${AUTHDATABASE}"
    elif [ -n "${TLSCAFILE}" ] && [ -z "${USERNAME}" ]; then
        SECUREOPTIONS="--tls --tlsCAFile=${TLSCAFILE}"
    else
        SECUREOPTIONS="--tls --tlsCAFile=${TLSCAFILE} -u ${USERNAME} -p ${PASSWORD} --authenticationDatabase=${AUTHDATABASE}"
    fi
fi

# Gather data from Mongo
# Wait until Mongo is ready to accept connections, exit if this does not happen within 10 seconds
COUNTER=0
until mongo ${SECUREOPTIONS} --host=${DATABASE_HOST} --port=${DATABASE_PORT} --eval 'printjson(db.serverStatus())' >& /dev/null
do
  sleep 1
  COUNTER=$((COUNTER+1))
  if [[ ${COUNTER} -eq 10 ]]; then
    echo "Exiting.... Mongo server ${DATABASE_HOST}:${DATABASE_PORT} ${SECUREOPTIONS} is not responding"
    exit 2
  fi
  echo "Waiting for Mongo server ${DATABASE_HOST}:${DATABASE_PORT} ${SECUREOPTIONS} to respond... ${COUNTER}/10"
done

CFG_COUNT=0
STATE_COUNT=0

# Find the Controller for the ONU
COMMAND="db[\"ONU-STATE\"].find( { _id: \"${ONU_MAC}\" }, { \"CNTL.MAC Address\": 1, \"MAC Address\": 1 } )"
CNTL_MAC="$(mongo ${SECUREOPTIONS} ${DATABASE_HOST}:${DATABASE_PORT}/${DATABASE_NAME} --eval "${COMMAND}" | awk -F \" 'NF>=10 { print $10 }')"
CNTL_ID=${CNTL_MAC//:}

#Find the OLT for the ONU
COMMAND="db[\"ONU-STATE\"].find( { _id: \"${ONU_MAC}\" }, { \"OLT.MAC Address\": 1, \"MAC Address\": 1 } )"
OLT_MAC="$(mongo ${SECUREOPTIONS} ${DATABASE_HOST}:${DATABASE_PORT}/${DATABASE_NAME} --eval "${COMMAND}" | awk -F \" 'NF>=10 { print $10 }')"
OLT_ID=${OLT_MAC//:}

#Find the Service Config for the ONU
COMMAND="db[\"ONU-CFG\"].find( { _id: \"${ONU_MAC}\" }, { \"ONU.SRV-CFG\": 1, \"SRV-CFG\": 1 } )"
SRV_CFG="$(mongo ${SECUREOPTIONS} ${DATABASE_HOST}:${DATABASE_PORT}/${DATABASE_NAME} --eval "${COMMAND}" | awk -F \" 'NF>=10 { print $10 }')"

#Trigger Pon Controller to generate OLT Debug Dump
echo -n "Initiating OLT Debug Dump " 2>&1 | tee -a ${FILENAME}/command.log
COMMAND="db[\"OLT-CFG\"].find( { _id: \"${OLT_MAC}\" }, { \"OLT.Debug Dump Count\": 1, \"Debug Dump Count\": 1 } )"
CFG_COUNT_INIT="$(mongo ${SECUREOPTIONS} ${DATABASE_HOST}:${DATABASE_PORT}/${DATABASE_NAME} --eval "${COMMAND}" | awk 'NF>=12 { print $12 }')"
COMMAND="db[\"OLT-CFG\"].update( { _id: \"${OLT_MAC}\" }, { \$inc : {\"OLT.Debug Dump Count\": 1 } })"
mongo ${SECUREOPTIONS} ${DATABASE_HOST}:${DATABASE_PORT}/${DATABASE_NAME} --eval "${COMMAND}" >& /dev/null
sleep 1
COMMAND="db[\"OLT-CFG\"].find( { _id: \"${OLT_MAC}\" }, { \"OLT.Debug Dump Count\": 1, \"Debug Dump Count\": 1 } )"
CFG_COUNT="$(mongo ${SECUREOPTIONS} ${DATABASE_HOST}:${DATABASE_PORT}/${DATABASE_NAME} --eval "${COMMAND}" | awk 'NF>=12 { print $12 }')"

if [ "${CFG_COUNT}" == "${CFG_COUNT_INIT}" ]; then
    echo " ERROR - OLT Debug not triggered" 2>&1 | tee -a ${FILENAME}/command.log
else
    # Wait for the Pon Controller to finish 
    COUNTER=0
    COMMAND="db[\"OLT-STATE\"].find( { _id: \"${OLT_MAC}\" }, { \"OLT.Debug Dump Count\": 1, \"Debug Dump Count\": 1 } )"

    while [ "$COUNTER" -lt 20 ] && [ "$STATE_COUNT" != "$CFG_COUNT" ]
    do
      STATE_COUNT="$(mongo ${SECUREOPTIONS} ${DATABASE_HOST}:${DATABASE_PORT}/${DATABASE_NAME} --eval "${COMMAND}" | awk 'NF>=12 { print $12 }')"
      echo -n "." 2>&1 | tee -a ${FILENAME}/command.log
      COUNTER=$((COUNTER+1))
      sleep 5
    done
    echo " done" 2>&1 | tee -a ${FILENAME}/command.log
fi

# Export ONU information
echo -n "Pulling ONU Configuration information for ${ONU_MAC}" 2>&1 | tee -a ${FILENAME}/command.log
COMMAND="db[\"ONU-CFG\"].find( { _id: \"${ONU_ID}\" } ).pretty()"
mongo ${SECUREOPTIONS} ${DATABASE_HOST}:${DATABASE_PORT}/${DATABASE_NAME} --eval "${COMMAND}" > ${FILENAME}/ONU-CFG-${ONU_ID}.json
echo 2>&1 | tee -a ${FILENAME}/command.log
echo -n "Pulling ONU State information for ${ONU_MAC}" 2>&1 | tee -a ${FILENAME}/command.log
COMMAND="db[\"ONU-STATE\"].find( { _id: \"${ONU_ID}\" } ).pretty()"
mongo ${SECUREOPTIONS} ${DATABASE_HOST}:${DATABASE_PORT}/${DATABASE_NAME} --eval "${COMMAND}" > ${FILENAME}/ONU-STATE-${ONU_ID}.json
echo 2>&1 | tee -a ${FILENAME}/command.log
echo -n "Pulling ONU Syslog information for ${ONU_MAC}" 2>&1 | tee -a ${FILENAME}/command.log
COMMAND="db[\"SYSLOG-ONU-${ONU_ID}\"].find()"
mongo ${SECUREOPTIONS} ${DATABASE_HOST}:${DATABASE_PORT}/${DATABASE_NAME} --eval "${COMMAND}" > ${FILENAME}/SYSLOG-ONU-${ONU_ID}.json
echo 2>&1 | tee -a ${FILENAME}/command.log
echo -n "Pulling ONU Statistics for ${ONU_MAC}" 2>&1 | tee -a ${FILENAME}/command.log
COMMAND="db[\"STATS-ONU-${ONU_ID}\"].find()"
mongo ${SECUREOPTIONS} ${DATABASE_HOST}:${DATABASE_PORT}/${DATABASE_NAME} --eval "${COMMAND}" > ${FILENAME}/STATS-ONU-${ONU_ID}.json
echo 2>&1 | tee -a ${FILENAME}/command.log
echo -n "Pulling ONU MIB Current State for ${ONU_MAC}" 2>&1 | tee -a ${FILENAME}/command.log
COMMAND="db[\"ONU-MIB-CUR-STATE\"].find( { _id: \"${ONU_ID}\" } ).pretty()"
mongo ${SECUREOPTIONS} ${DATABASE_HOST}:${DATABASE_PORT}/${DATABASE_NAME} --eval "${COMMAND}" > ${FILENAME}/ONU-MIB-CUR-STATE-${ONU_ID}.json
echo 2>&1 | tee -a ${FILENAME}/command.log
echo -n "Pulling ONU MIB Reset State for ${ONU_MAC}" 2>&1 | tee -a ${FILENAME}/command.log
COMMAND="db[\"ONU-MIB-RST-STATE\"].find( { _id: \"${ONU_ID}\" } ).pretty()"
mongo ${SECUREOPTIONS} ${DATABASE_HOST}:${DATABASE_PORT}/${DATABASE_NAME} --eval "${COMMAND}" > ${FILENAME}/ONU-MIB-RST-STATE-${ONU_ID}.json
echo 2>&1 | tee -a ${FILENAME}/command.log
echo -n "Pulling ONU Service Configurations" 2>&1 | tee -a ${FILENAME}/command.log
COMMAND="db[\"SRV-CFG\"].find().pretty()"
mongo ${SECUREOPTIONS} ${DATABASE_HOST}:${DATABASE_PORT}/${DATABASE_NAME} --eval "${COMMAND}" > ${FILENAME}/SRV-CFG.json
echo 2>&1 | tee -a ${FILENAME}/command.log
echo -n "Pulling SLA information" 2>&1 | tee -a ${FILENAME}/command.log 
COMMAND="db[\"SLA-CFG\"].find().pretty()"
mongo ${SECUREOPTIONS} ${DATABASE_HOST}:${DATABASE_PORT}/${DATABASE_NAME} --eval "${COMMAND}" > ${FILENAME}/SLA-CFG.json
echo 2>&1 | tee -a ${FILENAME}/command.log

# Export OLT information
echo -n "Pulling OLT Configuration information for ${OLT_MAC}" 2>&1 | tee -a ${FILENAME}/command.log
COMMAND="db[\"OLT-CFG\"].find( { _id: \"${OLT_MAC}\" } ).pretty()"
mongo ${SECUREOPTIONS} ${DATABASE_HOST}:${DATABASE_PORT}/${DATABASE_NAME} --eval "${COMMAND}" > ${FILENAME}/OLT-CFG-${OLT_ID}.json
echo 2>&1 | tee -a ${FILENAME}/command.log
echo -n "Pulling OLT State information for ${OLT_MAC}" 2>&1 | tee -a ${FILENAME}/command.log
COMMAND="db[\"OLT-STATE\"].find( { _id: \"${OLT_MAC}\" } ).pretty()"
mongo ${SECUREOPTIONS} ${DATABASE_HOST}:${DATABASE_PORT}/${DATABASE_NAME} --eval "${COMMAND}" > ${FILENAME}/OLT-STATE-${OLT_ID}.json
echo 2>&1 | tee -a ${FILENAME}/command.log
echo -n "Pulling OLT Debug information for ${OLT_MAC}" 2>&1 | tee -a ${FILENAME}/command.log
COMMAND="db[\"OLT-DEBUG-STATE\"].find( { _id: \"${OLT_MAC}\" } ).pretty()"
mongo ${SECUREOPTIONS} ${DATABASE_HOST}:${DATABASE_PORT}/${DATABASE_NAME} --eval "${COMMAND}" > ${FILENAME}/OLT-DEBUG-STATE-${OLT_ID}.json
echo 2>&1 | tee -a ${FILENAME}/command.log
echo -n "Pulling OLT Syslog information for ${OLT_MAC}" 2>&1 | tee -a ${FILENAME}/command.log
COMMAND="db[\"SYSLOG-OLT-${OLT_ID}\"].find()"
mongo ${SECUREOPTIONS} ${DATABASE_HOST}:${DATABASE_PORT}/${DATABASE_NAME} --eval "${COMMAND}" > ${FILENAME}/SYSLOG-OLT-${OLT_ID}.json
echo 2>&1 | tee -a ${FILENAME}/command.log
echo -n "Pulling OLT Statistics for ${OLT_MAC}" 2>&1 | tee -a ${FILENAME}/command.log
COMMAND="db[\"STATS-OLT-${OLT_ID}\"].find()"
mongo ${SECUREOPTIONS} ${DATABASE_HOST}:${DATABASE_PORT}/${DATABASE_NAME} --eval "${COMMAND}" > ${FILENAME}/STATS-OLT-${OLT_ID}.json
echo 2>&1 | tee -a ${FILENAME}/command.log

#Export Controller Information
echo -n "Pulling Controller Configuration information for ${CNTL_MAC}" 2>&1 | tee -a ${FILENAME}/command.log
COMMAND="db[\"CNTL-CFG\"].find( { _id: \"${CNTL_MAC}\" } ).pretty()"
mongo ${SECUREOPTIONS} ${DATABASE_HOST}:${DATABASE_PORT}/${DATABASE_NAME} --eval "${COMMAND}" > ${FILENAME}/CNTL-CFG-${CNTL_ID}.json
echo 2>&1 | tee -a ${FILENAME}/command.log
echo -n "Pulling Controller State information for ${CNTL_MAC}" 2>&1 | tee -a ${FILENAME}/command.log
COMMAND="db[\"CNTL-STATE\"].find( { _id: \"${CNTL_MAC}\" } ).pretty()"
mongo ${SECUREOPTIONS} ${DATABASE_HOST}:${DATABASE_PORT}/${DATABASE_NAME} --eval "${COMMAND}" > ${FILENAME}/CNTL-STATE-${CNTL_ID}.json
echo 2>&1 | tee -a ${FILENAME}/command.log
echo -n "Pulling Controller Syslog information for ${CNTL_MAC}" 2>&1 | tee -a ${FILENAME}/command.log
COMMAND="db[\"SYSLOG-CNTL-${CNTL_ID}\"].find()"
mongo ${SECUREOPTIONS} ${DATABASE_HOST}:${DATABASE_PORT}/${DATABASE_NAME} --eval "${COMMAND}" > ${FILENAME}/SYSLOG-CNTL-${CNTL_ID}.json
echo 2>&1 | tee -a ${FILENAME}/command.log
echo -n "Pulling Controller Statistics information for ${CNTL_MAC}" 2>&1 | tee -a ${FILENAME}/command.log
COMMAND="db[\"STATS-CNTL-${CNTL_ID}\"].find()"
mongo ${SECUREOPTIONS} ${DATABASE_HOST}:${DATABASE_PORT}/${DATABASE_NAME} --eval "${COMMAND}" > ${FILENAME}/STATS-CNTL-${CNTL_ID}.json
echo 2>&1 | tee -a ${FILENAME}/command.log


#Export ONU Firmware file information
echo -n "Pulling ONU Firmware Files information" 2>&1 | tee -a ${FILENAME}/command.log
COMMAND="db[\"ONU-FIRMWARE.files\"].find().pretty()"
mongo ${SECUREOPTIONS} ${DATABASE_HOST}:${DATABASE_PORT}/${DATABASE_NAME} --eval "${COMMAND}" > ${FILENAME}/ONU-FIRMWARE.files.json
echo 2>&1 | tee -a ${FILENAME}/command.log

#Compress all the .json files into a zip file and delete files
echo "Saving debug information to ${FILENAME}.tgz:" 2>&1 | tee -a ${FILENAME}/command.log
tar czvf ${FILENAME}.tgz ${FILENAME}/ 2>&1


rm -rf ${FILENAME}/ 2>&1
echo
