#!/usr/bin/env python3
#--------------------------------------------------------------------------#
# Copyright (c) 2025, Ciena Corporation                                    #
# All rights reserved.                                                     #
#                                                                          #
#     _______ _____ __    __ ___                                           #
#    / _ __(_) ___//  |  / // _ |                                          #
#   / /   / / /__ / /|| / // / ||                                          #
#  / /___/ / /__ / / ||/ // /__||                                          #
# /_____/_/_____/_/  |__//_/   ||                                          #
#                                                                          #
# Distributed as Ciena-Customer confidential.                              #
#                                                                          #
#--------------------------------------------------------------------------#
""" Disable service on an ONU.

This MCMS REST API example script disables service on an ONU, which
includes removing the ONU from the OLT and disabling and clearing the
service configuration such that subscriber traffic is no longer forwarded.

Example:

  ./disable_service.py --url https://10.2.10.29/api --user <email> --password <password> \
     --olt e8:b4:70:70:0c:9c --olt_tag 200 --onu ALPHe30cadcf --onu_tag 25

usage: disable_service.py [-d DATABASE] [-h] [-l URL] --olt OLT
                          --olt_tag OLT_TAG --onu ONU --onu_tag
                          ONU_TAG [-p PASSWORD] [-u USER] [-v]

"""

import argparse
import datetime
import sys
from api_client import ApiClient
from api_utilities import dict_read, dict_to_list, list_to_dict

def main():
    """ Entry point for the script. """
    parser = argparse.ArgumentParser(add_help=False, formatter_class=argparse.ArgumentDefaultsHelpFormatter)
    parser.add_argument("-d", "--db", action="store", dest="database", default="Default", required=False, help="Name of the database.")
    parser.add_argument("-h", "--help", action="help", default=argparse.SUPPRESS, help="Show this help message and exit.")
    parser.add_argument("-l", "--url", action="store", dest="url", default="https://10.2.10.29/api", required=False, help="URL of the MCMS API server (e.g., https://10.2.10.29/api).")
    parser.add_argument("--olt", action="store", dest="olt", default=None, required=True, help="OLT MAC Address (e.g., e8:b4:70:70:0c:9c)")
    parser.add_argument("--olt_tag", action="store", dest="olt_tag", default=None, required=True, help="Tag to be added by the OLT")
    parser.add_argument("--onu", action="store", dest="onu", default=None, required=True, help="ONU Serial Number (e.g., TBITc84c00df)")
    parser.add_argument("--onu_tag", action="store", dest="onu_tag", default=None, required=True, help="Tag to be added by the ONU")
    parser.add_argument("-p", "--password", action="store", dest="password", default="tibit", required=False, help="User password to authenticate with.")
    parser.add_argument("-u", "--user", action="store", dest="user", default="tibit@tibitcom.com", required=False, help="User email to authenticate with.")
    parser.add_argument("-v", "--verbose", action="store_true", dest="verbose", default=False, required=False, help="Verbose output.")
    parser.parse_args()
    args = parser.parse_args()


    # Instantiate an API Client Connection
    api_client = ApiClient(args.url, args.verbose)

    # Login to the web server
    api_client.login(args.user, args.password)

    # Select the database to use for this session
    api_client.select_database(args.database)

    # Construct the NNI for this service
    nni_tags = f"s{args.olt_tag}.c{args.onu_tag}.c0"

    #
    # Clear the ONU configuration from the OLT
    #

    # Get the OLT configuration that this ONU is attached to
    status, olt_cfg = api_client.request("GET", f"/v1/olts/configs/{args.olt}/")
    if status != 200 or not olt_cfg:
        print(f"ERROR: Configuration for OLT {args.olt} does not exist.")
    else:
        # Remove the ONU from the OLT's ONU Inventory
        if args.onu in olt_cfg["ONUs"]:
            del olt_cfg["ONUs"][args.onu]

        # Remove the NNI Network from the OLT
        nni_networks = dict_read(olt_cfg, "NNI Networks", default=[])
        nni_networks = list_to_dict(nni_networks, "TAG MATCH")
        if nni_tags in nni_networks:
            del nni_networks[nni_tags]
        olt_cfg["NNI Networks"] = dict_to_list(nni_networks, "TAG MATCH")

        # Update the OLT configuration
        status, result = api_client.request("PUT", f"/v1/olts/configs/{args.olt}/", data={"data" : olt_cfg})
        if status != 200:
            print(f"ERROR: Configuration failed for OLT {args.olt}.")
            # Fall through and attempt to clean up the ONU


    #
    # Disable service for this ONU
    #

    # Get the configuration for this ONU
    status, onu_cfg = api_client.request("GET", f"/v1/onus/configs/{args.onu}/")
    if status != 200 or not onu_cfg:
        print(f"ERROR: Configuration for ONU {args.onu} does not exist.")
    else:
        # Clear the NNI Network, PON Network, SLA and disable OLT-Service 0
        onu_cfg["OLT-Service 0"]["Enable"] = False
        onu_cfg["OLT-Service 0"]["DHCP"]["Circuit ID"] = ""
        onu_cfg["OLT-Service 0"]["DHCP"]["Remote ID"] = ""
        onu_cfg["OLT-Service 0"]["Filter"]["DHCPv4"] = "pass"
        onu_cfg["OLT-Service 0"]["Filter"]["DHCPv6"] = "pass"
        onu_cfg["OLT-Service 0"]["NNI Network"] = []
        onu_cfg["OLT-Service 0"]["PON Network"] = []
        onu_cfg["OLT-Service 0"]["SLA-CFG"] = "Min"

        # Set the ONU Service Configuration to 'Disabled'
        onu_cfg["ONU"]["SRV-CFG"] = "Disabled"
        onu_cfg["ONU"]["CVID"] = 0

        # Clear the allowed OLT for this ONU
        onu_cfg["OLT"]["MAC Address"] = []

        # Update the ONU configuration
        status, result = api_client.request("PUT", f"/v1/onus/configs/{args.onu}/", data={"data" : onu_cfg})
        if status != 200:
            print(f"ERROR: Configuration failed for ONU {args.onu}.")

    # Logout of the web server to terminate the session
    api_client.logout()

if __name__ == '__main__':
    main()
