#!/usr/bin/env bash
#--------------------------------------------------------------------------#
#  Copyright (c) 2022-2024, Ciena Corporation                              #
#  All rights reserved.                                                    #
#                                                                          #
#     _______ _____ __    __ ___                                           #
#    / _ __(_) ___//  |  / // _ |                                          #
#   / /   / / /__ / /|| / // / ||                                          #
#  / /___/ / /__ / / ||/ // /__||                                          #
# /_____/_/_____/_/  |__//_/   ||                                          #
#                                                                          #
#  PROPRIETARY NOTICE                                                      #
#  This Software consists of confidential information.                     #
#  Trade secret law and copyright law protect this Software.               #
#  The above notice of copyright on this Software does not indicate        #
#  any actual or intended publication of such Software.                    #
#                                                                          #
#--------------------------------------------------------------------------#

# Usage
usage () {
    cat <<HELP_USAGE

    $0 onu-name

    Reset an ONU device.

    Options:
      -h|--help       Display this help and usage text.
      -p|--port       TCP port number used to connect to the NETCONF Server (default: 830).
      -s|--server     Hostname or IP address used to connect to the NETCONF Server (default: localhost).
      -u|--username   Username with access to the NETCONF Server (default: $USER).
      -w|--password   Password used to access to the NETCONF Server.

    Example:
      $0 ALPHe30cadcf

HELP_USAGE

exit 0
}

# Defaults
HOST=localhost
PORT=830

# Parse command line arguments
PARAMS=""
while (( "$#" )); do
    case "$1" in
        -s|--server)
            HOST=$2
            shift 2
            ;;
        -p|--port)
            PORT=$2
            shift 2
            ;;
        -u|--username)
            USER=$2
            shift 2
            ;;
        -w|--password)
            PASSWD="--password $2"
            shift 2
            ;;
        -h|--help)
            usage
            ;;
        --) # end argument parsing
            shift
            break
            ;;
        -*|--*=) # unsupported option
            echo "Error: Unsupported option $1" >&2
            usage
            ;;
        *) # preserve positional arguments
            PARAMS="$PARAMS $1"
            shift
            ;;
    esac
done
# set positional arguments in their proper place
eval set -- "$PARAMS"

[ -z "$1" ] && { usage; }

ONU="${1}"

echo "Resetting ONU '${ONU}'."

# NETCONF <rpc> for /tibitcntlr:onu-reset
source /opt/tibit/confd/confdrc
${CONFD_DIR}/bin/netconf-console --host ${HOST} --port ${PORT} --user ${USER} ${PASSWD} --rpc - << XMLDATA
<pon-onu-reset xmlns="urn:com:tibitcom:ns:yang:controller:db">
    <name>${ONU}</name>
</pon-onu-reset>
XMLDATA

if [ "$?" = "0" ]; then
    echo "Success."
else
    echo "Failed."
    exit 1
fi

exit 0