#!/usr/bin/env python3
#--------------------------------------------------------------------------#
# Copyright (C) 2022 by Tibit Communications, Inc.                         #
# All rights reserved.                                                     #
#                                                                          #
#    _______ ____  _ ______                                                #
#   /_  __(_) __ )(_)_  __/                                                #
#    / / / / __  / / / /                                                   #
#   / / / / /_/ / / / /                                                    #
#  /_/ /_/_____/_/ /_/                                                     #
#                                                                          #
#  Distributed as Tibit-Customer confidential.                             #
#                                                                          #
#--------------------------------------------------------------------------#
""" Get PON optical levels for an ONU.

This MCMS REST API example script retrieves PON optical levels for a
specific ONU. The script displays optical levels as reported by both the
OLT device and ONU device.

Example:

  ./get_onu_optical_levels.py --url https://10.2.10.29/api --user <email> --password <password> --onu ALPHe30cadcf


usage: get_onu_optical_levels.py [-d DATABASE] [-h] [-l URL] --onu ONU
                                 [-p PASSWORD] [-u USER] [-v]

optional arguments:
  -d DATABASE, --db DATABASE
                        Name of the database. (default: Default)
  -h, --help            Show this help message and exit.
  -l URL, --url URL     URL of the MCMS API server (e.g.,
                        https://10.2.10.29/api). (default:
                        https://10.2.10.29/api)
  --onu ONU             ONU Serial Number (e.g., TBITc84c00df) (default: None)
  -p PASSWORD, --password PASSWORD
                        User password to authenticate with. (default: tibit)
  -u USER, --user USER  User email to authenticate with. (default:
                        tibit@tibitcom.com)
  -v, --verbose         Verbose output. (default: False)

"""

import argparse
import sys
from api_client import ApiClient


def main():
    """ Entry point for the script. """
    parser = argparse.ArgumentParser(add_help=False, formatter_class=argparse.ArgumentDefaultsHelpFormatter)
    parser.add_argument("-d", "--db", action="store", dest="database", default="Default", required=False, help="Name of the database.")
    parser.add_argument("-h", "--help", action="help", default=argparse.SUPPRESS, help="Show this help message and exit.")
    parser.add_argument("-l", "--url", action="store", dest="url", default="https://10.2.10.29/api", required=False, help="URL of the MCMS API server (e.g., https://10.2.10.29/api).")
    parser.add_argument("--onu", action="store", dest="onu", default=None, required=True, help="ONU Serial Number (e.g., TBITc84c00df)")
    parser.add_argument("-p", "--password", action="store", dest="password", default="tibit", required=False, help="User password to authenticate with.")
    parser.add_argument("-u", "--user", action="store", dest="user", default="tibit@tibitcom.com", required=False, help="User email to authenticate with.")
    parser.add_argument("-v", "--verbose", action="store_true", dest="verbose", default=False, required=False, help="Verbose output.")
    parser.parse_args()
    args = parser.parse_args()


    # Instantiate an API Client Connection
    api_client = ApiClient(args.url, args.verbose)

    # Login to the web server
    api_client.login(args.user, args.password)

    # Select the database to use for this session
    api_client.select_database(args.database)

    # Get ONU state data
    status, onu_state = api_client.request("GET", f"/v1/onus/states/{args.onu}/")
    if status != 200 or not onu_state:
        print(f"ERROR: Failed to read state for ONU {args.onu}.")
        sys.exit(1)

    # Get the optical levels as reported by the OLT for this ONU
    if 'OLT-PON' in onu_state['STATS']:
        olt_optical_levels = {
            "tx": onu_state['STATS']['OLT-PON']['TX Optical Level'],
            "rx": onu_state['STATS']['OLT-PON']['RX Optical Level']
        }
    else:
        olt_optical_levels = {"tx": "-", "rx": "-"}

    # Get the optical levels as reported by the ONU
    if 'ONU-PON' in onu_state['STATS']:
        onu_optical_levels = {
            "tx": onu_state['STATS']['ONU-PON']['TX Optical Level'],
            "rx": onu_state['STATS']['ONU-PON']['RX Optical Level']
        }
    else:
        onu_optical_levels = {"tx": "-", "rx": "-"}

    # Display the optical levels for the ONU
    print(f"\nOptical Levels for ONU {args.onu}:")
    print("Downstream:")
    print(f"  OLT TX:  {olt_optical_levels['tx']:>{7}} dB")
    print(f"  ONU RX:  {onu_optical_levels['rx']:>{7}} dB")
    print("Upstream:")
    print(f"  OLT RX:  {olt_optical_levels['rx']:>{7}} dB")
    print(f"  ONU TX:  {onu_optical_levels['tx']:>{7}} dB")

if __name__ == '__main__':
    main()
