"""
#--------------------------------------------------------------------------#
# Copyright (C) 2022 by Tibit Communications, Inc.                         #
# All rights reserved.                                                     #
#                                                                          #
#    _______ ____  _ ______                                                #
#   /_  __(_) __ )(_)_  __/                                                #
#    / / / / __  / / / /                                                   #
#   / / / / /_/ / / / /                                                    #
#  /_/ /_/_____/_/ /_/                                                     #
#                                                                          #
#  Distributed as Tibit-Customer confidential.                             #
#                                                                          #
#--------------------------------------------------------------------------#
"""

import pymongo
from pymongo import monitoring

from log.PonManagerLogger import pon_manager_logger


class MongoServerHeartbeatHandler(monitoring.ServerHeartbeatListener):
    """ Custom handler for pymongo.monitoring.ServerHeartbeatListener responses """

    is_alive: bool = False
    do_log = False  # Makes the first time the sever comes online not create a log message
    stop_listening = False
    status: str
    details: str = "MongoDB server awaiting Heartbeat response"
    name: str

    def __init__(self, is_replica_set=False, name=""):
        """ Initializes the heartbeat listener """
        self.is_replica_set = is_replica_set
        if self.is_replica_set:
            self.hosts = {}
        self.name = name

    def stop(self):
        """ Stop listening for heartbeat to prevent unintended failed message when closing MongoClient """
        self.stop_listening = True

    def started(self, event: monitoring.ServerHeartbeatStartedEvent):
        """ Handle a heart beat started signal """
        if self.is_replica_set:
            if event.connection_id not in self.hosts:
                self.hosts[event.connection_id] = False

    def succeeded(self, event: monitoring.ServerHeartbeatSucceededEvent):
        """ Handle a successful heart beat response """
        if not self.stop_listening:
            host = event.connection_id[0]
            port = event.connection_id[1]

            if self.is_replica_set:
                if not self.hosts[event.connection_id] and self.do_log:
                    pon_manager_logger.info(f"MongoDB server at {host}:{port} has come online")

                self.hosts[event.connection_id] = True
            elif not self.is_alive and self.do_log:
                pon_manager_logger.info(f"MongoDB server at {host}:{port} has come online")

            self.is_alive = True
            self.do_log = True

    def failed(self, event: monitoring.ServerHeartbeatFailedEvent):
        """ Handle a failed heart beat response """
        if not self.stop_listening:
            host = event.connection_id[0]
            port = event.connection_id[1]
            log_message = f"MongoDB server at {host}:{port} has gone offline due to Heartbeat failure"
            self.status = "Offline"

            if not isinstance(event.reply, pymongo.errors.NotMasterError):
                log_message += f": {str(event.reply)}"

            self.details = log_message

            if self.is_replica_set:
                # If using a replica set the mongo database is still online as long as
                # at least one of the hosts is accessible. Each replica set host server
                # has its own heartbeat, so they must be tracked separately to not
                # override one another.
                if self.hosts[event.connection_id] and self.do_log:
                    pon_manager_logger.error(log_message)

                self.hosts[event.connection_id] = False
                self.is_alive = any(self.hosts.values())
            else:
                if self.is_alive and self.do_log:
                    pon_manager_logger.error(log_message)

                self.is_alive = False
