"""
#--------------------------------------------------------------------------#
# Copyright (C) 2022 by Tibit Communications, Inc.                         #
# All rights reserved.                                                     #
#                                                                          #
#    _______ ____  _ ______                                                #
#   /_  __(_) __ )(_)_  __/                                                #
#    / / / / __  / / / /                                                   #
#   / / / / /_/ / / / /                                                    #
#  /_/ /_/_____/_/ /_/                                                     #
#                                                                          #
# Distributed as Tibit-Customer confidential.                              #
#                                                                          #
#--------------------------------------------------------------------------#
"""

from django.contrib.auth.decorators import permission_required
from django.contrib.auth.mixins import LoginRequiredMixin
from django.utils.decorators import method_decorator
from rest_framework.generics import GenericAPIView
from drf_spectacular.utils import extend_schema
from rest_framework import status

from database_manager import database_manager
from utils.schema_helpers import ResponseExample
from utils.tools import PonManagerApiResponse, validate_query_params


# ==================================================
# =============== One CPE State View ===============
# ==================================================
class OneCpeState(LoginRequiredMixin, GenericAPIView):
    raise_exception = True
    queryset = ''

    @extend_schema(
        operation_id="get_one_cpe_state",
        responses={
            200: ResponseExample(200),
            403: ResponseExample(403),
            500: ResponseExample(500),
        },
        tags=['cpe', 'get']
    )
    @method_decorator(permission_required('network.can_read_network_onus', raise_exception=True))
    def get(self, request, cpe_id, version):
        """Get the specified CPE State"""
        res_data = database_manager.find_one(database_id=request.session.get('database'), collection="CPE-STATE", query={"_id": cpe_id})

        return PonManagerApiResponse(status=status.HTTP_200_OK, data=res_data)

    @extend_schema(
        operation_id="delete_one_cpe_state",
        responses={
            204: ResponseExample(204),
            403: ResponseExample(403),
            500: ResponseExample(500),
        },
        tags=['cpe', 'delete']
    )
    @method_decorator(permission_required('network.can_delete_network_onus', raise_exception=True))
    def delete(self, request, cpe_id, version):
        """Delete the specified CPE State"""
        database_manager.delete_one(database_id=request.session.get('database'), collection="CPE-STATE", query={"_id": cpe_id})

        return PonManagerApiResponse(status=status.HTTP_204_NO_CONTENT)


# ==================================================
# ================= CPE States View ================
# ==================================================
class CpeStates(LoginRequiredMixin, GenericAPIView):
    raise_exception = True
    queryset = ''

    @extend_schema(
        operation_id="get_cpe_states",
        responses={
            200: ResponseExample(200),
            403: ResponseExample(403),
            500: ResponseExample(500),
        },
        tags=['cpe', 'get']
    )
    @method_decorator(permission_required('network.can_read_network_onus', raise_exception=True))
    @validate_query_params(collection="CPE-STATE")
    def get(self, request, query, projection, sort, limit, skip, next, distinct, version):
        """Get all CPE States"""
        if distinct:
            res_data = database_manager.distinct(database_id=request.session.get('database'), collection="CPE-STATE",
                                             query=query, distinct = distinct)
        else:
            res_data = database_manager.find(database_id=request.session.get('database'), collection="CPE-STATE",
                                             query=query, projection=projection, sort=sort, limit=limit, skip=skip, next=next)

        return PonManagerApiResponse(status=status.HTTP_200_OK, data=res_data)
