"""
#--------------------------------------------------------------------------#
# Copyright (C) 2022 by Tibit Communications, Inc.                         #
# All rights reserved.                                                     #
#                                                                          #
#    _______ ____  _ ______                                                #
#   /_  __(_) __ )(_)_  __/                                                #
#    / / / / __  / / / /                                                   #
#   / / / / /_/ / / / /                                                    #
#  /_/ /_/_____/_/ /_/                                                     #
#                                                                          #
# Distributed as Tibit-Customer confidential.                              #
#                                                                          #
#--------------------------------------------------------------------------#
======================================================

Django project configurations
"""

import json

APPEND_SLASH = False
ALLOWED_HOSTS = ['*']
USE_X_FORWARDED_HOST = True

IN_PRODUCTION = True

# Check if in development mode
try:
    with open("/opt/tibit/ponmgr/api/api/settings.py") as json_file:
        IN_PRODUCTION = True
except FileNotFoundError:
    IN_PRODUCTION = False

DEBUG = not IN_PRODUCTION

# Grab secret key from user_database.json
try:
    if IN_PRODUCTION:
        with open("/opt/tibit/ponmgr/api/user_database.json") as json_file:
            user_db_data = json.load(json_file)
            if user_db_data and "django_key" in user_db_data:
                SECRET_KEY = user_db_data["django_key"]
    else:
        with open("user_database.json") as json_file:
            user_db_data = json.load(json_file)
            if user_db_data and "django_key" in user_db_data:
                SECRET_KEY = user_db_data["django_key"]
except FileNotFoundError:
    SECRET_KEY = ''

INSTALLED_APPS = [
    'django.contrib.admin',
    'django.contrib.auth',
    'django.contrib.contenttypes',
    'django.contrib.sessions',
    'django.contrib.messages',
    'rest_framework',
    'corsheaders',
    'rest_framework_swagger',
    'drf_spectacular',
    'databases',
    'users',
    'controllers',
    'switches',
    'olts',
    'onus',
    'cpes',
    'files',
    'slas',
    'service_configs',
    'downstream_maps',
    'pon_automation',
    'ponmgr',
    'docs',
    'DBdjango.apps.DiscoverConfig',
]

MIDDLEWARE = [
    'django.middleware.security.SecurityMiddleware',
    'django.contrib.sessions.middleware.SessionMiddleware',
    'corsheaders.middleware.CorsMiddleware',
    'django.middleware.common.CommonMiddleware',
    'django.middleware.csrf.CsrfViewMiddleware',
    'django.contrib.auth.middleware.AuthenticationMiddleware',
    'django.contrib.messages.middleware.MessageMiddleware',
    'django.middleware.clickjacking.XFrameOptionsMiddleware',
    'utils.middleware.set_headers',
    'utils.middleware.handle_server_errors',
    'utils.middleware.log_request',
    'utils.middleware.update_session_database_id'
]

# Retrieving logging configuration json file
try:
    if IN_PRODUCTION:
        with open("/var/www/html/api/logging_configuration.json") as json_file:
            LOGGING_OPTIONS = json.load(json_file)
    else:
        with open("logging_configuration.json") as json_file:
            LOGGING_OPTIONS = json.load(json_file)
except:
    # Set default values in case of file failure
    LOGGING_OPTIONS = {
        "Directory": "/var/log/tibit",
        "FileCount": 3,
        "FileSize": 1024000
    }

if LOGGING_OPTIONS['Directory'].endswith("/"):
    log_file_path = LOGGING_OPTIONS['Directory'] + 'ponMgr.log'
else:
    log_file_path = LOGGING_OPTIONS['Directory'] + '/ponMgr.log'

# Settings here are only to disable default django logging
LOGGING = {
    'version': 1,
    'disable_existing_loggers': False,
    'filters': {
        'require_debug_true': {
            '()': 'django.utils.log.RequireDebugTrue',
        }
    },
    'handlers': {
        # Used to silence unwanted logs
        'null': {
            'class': 'logging.NullHandler',
        },
    },
    'loggers': {
        'django': {
            'propagate': False
        },
        # Silence default error logs to use custom logging
        'django.request': {
            'handlers': ['null'],
            'propagate': False
        },
        # Silence logs when running via runserver
        'django.server': {
            'filters': ['require_debug_true'],
            'propagate': False
        }
    }
}

AUTHENTICATION_BACKENDS = [
    'django.contrib.auth.backends.ModelBackend'
]

REST_FRAMEWORK = {
    'DEFAULT_AUTHENTICATION_CLASSES': ['rest_framework.authentication.SessionAuthentication'],
    'DEFAULT_SCHEMA_CLASS': 'drf_spectacular.openapi.AutoSchema',
    'DEFAULT_VERSIONING_CLASS': 'rest_framework.versioning.URLPathVersioning',
    'ALLOWED_VERSIONS': 'v1|v2',
    'DEFAULT_VERSION': 'v2'
}

# Used for generating OpenAPI yaml Swagger and ReDoc UIs
SPECTACULAR_SETTINGS = {
    'TITLE': 'MCMS PON Manager REST API',
    'DESCRIPTION': 'REST API for interfacing with Tibit MCMS PON Controller databases',
    'VERSION': 'R2.3.0',
    'CONTACT': {
        'url': 'https://tibitcom.com/',
        'name': 'TiBiT Communications Inc.'
    },
    'SWAGGER_UI_SETTINGS': {  # Only applies to Swagger UI, not ReDoc
        'filter': True,  # Adds filter input
        'docExpansion': 'none',  # Collapses the tag entries by default
        'deepLinking': True,  # Retains users spot on the page, allows link sharing to a specific tag/operation
        'tagsSorter': 'alpha'  # Sorts the tags alphabetically
    },
    'PREPROCESSING_HOOKS': [
        'utils.drf_hooks.exclude_private_endpoints',
        'drf_spectacular.hooks.preprocess_exclude_path_format'
    ]
}

ROOT_URLCONF = 'api.urls'

TEMPLATES = [
    {
        'BACKEND': 'django.template.backends.django.DjangoTemplates',
        'DIRS': [
            '/var/www/html',
            '/var/www/tibitdev.localhost/public_html'
        ],
        'APP_DIRS': True,
        'OPTIONS': {
            'context_processors': [
                'django.template.context_processors.debug',
                'django.template.context_processors.request',
                'django.contrib.auth.context_processors.auth',
                'django.contrib.messages.context_processors.messages',
            ],
        },
    },
]

WSGI_APPLICATION = 'api.wsgi.application'

# Retrieving database json files
if IN_PRODUCTION:
    with open("/var/www/html/api/user_database.json") as json_file:
        database_options = json.load(json_file)
    with open("/var/www/html/api/recovery_email_configuration.json") as json_file:
        recovery_email_options = json.load(json_file)
else:
    with open("user_database.json") as json_file:
        database_options = json.load(json_file)
    with open("recovery_email_configuration.json") as json_file:
        recovery_email_options = json.load(json_file)

user_database_info = {
    "host": "127.0.0.1",
    "name": "tibit_users",
    "port": "27017",
    "auth_enable": False,
    "auth_db": "tibit_users",
    "username": "",
    "password": "",
    "tls_enable": False,
    "ca_cert_path": "",
    "dns_srv": False,
    "db_uri": "",
    "replica_set_enable": False,
    "replica_set_name": "",
    "replica_set_hosts": []
}

user_database_info.update(database_options)

DATABASES = {
    'default': {
        'ENGINE': 'djongo',
        'NAME': user_database_info["name"]
    }
}

replica_set_enabled = user_database_info["replica_set_enable"]
if replica_set_enabled:
    DATABASES['default']['HOST'] = ",".join(user_database_info["replica_set_hosts"])
    DATABASES['default']['REPLICASET'] = user_database_info["replica_set_name"]
else:
    DATABASES['default']['HOST'] = user_database_info["host"]
    DATABASES['default']['PORT'] = int(user_database_info["port"])

auth_enabled = user_database_info["auth_enable"]
if auth_enabled:
    DATABASES['default']['USER'] = user_database_info["username"]
    DATABASES['default']['PASSWORD'] = user_database_info["password"]
    DATABASES['default']['AUTH_SOURCE'] = user_database_info["auth_db"]

tls_enable = user_database_info["tls_enable"]
if tls_enable:
    DATABASES['default']['SSL'] = True
    DATABASES['default']['SSL_CA_CERTS'] = user_database_info["ca_cert_path"]

AUTH_PASSWORD_VALIDATORS = [
    {
        'NAME': 'django.contrib.auth.password_validation.UserAttributeSimilarityValidator',
    },
    {
        'NAME': 'django.contrib.auth.password_validation.MinimumLengthValidator',
    },
    {
        'NAME': 'django.contrib.auth.password_validation.CommonPasswordValidator',
    },
    {
        'NAME': 'django.contrib.auth.password_validation.NumericPasswordValidator',
    },
]

# Internationalization
# https://docs.djangoproject.com/en/1.9/topics/i18n/
LANGUAGE_CODE = 'en-us'
TIME_ZONE = 'UTC'
USE_I18N = True
USE_L10N = True
USE_TZ = True

# SMTP server config for recovery email
EMAIL_BACKEND = 'django.core.mail.backends.smtp.EmailBackend'
EMAIL_HOST = recovery_email_options['host']
EMAIL_PORT = recovery_email_options['port']
EMAIL_USE_TLS = recovery_email_options['use_tls'].lower() == "true"
EMAIL_HOST_USER = recovery_email_options['user']
EMAIL_HOST_PASSWORD = recovery_email_options['password']

# Session Settings
SESSION_SAVE_EVERY_REQUEST = True
SESSION_COOKIE_AGE = 1200  # Default 20 minutes
SESSION_COOKIE_SAMESITE = 'Strict'
SESSION_COOKIE_NAME = '__Host-sessionid'
SESSION_COOKIE_SECURE = True

# CSRF Settings
CSRF_COOKIE_SAMESITE = 'Strict'
CSRF_COOKIE_NAME = '__Host-csrftoken'
CSRF_COOKIE_SECURE = True

# If in development mode, overwrite settings above.
# NOTE: None of this section needs to be modified if enabling non-secure HTTP
if IN_PRODUCTION is False:
    CSRF_COOKIE_NAME = 'csrftoken'
    CSRF_COOKIE_SECURE = False
    SESSION_COOKIE_NAME = 'sessionid'
    SESSION_COOKIE_SECURE = False

# Swagger api documentation for version 2.2
# https://django-rest-swagger.readthedocs.io/en/stable-0.3.x/settings.html
SWAGGER_SETTINGS = {
    'api_version': '0.1',
    'JSON_EDITOR': True,
    'enabled_methods': [
        'get',
        'post',
        'put',
        'patch',
        'delete'
    ],
}
