"""
#--------------------------------------------------------------------------#
# Copyright (C) 2022 by Tibit Communications, Inc.                         #
# All rights reserved.                                                     #
#                                                                          #
#    _______ ____  _ ______                                                #
#   /_  __(_) __ )(_)_  __/                                                #
#    / / / / __  / / / /                                                   #
#   / / / / /_/ / / / /                                                    #
#  /_/ /_/_____/_/ /_/                                                     #
#                                                                          #
# Distributed as Tibit-Customer confidential.                              #
#                                                                          #
#--------------------------------------------------------------------------#

Maps url endpoints to database connector
"""

import json
import os
import traceback
import tarfile
import threading
import hashlib

import django.contrib.auth.models
from django.contrib.auth.mixins import LoginRequiredMixin
from django.http import JsonResponse, FileResponse
from django.http import HttpResponse
from django.contrib.auth.models import User, Permission, ContentType, Group
from django.contrib.sessions.models import Session
from django.contrib.auth import authenticate, login, logout, update_session_auth_hash
from djongo.sql2mongo import SQLDecodeError
from rest_framework.views import APIView
from rest_framework.response import Response
from rest_framework import status, generics
from django.conf import settings
import django.core.serializers as serializers
import uuid
from datetime import datetime, timedelta, timezone
from django.core.mail import send_mail
import sys
from django.forms.models import model_to_dict
from django.template import loader

from database_manager import database_manager
from zxcvbn import zxcvbn

from log.PonManagerLogger import pon_manager_logger
from utils.tools import PonManagerApiResponse

sys.path.append('DBdjango/')  # Allows use of module outside of IDE
from DBdjango.database_connector import MongoDBConnector
from utils.tools import get_nested_value
from manage import BUILDING_DOCUMENTATION

UserRecoveryCodes = {}
dbs = {}
clients = {}
sessionPurgeTime = datetime.now()
development_databases_files = "databases.json"
development_omci_info = "omci.json"
development_search_states = [
    "searchColls/CNTL-STATE.json",
    "searchColls/OLT-STATE.json",
    "searchColls/ONU-STATE.json",
    "searchColls/ONU-MIB-CUR-STATE.json",
    "searchColls/ONU-MIB-RST-STATE.json"
]
production_databases_files = "/var/www/html/api/databases.json"
production_omci_info = "/var/www/html/api/omci.json"
production_search_states = [
    "/var/www/html/api/searchColls/CNTL-STATE.json",
    "/var/www/html/api/searchColls/OLT-STATE.json",
    "/var/www/html/api/searchColls/ONU-STATE.json",
    "/var/www/html/api/searchColls/ONU-MIB-CUR-STATE.json",
    "/var/www/html/api/searchColls/ONU-MIB-RST-STATE.json"
]


def start_up():
    """Creates an instance of MongoDBConnector for each database defined in databases.json"""
    for database in database_manager.list_databases():
        dbs[database] = MongoDBConnector(database)
    return True

if BUILDING_DOCUMENTATION:
    print('Skipping Startup for Documentation')
else:
    start_up()

# Returns the ID of the database that the given client is using
def get_client_db(client):
    try:
        if client not in clients:
            clients[client] = database_manager.get_users_selected_database(client)
    except KeyError:
        clients[client] = "Default"

    return dbs[clients[client]]


# Purges expired sessions IF one day or more has elapsed since last purging. Triggered by API call(s)
def purge_expired_sessions(request):
    global sessionPurgeTime
    if datetime.now() >= sessionPurgeTime:
        request.session.clear_expired()
        sessionPurgeTime = datetime.now() + timedelta(days=1)  # Updating purge time to be one day from now

        try:
            database_manager.remove_session_database_id(session_key=request.session.session_key)
        except KeyError:
            pass


# Returns dictionary for Headers to be added to HTTP Response
def set_response_headers(request):
    cookie_id = "__Host-sessionexpire="
    secure_flag = "Secure"

    if  "__Host-" not in settings.CSRF_COOKIE_NAME:
        cookie_id = "sessionexpire="
        secure_flag = ""
    headers = {'set-cookie': cookie_id + request.session.get_expiry_date().strftime('%Y-%m-%dT%H:%M:%SZ') + "; expires=" + request.session.get_expiry_date().strftime('%a, %d %b %Y %H:%M:%S GMT') + "; Max-Age=31449600; Path=/; SameSite=Strict; " + secure_flag}
    return headers


# Create your views here.
# Django HTTP status codes reference: https://www.django-rest-framework.org/api-guide/status-codes/

class ApiLanding(generics.GenericAPIView):
    def get(self, request):
        template = loader.get_template("api/apiStatus.html")
        return HttpResponse(template.render())


class ReplaceSla(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def put(self, request, sla):
        if "global_config.can_delete_global_config_slas" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            response = db.replace_sla(sla, request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to update this data. See a system administrator.", None, None]
        return PonManagerApiResponse(status=response[0], data=response[1], new_data=response[2], old_data=response[3], headers=set_response_headers(request))


class ReplaceSrv(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def put(self, request, srv):
        if "global_config.can_delete_global_config_services" in User.get_all_permissions(User.objects.get(email=request.user.email)) or "global_config.can_delete_global_config_files" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            response = db.replace_srv(srv, request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to update this data. See a system administrator.", None, None]
        return PonManagerApiResponse(status=response[0], data=response[1], new_data=response[2], old_data=response[3], headers=set_response_headers(request))


class GetAllOnuLabels(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        if "network.can_read_network_onus" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            response = db.get_all_onu_labels(request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class GetAllOltLabels(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        if "network.can_read_network_olts" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            response = db.get_all_olt_labels(request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))

class GetFirstAssignedOltFirmware(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request, firmware):
        if "network.can_read_network_olts" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            response = db.get_first_assigned_olt_firmware(request.user.email, firmware, request.GET.get('limit', 1))
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))

class GetAllControllerLabels(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        if "network.can_read_network_controllers" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            response = db.get_all_cntl_labels(request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class GetAllSwitchLabels(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        if "network.can_read_network_switches" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            response = db.get_all_switch_labels(request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class GetAllSwitchRegionsPops(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        if "network.can_read_network_switches" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            response = db.get_all_switch_regions_pops(request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class GetSwitchRegionPop(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request, mac_address):
        if "network.can_read_network_switches" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            response = db.get_switch_region_pop(request.user.email, mac_address)
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class UploadImage(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def post(self, request):
        if "global_config.can_create_global_config_files" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            response = db.upload_image(request.data, request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to create this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))

    def put(self, request):
        if "global_config.can_update_global_config_files" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            response = db.update_image(request.data['fileId'], request.data['metadata'], request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to update this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))

    def delete(self, request, file_id):
        if "global_config.can_delete_global_config_files" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            response = db.delete_picture(file_id, request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to delete this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))

    def get(self, request):
        if "global_config.can_read_global_config_files" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            response = db.get_all_pictures_info(request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class UploadLogoImage(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def post(self, request):
        if "other.can_create_other_branding" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            response = db.upload_logo_image(request.data, request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to write this data. See a system administrator.", None, None]
        return PonManagerApiResponse(status=response[0], data=response[1], new_data=response[2], old_data=response[3], headers=set_response_headers(request))


class GetControllerSummary(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request, mac_address):
        if 'network.can_read_network_controllers' in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            response = db.get_cntl_summary(request.user.email, mac_address)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to write this data. See a system administrator.", None, None]
        return PonManagerApiResponse(status=response[0], data=response[1], headers=set_response_headers(request))


class GetOltSummary(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request, mac_address):
        if 'network.can_read_network_olts' in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            response = db.get_olt_summary(request.user.email, mac_address)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to write this data. See a system administrator.", None, None]
        return PonManagerApiResponse(status=response[0], data=response[1], headers=set_response_headers(request))


class GetOnuSummary(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request, mac_address):
        if 'network.can_read_network_onus' in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            response = db.get_onu_summary(request.user.email, mac_address)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to write this data. See a system administrator.", None, None]
        return PonManagerApiResponse(status=response[0], data=response[1], headers=set_response_headers(request))


class GetPonAutoStateCount(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        if 'global_config.can_read_automation' in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            response = db.get_pon_auto_state_count()
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class GetPonAutoServicesForTree(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        if 'global_config.can_read_automation' in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            response = db.get_pon_auto_services_for_tree()
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class GetPonAutoStatus(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        if 'global_config.can_read_automation' in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            response = db.get_pon_auto_status(request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


# TODO No permissions check on this endpoint as of now. Not sure what an appropriate check(s) is?
class GetMgmtLansForTree(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        db = get_client_db(request.user.email)
        response = db.get_mgmt_lans_for_tree(request.user.email)
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class GetAllCntlsForTree(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request, mgmt_lan):
        if "network.can_read_network_controllers" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            response = db.get_cntls_for_tree(request.user.email, mgmt_lan)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class GetSwitchRegionsForTree(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        if "network.can_read_network_switches" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            response = db.get_regions_for_tree(request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class GetPopsUnderRegionForTree(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request, region):
        if "network.can_read_network_switches" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            response = db.get_pops_under_region_for_tree(request.user.email, region)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class GetSwitchesUnderPopForTree(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request, region, pop):
        if "network.can_read_network_switches" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            response = db.get_switches_under_pop_for_tree(request.user.email, region, pop)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class GetOltsUnderSwitchForTree(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request, switch_id):
        if "network.can_read_network_olts" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            response = db.get_olts_under_switch_for_tree(request.user.email, switch_id)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class GetOltsUnderControllerForTree(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request, controller_id):
        if "network.can_read_network_olts" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            response = db.get_olts_under_controller_for_tree(request.user.email, controller_id)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class GetOnusUnderOltForTree(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request, olt_id):
        if "network.can_read_network_onus" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            response = db.get_onus_under_olt_for_tree(request.user.email, olt_id)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class UploadOltFirmware(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def post(self, request):
        if "global_config.can_create_global_config_files" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            response = db.upload_olt_firmware(request.data, request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to write this data. See a system administrator.", None, None]
        return PonManagerApiResponse(status=response[0], data=response[1], new_data=response[2], old_data=response[3], headers=set_response_headers(request))

    def put(self, request):
        if "global_config.can_update_global_config_files" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            response = db.update_olt_fw_metadata(request.data, request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to update this data. See a system administrator.", None, None]
        return PonManagerApiResponse(status=response[0], data=response[1], new_data=response[2], old_data=response[3], headers=set_response_headers(request))

    def get(self, request):
        if "global_config.can_read_global_config_files" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            response = db.get_olt_compatibility_data(request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))

    def delete(self, request, file_id):
        if "global_config.can_delete_global_config_files" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            response = db.delete_olt_firmware(file_id, request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to delete this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class UploadOnuFirmware(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def post(self, request):
        if "global_config.can_create_global_config_files" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            response = db.upload_onu_firmware(request.data, request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to write this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))

    def put(self, request):
        if "global_config.can_update_global_config_files" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            response = db.update_onu_fw_metadata(request.data, request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to update this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))

    def delete(self, request, file_id):
        if "global_config.can_delete_global_config_files" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            response = db.delete_onu_firmware(file_id, request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to delete this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class GetOlts(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        if "global_config.can_read_global_config_files" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            response = db.get_all_olts(request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class OltFirmwareMetaData(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        if "global_config.can_read_global_config_files" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            response = db.get_all_olt_firmwares_info(request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class OnuFirmwareMetaData(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        if "global_config.can_read_global_config_files" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            response = db.get_all_onu_firmwares_info(request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class GetOltsForUnknownSwitch(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        if "network.can_read_network_olts" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            response = db.get_all_olts_for_unknown_switch(request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class GetCntls(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        if "network.can_read_network_controllers" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            response = db.get_all_controllers(request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class GetAllCntlStates(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        if "network.can_read_network_controllers" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            response = db.get_all_cntl_states(request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class GetOltsForSwitch(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request, mac_address):
        if "network.can_read_network_olts" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            response = db.get_all_olts_for_switch(mac_address, request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class GetOltsInSwitchInventory(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request, mac_address):
        if "network.can_read_network_switches" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            response = db.get_all_olts_in_switch_inventory(mac_address, request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class GetAllOltsForController(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request, mac_address):
        if "network.can_read_network_olts" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            response = db.get_all_olts_for_controller(mac_address, request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class GetAllOltsWithVersions(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        if "network.can_read_network_olts" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            response = db.get_all_olts_with_versions(request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class GetSwitches(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        if "network.can_read_network_switches" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            response = db.get_all_switches(request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class GetAllSwitchesWithVersions(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        if "network.can_read_network_switches" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            response = db.get_all_switches_with_versions(request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class GetSwitchesForCntl(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request, mac_address):
        if "network.can_read_network_switches" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            response = db.get_all_switches_for_cntl(mac_address, request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class GetAllOltStates(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        if "network.can_read_network_olts" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            response = db.get_all_olt_states(request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class OltDebugState(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request, mac_address):
        if "network.can_read_network_olts" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            response = db.download_olt_debug_state(mac_address, request.user.email)

            if response[0] == status.HTTP_200_OK:
                return FileResponse(json.dumps(response[1]), status=response[0], content_type="text/json", as_attachment=True,
                                    filename=f"OLT-{mac_address.replace(':', '_')}-DEBUG-STATE.json")
            else:
                return Response(status=response[0], data=response[1], headers=set_response_headers(request))
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to read this data. See a system administrator."]
            return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class GetOnus(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        if "network.can_read_network_onus" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            response = db.get_all_onus(request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))

class ProtectionAutomaticSwitchover(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def put(self, request, mac_address):
        if "network.can_update_network_olts" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            body = json.loads(request.body)
            auto_switchover_status = body['Auto Switchover Status']
            response = db.put_protection_automatic_switchover(request.user.email, mac_address, auto_switchover_status)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to read this data. See a system administrator.", None, None]
        return PonManagerApiResponse(status=response[0], data=response[1], new_data=response[2], old_data=response[3], headers=set_response_headers(request))

class ProtectionForcedSwitchover(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def put(self, request, mac_address):
        if "network.can_update_network_olts" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            response = db.put_protection_forced_switchover(request.user.email, mac_address)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to read this data. See a system administrator.", None, None]
        return PonManagerApiResponse(status=response[0], data=response[1], new_data=response[2], old_data=response[3], headers=set_response_headers(request))

class ProtectionCfg(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def put(self, request, mac_address):
        if "network.can_update_network_olts" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            body = json.loads(request.body)
            peer = body['Peer']
            inactivity_detection_time_active = body['Inactivity Detection Time When Active']
            inactivity_detection_time_standby = body['Inactivity Detection Time When Standby']
            response = db.put_protection_cfg(request.user.email, mac_address, peer, inactivity_detection_time_active, inactivity_detection_time_standby)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to read this data. See a system administrator.", None, None]
        return PonManagerApiResponse(status=response[0], data=response[1], new_data=response[2], old_data=response[3], headers=set_response_headers(request))

class GetAllGponOnuSerialNumbers(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        if "network.can_read_network_onus" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            response = db.get_all_gpon_onu_serial_numbers(request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class GetAllOnusWithVersions(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        if "network.can_read_network_onus" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            response = db.get_all_onus_with_versions(request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class GetUnspecifiedOnus(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        if "network.can_read_network_onus" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            response = db.get_all_unspecified_onus(request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class GetOnusForOlt(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request, mac_address):
        if "network.can_read_network_onus" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            response = db.get_all_onus_for_olt(mac_address, request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class GetOnlineOnusForController(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request, mac_address):
        if "network.can_read_network_onus" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            response = db.get_online_onus_for_controller(mac_address, request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class GetOnusUnderOlt(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request, mac_address):
        if "network.can_read_network_onus" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            response = db.get_all_onus_under_olt(mac_address, request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class GetOnuStates(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        if "network.can_read_network_onus" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            response = db.get_all_onu_states(request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class GetAllOnuStatesForOlt(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request, mac_address):
        if "network.can_read_network_onus" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            response = db.get_all_onu_states_for_olt(mac_address, request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class GetFirstAssignedOnuFirmware(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request, firmware):
        if "network.can_read_network_onus" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            response = db.get_first_assigned_onu_firmware(request.user.email, firmware, request.GET.get('limit', 1))
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class GetFirstAssignedOnuSla(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request, sla):
        if "network.can_read_network_onus" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            response = db.get_first_assigned_onu_sla(request.user.email, sla)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class CntlState(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request, mac_address):
        if "network.can_read_network_controllers" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            response = db.get_cntl_state(mac_address, request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class SwitchState(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request, mac_address):
        if "network.can_read_network_switches" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            response = db.get_switch_state(mac_address, request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class OltState(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request, mac_address):
        if "network.can_read_network_olts" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            response = db.get_olt_state(mac_address, request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class OnuState(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request, mac_address):
        if "network.can_read_network_onus" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            response = db.get_onu_state(mac_address, request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class ControllerStatistics(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request, mac_address, since_utc_time):
        if "network.can_read_network_controllers" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            response = db.get_cntl_stats(mac_address, request.user.email, since_utc_time)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class GetRangedControllerStatistics(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request, mac_address, since_utc_time, to_utc_time):
        if "network.can_read_network_onus" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            response = db.get_cntl_stats(mac_address, request.user.email, since_utc_time, to_utc_time)
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class OltStatistics(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request, mac_address, since_utc_time):
        if "network.can_read_network_olts" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            response = db.get_olt_stats(mac_address, request.user.email, since_utc_time)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class GetRangedOltStatistics(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request, mac_address, since_utc_time, to_utc_time):
        if "network.can_read_network_onus" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            response = db.get_olt_stats(mac_address, request.user.email, since_utc_time, to_utc_time)
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class GetAllOltStatistics(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request, since_utc_time):
        if "network.can_read_network_olts" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            response = db.get_all_olt_stats(since_utc_time, request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class OnuStatistics(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request, mac_address, since_utc_time):
        if "network.can_read_network_onus" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            response = db.get_onu_stats(mac_address, request.user.email, since_utc_time)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class GetRangedOnuStatistics(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request, mac_address, since_utc_time, to_utc_time):
        if "network.can_read_network_onus" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            response = db.get_onu_stats(mac_address, request.user.email, since_utc_time, to_utc_time)
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class GetManyOnuStatistics(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        params = request.query_params
        onus = params["onus"]
        since_utc_time = params["start-time"]
        to_utc_time = get_nested_value(params, ["end-time"], None)

        if "network.can_read_network_onus" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            response = db.get_many_onu_stats(onus, request.user.email, since_utc_time, to_utc_time)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))

class GetManyDistinctOnuStatistics(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        params = request.query_params
        onus = params["onus"]
        stats = params["stats"]
        since_utc_time = params["start-time"]
        to_utc_time = get_nested_value(params, ["end-time"], None)

        if "network.can_read_network_onus" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            response = db.get_many_distinct_onu_stats(onus, stats, request.user.email, since_utc_time, to_utc_time)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class GetOnuStatisticsTotals(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        params = request.query_params
        onus = params["onus"]
        since_utc_time = params["start-time"]
        to_utc_time = get_nested_value(params, ["end-time"], None)

        if "network.can_read_network_onus" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            response = db.get_onu_stats_totals(onus, request.user.email, since_utc_time, to_utc_time)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class GetAllOnuStatistics(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request, since_utc_time):
        if "network.can_read_network_onus" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            response = db.get_all_onu_stats(since_utc_time, request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class GetAllCntlCfgs(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        if "network.can_read_network_controllers" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            response = db.get_all_cntl_cfgs(request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class GetAllOltCfgs(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        if "network.can_read_network_olts" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            response = db.get_all_olt_cfgs(request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class GetAllOnuCfgs(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        if "network.can_read_network_onus" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            response = db.get_all_onu_cfgs(request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class CntlCfg(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request, mac_address):
        if "network.can_read_network_controllers" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            response = db.get_cntl_cfg(mac_address, request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))

    def put(self, request, mac_address):
        if "network.can_update_network_controllers" in User.get_all_permissions(User.objects.get(email=request.user.email)) or "network.can_create_network_olts" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            cntl_config_data = request.data
            res1 = db.put_cntl_cfg(mac_address, cntl_config_data, request.user.email)
            if res1[0] == 200:
                new_mgmt_lan_name = cntl_config_data['data']['cntlConfig']['MGMT LAN']['Name']
                res2 = db.update_mgmt_lan_names(mac_address, new_mgmt_lan_name, request.user.email)
                if res2[0] == status.HTTP_500_INTERNAL_SERVER_ERROR:
                    return Response(status=res2[0], data=res2[1])
        else:
            res1 = [status.HTTP_403_FORBIDDEN, "You are not permitted to update this data. See a system administrator."]
        return Response(status=res1[0], data=res1[1], headers=set_response_headers(request))

    def post(self, request, mac_address):
        if "network.can_create_network_controllers" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            cntl_config_data = request.data
            res1 = db.put_cntl_cfg(mac_address, cntl_config_data, request.user.email)
            if res1[0] == 200:
                new_mgmt_lan_name = cntl_config_data['data']['cntlConfig']['MGMT LAN']['Name']
                res2 = db.update_mgmt_lan_names(mac_address, new_mgmt_lan_name, request.user.email)
                if res2[0] == status.HTTP_500_INTERNAL_SERVER_ERROR:
                    return Response(status=res2[0], data=res2[1])
        else:
            res1 = [status.HTTP_403_FORBIDDEN, "You are not permitted to write this data. See a system administrator."]
        return Response(status=res1[0], data=res1[1], headers=set_response_headers(request))

    def delete(self, request, mac_address):
        if "network.can_delete_network_controllers" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            response = db.delete_cntl_cfg(mac_address, request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to delete this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class SwitchCfg(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request, mac_address):
        if "network.can_read_network_switches" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            response = db.get_switch_cfg(mac_address, request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))

    def put(self, request, mac_address):
        if "network.can_update_network_switches" in User.get_all_permissions(User.objects.get(email=request.user.email)) or "network.can_create_network_olts" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            switch_config_data = request.data
            response = db.put_switch_cfg(mac_address, switch_config_data, request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to update this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))

    def post(self, request, mac_address):
        if "network.can_create_network_switches" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            switch_config_data = request.data
            response = db.put_switch_cfg(mac_address, switch_config_data, request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to write this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))

    def delete(self, request, mac_address):
        if "network.can_delete_network_switches" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            response = db.delete_switch_cfg(mac_address, request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to delete this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class OltCfg(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request, mac_address):
        if "network.can_read_network_olts" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            response = db.get_olt_cfg(mac_address, request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))

    def put(self, request, mac_address):
        if "network.can_update_network_olts" in User.get_all_permissions(User.objects.get(email=request.user.email)) or "network.can_create_network_onus" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            olt_config_data = request.data
            response = db.put_olt_cfg(mac_address, olt_config_data, request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to update this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))

    def post(self, request, mac_address):
        if "network.can_create_network_olts" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            olt_config_data = request.data
            response = db.put_olt_cfg(mac_address, olt_config_data, request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to create this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))

    def delete(self, request, mac_address):
        if "network.can_delete_network_olts" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            response = db.delete_olt_cfg(mac_address, request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to delete this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class OnuCfg(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request, mac_address):
        if "network.can_read_network_onus" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            response = db.get_onu_cfg(mac_address, request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))

    def put(self, request, mac_address):
        if "network.can_update_network_onus" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            onu_config_data = request.data
            response = db.put_onu_cfg(mac_address, onu_config_data, request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to update this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))

    def post(self, request, mac_address):
        if "network.can_create_network_onus" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            onu_config_data = request.data
            response = db.put_onu_cfg(mac_address, onu_config_data, request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to write this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))

    def delete(self, request, mac_address):
        if "network.can_delete_network_onus" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            response = db.delete_onu_cfg(mac_address, request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to delete this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class CntlLog(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request, mac_address, since_utc_time):
        if "network.can_read_network_controllers" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            response = db.get_cntl_log(mac_address, since_utc_time, request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class OltLog(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request, mac_address, since_utc_time):
        if "network.can_read_network_olts" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            response = db.get_olt_log(mac_address, since_utc_time, request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class OnuLog(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request, mac_address, since_utc_time):
        if "network.can_read_network_onus" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            response = db.get_onu_log(mac_address, since_utc_time, request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class GetAllCntlAlarms(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        if "network.can_read_network_controllers" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            response = db.get_all_cntl_alarms(request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class GetAllOltAlarms(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        if "network.can_read_network_olts" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            response = db.get_all_olt_alarms(request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class GetAllOnuAlarms(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        if "network.can_read_network_onus" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            response = db.get_all_onu_alarms(request.user.email, request.GET.get('attribute', None))
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class CntlAlarmsCfg(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request, id):
        if "global_config.can_read_global_config_alarms" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            response = db.get_cntl_alarms_cfg(id, request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))

    def put(self, request, id):
        if "global_config.can_update_global_config_alarms" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            body = request.data
            response = db.put_cntl_alarms_cfg(id, body, request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to update this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))

    def post(self, request, id):
        if "global_config.can_create_global_config_alarms" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            body = request.data
            response = db.put_cntl_alarms_cfg(id, body, request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to write this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))

    def delete(self, request, id):
        if "global_config.can_delete_global_config_alarms" in User.get_all_permissions(User.objects.get(email=request.user.email)) or "global_config.can_update_global_config_alarms" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            response = db.delete_cntl_alarms_cfg(id, request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to delete this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class OltAlarmsCfg(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request, id):
        if "global_config.can_read_global_config_alarms" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            response = db.get_olt_alarms_cfg(id, request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))

    def put(self, request, id):
        if "global_config.can_update_global_config_alarms" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            body = request.data
            response = db.put_olt_alarms_cfg(id, body, request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to update this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))

    def post(self, request, id):
        if "global_config.can_create_global_config_alarms" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            body = request.data
            response = db.put_olt_alarms_cfg(id, body, request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to write this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))

    def delete(self, request, id):
        if "global_config.can_delete_global_config_alarms" in User.get_all_permissions(User.objects.get(email=request.user.email)) or "global_config.can_update_global_config_alarms" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            response = db.delete_olt_alarms_cfg(id, request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to delete this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class OnuAlarmsCfg(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request, id):
        if "global_config.can_read_global_config_alarms" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            response = db.get_onu_alarms_cfg(id, request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))

    def put(self, request, id):
        if "global_config.can_update_global_config_alarms" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            body = request.data
            response = db.put_onu_alarms_cfg(id, body, request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to update this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))

    def post(self, request, id):
        if "global_config.can_create_global_config_alarms" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            body = request.data
            response = db.put_onu_alarms_cfg(id, body, request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to write this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))

    def delete(self, request, id):
        if "global_config.can_delete_global_config_alarms" in User.get_all_permissions(User.objects.get(email=request.user.email)) or "global_config.can_update_global_config_alarms" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            response = db.delete_onu_alarms_cfg(id, request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to delete this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))

class OnuRegistrationAllow(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    # Checking if ONU is disallowed on any OLT AND if there is a different Reg Allow ONU pending on the given OLT
    def get(self, request, id):
        if "network.can_read_network_onus" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            response = db.get_onu_registration_allow(id, request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to reade this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))

    # Performs a Reg Allow ONU an all OLTs an OLT is disallowed on
    def put(self, request, id):
        if "network.can_update_network_onus" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            response = db.put_onu_registration_allow(id, request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to update this data. See a system administrator.", None, None]
        return PonManagerApiResponse(status=response[0], data=response[1], new_data=response[2], old_data=response[3], headers=set_response_headers(request))

class GetAllCntlAlarmsCfg(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        if "global_config.can_read_global_config_alarms" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            response = db.get_all_cntl_alarms_cfgs(request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class GetAllCntlAlarmsIds(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        if "global_config.can_read_global_config_alarms" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            response = db.get_all_cntl_alarms_ids(request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class GetAllOltAlarmsCfg(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        if "global_config.can_read_global_config_alarms" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            response = db.get_all_olt_alarms_cfgs(request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class GetAllOltAlarmsIds(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        if "global_config.can_read_global_config_alarms" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            response = db.get_all_olt_alarms_ids(request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class GetAllOnuAlarmsCfg(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        if "global_config.can_read_global_config_alarms" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            response = db.get_all_onu_alarms_cfgs(request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class GetAllOnuAlarmsIds(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        if "global_config.can_read_global_config_alarms" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            response = db.get_all_onu_alarms_ids(request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class SlaCfg(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request, id):
        if "global_config.can_read_global_config_slas" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            response = db.get_sla_cfg(id, request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))

    def put(self, request, id):
        if "global_config.can_update_global_config_slas" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            body = request.data
            response = db.put_sla_cfg(id, body, request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to update this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))

    def post(self, request, id):
        if "global_config.can_create_global_config_slas" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            body = request.data
            response = db.put_sla_cfg(id, body, request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to write this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))

    def delete(self, request, id):
        if "global_config.can_delete_global_config_slas" in User.get_all_permissions(User.objects.get(email=request.user.email)) or "global_config.can_update_global_config_slas" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            response = db.delete_sla_cfg(id, request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to update this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class GetAllSlaCfg(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        if "global_config.can_read_global_config_slas" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            response = db.get_all_sla_cfgs(request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class GetAllSlaIds(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        if "global_config.can_read_global_config_slas" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            response = db.get_all_sla_ids(request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class ServiceCfg(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def put(self, request, id):
        if "global_config.can_update_global_config_services" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            body = request.data
            response = db.put_service_cfg(id, body, request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to update this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))

    def post(self, request, id):
        if "global_config.can_create_global_config_services" in User.get_all_permissions(User.objects.get(email=request.user.email)) or "global_config.can_create_global_config_files" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            body = request.data
            response = db.put_service_cfg(id, body, request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to write this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))

    def delete(self, request, id):
        if "global_config.can_delete_global_config_services" in User.get_all_permissions(User.objects.get(email=request.user.email)) or "global_config.can_update_global_config_services" in User.get_all_permissions(User.objects.get(email=request.user.email)) or "global_config.can_delete_global_config_files" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            response = db.delete_service_cfg(id, request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to delete this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))

    def get(self, request, id):
        if "global_config.can_read_global_config_services" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            response = db.get_service_cfg(id, request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class GetAllServiceCfg(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        if "global_config.can_read_global_config_services" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            response = db.get_all_service_cfgs(request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class GetAllServiceIds(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        if "global_config.can_read_global_config_services" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            response = db.get_all_service_ids(request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class GetAllDownstreamMapIds(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        if "global_config.can_read_global_config_services" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            response = db.get_all_downstream_map_ids(request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class GetAllOmciServiceIds(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        if "global_config.can_read_global_config_services" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            response = db.get_all_omci_service_ids(request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class Images(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request, filename):
        if "global_config.can_read_global_config_files" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            response = db.download_image(filename, request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class ImageNames(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):

        if "global_config.can_read_global_config_files" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            response = db.get_image_names(request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))

class DeviceImageNames(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        if "global_config.can_read_global_config_files" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            if request.query_params and request.query_params['image_name'] and request.query_params['device_type']:
                params = request.query_params
                db = get_client_db(request.user.email)
                response = db.get_devices_from_image(request.user.email, params['image_name'], params['device_type'])
            else:
                db = get_client_db(request.user.email)
                response = db.get_device_image_names(request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))

class Cpes(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def delete(self, request, id):
        if "network.can_delete_network_onus" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            response = db.delete_cpe(id, request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to delete this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class CpesForOnu(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request, mac_address):  # Get all for ONU
        if "network.can_read_network_onus" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            response = db.get_all_cpes_for_onu(mac_address, request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))

    def delete(self, request, mac_address):  # Delete All for ONU
        if "network.can_delete_network_onus" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            response = db.delete_all_cpes_for_onu(mac_address, request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to delete this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class ExpiredCpesForOnu(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def delete(self, request, mac_address):  # Delete All Expired for ONU
        if "network.can_delete_network_onus" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            response = db.delete_all_expired_cpes_for_onu(mac_address, request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to delete this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class OltFwNames(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        if "global_config.can_read_global_config_files" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            response = db.get_olt_fw_filenames(request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class OnuFwNames(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        if "global_config.can_read_global_config_files" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            response = db.get_onu_fw_filenames(request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class Names(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request, id, device_type):
        db = get_client_db(request.user.email)
        if device_type == 'onu':
            if "network.can_read_network_onus" in User.get_all_permissions(User.objects.get(email=request.user.email)):
                response = db.get_onu_name(id, request.user.email)
            else:
                response = [status.HTTP_403_FORBIDDEN, "You are not permitted to read this data. See a system administrator."]
        elif device_type == 'olt':
            if "network.can_read_network_olts" in User.get_all_permissions(User.objects.get(email=request.user.email)):
                response = db.get_olt_name(id, request.user.email)
            else:
                response = [status.HTTP_403_FORBIDDEN, "You are not permitted to read this data. See a system administrator."]
        else:
            if "network.can_read_network_controllers" in User.get_all_permissions(User.objects.get(email=request.user.email)):
                response = db.get_cntl_name(id, request.user.email)
            else:
                response = [status.HTTP_403_FORBIDDEN, "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class GetAllMibCurStates(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        if "network.can_read_network_onus" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            response = db.get_onu_mib_cur_states(request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class GetMibCurState(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request, id):
        if "network.can_read_network_onus" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            response = db.get_onu_mib_cur_state(id, request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class OnuFirmwareUpgradeStatus(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request, id):
        if "network.can_read_network_onus" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            response = db.get_onu_upgrade_status(id, request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class OltOnuFirmwareUpgradeStatus(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request, id):
        if "network.can_read_network_onus" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            response = db.get_onu_upgrade_statuses_for_olt(id, request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class ControllerOnuFirmwareUpgradeStatus(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request, id):
        if "network.can_read_network_onus" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            response = db.get_onu_upgrade_statuses_for_controller(id, request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class GetAllMibRstStates(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        if "network.can_read_network_onus" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            response = db.get_onu_mib_rst_states(request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class GetMibRstState(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request, id):
        if "network.can_read_network_onus" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            response = db.get_onu_mib_rst_state(id, request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


# TODO No permissions check on this endpoint as of now. Not sure what an appropriate check(s) is?
class CustomDeviceQuery(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request, collection, attribute, operator, value):
        db = get_client_db(request.user.email)
        response = db.get_custom_device_query(collection, attribute, operator, value, request.user.email)
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class UpgradeConfigurationsToNewVersion(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        if "global_config.can_read_global_config_databases" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            response = db.get_device_upgrade_hierarchy(request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to update this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))

    def put(self, request):
        if "global_config.can_update_global_config_databases" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            body = request.data
            response = db.upgrade_configurations_versions(body["data"], request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to update this data. See a system administrator.", None, None]
        return PonManagerApiResponse(status=response[0], data=response[1], new_data=response[2], old_data=response[3], headers=set_response_headers(request))


# TODO No permissions check on this endpoint as of now. Not sure what an appropriate check(s) is?
class SwitchThermalTestData(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request, switch_id):
        db = get_client_db(request.user.email)
        response = db.get_switch_thermal_test_data(switch_id, request.user.email)
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class DeleteOnu(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def delete(self, request, mac_address):
        if "network.can_delete_network_onus" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            delete_from_olt_inv = request.GET.get('deleteFromOltInv')
            response = db.delete_onu_all(mac_address, request.user.email, delete_from_olt_inv)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to delete this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class DeleteOnuLogs(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def delete(self, request, mac_address):
        if "network.can_delete_network_onus" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            response = db.delete_onu_logs(mac_address, request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to delete this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class DeleteOnuStats(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def delete(self, request, mac_address):
        if "network.can_delete_network_onus" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            response = db.delete_onu_stats(mac_address, request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to delete this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class DeleteOlt(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def delete(self, request, mac_address):
        if "network.can_delete_network_olts" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            response = db.delete_olt_all(mac_address, request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to delete this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class DeleteOltLogs(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def delete(self, request, mac_address):
        if "network.can_delete_network_olts" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            response = db.delete_olt_logs(mac_address, request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to delete this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class DeleteOltStats(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def delete(self, request, mac_address):
        if "network.can_delete_network_olts" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            response = db.delete_olt_stats(mac_address, request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to delete this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class DeleteController(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def delete(self, request, mac_address):
        if "network.can_delete_network_controllers" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            response = db.delete_cntl(mac_address, request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to delete this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class DeleteControllerLogs(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def delete(self, request, mac_address):
        if "network.can_delete_network_controllers" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            response = db.delete_cntl_logs(mac_address, request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to delete this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class DeleteControllerStats(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def delete(self, request, mac_address):
        if "network.can_delete_network_controllers" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            response = db.delete_cntl_stats(mac_address, request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to delete this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class DeleteSwitch(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def delete(self, request, mac_address):
        if "network.can_delete_network_switches" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            response = db.delete_switch_all(mac_address, request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to delete this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class Databases(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        if "global_config.can_read_global_config_databases" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            response = db.get_databases(request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))

    def put(self, request):
        if "global_config.can_update_global_config_databases" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            body = request.data
            response = db.edit_database(body['data'], request.user.email)
            response_status = response[0]
            message = response[1]

            if response_status == status.HTTP_200_OK:
                added = start_up()
                if not added:
                    response_status = status.HTTP_500_INTERNAL_SERVER_ERROR
                    message = "Could not connect to the specified MongoDB Database or Server."
        else:
            response_status = status.HTTP_403_FORBIDDEN
            message = "You are not permitted to update this data. See a system administrator."

        return Response(status=response_status, data=message, headers=set_response_headers(request))

    def post(self, request):
        if "global_config.can_create_global_config_databases" in User.get_all_permissions(User.objects.get(email=request.user.email)) or "global_config.can_update_global_config_databases" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            body = request.data
            response = db.add_database(body['data'], request.user.email)
            response_status = response[0]
            message = response[1]

            if response_status == status.HTTP_201_CREATED:
                added = start_up()
                if not added:
                    response_status = status.HTTP_500_INTERNAL_SERVER_ERROR
                    message = "Could not connect to the specified MongoDB Database or Server."
        else:
            response_status = status.HTTP_403_FORBIDDEN
            message = "You are not permitted to write this data. See a system administrator."

        return Response(status=response_status, data=message)

    def delete(self, request):
        if "global_config.can_delete_global_config_databases" in User.get_all_permissions(User.objects.get(email=request.user.email)) or "global_config.can_update_global_config_databases" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            body = json.loads(request.body)
            if body['key'] == "Default":
                response = [status.HTTP_400_BAD_REQUEST, "You may not delete the Default database entry.::You may not delete the Default database entry."]
            else:
                response = db.delete_database(body['key'], request.user.email)

            if str(response[0]).startswith("2"):
                # Switch users using this db to Default database
                for client in clients:
                    if clients[client] == body['key']:
                        clients[client] = 'Default'
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to delete this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


# TODO No permissions check on this endpoint as of now. Not sure what an appropriate check(s) is?
class ModifyDatabases(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        get_client_db(request.user.email)
        response = clients[request.user.email]
        return Response(status=status.HTTP_200_OK, data=response)

    # REPLACED BY PUT IN "/databases/selection/"
    def put(self, request):
        response = [status.HTTP_200_OK, ""]
        try:
            new_db_key = json.loads(request.body)["key"]
            if new_db_key in list(dbs.keys()):
                clients[request.user.email] = new_db_key
                new_db = clients[request.user.email]
                response[1] = "{} source database changed to {}".format(request.user.email, new_db)
            else:
                response = [status.HTTP_404_NOT_FOUND,
                            f"Database {new_db_key} was not found in active databases. Please verify the {new_db_key} entry fields.::Database {new_db_key} entry likely has invalid fields."]
        except Exception as err:
            response[0] = status.HTTP_500_INTERNAL_SERVER_ERROR
            response[1] = "Could not change databases due to {}::{}".format(type(err).__name__, sys.exc_info()[1])
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class DatabaseSeeding(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        if "global_config.can_read_global_config_databases" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            response = db.get_seed_files(request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))

    def put(self, request):
        if "global_config.can_update_global_config_databases" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db_to_seed = json.loads(request.body)["toSeed"]
            to_include = json.loads(request.body)["toInclude"]
            db = dbs[db_to_seed]
            response = db.seed_database(to_include, request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to update this data. See a system administrator.", None, None]
        return PonManagerApiResponse(status=response[0], data=response[1], new_data=response[2], old_data=response[3], headers=set_response_headers(request))

class DatabaseSeedOltCFG(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def put(self, request):
        if "global_config.can_update_global_config_databases" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            pon_mode = json.loads(request.body)["ponMode"]
            db = get_client_db(request.user.email)
            response = db.set_seed_pon_mode(pon_mode)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to update this data. See a system administrator.", None, None]
        return PonManagerApiResponse(status=response[0], data=response[1], new_data=response[2], old_data=response[3], headers=set_response_headers(request))


class DatabaseBackup(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def post(self, request):
        try:
            result = None
            if "global_config.can_read_global_config_databases" in User.get_all_permissions(User.objects.get(email=request.user.email)):
                db = dbs[request.data['databaseID']]
                result = db.backup_database(request.user.email, request.data['toExclude'])
                if result[0] == status.HTTP_201_CREATED:
                    # Read file created in backup
                    file = tarfile.open(result[1]["path"], 'r')
                    response = FileResponse(file.fileobj, content_type='application/gzip')
                    response.status_code = status.HTTP_201_CREATED
                    response['Content-Disposition'] = f'attachment; filename="{result[1]["filename"]}"'
                else:
                    response = Response(status=result[0], data=result[1])
            else:
                response = Response(status=status.HTTP_403_FORBIDDEN, data="You are not permitted to access this data. See a system administrator.")

            try:
                # Callback function to delete the backup file after 15 seconds (allows time for response)
                def delete_file_callback():
                    try:
                        if result is not None:
                            mongodb_backup_dir = "/var/www/html/api/mongodb_backup"
                            os.remove(f"{mongodb_backup_dir}/{result[1]['filename']}")
                    except Exception:
                        try:
                            if result is not None:
                                mongodb_backup_dir = "mongodb_backup"
                                os.remove(f"{mongodb_backup_dir}/{result[1]['filename']}")
                        except Exception as ex:
                            print(ex)
                            pass

                timer = threading.Timer(15, delete_file_callback)
                timer.start()
            except Exception as err:
                print(err)
        except Exception as err:
            response = Response(status=status.HTTP_500_INTERNAL_SERVER_ERROR, data="Could not backup database due to {}::{}".format(type(err).__name__, sys.exc_info()[1]))

        return response

    def delete(self, request, filename):
        response = [status.HTTP_200_OK, "Backup file was deleted."]
        try:
            if "global_config.can_delete_global_config_databases" in User.get_all_permissions(User.objects.get(email=request.user.email)):
                try:
                    os.remove(f"/var/www/html/{filename}")
                except FileNotFoundError:
                    os.remove(f"../{filename}")
            else:
                response = [status.HTTP_403_FORBIDDEN, "You are not permitted to delete this data. See a system administrator."]
        except Exception as err:
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR, "Could not delete database backup file due to {}::{}".format(type(err).__name__, sys.exc_info()[1])]

        return Response(data=response[1], status=response[0])


class DatabaseRestore(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def post(self, request):
        try:
            if "global_config.can_create_global_config_databases" in User.get_all_permissions(User.objects.get(email=request.user.email)):
                data = request.data['tgzFile'].split(',')[1]
                del request.data['tgzFile']
                response = dbs[request.data['databaseID']].restore_database(request.user.email, data=data)
            else:
                response = [status.HTTP_403_FORBIDDEN, "You are not permitted to create this data. See a system administrator."]
        except Exception as err:
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR, "Could not restore database due to {}::{}".format(type(err).__name__, sys.exc_info()[1])]

        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class ExcessiveLoginAttempts(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def post(self, request):
        user_email = request.data["data"]["user"]
        pon_manager_logger.warning(f"Detected excessive number of login attempts for user {user_email}")
        return Response(status=status.HTTP_200_OK, headers=set_response_headers(request))


class GetAllUnattachedControllers(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        if "network.can_read_network_controllers" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            response = db.get_all_unattached_controllers(request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class GetAllControllersWithVersions(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        if "network.can_read_network_controllers" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            response = db.get_all_controllers_with_versions(request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class GetAllUnattachedSwitches(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        if "network.can_read_network_switches" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            response = db.get_all_unattached_switches(request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class GetAllUnattachedOlts(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        if "network.can_read_network_olts" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            response = db.get_all_unattached_olts(request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class GetAllUnattachedOnus(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        if "network.can_read_network_onus" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            response = db.get_all_unattached_onus(request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class GetAllSlaOnuUsage(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request, sla):
        if "global_config.can_read_global_config_slas" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            response = db.get_all_sla_onu_usage(sla, request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class GetAllSrvOnuUsage(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request, srv):
        if "global_config.can_read_global_config_services" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            response = db.get_all_srv_onu_usage(srv, request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class OmciInfo(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        if "global_config.can_read_global_config_services" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            response = [status.HTTP_200_OK, ""]
            try:
                try:
                    with open(production_omci_info, 'r', encoding='utf-8') as omci_file:
                        omci_data = json.load(omci_file)
                except FileNotFoundError:
                    with open(development_omci_info, 'r', encoding='utf-8') as omci_file:
                        omci_data = json.load(omci_file)

                response[1] = omci_data
            except Exception as err:
                response[0] = status.HTTP_500_INTERNAL_SERVER_ERROR
                response[1] = "Could not get OMCI information due to {}::{}".format(type(err).__name__, sys.exc_info()[1])
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


# Create User
# POST support only
# /user/
# This will create a user
# Returns error message if it fails, otherwise, nothing
class CreateUsers(LoginRequiredMixin, APIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        pass

    def post(self, request):
        if "accounts.can_create_accounts_admin" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            userData = request.data

            try:
                user_email = userData['data']['email'].lower()
                user_password = hashlib.md5(userData['data']['password'].encode('utf-8')).hexdigest()


                # Verify email is not in use
                email_in_use = User.objects.filter(email=user_email).exists()
                # creating of user
                if email_in_use:
                    return Response(status=status.HTTP_409_CONFLICT, data="The provided email address is already in use.")


                user = User.objects.create_user(username=user_email, email=user_email,
                                                password=user_password
                                                )

            except SQLDecodeError as e:
                print(e)
                if 'duplicate key error' in e.args[0]:
                    data = {
                        'err': 'User Already exists',
                        'code': 1}
                else:
                    data = {
                        'err': 'Unable to connect to database',
                        'code': 0}
                return JsonResponse(data)
            user.first_name = userData['data']['fName']
            user.last_name = userData['data']['lName']
            for role in userData['data']['roles']:
                if Group.objects.filter(name=role).exists():
                    user.groups.add(Group.objects.get(name=role))
            user.backend = 'django.contrib.auth.backends.ModelBackend'

            user.save()

            db = get_client_db(request.user.email)
            db.create_user_log(user, request.user.email)
            response = [status.HTTP_201_CREATED, None, user_email, None]
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to write this data. See a system administrator.", None, None]
        return PonManagerApiResponse(status=response[0], new_data=response[2], old_data=response[3])


class UserPermissions(LoginRequiredMixin, APIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        if "accounts.can_read_accounts_admin" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            status_code = status.HTTP_200_OK
            try:
                content_types = {}
                for content_type in json.loads(serializers.serialize('json', ContentType.objects.all())):
                    content_types[content_type['pk']] = (content_type['fields']['app_label'], content_type['fields']['model'])
                data = {}
                for permission in json.loads(serializers.serialize('json', Permission.objects.all())):
                    # Build permissions response
                    app_label = ""
                    for token in content_types[permission['fields']['content_type']][0].split("_"):
                        app_label += token.capitalize() + " "
                    app_label = app_label.strip()
                    model = ""
                    for token in content_types[permission['fields']['content_type']][1].split("_"):
                        model += token.capitalize() + " "
                    model = model.strip()
                    if app_label not in data:
                        data[app_label] = {}
                    if model not in data[app_label]:
                        data[app_label][model] = {}
                    if "read" in permission['fields']['codename']:
                        data[app_label][model]['read'] = permission['fields']['name']
                    elif "create" in permission['fields']['codename']:
                        data[app_label][model]['create'] = permission['fields']['name']
                    elif "update" in permission['fields']['codename']:
                        data[app_label][model]['update'] = permission['fields']['name']
                    elif "delete" in permission['fields']['codename']:
                        data[app_label][model]['delete'] = permission['fields']['name']
                    elif "view" in permission['fields']['codename']:
                        data[app_label][model]['view'] = permission['fields']['name']
            except Exception as err:
                traceback.print_exc(file=sys.stdout)
                status_code = status.HTTP_500_INTERNAL_SERVER_ERROR
                data = "Could not get permissions list due to {}::{}".format(type(err).__name__, sys.exc_info()[1])
        else:
            status_code = status.HTTP_403_FORBIDDEN
            data = "You are not permitted to read this data. See a system administrator."
        return Response(status=status_code, data=data)


class UserRoles(LoginRequiredMixin, APIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        if "accounts.can_read_accounts_admin" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            status_code = status.HTTP_200_OK
            try:
                db = get_client_db(request.user.email)
                groups = json.loads(serializers.serialize('json', Group.objects.all()))
                data = []
                for group in groups:
                    users = map(lambda u: u["fields"]["username"], json.loads(serializers.serialize('json', User.objects.filter(groups=group["pk"]))))
                    permissions = map(lambda p: p["fields"]["name"], json.loads(serializers.serialize('json', Permission.objects.filter(group=group["pk"]))))
                    role_info = db.get_role_session_expiry(group["pk"])[1]
                    if role_info and "timeout" in role_info:
                        timeout = role_info["timeout"]
                        override = role_info["override"]
                    else:
                        timeout = db.get_global_session_time()[1]['timeout']
                        override = False
                    data.append({"name": group["fields"]["name"], "users": users, "pk": group["pk"], "permissions": permissions, "timeout": timeout, "override": override})
            except Exception as err:
                traceback.print_exc(file=sys.stdout)
                status_code = status.HTTP_500_INTERNAL_SERVER_ERROR
                data = "Could not get roles list due to {}::{}".format(type(err).__name__, sys.exc_info()[1])
        else:
            status_code = status.HTTP_403_FORBIDDEN
            data = "You are not permitted to read this data. See a system administrator."
        return Response(status=status_code, data=data)

    def post(self, request):
        if "accounts.can_create_accounts_admin" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            status_code = status.HTTP_201_CREATED
            try:
                body = request.data["data"]
                name = body["name"]
                users = body["users"]
                permissions = body["permissions"]
                new_group, created = Group.objects.get_or_create(name=name)
                if created:
                    # Add users
                    for user in users:
                        if user is not None and User.objects.filter(email=user).exists():
                            new_group.user_set.add(User.objects.get(email=user))

                    # Add permissions
                    for permission in permissions:
                        if permissions is not None and Permission.objects.filter(name=permission).exists():
                            new_group.permissions.add(Permission.objects.get(name=permission))
                    new_group.save()
                    data = f"Role '{name}' was created."
                else:
                    status_code = status.HTTP_409_CONFLICT
                    data = f"Role '{name}' cannot be created because is already exists."
            except Exception as err:
                traceback.print_exc(file=sys.stdout)
                status_code = status.HTTP_500_INTERNAL_SERVER_ERROR
                data = "Could not create role due to {}::{}".format(type(err).__name__, sys.exc_info()[1])
        else:
            status_code = status.HTTP_403_FORBIDDEN
            data = "You are not permitted to write this data. See a system administrator."
        return Response(status=status_code, data=data)

    def put(self, request):
        if "accounts.can_update_accounts_admin" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            status_code = status.HTTP_200_OK
            try:
                body = json.loads(request.body)
                name = body["name"]
                group = Group.objects.get(name=name)
                if "users" in body:
                    if name == "Administrators" and request.user.email not in body['users']:
                        status_code = status.HTTP_403_FORBIDDEN
                        data = "A user assigned the Administrator role may not remove themselves from that role."
                    else:
                        for user in group.user_set.all():
                            if user.email not in body['users']:
                                group.user_set.remove(User.objects.get(email=user.email))
                        for user in body['users']:
                            if user is not None:
                                group.user_set.add(User.objects.get(email=user))
                        data = f"Updated role {name} members."
                elif "permissions" in body:
                    for permission in group.permissions.all():
                        if permission.name not in body['permissions']:
                            group.permissions.remove(Permission.objects.get(name=permission.name))
                    for permission in body['permissions']:
                        if permission is not None:
                            group.permissions.add(Permission.objects.get(name=permission))
                    data = f"Updated role {name} permissions."
                else:
                    status_code = status.HTTP_400_BAD_REQUEST
                    data = f"Role {name} could not be updated."
            except Exception as err:
                traceback.print_exc(file=sys.stdout)
                status_code = status.HTTP_500_INTERNAL_SERVER_ERROR
                data = "Could not update role due to {}::{}".format(type(err).__name__, sys.exc_info()[1])
        else:
            status_code = status.HTTP_403_FORBIDDEN
            data = "You are not permitted to update this data. See a system administrator."
        return Response(status=status_code, data=data)

    def delete(self, request):
        if "accounts.can_delete_accounts_admin" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            status_code = status.HTTP_200_OK
            try:
                user = User.objects.get(username=request.user.email)
                body = json.loads(request.body)
                name = body["name"]
                if Group.objects.filter(name=name).exists():
                    group = Group.objects.get(name=name)
                    if name == "Administrators":
                        status_code = status.HTTP_403_FORBIDDEN
                        data = f"The Administrators role cannot be deleted."
                    elif group.user_set.count() > 0:
                        status_code = status.HTTP_428_PRECONDITION_REQUIRED
                        data = f"The role '{name}' cannot be deleted because it contains users."
                    else:
                        group.delete()
                        data = f"Role '{name}' was deleted."
                else:
                    data = f"Role '{name}' was deleted."

            except Exception as err:
                traceback.print_exc(file=sys.stdout)
                status_code = status.HTTP_500_INTERNAL_SERVER_ERROR
                data = "Could not delete role due to {}::{}".format(type(err).__name__, sys.exc_info()[1])
        else:
            status_code = status.HTTP_403_FORBIDDEN
            data = "You are not permitted to delete this data. See a system administrator."

        return Response(status=status_code, data=data, headers=set_response_headers(request))


# Create Initial User
# POST support only
# /user/initialize/
# This will create a user ONLY when no other users exist
# Returns error message if it fails, otherwise, nothing
class CreateInitialUser(APIView):
    def get(self, request):
        pass

    def post(self, request):

        if len(User.objects.all()) == 0:

            userData = request.data

            try:
                # Initializing user parameters to be used for password validation
                user_email = ''
                user_first_name = ''
                user_last_name = ''

                # Verifying that request has all parameters for password validation, and setting local vars
                if 'email' in userData['data']:
                    user_email = userData['data']['email'].lower()

                if 'fName' in userData['data']:
                    user_first_name = userData['data']['fName']

                if 'lName' in userData['data']:
                    user_last_name = userData['data']['lName']

                # Setting and hashing password
                user_password = hashlib.md5(userData['data']['password'].encode('utf-8')).hexdigest()

                # Validating password meets minimum requirements
                password_requirement_response = validateMinimumPasswordRequirements(request, userData['data']['password'], user_email, user_first_name,  user_last_name)

                # If password does not meet requirements, return
                if password_requirement_response['password_meets_requirements'] is False:
                    return Response(status=status.HTTP_400_BAD_REQUEST, data=password_requirement_response['missing_criteria_message'])

                # creating of user
                user = User.objects.create_user(username=user_email, email=user_email,
                                                password=user_password
                                                )

            except SQLDecodeError as e:
                print(e)
                if 'duplicate key error' in e.args[0]:
                    data = {
                        'err': 'User Already exists',
                        'code': 1}
                else:
                    data = {
                        'err': 'Unable to connect to database',
                        'code': 0}
                return JsonResponse(data)

            # If password DOES meet minimum reqs, configure and safe the user, else return 400
            user.first_name = userData['data']['fName']
            user.last_name = userData['data']['lName']
            user.backend = 'django.contrib.auth.backends.ModelBackend'

            user.groups.add(Group.objects.get(name="Administrators"))

            user.save()

            db = get_client_db(user_email)
            db.create_user_log(user, user_email)

            return PonManagerApiResponse(status=status.HTTP_204_NO_CONTENT, new_data=user_email)
        else:
            return Response(status=status.HTTP_403_FORBIDDEN)


# Get All Users
# GET support only
# /getAllUsers
# This will return all users
class GetAllUsers(LoginRequiredMixin, APIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        all_users = []
        try:
            if "accounts.can_read_accounts_admin" in User.get_all_permissions(User.objects.get(email=request.user.email)):
                for user in User.objects.all():
                    user = model_to_dict(user)
                    if user['groups']:
                        user['groups'] = list(map(lambda g: model_to_dict(g)['name'], user['groups']))
                        del user['password']
                    all_users.append(user)
                data = {'users': all_users}
                return JsonResponse(data, status=status.HTTP_200_OK)
            else:
                return JsonResponse({}, status=status.HTTP_403_FORBIDDEN)
        except:
            data = {'err': 'Server Error'}
            return JsonResponse(data, status=status.HTTP_500_INTERNAL_SERVER_ERROR)  # if failed, we return error message# Get All Users


# GET support only
# /user/active/
# Returns an array of all current active users, by ID
class GetAllActiveUsers(LoginRequiredMixin, APIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        all_users = set()
        try:
            if "accounts.can_read_accounts_admin" in User.get_all_permissions(User.objects.get(email=request.user.email)):
                request.session.clear_expired()
                for session in Session.objects.all():
                    user_auth_id = session.get_decoded().get('_auth_user_id')
                    if user_auth_id is not None:
                        try:
                            user_email = User.objects.get(pk=user_auth_id)
                            if user_email is not None:
                                all_users.add(str(user_email))
                        except django.contrib.auth.models.User.DoesNotExist: # handle session with user that no longer exists
                            continue

                data = {'users': list(all_users)}
                return JsonResponse(data, status=status.HTTP_200_OK)
            else:
                return JsonResponse({}, status=status.HTTP_403_FORBIDDEN)
        except Exception as err:
            data = {'err': 'Server Error'}
            return JsonResponse(data, status=status.HTTP_500_INTERNAL_SERVER_ERROR)  # if failed, we return error message


# Get users exist (bool)
# GET support only
# /user/exist/
# This will return True if one or more users exist in database, false if not
class GetUsersExist(APIView):
    def get(self, request):
        try:
            users_exist = len(User.objects.all()) > 0
            data = {'usersExist': users_exist}
            return JsonResponse(data)
        except:
            data = {'err': 'Server Error'}
            return JsonResponse(data)  # if failed, we return error message


# Updates user session
# GET support only
# /user/session/
# Session is updated automatically in middleware
class RefreshSession(LoginRequiredMixin, APIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        return Response(status=status.HTTP_200_OK, data={'session': True}, headers=set_response_headers(request))


# Global Session Age Setting
# GET/PUT support
# /user/session/age/
class GlobalSessionAgeSetting(LoginRequiredMixin, APIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        db = get_client_db(request.user.email)
        response = db.get_global_session_time()
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))

    def put(self, request):
        if "accounts.can_update_accounts_admin" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            body = request.data
            response = db.set_global_session_time(body['data']['timeout'], request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to update this data. See a system administrator.", None, None]
        return PonManagerApiResponse(status=response[0], data=response[1], new_data=response[2], old_data=response[3], headers=set_response_headers(request))


class UserSessionAgeSetting(LoginRequiredMixin, APIView):
    raise_exception = True

    def get(self, request):
        db = get_client_db(request.user.email)
        response = db.get_user_session_expiry(request.user.email)
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))

    def put(self, request):
        if "accounts.can_update_accounts_admin" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            body = request.data
            response = db.set_role_session_expiry(body['data']['role_id'], body['data']['override'], body['data']['timeout'], request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to update this data. See a system administrator.", None, None]
        return PonManagerApiResponse(status=response[0], data=response[1], new_data=response[2], old_data=response[3], headers=set_response_headers(request))


class GetAllSessionAges(LoginRequiredMixin, APIView):
    raise_exception = True

    def get(self, request):
        db = get_client_db(request.user.email)
        response = db.get_all_session_expiry()
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


# Global timestamp locale format setting
# GET/PUT support
# /user/locale/timestamp/
class LocaleTimestampFormat(LoginRequiredMixin, APIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        db = get_client_db(request.user.email)
        response = db.get_locale_timestamp_format(request.user.email)
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))

    def put(self, request):
        if "accounts.can_update_accounts_admin" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            body = request.data
            response = db.set_locale_timestamp_format(body['data']['Locale Timestamp'], request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to update this data. See a system administrator.", None, None]
        return PonManagerApiResponse(status=response[0], data=response[1], new_data=response[2], old_data=response[3], headers=set_response_headers(request))

# Global pon mode setting
# GET/PUT support
# /user/pon/mode
class PonMode(LoginRequiredMixin, APIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        db = get_client_db(request.user.email)
        response = db.get_pon_mode(request.user.email)
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))

    def put(self, request):
        if "accounts.can_update_accounts_admin" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            body = request.data
            response = db.set_pon_mode(body['data']['PON Mode'], request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to update this data. See a system administrator.", None, None]
        return PonManagerApiResponse(status=response[0], data=response[1], new_data=response[2], old_data=response[3], headers=set_response_headers(request))

# Authenticate User
# POST support only
# /user/authenticate/
# This will authenticate a user based on credentials
# Returns success status(bool) and userdata(only on success)
class AuthenticateUsers(APIView):
    def post(self, request):
        purge_expired_sessions(request)
        userData = request.data
        email = userData['data']['email'].lower()
        user_password = hashlib.md5(userData['data']['password'].encode('utf-8')).hexdigest()

        user = authenticate(username=email,
                            password=user_password)  # This is responsible for actual authentication. If it succeeds, it returns an object of user data. IF it fails, it returns nothing

        if user is not None:  # IF usr is not null, that means it returned user data, which means user authentication was successful.
            db = get_client_db(email)

            user = User.objects.get(username=email)  # Getting user object based on primary key Username( which is email)
            user.last_login = datetime.now(timezone.utc)

            user.save()

            try:
                response = db.get_user_session_expiry(email)
                timeout_in_minutes = response[1] * 60  # Database value is in minutes. However, the set_expiry method takes seconds.
                request.session.set_expiry(timeout_in_minutes)

            except Exception as err:
                traceback.print_exc(file=sys.stdout)
                data = "Could not user session expiry age due to {}::{}".format(type(err).__name__, sys.exc_info()[1])

            login(request, user)  # logging in user

            data = {'auth': 'True', 'fName': user.first_name, 'lName': user.last_name,
                    'email': user.email, 'lastLogin': user.last_login,
                    'dateJoined': user.date_joined,
                    'roles': list(map(lambda u: u['fields']['name'], json.loads(serializers.serialize('json', user.groups.all())))),
                    'permissions': list(User.get_all_permissions(user))}  # returning user data to be used in view context

            try:
                db = get_client_db(email)
                db.sys_log({"user": email, "collection": "users"}, "login", True)
            except Exception as err:  # Failed to retrieve database associated with user / possibly does not exist anymore
                try:
                    db = dbs['Default']  # Will attempt to write log using Default database connector
                    db.sys_log({"user": email, "collection": "users"}, "login", True)
                except Exception as defaultErr:
                    print("Failed to write log. Action: Authenticate: ", err)

            return JsonResponse(data)
        else:
            data = {'auth': 'False'}

            db = get_client_db(email)
            db.sys_log({"user": email, "collection": "users"}, "login", False)

            return JsonResponse(data)  # return fail message


# Authenticate User based on session token, not credentials
# POST support only
# /user/authenticate/token/
# This will authenticate a user
# Returns success status(bool) and userdata(only on success)
class AuthenticateUsersToken(LoginRequiredMixin, APIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def post(self, request):
        response = [status.HTTP_200_OK, {'auth': 'False'}]
        purge_expired_sessions(request)
        request.user.last_login = datetime.now(timezone.utc)
        request.user.save()

        response[1] = {'auth': 'True', 'fName': request.user.first_name, 'lName': request.user.last_name,
                       'email': request.user.email, 'lastLogin': request.user.last_login,
                       'dateJoined': request.user.date_joined,
                       'roles': list(map(lambda u: u['fields']['name'], json.loads(serializers.serialize('json', request.user.groups.all())))),
                       'permissions': list(User.get_all_permissions(User.objects.get(email=request.user.email)))}  # returning user data to be used in view context

        try:
            db = get_client_db(request.user.email)
            session_age_response = db.get_user_session_expiry(request.user.email)
            timeout_in_minutes = session_age_response[1] * 60  # Database value is in minutes. However, the set_expiry method takes seconds.
            request.session.set_expiry(timeout_in_minutes)
            db.sys_log({"user": request.user.email, "collection": "users"}, "login", True)
        except Exception as err:  # Failed to retrieve database associated with user / possibly does not exist anymore
            try:
                db = dbs['Default']  # Will attempt to write log using Default database connector
                db.sys_log({"user": request.user.email, "collection": "users"}, "login", True)
            except Exception as defaultErr:
                print("Failed to write log. Action: Authenticate: ", err)

        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


# Logout User
# GET support only
# user/logout/
# This will logout a user
# Returns nothing. Might want to change, to return a status of some kind, but i'm skeptical there will ever be a need. USER not existing              Thu, 22 Jul 2021 21:17:58 GMT
class LogoutUser(APIView):

    def get(self, request):
        #Logout always return successfully regardless if user is logged in/exists
        logout(request)
        response = Response(status=status.HTTP_204_NO_CONTENT)
        response.delete_cookie('sessionexpire')
        response.delete_cookie('csrftoken')
        response.delete_cookie('sessionid')
        response.delete_cookie('__Host-sessionexpire')
        response.delete_cookie('__Host-csrftoken')
        response.delete_cookie('__Host-sessionid')
        return response


# Triggers automatic logout of user, such as in the case of closing browser session
# PUT support only
# user/logout/auto/
class AutoLogoutUser(LoginRequiredMixin, APIView):

    def get(self, request):
        pass

    def put(self, request):
        try:
            request.session.set_expiry(60)  # User will have 60 seconds to make another API request before their session is invalidated
        except Exception as err:
            traceback.print_exc(file=sys.stdout)
            data = "Could not user session expiry age due to {}::{}".format(type(err).__name__, sys.exc_info()[1])

        db = get_client_db(request.user.email)
        db.sys_log({"user": request.user.email}, "logout", True)
        return Response(status=status.HTTP_204_NO_CONTENT)


# Send Recovery Email
# POST support only
# /sendRecoveryEmail
# This will Send a recovery email to the specified user. Invoked from forgot my password
# Returns error message if it fails, otherwise, nothing
# Good reference on how to properly configure gmail smtp server: https://www.siteground.com/kb/google_free_smtp_server/
class SendRecoveryEmail(APIView):
    def get(self, request):
        pass

    def post(self, request):
        data = {}  # Fail message
        # Getting email
        userData = request.data
        userEmail = userData['data']['email'].lower()

        # Try/catch Send email

        # Generate new Challenge Code
        recoveryHash = uuid.uuid4().hex
        recoveryCode = recoveryHash[:6]
        UserRecoveryCodes[userEmail] = [recoveryCode, datetime.now()]

        try:
            subject = 'MCMS PON Manager Password Recovery Code'  # Subject of email
            body = 'Your Code Is: ' + UserRecoveryCodes[userEmail][
                0] + ' (Case-Sensitive) This code will expire within 10 minutes of receipt'  # Body of email, this will contain the given recovery Code
            emailFrom = settings.EMAIL_HOST_USER  # Sender. Currently configured in settings.py
            emailTo = [userEmail]  # Who we are sending to. Gotten from the payload

            send_mail(subject, body, emailFrom, emailTo,
                      fail_silently=False, )  # Sending email. We want this to not fail silently. That way, we can catch this and trigger warning to display in index(UI)
            data = {'Sent': 'True'}  # This code denotes that the message was sent properly
        except:
            data = {'Sent': 'False',
                    'Message': 'Recovery Email Failed To Send'}  # If the try/catch on sending fails, all that matters is that it failed to send, not the specific error. Thus, we send an error message back that the email failed to send
            db = dbs['Default']
            db.sys_log({"user": userEmail}, "recovery", False)
            print("Error: Recovery email failed to send.")

        return JsonResponse(data)  # return status


# Validate Recovery Code
# POST support only
# /validateRecoveryCode
# This will validate that a recovery code supplied by user is correct and valid
class ValidateRecoveryCode(APIView):
    def post(self, request):
        data = {}  # message

        userData = request.data
        userCode = userData['data']['resetCode']
        userEmail = userData['data']['email'].lower()
        storedCode = UserRecoveryCodes[userEmail][0]  # getting zero'ith element value from userEmail key, which is the code

        currentDate = datetime.now()
        if (currentDate - timedelta(minutes=10) <= UserRecoveryCodes[userEmail][
            1] <= currentDate):  # getting first element value from userEmail key, which is date the code was instanciated and checking that it is within 10 minutes old

            if (userCode == storedCode):  # This will eventually be replaced with the appropriate way of validating whether the recovery code was entered in properly
                user = User.objects.get(username=userEmail)  # Getting user object based on primary key Username( which is email)
                data = {'valid': 'True', 'fname': user.first_name, 'lname': user.last_name}  # If code was valid
            else:
                data = {'valid': 'False'}  # If code was not valid

            return JsonResponse(data)

        else:
            # code is no longer valid
            # Generate new Challenge Code
            recoveryHash = uuid.uuid4().hex
            recoveryCode = recoveryHash[:6]
            UserRecoveryCodes[userEmail] = [recoveryCode, datetime.datetime.now()]

            try:
                subject = 'MCMS PON Manager Password Recovery Code'  # Subject of email
                body = 'Your previous code expired :( Your New Code Is: ' + UserRecoveryCodes[userEmail][
                    0] + ' (Case-Sensitive) This code will expire within 10 minutes of receipt'  # Body of email, this will contain the given recovery Code
                emailFrom = settings.EMAIL_HOST_USER  # Sender. Currently configured in settings.py
                emailTo = [userEmail]  # Who we are sending to. Gotten from the payload

                send_mail(subject, body, emailFrom, emailTo,
                          fail_silently=False, )  # Sending email. We want this to not fail silently. That way, we can catch this and trigger warning to display in index(UI)
                data = {'Sent': 'True', 'valid': 'False'}  # This code denotes that the message was sent properly
            except:
                data = {'Sent': 'False', 'Message': 'Recovery Email Failed To Send',
                        'valid': 'False'}  # If the try/catch on sending fails, all that matters is that it failed to send, not the specific error. Thus, we send an error message back that the email failed to send

        return JsonResponse(data)  # return status


#########Reset Password #########################
# POST support only
# Reset Password
# POST support only
# /resetPassword
# This will reset a given users password
class ResetPassword(APIView):
    def get(self, request):
        pass

    def put(self, request):
        try:
            data = {}  # message
            userData = request.data
            userEmail = userData['data']['username'].lower()  # Username
            userCode = userData['data']['resetCode']
            user_password = hashlib.md5(userData['data']['passwordHash'].encode('utf-8')).hexdigest()
            storedCode = UserRecoveryCodes[userEmail][0]  # getting zero'ith element value from userEmail key, which is the code
            # validating that request has valid recovery token
            currentDate = datetime.now()
            user = User.objects.get(
                username=userEmail)  # Getting user object based on primary key Username( which is email)

            # Validating password meets minimum requirements
            password_requirement_response = validateMinimumPasswordRequirements(request, userData['data']['passwordHash'], userEmail,
                                                                                user.first_name, user.last_name)

            # If password does not meet requirements, return
            if password_requirement_response['password_meets_requirements'] is False:
                return Response(status=status.HTTP_400_BAD_REQUEST, data=password_requirement_response['missing_criteria_message'])

            if (currentDate - timedelta(minutes=10) <= UserRecoveryCodes[userEmail][1] <= currentDate):  # getting first element value from userEmail key, which is date the code was instantiated and checking that it is within 10 minutes old

                if (userCode == storedCode):  # This will eventually be replaced with the appropriate way of validating whether the recovery code was entered in properly
                    storedCode = ""
                    del UserRecoveryCodes[userEmail]
                    user.set_password(user_password)  # Setting the password of that user to the new password hash
                    user.save()  # Saving the state of that user
                    data = {'valid': 'True'}  # Resetting was successful
                else:
                    raise Exception
            else:
                raise Exception
        except Exception as err:
            return Response(status=status.HTTP_400_BAD_REQUEST, data={'valid': 'False'})
        return JsonResponse(data)

#########  Getting Password Requirements #########################
# GET support only
# /user/password/requirements/unauthenticated
# Get/Set the minimum requirements for a valid password
class GetPasswordRequirements(generics.GenericAPIView):

    def get(self, request):
        # Setting requesters email, if it exists
        try:
            requesting_email = request.user.email
        except Exception:
            requesting_email = 'AnonymousUser'

        db = get_client_db(requesting_email)
        response = db.get_minimum_password_requirements(requesting_email)
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))

#########  Setting Password Requirements #########################
# PUT support only
# /user/password/requirements/authenticated
# Get/Set the minimum requirements for a valid password
class SetPasswordRequirements(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def put(self, request):
        if "accounts.can_update_accounts_admin" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            password_requirements = request.data['data']['passwordRequirementsObject']
            response = db.set_minimum_password_requirements(password_requirements, request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to update this data. See a system administrator.", None, None]
        return PonManagerApiResponse(status=response[0], data=response[1], new_data=response[2], old_data=response[3], headers=set_response_headers(request))

######### Delete User #########################
# DELETE support only
# Delete User
# /user/<email>
# This will take in the email of the user who will be deleted, and removes it
class DeleteUser(LoginRequiredMixin, APIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        pass

    def delete(self, request, email):
        if "accounts.can_delete_accounts_admin" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            data = {}

            try:
                user = User.objects.get(username=email)
                if request.user.email == email:
                    return Response(status=status.HTTP_403_FORBIDDEN, data="User cannot delete themselves.")

                user.delete()
                data = {'valid': 'True'}

                db = get_client_db(request.user.email)
                db.delete_user(request.user.email)

            except Exception:
                data = {'valid': 'False'}
        else:
            status_code = status.HTTP_403_FORBIDDEN
            data = "You are not permitted to update this data. See a system administrator"
        return JsonResponse(data)


######### Get Logs For User #########################
# GET support only
# /userLogs/<email>/<since_utc_time>
# This will take in the email of the user and return the logs associated
# TODO No permissions check on this endpoint as of now. Not sure what an appropriate check(s) is?
class GetSysLogsForUser(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request, email, since_utc_time):
        db = get_client_db(request.user.email)
        response = db.get_sys_logs_for_user(email, since_utc_time, request.user.email)
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


######### Get Returned Data for log#########################
# GET support only
# /userLogs/<logId>
# This will take in the log id and return the relevent returned log data
# TODO No permissions check on this endpoint as of now. Not sure what an appropriate check(s) is?
class GetSysLogReturned(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request, logId):
        db = get_client_db(request.user.email)
        response = db.get_sys_log_returned(logId)
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


"""
Angular 2 new endpoints
"""


class AdminEditUsers(LoginRequiredMixin, APIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def put(self, request):
        status_code = status.HTTP_200_OK
        data = "Users updated"

        has_permission = "accounts.can_update_accounts_admin" in User.get_all_permissions(User.objects.get(email=request.user.email))

        if has_permission:
            for user_info in request.data['users_list']:
                if User.objects.filter(email=user_info['email']).exists():
                    user = User.objects.get(email=user_info['email'])

                    if 'delete' in user_info and user_info['delete'] is True:
                        user.delete()
                        db = get_client_db(request.user.email)
                        db.delete_user(request.user.email)
                    else:
                        user.first_name = user_info['firstName']
                        user.last_name = user_info['lastName']

                        if 'newPassword' in user_info:
                            user.set_password(hashlib.md5(user_info['newPassword'].encode('utf-8')).hexdigest())
                            if hasattr(user, 'get_session_auth_hash') and request.user == user:
                                # Gets new session if user changed their own password
                                update_session_auth_hash(request, user)

                        user.groups.clear()
                        for role in user_info['roles']:
                            if Group.objects.filter(name=role).exists():
                                user.groups.add(Group.objects.get(name=role))

                        user.save()
                else:
                    status_code = status.HTTP_500_INTERNAL_SERVER_ERROR
                    data = f"User {user_info['email']} was not found"
                    break
        else:
            status_code = status.HTTP_403_FORBIDDEN
            data = "You are not permitted to update this data. See a system administrator"
        return Response(status=status_code, data=data, headers=set_response_headers(request))


class UserDetails(LoginRequiredMixin, APIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request, user_email):
        status_code = status.HTTP_200_OK
        data = {}

        try:
            has_permission = "accounts.can_read_accounts_admin" in User.get_all_permissions(User.objects.get(email=request.user.email))
            is_current_user = request.user.email == user_email

            # Update details if they are for the current user OR if they have permission
            if is_current_user or has_permission:
                user = User.objects.get(username=user_email)
                data['email'] = user_email
                data['firstName'] = user.first_name
                data['lastName'] = user.last_name
                data['roles'] = list(map(lambda u: u['fields']['name'], json.loads(serializers.serialize('json', user.groups.all()))))
                data['permissions'] = list(User.get_all_permissions(User.objects.get(email=request.user.email)))
                data['lastLogin'] = user.last_login
                data['dateJoined'] = user.date_joined
            else:
                status_code = status.HTTP_403_FORBIDDEN
                data = "You are not permitted to read this data. See a system administrator"
        except Exception as err:
            traceback.print_exc(file=sys.stdout)
            status_code = status.HTTP_500_INTERNAL_SERVER_ERROR
            data = "Could not update user details due to {}::{}".format(type(err).__name__, sys.exc_info()[1])

        return Response(status=status_code, data=data, headers=set_response_headers(request))

    def put(self, request, user_email):
        status_code = status.HTTP_200_OK

        try:
            body = json.loads(request.body)
            user_info = body['userInfo']
            user_to_change = user_email

            has_permission = "accounts.can_update_accounts_admin" in User.get_all_permissions(User.objects.get(email=request.user.email))
            is_current_user = request.user.email == user_to_change

            # Update details if they are for the current user OR if they have permission
            if is_current_user or has_permission:

                # Validating password meets minimum requirements
                if 'password' in user_info:
                    # Setting first name, if present. Else set based on DB user model
                    if 'firstName' in user_info:
                        first_name = user_info['firstName']
                    else:
                        first_name = request.user.first_name

                    # Setting last name, if present. Else set based on DB user model
                    if 'lastName' in user_info:
                        last_name = user_info['lastName']
                    else:
                        last_name = request.user.last_name

                    # Setting current password if present. Else set to blank
                    if 'currentPassword' in user_info:
                        current_password = user_info['currentPassword']
                    else:
                        current_password = ''

                    # Validating password criteria
                    password_requirement_response = validateMinimumPasswordRequirements(request, user_info['password'], request.user.email,
                                                                                        first_name, last_name, current_password)
                # If password is not being configured, it passes requirements
                else:
                    password_requirement_response = {'password_meets_requirements': True}
                # END Validating password meets minimum requirements

                # If password meets reqs, continue
                if password_requirement_response['password_meets_requirements'] is True:
                    user = User.objects.get(email=user_to_change)
                    old = {
                        'firstName': user.first_name,
                        'lastName': user.last_name,
                    }

                    if 'firstName' in user_info:
                        user.first_name = user_info['firstName']
                    if 'lastName' in user_info:
                        user.last_name = user_info['lastName']
                    if 'password' in user_info:
                        user.set_password(hashlib.md5(user_info['password'].encode('utf-8')).hexdigest())
                        user.save()
                        user = authenticate(username=request.user.email, password=hashlib.md5(user_info['password'].encode('utf-8')).hexdigest())
                        update_session_auth_hash(request, user)
                    user.save()
                    data = {
                        'newFirstName': user.first_name,
                        'newLastName': user.last_name
                    }

                    db = get_client_db(request.user.email)
                    action_data = {"collection": "tibit_users", "old": old, "new": {'firstName': user.first_name, 'lastName': user.last_name}, "user": request.user.email}
                    db.sys_log(action_data, "put", status_code == status.HTTP_200_OK)
                else:
                    status_code = status.HTTP_400_BAD_REQUEST
                    data = password_requirement_response['missing_criteria_message']
            else:
                status_code = status.HTTP_403_FORBIDDEN
                data = "You are not permitted to update this data. See a system administrator"

        except Exception as err:
            traceback.print_exc(file=sys.stdout)
            status_code = status.HTTP_500_INTERNAL_SERVER_ERROR
            data = "Could not update user details due to {}::{}".format(type(err).__name__, sys.exc_info()[1])

        return Response(status=status_code, data=data, headers=set_response_headers(request))


class SwitchAutoCounts(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        if "global_config.can_read_automation" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            response = db.get_switch_automation_counts(request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class ControllerCounts(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        if "network.can_read_network_controllers" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            response = db.get_controller_counts(request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class ControllerAutoCounts(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        if "global_config.can_read_automation" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            response = db.get_cntl_automation_counts(request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class OltCounts(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        if "network.can_read_network_olts" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            response = db.get_olt_counts(request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class OltAutoCounts(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        if "global_config.can_read_automation" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            response = db.get_olt_automation_counts(request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class OnuCounts(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        if "network.can_read_network_onus" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            response = db.get_onu_counts(request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class OnuAutoCounts(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        if "global_config.can_read_automation" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            response = db.get_onu_automation_counts(request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


# TODO No permissions check on this endpoint as of now. Not sure what an appropriate check(s) is?
# TODO This request does not look like it should be of type Put.
class SearchTreeDevices(LoginRequiredMixin, APIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        pass

    def put(self, request):
        userData = request.data
        device_type = userData['data']['device']
        filter_str = userData['data']['filter']
        db = get_client_db(request.user.email)
        response = db.search_tree_device(request.user.email, device_type, filter_str)
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class GetOnuConfigsUnderOlt(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request, olt_id):
        if "network.can_read_network_onus" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            response = db.get_onu_configs_for_olt(request.user.email, olt_id)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class GetOnuStatesUnderOlt(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request, olt_id):
        if "network.can_read_network_onus" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            response = db.get_onu_states_for_olt(request.user.email, olt_id)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class GetAllCntlCfgIds(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        if "network.can_read_network_controllers" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            response = db.get_all_cntl_cfg_ids(request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class SwitchCfgIds(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        if "network.can_read_network_switches" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            response = db.get_all_switch_cfg_ids(request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class GetAllOltCfgIds(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        if "network.can_read_network_olts" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            response = db.get_all_olt_cfg_ids(request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))

class GetAllAvailableOltPeers(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request, olt_id):
        if "network.can_read_network_olts" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            response = db.get_all_available_peer_olt_ids(request.user.email, olt_id)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))

class GetAllOnuCfgIds(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        if "network.can_read_network_onus" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            response = db.get_all_onu_cfg_ids(request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


# Makes a config change for ALL the selected devices
class MakeBulkChange(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def post(self, request):
        response = [status.HTTP_200_OK, "", None, {}]
        can_update = False

        try:
            doc_type = request.data['changesObj']['collection']
            # First check appropriate permisssions
            if doc_type == 'ONU Configurations' and "network.can_update_network_onus" in User.get_all_permissions(
                    User.objects.get(email=request.user.email)):
                can_update = True
            elif doc_type == 'OLT Configurations' and "network.can_update_network_olts" in User.get_all_permissions(
                    User.objects.get(email=request.user.email)):
                can_update = True
            elif doc_type == 'Switch Configurations' and "network.can_update_network_controllers" in User.get_all_permissions(
                    User.objects.get(email=request.user.email)):
                can_update = True
            elif doc_type == 'Controller Configurations' and "network.can_update_network_switches" in User.get_all_permissions(
                    User.objects.get(email=request.user.email)):
                can_update = True

            # If one of the above situations is True
            if can_update:
                db = get_client_db(request.user.email)
                changes = request.data
                response = db.make_bulk_change(changes, request.user.email)
            else:
                response = [status.HTTP_403_FORBIDDEN, "You are not permitted to update this data. See a system administrator.", None, {}]
        except Exception as err:
            response[0] = status.HTTP_500_INTERNAL_SERVER_ERROR
            response[1] = "Could not get Search State information due to {}::{}".format(type(err).__name__, sys.exc_info()[1])

        return PonManagerApiResponse(status=response[0], data=response[1], new_data=response[2], old_data=response[3], headers=set_response_headers(request))


class GetAllOlts(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        if "network.can_read_network_olts" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            response = db.get_all_olts(request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


# TODO No permissions check on this endpoint as of now. Not sure what an appropriate check(s) is?
class SearchStates(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request, collection):
        response = [status.HTTP_200_OK, ""]

        try:
            try:
                with open(production_search_states[int(collection)], 'r', encoding='utf-8') as state_file:
                    state_data = json.load(state_file)
            except FileNotFoundError:
                with open(development_search_states[int(collection)], 'r', encoding='utf-8') as state_file:
                    state_data = json.load(state_file)

            response[1] = state_data
        except Exception as err:
            response[0] = status.HTTP_500_INTERNAL_SERVER_ERROR
            response[1] = "Could not get Search State information due to {}::{}".format(type(err).__name__, sys.exc_info()[1])

        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class OnuDistinctIdentifiers(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        if "network.can_read_network_onus" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            response = db.get_distinct_onu_state_identifiers(request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class OltDistinctIdentifiers(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        if "network.can_read_network_olts" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            response = db.get_distinct_olt_state_identifiers(request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


# Handles getting, creating, editing, and deleting PON Automation templates
class AutomationStates(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request, device_type, device_id):
        if "global_config.can_read_automation" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            response = db.get_automation_state(device_type, device_id, request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to read this data. See a system administrator."]

        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class OnuAutomationStatesForOlt(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request, olt_id):
        if "global_config.can_read_automation" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            response = db.get_onu_automation_states_for_olt(olt_id, request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to read this data. See a system administrator."]

        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


# Handles getting, creating, editing, and deleting PON Automation templates
class AutomationGlobalConfigs(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request, device_type):
        if "global_config.can_read_automation" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            response = db.get_automation_cfg(device_type, "Global", request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to read this data. See a system administrator."]

        return Response(status=response[0], data=response[1], headers=set_response_headers(request))

    # Update the global automation config
    def put(self, request, device_type):
        if "global_config.can_update_automation" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            response = db.update_automation_cfg(device_type, "Global", json.loads(request.body), request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to update this data. See a system administrator."]

        return Response(status=response[0], data=response[1], headers=set_response_headers(request))

    # Delete the specified reference from the Global document for the given device type
    def delete(self, request, device_type, automation_step, reference_name):
        if "global_config.can_delete_automation" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            response = db.delete_automation_reference(device_type, automation_step, reference_name, request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to delete this data. See a system administrator."]

        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


# Handles getting, creating, editing, and deleting PON Automation templates
class AutomationDeviceConfigs(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    # Get the specified device's automation configuration
    def get(self, request, device_type, device_id):
        if "global_config.can_read_automation" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            response = db.get_automation_cfg(device_type, device_id, request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to read this data. See a system administrator."]

        return Response(status=response[0], data=response[1], headers=set_response_headers(request))

    # Update the specified device's automation configuration
    def put(self, request, device_type, device_id):
        if "global_config.can_update_automation" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            response = db.update_automation_cfg(device_type, device_id, json.loads(request.body), request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to update this data. See a system administrator."]

        return Response(status=response[0], data=response[1], headers=set_response_headers(request))

    # Delete the specified reference from the Global document for the given device type
    def delete(self, request, device_type, device_id):
        if "global_config.can_delete_automation" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            response = db.delete_automation_cfg(device_type, device_id, request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to delete this data. See a system administrator."]

        return Response(status=response[0], data=response[1], headers=set_response_headers(request))

# Utility function
# Validates that password parameter meets all minimum password requirements
# TODO How do I handle if password requirements in MongoDB do NOT exist?
def validateMinimumPasswordRequirements(request, password, email=None, first_name=None, last_name=None, current_password=None):
    password_meets_requirements = True
    missing_criteria_message = []

    lowercase_character_count = 0
    uppercase_character_count = 0
    number_character_count = 0
    special_character_count = 0

    # Setting requesters email, if it exists
    try:
        requesting_email = request.user.email
    except Exception as err:
        requesting_email = 'AnonymousUser'


    # Retrieve minimum password criteria from MongoDB
    db = get_client_db(requesting_email)
    minimum_password_criteria = db.get_minimum_password_requirements(requesting_email)[1]

    # Minimum Length
    if len(password) < minimum_password_criteria['Minimum Length']:
        password_meets_requirements = False
        missing_criteria_message.append("Password is too short. Password contains only " + str(len(password)) + " character(s); Atleast " + str(minimum_password_criteria['Minimum Length']) + " characters are required.")

    # Maximum Length
    if len(password) > minimum_password_criteria['Maximum Length']:
        password_meets_requirements = False
        missing_criteria_message.append("Password is too long. Password contains " + str(len(password)) + " character(s); " + str(minimum_password_criteria['Maximum Length']) + " characters are allowed.")


    # Looping through each character in the password
    for char in password:
        # Counting lowercase characters
        if char.islower():
            lowercase_character_count+=1

        # Counting uppercase characters
        if char.isupper():
            uppercase_character_count+=1

        # Counting numeric characters
        if char.isnumeric():
            number_character_count+=1

        # Counting special characters
        if not (char.isnumeric() and char.isspace() and char.isalpha()):
            special_character_count+=1

        # Does not contain excluded Characters
        if char in minimum_password_criteria['Excluded Characters']:
            password_meets_requirements = False
            missing_criteria_message.append("Password contains invalid character(s). Password is not allowed to contain " + char)

    # Verifying lowercase count
    if lowercase_character_count < minimum_password_criteria['Minimum Lowercase Character Count']:
        password_meets_requirements = False
        missing_criteria_message.append("Password requires atleast " + str(minimum_password_criteria['Minimum Lowercase Character Count']) + " lowercase character(s); " + str(lowercase_character_count) + " lowercase characters are present.")

    # Verifying uppercase count
    if uppercase_character_count < minimum_password_criteria['Minimum Uppercase Character Count']:
        password_meets_requirements = False
        missing_criteria_message.append("Password requires atleast " + str(minimum_password_criteria['Minimum Uppercase Character Count']) + " uppercase character(s); " + str(uppercase_character_count) + " uppercase characters are present.")

    # Verifying number count
    if number_character_count < minimum_password_criteria['Minimum Number Count']:
        password_meets_requirements = False
        missing_criteria_message.append("Password requires atleast " + str(minimum_password_criteria['Minimum Number Count']) + " number(s); " + str(number_character_count) + " numbers are present.")

    # Verifying special character count
    if special_character_count < minimum_password_criteria['Minimum Special Character Count']:
        password_meets_requirements = False
        missing_criteria_message.append("Password requires atleast " + str(minimum_password_criteria['Minimum Special Character Count']) + " special character(s);" + str(special_character_count) + " special characters are present.")

    # ASCII Standard Characters Only
    found_non_ascii_character = False
    if minimum_password_criteria['ASCII Standard Characters Only'] is True:
        try:
            password.encode('ascii')
        except UnicodeEncodeError:
            found_non_ascii_character = True

        if found_non_ascii_character is True:
            password_meets_requirements = False
            missing_criteria_message.append("Invalid password characters.")

    # Does not include email
    if minimum_password_criteria['Exclude Email'] is True and email is not None:
        email_components = email.split('@')
        if len(email_components) < 2 or email_components[0].lower() in password.lower() or email_components[1].lower() in password.lower():
            password_meets_requirements = False
            missing_criteria_message.append("Password cannot contain email.")

    # Does not include Name
    # TODO Might want to think about this more. What if users name is a single name that may naturally occur in a password?
    if minimum_password_criteria['Exclude Name'] is True and first_name is not None and last_name is not None:
        if first_name.lower() in password.lower() or last_name.lower() in password.lower():
            password_meets_requirements = False
            missing_criteria_message.append("Password cannot contain name.")

    # Current Password is correct
    if current_password is not None:
        user = authenticate(username=email, password=hashlib.md5(current_password.encode('utf-8')).hexdigest())

        if user is None:
            password_meets_requirements = False
            missing_criteria_message.append("Current password is invalid.")

    # Meets minimum strength
    if minimum_password_criteria['Minimum Password Strength'].lower() != 'none':
        # Password Strength enum
        password_strength_enum = {
            'none': 0,
            'weak': 1,
            'fair': 2,
            'strong': 3,
            'very strong': 4
        }
        password_strength_score = zxcvbn(password)['score']
        minimum_password_strength_as_int = password_strength_enum[minimum_password_criteria['Minimum Password Strength'].lower()]

        if password_strength_score < minimum_password_strength_as_int:
            password_meets_requirements = False
            missing_criteria_message.append("Password is not strong enough. A strength of at least " + minimum_password_criteria['Minimum Password Strength'] + " is required.")

    return {"password_meets_requirements" : password_meets_requirements,
            "missing_criteria_message" : missing_criteria_message}


class GrafanaOltLinks(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def put(self, request):
        has_permissions = False

        if "network.can_update_network_olts" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            has_permissions = True

        if has_permissions:
            db = get_client_db(request.user.email)
            body = request.data
            response = db.put_grafana_olt_links(body, request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to update this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))

    def get(self, request):
        has_permissions = False

        if "network.can_read_network_olts" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            has_permissions = True

        if has_permissions:
            db = get_client_db(request.user.email)
            response = db.get_grafana_olt_links(request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to see this data. See a system administrator."]

        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class GrafanaOnuLinks(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def put(self, request):
        has_permissions = False

        if "network.can_update_network_onus" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            has_permissions = True

        if has_permissions:
            db = get_client_db(request.user.email)
            body = request.data
            response = db.put_grafana_onu_links(body, request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to update this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))

    def get(self, request):
        has_permissions = False

        if "network.can_read_network_onus" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            has_permissions = True

        if has_permissions:
            db = get_client_db(request.user.email)
            response = db.get_grafana_onu_links(request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to see this data. See a system administrator."]

        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class GetOnuInventory(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        params = request.query_params
        switch_id = params['switch_id']
        if "network.can_read_network_onus" in User.get_all_permissions(User.objects.get(email=request.user.email)):
            db = get_client_db(request.user.email)
            response = db.get_onu_inventory(switch_id, request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))
