#!/usr/bin/env python3
#--------------------------------------------------------------------------#
# Copyright (C) 2022 by Tibit Communications, Inc.                         #
# All rights reserved.                                                     #
#                                                                          #
#    _______ ____  _ ______                                                #
#   /_  __(_) __ )(_)_  __/                                                #
#    / / / / __  / / / /                                                   #
#   / / / / /_/ / / / /                                                    #
#  /_/ /_/_____/_/ /_/                                                     #
#                                                                          #
#--------------------------------------------------------------------------#

import argparse
import sys
import os
sys.path.append(os.path.join(os.path.dirname(os.path.realpath(__file__)), ".."))
from netconf_driver import NetconfDriver


if __name__ == '__main__':

    # Command line arguments
    parser = argparse.ArgumentParser(add_help=False,formatter_class=argparse.ArgumentDefaultsHelpFormatter)
    parser.add_argument(      "--help", action="help", default=argparse.SUPPRESS, help="Show this help message and exit.")
    parser.add_argument("-h", "--host", action="store", dest="host", default='127.0.0.1', required=False, help="NETCONF Server IP address or hostname.")
    parser.add_argument(      "--onu", action="store", dest="onu", default=None, required=True, help="ONU Serial Number (e.g., TBITc84c00df)")
    parser.add_argument("-p", "--port", action="store", dest="port", default='830', required=False, help="NETCONF Server port number.")
    parser.add_argument("-u", "--user", action="store", dest="user", default=None, required=False, help="Username.")
    parser.add_argument("-w", "--passwd", action="store", dest="passwd", default=None, required=False, help="Password. If no password is provided, attempt to read it from .nc_edit_auth.")
    parser.add_argument("-v", "--verbose", action="store_true", dest="verbose", default=False, required=False, help="Verbose output.")
    parser.parse_args()
    args = parser.parse_args()

    nc = NetconfDriver(host=args.host, port=args.port, user=args.user, passwd=args.passwd, verbose=args.verbose)
    if not nc:
        # Error
        print(f"ERROR: Failed to connect to Netconf server {args.host}:{args.port}.")
        sys.exit(1)

    # Build an options dictionary from the command line arguments
    # The variables {{VAR}} be substituted in the Netconf requests from the .xml files sent below
    options = {
        "{{ONU}}" : args.onu,
    }

    # Reset an OLT Device by MAC Address
    print(f"Resetting ONU {args.onu}.")
    nc.rpc(
        data_xml='''
<rpc xmlns="urn:ietf:params:xml:ns:netconf:base:1.0"
xmlns:yang-1_1="urn:ietf:params:xml:ns:yang:1"
xmlns:bbf-hw="urn:bbf:yang:bbf-hardware"
message-id="1">
<action xmlns="urn:ietf:params:xml:ns:yang:1">
    <hardware xmlns="urn:ietf:params:xml:ns:yang:ietf-hardware">
        <component>
            <name>XGS-PON ONU {{ONU}}</name>
            <reset xmlns="urn:bbf:yang:bbf-hardware">
                <reset-type>bbf-hw:hardware-reset</reset-type>
            </reset>
        </component>
    </hardware>
</action>
</rpc>
        ''',
        options=options
    )

