#!/usr/bin/env python3
#--------------------------------------------------------------------------#
# Copyright (C) 2022 by Tibit Communications, Inc.                         #
# All rights reserved.                                                     #
#                                                                          #
#    _______ ____  _ ______                                                #
#   /_  __(_) __ )(_)_  __/                                                #
#    / / / / __  / / / /                                                   #
#   / / / / /_/ / / / /                                                    #
#  /_/ /_/_____/_/ /_/                                                     #
#                                                                          #
#--------------------------------------------------------------------------#

import argparse
import sys
import os
from lxml import etree
sys.path.append(os.path.join(os.path.dirname(os.path.realpath(__file__)), ".."))
from netconf_driver import NetconfDriver


def get_ietf_hardware_name_for_olt(nc, olt):
    ietf_hardware_name = None

    # Send a Netconf <get> request to retreive the ietf-hardware names
    IETF_HARDWARE_NAMES = '''
<rpc xmlns="urn:ietf:params:xml:ns:netconf:base:1.0"
    xmlns:hw="urn:ietf:params:xml:ns:yang:ietf-hardware"
    message-id="1">
    <get>
        <filter type="subtree">
            <hw:hardware >
                <hw:component>
                    <hw:name/>
                </hw:component>
            </hw:hardware>
        </filter>
    </get>
</rpc>
    '''
    rsp_xml = nc.get(data_xml=IETF_HARDWARE_NAMES, message="/ietf-hardware:hardware/ietf-hardware:component/ietf-hardware:name")

    # Parse the Netconf response and find the ietf-hardware entry for this OLT.
    if rsp_xml:
        NSMAP = {
            'nc' : "urn:ietf:params:xml:ns:netconf:base:1.0",
            'hw' : "urn:ietf:params:xml:ns:yang:ietf-hardware",
            }
        root = etree.fromstring(rsp_xml)
        for name in root.findall(f"nc:data/hw:hardware/hw:component/hw:name", namespaces=NSMAP):
            name = name.text
            if olt in name:
                ietf_hardware_name = name
                break

    return ietf_hardware_name

if __name__ == '__main__':

    # Command line arguments
    parser = argparse.ArgumentParser(add_help=False,formatter_class=argparse.ArgumentDefaultsHelpFormatter)
    parser.add_argument(      "--help", action="help", default=argparse.SUPPRESS, help="Show this help message and exit.")
    parser.add_argument("-h", "--host", action="store", dest="host", default='127.0.0.1', required=False, help="NETCONF Server IP address or hostname.")
    parser.add_argument(      "--olt", action="store", dest="olt", default=0, required=False, help="OLT device MAC address")
    parser.add_argument("-p", "--port", action="store", dest="port", default='830', required=False, help="NETCONF Server port number.")
    parser.add_argument("-u", "--user", action="store", dest="user", default=None, required=False, help="Username.")
    parser.add_argument("-w", "--passwd", action="store", dest="passwd", default=None, required=False, help="Password. If no password is provided, attempt to read it from .nc_edit_auth.")
    parser.add_argument("-v", "--verbose", action="store_true", dest="verbose", default=False, required=False, help="Verbose output.")
    parser.parse_args()
    args = parser.parse_args()


    nc = NetconfDriver(host=args.host, port=args.port, user=args.user, passwd=args.passwd, verbose=args.verbose)
    if not nc:
        # Error
        print(f"ERROR: Failed to connect to Netconf server {args.host}:{args.port}.")
        sys.exit(1)

    # Find the ietf-hardware entry that represents this OLT
    ietf_hardware_name = get_ietf_hardware_name_for_olt(nc, args.olt)

    # Reset an OLT Device by MAC Address
    if ietf_hardware_name:
        print(f"Resetting OLT {args.olt}.")
        nc.rpc(
            data_xml=f'''
<rpc xmlns="urn:ietf:params:xml:ns:netconf:base:1.0"
xmlns:yang-1_1="urn:ietf:params:xml:ns:yang:1"
xmlns:bbf-hw="urn:bbf:yang:bbf-hardware"
message-id="1">
<action xmlns="urn:ietf:params:xml:ns:yang:1">
    <hardware xmlns="urn:ietf:params:xml:ns:yang:ietf-hardware">
        <component>
            <name>{ietf_hardware_name}</name>
            <reset xmlns="urn:bbf:yang:bbf-hardware">
                <reset-type>bbf-hw:hardware-reset</reset-type>
            </reset>
        </component>
    </hardware>
</action>
</rpc>
            '''
        )
    else:
        # Error
        print(f"ERROR: ietf-hardware entry for OLT {args.olt} not found.")

