#!/usr/bin/env python3
#--------------------------------------------------------------------------#
# Copyright (C) 2020 - 2022 by Tibit Communications, Inc.                  #
# All rights reserved.                                                     #
#                                                                          #
#    _______ ____  _ ______                                                #
#   /_  __(_) __ )(_)_  __/                                                #
#    / / / / __  / / / /                                                   #
#   / / / / /_/ / / / /                                                    #
#  /_/ /_/_____/_/ /_/                                                     #
#                                                                          #
#   MicroPlug PON Controller                                               #
#                                                                          #
#                                                                          #
#--------------------------------------------------------------------------#

import argparse
import json
import os
import shutil
import sys

def _merge_dict(source, destination):
    for key, value in source.items():
        if isinstance(value, dict):
            # get node or create one
            node = destination.setdefault(key, {})
            _merge_dict(value, node)
        else:
            destination[key] = value

    return destination

if __name__ == '__main__':

    # Example command line invocation of the migration script
    #
    # sudo ./migrate_poncntlinit.py -s /etc/tibit/poncntl/PonCntlInit.json.bak -d /etc/tibit/poncntl/PonCntlInit.json
    #
    parser = argparse.ArgumentParser(add_help=True,formatter_class=argparse.ArgumentDefaultsHelpFormatter)
    parser.add_argument("-d", "--dest", action="store", dest="dst", default="/etc/tibit/poncntl/PonCntlInit.json", required=False, help="Migrated PonCntlInit.json file")
    parser.add_argument("-s", "--src", action="store", dest="src", default="/etc/tibit/poncntl/PonCntlInit.json.bak", required=False, help="Source PonCntlInit.json file")
    args = parser.parse_args()

    dst = args.dst
    src = args.src

    rc = 0

    if os.path.exists(src) and os.path.isfile(src):
        if os.path.exists(dst) and os.path.isfile(dst):
            try:
                with open(src) as in_file:
                    ponCntlInit = json.load(in_file)

                    cfgInfo = {}

                    cfgInfo["CNTL"]                     = {}
                    cfgInfo["CNTL"]["CFG Version"]      = "R1.3.0"
                    cfgInfo["CNTL"]["UMT interface"]    = "tibitvirt"
                    cfgInfo["CNTL"]["Auth"]             = False
                    cfgInfo["CNTL"]["DHCPv4"]           = False

                    cfgInfo["DEBUG"]                    = {}

                    cfgInfo["MongoDB"] = {}
                    cfgInfo["MongoDB"]["name"]          = "tibit_pon_controller"
                    cfgInfo["MongoDB"]["host"]          = "127.0.0.1"
                    cfgInfo["MongoDB"]["port"]          = "27017"
                    cfgInfo["MongoDB"]["auth_enable"]   = False
                    cfgInfo["MongoDB"]["auth_db"]       = "tibit_users"
                    cfgInfo["MongoDB"]["username"]      = ""
                    cfgInfo["MongoDB"]["password"]      = ""
                    cfgInfo["MongoDB"]["tls_enable"]    = False
                    cfgInfo["MongoDB"]["ca_cert_path"]  = "/etc/tibit/ca.pem"

                    cfgInfo["JSON"] = {}
                    cfgInfo["JSON"]["defaultDir"]       = "database/"
                    cfgInfo["JSON"]["databaseDir"]      = "database/"

                    cfgInfo["Local Copy"] = {}
                    cfgInfo["Local Copy"]["CNTL-STATE"] = False
                    cfgInfo["Local Copy"]["OLT-STATE"]  = False
                    cfgInfo["Local Copy"]["ONU-STATE"]  = False

                    cfgInfo["Logging"] = {}
                    cfgInfo["Logging"]["Directory"]     = "/var/log/tibit"
                    cfgInfo["Logging"]["FileSize"]      = 1024000
                    cfgInfo["Logging"]["FileCount"]     = 3

                    cfgInfo["databaseType"]             = "MongoDB"
                    cfgInfo["interface"]                = "enp0s8.4090"

                    cfgInfo = _merge_dict(ponCntlInit, cfgInfo)

                    # Migrate the database directories
                    cfgInfo["CNTL"]["CFG Version"]      = "R1.3.0"
                    cfgInfo["DEBUG"]                    = {}
                    cfgInfo["JSON"]["defaultDir"]       = "/opt/tibit/poncntl/database/"
                    cfgInfo["JSON"]["databaseDir"]      = "/opt/tibit/poncntl/database/"

                    with open(dst, 'w') as out_file:
                        json.dump(cfgInfo, out_file, indent=4, sort_keys=True, ensure_ascii=False)

            except Exception as err:
                print("ERROR: Migration failed for source {}, destination {}, error {}".format(src, dst, err))
                rc = 1
        else:
            print("ERROR: Destination file {} is missing.".format(dst))
            rc = 1
    else:
        print("ERROR: Source file {} is missing.".format(src))
        rc = 1

    sys.exit(rc)
