#!/usr/bin/env python3
#--------------------------------------------------------------------------#
# Copyright (C) 2022 by Tibit Communications, Inc.                         #
# All rights reserved.                                                     #
#                                                                          #
#    _______ ____  _ ______                                                #
#   /_  __(_) __ )(_)_  __/                                                #
#    / / / / __  / / / /                                                   #
#   / / / / /_/ / / / /                                                    #
#  /_/ /_/_____/_/ /_/                                                     #
#                                                                          #
#--------------------------------------------------------------------------#

""" Configure Firmware Upgrade for an ONU.

This Tibit YANG Example script configures firmware upgrade for an ONU. The script
provides examples for configuring firmware images for an ONU.

Example - Configure the firmware image (filename + version string) in bank 1:

  ./firmware_upgrade/edit_config_onu_firmware.py \
      --onu ALPHe30cadcf \
      --upgrade-bank 1 \
      --upgrade-file FW-GPON-ALPH-34000-5025_007_SFU24.bin \
      --upgrade-version 5025_007_SFU24

usage: edit_config_onu_firmware.py [--help]
                                   [-h HOST] --onu ONU [-w PASSWD] [-p PORT]
                                   [--upgrade-bank UPGRADE_BANK]
                                   [--upgrade-file UPGRADE_FILE]
                                   [--upgrade-version UPGRADE_VERSION]
                                   [-u USER] [-v]

optional arguments:
  --help                Show this help message and exit.
  -h HOST, --host HOST  NETCONF Server IP address or hostname. (default:
                        127.0.0.1)
  --onu ONU             ONU Serial Number (e.g., TBITc84c00df) (default: None)
  -w PASSWD, --passwd PASSWD
                        Password. If no password is provided, the user will be
                        prompted to enter. (default: None)
  -p PORT, --port PORT  NETCONF Server port number. (default: 830)
  --upgrade-bank UPGRADE_BANK
                        Specify the bank for that the --upgrade-version and
                        --upgrade-file options apply to. (default: None)
  --upgrade-file UPGRADE_FILE
                        Firmware filename to set for the specified bank.
                        (default: None)
  --upgrade-version UPGRADE_VERSION
                        Firmware version to set for the specified bank.
                        (default: None)
  -u USER, --user USER  Username. (default: None)
  -v, --verbose         Verbose output. (default: False)

"""

import argparse
from lxml import etree
import os
import sys
sys.path.append(os.path.join(os.path.dirname(os.path.realpath(__file__)), ".."))
from netconf_driver import NetconfDriver


def get_fw_upgrade_config(nc, options):
    """
    Get the firmware upgrade configuration for an ONU.

    Args:
        nc: Netconf driver object.
        options: Netconf request options ({{VAR}}=value) dictionary.

    Returns:
        firmware_upgrade_config: Dictionary containing the firmware upgrade information for the ONU.
    """
    firmware_upgrade_config = {
        'fw-bank-ptr': 65535,
        'fw-bank-files': ["", ""],
        'fw-bank-versions': ["", ""],
    }

    # Send a Netconf <get> request to retreive the ONU configuration
    ONU_CFG = '''
    <rpc
        xmlns="urn:ietf:params:xml:ns:netconf:base:1.0"
        xmlns:tibitcntlr="urn:com:tibitcom:ns:yang:controller:db"
        message-id="34566760">
        <get>
            <filter type="subtree">
                <tibitcntlr:onu>
                    <tibitcntlr:onu>
                        <tibitcntlr:name>{{ONU}}</tibitcntlr:name>
                        <tibitcntlr:onu>
                            <tibitcntlr:fw-bank-ptr/>
                            <tibitcntlr:fw-bank-files/>
                            <tibitcntlr:fw-bank-versions/>
                        </tibitcntlr:onu>
                    </tibitcntlr:onu>
                </tibitcntlr:onu>
            </filter>
        </get>
    </rpc>
    '''
    rsp_xml = nc.get(data_xml=ONU_CFG, options=options, message="/tibit-pon-controller-db::onu/tibit-pon-controller-db:onu/tibit-pon-controller-db:onu")

    # Parse the Netconf response and retrieve the firmware bank pointer from the XML response data.
    if rsp_xml:
        NSMAP = {
            'nc' : "urn:ietf:params:xml:ns:netconf:base:1.0",
            'tibitcntlr' : "urn:com:tibitcom:ns:yang:controller:db",
            }
        root = etree.fromstring(rsp_xml)
        for fw_bank_ptr in root.findall("nc:data/tibitcntlr:onu/tibitcntlr:onu/tibitcntlr:onu/tibitcntlr:fw-bank-ptr", namespaces=NSMAP):
            firmware_upgrade_config['fw-bank-ptr'] = int(fw_bank_ptr.text)

    # Parse the Netconf response and retrieve the firmware image file names for all banks from the XML response data.
    if rsp_xml:
        for fw_bank_file in root.findall("nc:data/tibitcntlr:onu/tibitcntlr:onu/tibitcntlr:onu/tibitcntlr:fw-bank-files", namespaces=NSMAP):
            fw_bank_file_id = fw_bank_file.find("tibitcntlr:id", namespaces=NSMAP)
            if fw_bank_file_id is not None:
                fw_bank_file_id = int(fw_bank_file_id.text)
                fw_bank_file_value = fw_bank_file.find("tibitcntlr:file", namespaces=NSMAP)
                if fw_bank_file_value is not None:
                    fw_bank_file_value = fw_bank_file_value.text
                    if fw_bank_file_value:
                        firmware_upgrade_config['fw-bank-files'][fw_bank_file_id] = fw_bank_file_value

    # Parse the Netconf response and retrieve the firmware image versions for all banks from the XML response data.
    if rsp_xml:
        for fw_bank_version in root.findall("nc:data/tibitcntlr:onu/tibitcntlr:onu/tibitcntlr:onu/tibitcntlr:fw-bank-versions", namespaces=NSMAP):
            fw_bank_version_id = fw_bank_version.find("tibitcntlr:id", namespaces=NSMAP)
            if fw_bank_version_id is not None:
                fw_bank_version_id = int(fw_bank_version_id.text)
                fw_bank_version_value = fw_bank_version.find("tibitcntlr:version", namespaces=NSMAP)
                if fw_bank_version_value is not None:
                    fw_bank_version_value = fw_bank_version_value.text
                    if fw_bank_version_value:
                        firmware_upgrade_config['fw-bank-versions'][fw_bank_version_id] = fw_bank_version_value

    return firmware_upgrade_config


def get_fw_upgrade_state(nc, options):
    """
    Get the firmware upgrade state for an ONU.

    Args:
        nc: Netconf driver object.
        options: Netconf request options ({{VAR}}=value) dictionary.

    Returns:
        firmware_upgrade_state: Dictionary containing the firmware upgrade information for the ONU.
    """
    firmware_upgrade_state = {
        'fw-bank-ptr': 65535,
        'fw-bank-versions': ["", ""],
    }

    # Send a Netconf <get> request to retreive the state for an ONU
    ONU_STATE = '''
    <rpc
        xmlns="urn:ietf:params:xml:ns:netconf:base:1.0"
        xmlns:tibitcntlr="urn:com:tibitcom:ns:yang:controller:db"
        message-id="34566760">
        <get>
            <filter type="subtree">
                <tibitcntlr:onu-state>
                    <tibitcntlr:onu>
                        <name>{{ONU}}</name>
                        <tibitcntlr:onu>
                            <tibitcntlr:fw-bank-ptr/>
                            <tibitcntlr:fw-bank-files/>
                            <tibitcntlr:fw-bank-versions/>
                        </tibitcntlr:onu>
                    </tibitcntlr:onu>
                </tibitcntlr:onu-state>
            </filter>
        </get>
    </rpc>
    '''
    rsp_xml = nc.get(data_xml=ONU_STATE, options=options, message="/tibit-pon-controller-db::onu-state/tibit-pon-controller-db:onu/tibit-pon-controller-db:onu")

    # Parse the Netconf response and retrieve the firmware bank pointer from the XML response data.
    if rsp_xml:
        NSMAP = {
            'nc' : "urn:ietf:params:xml:ns:netconf:base:1.0",
            'tibitcntlr' : "urn:com:tibitcom:ns:yang:controller:db",
            }
        root = etree.fromstring(rsp_xml)
        for fw_bank_ptr in root.findall("nc:data/tibitcntlr:onu-state/tibitcntlr:onu/tibitcntlr:onu/tibitcntlr:fw-bank-ptr", namespaces=NSMAP):
            firmware_upgrade_state['fw-bank-ptr'] = int(fw_bank_ptr.text)

    # Parse the Netconf response and retrieve the firmware image versions for all banks from the XML response data.
    if rsp_xml:
        for fw_bank_version in root.findall("nc:data/tibitcntlr:onu-state/tibitcntlr:onu/tibitcntlr:onu/tibitcntlr:fw-bank-versions", namespaces=NSMAP):
            fw_bank_version_id = fw_bank_version.find("tibitcntlr:id", namespaces=NSMAP)
            if fw_bank_version_id is not None:
                fw_bank_version_id = int(fw_bank_version_id.text)
                fw_bank_version_value = fw_bank_version.find("tibitcntlr:version", namespaces=NSMAP)
                if fw_bank_version_value is not None:
                    fw_bank_version_value = fw_bank_version_value.text
                    if fw_bank_version_value:
                        firmware_upgrade_state['fw-bank-versions'][fw_bank_version_id] = fw_bank_version_value

    return firmware_upgrade_state


if __name__ == '__main__':
    # Command line arguments
    parser = argparse.ArgumentParser(add_help=False,formatter_class=argparse.ArgumentDefaultsHelpFormatter)
    parser.add_argument(      "--help", action="help", default=argparse.SUPPRESS, help="Show this help message and exit.")
    parser.add_argument("-h", "--host", action="store", dest="host", default='127.0.0.1', required=False, help="NETCONF Server IP address or hostname.")
    parser.add_argument(      "--onu", action="store", dest="onu", default=None, required=True, help="ONU Serial Number (e.g., TBITc84c00df)")
    parser.add_argument("-w", "--passwd", action="store", dest="passwd", default=None, required=False, help="Password. If no password is provided, the user will be prompted to enter.")
    parser.add_argument("-p", "--port", action="store", dest="port", default='830', required=False, help="NETCONF Server port number.")
    parser.add_argument(      "--upgrade-bank", action="store", dest="upgrade_bank", default=None, required=True, help="Specify the bank for that the --upgrade-version and --upgrade-file options apply to.")
    parser.add_argument(      "--upgrade-file", action="store", dest="upgrade_file", default=None, required=True, help="Firmware filename to set for the specified bank.")
    parser.add_argument(      "--upgrade-version", action="store", dest="upgrade_version", default=None, required=True, help="Firmware version to set for the specified bank.")
    parser.add_argument("-u", "--user", action="store", dest="user", default=None, required=False, help="Username.")
    parser.add_argument("-v", "--verbose", action="store_true", dest="verbose", default=False, required=False, help="Verbose output.")
    parser.parse_args()
    args = parser.parse_args()

    # Validate parameters
    upgrade_bank = int(args.upgrade_bank)
    if upgrade_bank != 0 and upgrade_bank != 1:
        print("ERROR: The --upgrade-bank must be set to '0' or '1'.")
        sys.exit()

    nc = NetconfDriver(host=args.host, port=args.port, user=args.user, passwd=args.passwd, verbose=args.verbose)
    if not nc:
        # Error
        print(f"ERROR: Failed to connect to Netconf server {args.host}:{args.port}.")
        sys.exit(1)

    # Build an options dictionary from the command line arguments
    options = {
        "{{ONU}}" : args.onu,
        "{{ACTIVE_BANK}}" : 65535,
        "{{FW_FILE_BANK0}}" : "",
        "{{FW_VERSION_BANK0}}" : "",
        "{{FW_FILE_BANK1}}" : "",
        "{{FW_VERSION_BANK1}}" : "",
    }

    # PON Controller requires the FW bank pointer to be set in order to
    # upgrade the ONU. If the ONU is configured with a firmware bank pointer
    # value disabled(65535), read and use the value from ONU-STATE in to
    # initiate the download.
    firmware_upgrade_config = get_fw_upgrade_config(nc, options)
    firmware_upgrade_state = get_fw_upgrade_state(nc, options)

    # Set update the initial firmware configuration based on state
    if firmware_upgrade_config['fw-bank-ptr'] == 65535:
        firmware_upgrade_config['fw-bank-ptr'] = firmware_upgrade_state['fw-bank-ptr']
    if not firmware_upgrade_config['fw-bank-versions'][0]:
        firmware_upgrade_config['fw-bank-versions'][0] = firmware_upgrade_state['fw-bank-versions'][0]
    if not firmware_upgrade_config['fw-bank-versions'][1]:
        firmware_upgrade_config['fw-bank-versions'][1] = firmware_upgrade_state['fw-bank-versions'][1]

    # Update the firmware upgrade settings based on the values passed into the script
    firmware_upgrade_config['fw-bank-files'][upgrade_bank] = args.upgrade_file
    firmware_upgrade_config['fw-bank-versions'][upgrade_bank] = args.upgrade_version

    # Update the options with the FW configuration information
    options["{{ACTIVE_BANK}}"] = firmware_upgrade_config['fw-bank-ptr']
    options["{{FW_FILE_BANK0}}"] = firmware_upgrade_config['fw-bank-files'][0]
    options["{{FW_VERSION_BANK0}}"] = firmware_upgrade_config['fw-bank-versions'][0]
    options["{{FW_FILE_BANK1}}"] = firmware_upgrade_config['fw-bank-files'][1]
    options["{{FW_VERSION_BANK1}}"] = firmware_upgrade_config['fw-bank-versions'][1]

    # Configure the firmware image for a bank
    nc.edit_config(filename="1-onu-cfg-firmware-upgrade.xml", options=options)

    # Display a summary of what was configured
    print(f"\nConfigured Firmware Upgrade for ONU {options['{{ONU}}']}")
    print(f" Firmware Bank Pointer:  {options['{{ACTIVE_BANK}}']}")
    print(f" Bank 0:")
    print(f"   Version: {options['{{FW_VERSION_BANK0}}']}")
    print(f"   File:    {options['{{FW_FILE_BANK0}}']}")
    print(f" Bank 1:")
    print(f"   Version: {options['{{FW_VERSION_BANK1}}']}")
    print(f"   File:    {options['{{FW_FILE_BANK1}}']}")