#!/usr/bin/env python3
#--------------------------------------------------------------------------#
# Copyright (C) 2022 by Tibit Communications, Inc.                         #
# All rights reserved.                                                     #
#                                                                          #
#    _______ ____  _ ______                                                #
#   /_  __(_) __ )(_)_  __/                                                #
#    / / / / __  / / / /                                                   #
#   / / / / /_/ / / / /                                                    #
#  /_/ /_/_____/_/ /_/                                                     #
#                                                                          #
#--------------------------------------------------------------------------#

import argparse
import sys
import os
sys.path.append(os.path.join(os.path.dirname(os.path.realpath(__file__)), ".."))
from netconf_driver import NetconfDriver


if __name__ == '__main__':

    # Command line arguments
    parser = argparse.ArgumentParser(add_help=False,formatter_class=argparse.ArgumentDefaultsHelpFormatter)
    parser.add_argument(      "--help", action="help", default=argparse.SUPPRESS, help="Show this help message and exit.")
    parser.add_argument("-h", "--host", action="store", dest="host", default='127.0.0.1', required=False, help="NETCONF Server IP address or hostname.")
    parser.add_argument(      "--olt_port", action="store", dest="olt_port", default=None, required=False, help="OLT Port number. This could be a logical port number or a physical port number representing the switch port (e.g., LLDP switch port ID)")
    parser.add_argument(      "--olt_tag", action="store", dest="olt_tag", default=None, required=True, help="Tag to be added by the OLT")
    parser.add_argument(      "--onu", action="store", dest="onu", default=None, required=True, help="ONU Serial Number (e.g., TBITc84c00df)")
    parser.add_argument("-p", "--port", action="store", dest="port", default='830', required=False, help="NETCONF Server port number.")
    parser.add_argument(      "--uni", action="store", dest="uni_port", default='1', required=False, help="UNI port number 1..5")
    parser.add_argument("-u", "--user", action="store", dest="user", default=None, required=False, help="Username.")
    parser.add_argument("-w", "--passwd", action="store", dest="passwd", default=None, required=False, help="Password. If no password is provided, attempt to read it from .nc_edit_auth.")
    parser.add_argument("-v", "--verbose", action="store_true", dest="verbose", default=False, required=False, help="Verbose output.")
    parser.parse_args()
    args = parser.parse_args()

    nc = NetconfDriver(host=args.host, port=args.port, user=args.user, passwd=args.passwd, verbose=args.verbose)
    if not nc:
        # Error
        print(f"ERROR: Failed to connect to Netconf server {args.host}:{args.port}.")
        sys.exit(1)

    # Build an options dictionary from the command line arguments
    # The variables {{VAR}} be substituted in the Netconf requests from the
    # .xml files sent below
    options = {
        "{{OLT_PORT}}" : args.olt_port,
        "{{OLT_TAG}}" : args.olt_tag,
        "{{ONU}}" : args.onu,
        "{{UNI_PORT}}" : args.uni_port,
    }

    # Delete OLT forwarding and VLAN configuration
    nc.edit_config(
        message = "5-olt-1to1-forwarding",
        data_xml = '''
        <rpc xmlns="urn:ietf:params:xml:ns:netconf:base:1.0" message-id="34566760">
        <edit-config xmlns="urn:ietf:params:xml:ns:netconf:base:1.0">
            <target>
                <running/>
            </target>
            <config>
                <!-- Delete the NNI Network on the OLT (???) -->
                <bbf-l2-fwd:forwarding xmlns:bbf-l2-fwd="urn:bbf:yang:bbf-l2-forwarding">
                    <bbf-l2-fwd:forwarders>
                        <!-- forwarder configuration from User 1 to S-TAG 201 -->
                        <bbf-l2-fwd:forwarder>
                            <bbf-l2-fwd:name>shared_vlan-s{{OLT_TAG}}.cX.cX</bbf-l2-fwd:name>
                            <bbf-l2-fwd:ports>
                                <bbf-l2-fwd:port nc:operation="remove" xmlns:nc="urn:ietf:params:xml:ns:netconf:base:1.0">
                                    <bbf-l2-fwd:name>pon_network-{{ONU}}-eth.{{UNI_PORT}}.any</bbf-l2-fwd:name>
                                </bbf-l2-fwd:port>
                            </bbf-l2-fwd:ports>
                        </bbf-l2-fwd:forwarder>
                    </bbf-l2-fwd:forwarders>
                </bbf-l2-fwd:forwarding>
                <!-- Delete OLT VLAN sub-interfaces -->
                <if:interfaces xmlns:if="urn:ietf:params:xml:ns:yang:ietf-interfaces">
                    <!-- Delete OLT VLAN PON sub-interfaces -->
                    <if:interface nc:operation="remove" xmlns:nc="urn:ietf:params:xml:ns:netconf:base:1.0">
                        <if:name>pon_network-{{ONU}}-eth.{{UNI_PORT}}.any</if:name>
                    </if:interface>
                </if:interfaces>
                <!-- Delete the Subscriber Profile for this ONU -->
                <subscriber-profiles xmlns="urn:bbf:yang:bbf-subscriber-profiles">
                    <subscriber-profile nc:operation="remove" xmlns:nc="urn:ietf:params:xml:ns:netconf:base:1.0">
                        <name>{{ONU}}</name>
                    </subscriber-profile>
                </subscriber-profiles>
            </config>
        </edit-config>
        </rpc>
        ''',
        options=options
    )

    # Delete Tconts, Gemports, and Upstream SLAs
    nc.edit_config(
        message = "4-gemports",
        data_xml = '''
        <rpc xmlns="urn:ietf:params:xml:ns:netconf:base:1.0" message-id="34566760">
        <edit-config xmlns="urn:ietf:params:xml:ns:netconf:base:1.0">
            <target>
                <running />
            </target>
            <config>
                <bbf-xpongemtcont:xpongemtcont xmlns:bbf-xpongemtcont="urn:bbf:yang:bbf-xpongemtcont">
                    <!-- Delete tconts -->
                    <bbf-xpongemtcont:tconts>
                    <bbf-xpongemtcont:tcont nc:operation="remove" xmlns:nc="urn:ietf:params:xml:ns:netconf:base:1.0">
                        <bbf-xpongemtcont:name>vani-{{ONU}}-tcont.1</bbf-xpongemtcont:name>
                    </bbf-xpongemtcont:tcont>
                    </bbf-xpongemtcont:tconts>
                    <!-- Delete gemports -->
                    <bbf-xpongemtcont:gemports>
                    <bbf-xpongemtcont:gemport nc:operation="remove" xmlns:nc="urn:ietf:params:xml:ns:netconf:base:1.0">
                        <bbf-xpongemtcont:name>gem-{{ONU}}-eth.{{UNI_PORT}}-tc0</bbf-xpongemtcont:name>
                    </bbf-xpongemtcont:gemport>
                    </bbf-xpongemtcont:gemports>
                    <!-- Delete the Upstream SLA -->
                    <!-- <bbf-xpongemtcont:traffic-descriptor-profiles> -->
                    <!-- <bbf-xpongemtcont:traffic-descriptor-profile nc:operation="remove" xmlns:nc="urn:ietf:params:xml:ns:netconf:base:1.0"> -->
                    <!--     <bbf-xpongemtcont:name>sla-add-ctag</bbf-xpongemtcont:name> -->
                    <!-- </bbf-xpongemtcont:traffic-descriptor-profile> -->
                    <!-- </bbf-xpongemtcont:traffic-descriptor-profiles> -->
                </bbf-xpongemtcont:xpongemtcont>
            </config>
        </edit-config>
        </rpc>
        ''',
        options=options
    )

    # Delete ONU sub-interfaces
    nc.edit_config(
        message = "3-onu-vlan-sub-interfaces",
        data_xml = '''
        <rpc xmlns="urn:ietf:params:xml:ns:netconf:base:1.0" message-id="34566757">
        <edit-config xmlns="urn:ietf:params:xml:ns:netconf:base:1.0">
            <target>
                <running/>
            </target>
            <config>
                <interfaces xmlns="urn:ietf:params:xml:ns:yang:ietf-interfaces">
                    <!-- Delete the ONU vlan sub-interface -->
                    <interface nc:operation="remove" xmlns:nc="urn:ietf:params:xml:ns:netconf:base:1.0">
                    <name>onu-{{ONU}}-eth.{{UNI_PORT}}.any</name>
                    </interface>
                </interfaces>
            </config>
        </edit-config>
        </rpc>
        ''',
        options=options
    )

    # Delete Link Table entries
    nc.edit_config(
        message = "2-link-table",
        data_xml = '''
        <rpc xmlns="urn:ietf:params:xml:ns:netconf:base:1.0" message-id="34566754">
        <edit-config xmlns="urn:ietf:params:xml:ns:netconf:base:1.0">
            <target>
                <running/>
            </target>
            <config>
                <bbf-lt:link-table xmlns:bbf-lt="urn:bbf:yang:bbf-link-table">
                    <!-- Delete v-ani to ani mapping -->
                    <bbf-lt:link-table nc:operation="remove" xmlns:nc="urn:ietf:params:xml:ns:netconf:base:1.0">
                    <bbf-lt:from-interface>ani-{{ONU}}</bbf-lt:from-interface>
                    </bbf-lt:link-table>
                    <!-- Delete v-enet to uni mapping -->
                    <bbf-lt:link-table nc:operation="remove" xmlns:nc="urn:ietf:params:xml:ns:netconf:base:1.0">
                    <bbf-lt:from-interface>onu-{{ONU}}-eth.{{UNI_PORT}}</bbf-lt:from-interface>
                    </bbf-lt:link-table>
                </bbf-lt:link-table>
            </config>
        </edit-config>
        </rpc>
        ''',
        options=options
    )

    # Delete ONU Interfaces
    nc.edit_config(
        message = "1-onu-interfaces",
        data_xml = '''
        <rpc xmlns="urn:ietf:params:xml:ns:netconf:base:1.0" message-id="34566754">
        <edit-config xmlns="urn:ietf:params:xml:ns:netconf:base:1.0">
            <target>
                <running/>
            </target>
            <config>
                <if:interfaces xmlns:if="urn:ietf:params:xml:ns:yang:ietf-interfaces">
                    <!-- Delete the v-ani interface -->
                    <if:interface nc:operation="remove" xmlns:nc="urn:ietf:params:xml:ns:netconf:base:1.0">
                    <if:name>vani-{{ONU}}</if:name>
                    </if:interface>
                    <!-- Delete the ani interface -->
                    <if:interface nc:operation="remove" xmlns:nc="urn:ietf:params:xml:ns:netconf:base:1.0">
                    <if:name>ani-{{ONU}}</if:name>
                    </if:interface>
                    <!-- Delete the v-enet interface -->
                    <if:interface nc:operation="remove" xmlns:nc="urn:ietf:params:xml:ns:netconf:base:1.0">
                    <if:name>olt-v-enet-{{ONU}}.{{UNI_PORT}}</if:name>
                    </if:interface>
                    <!-- Delete the uni/veip interface -->
                    <if:interface nc:operation="remove" xmlns:nc="urn:ietf:params:xml:ns:netconf:base:1.0">
                    <if:name>onu-{{ONU}}-eth.{{UNI_PORT}}</if:name>
                    </if:interface>
                </if:interfaces>
            </config>
        </edit-config>
        </rpc>
        ''',
        options=options
    )
